﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class StreamString : Stream, IDeepCopyable<StreamString>, IDeepCopyFrom<StreamString>
    {
        private string @value = string.Empty;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public StreamString()
        {
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public StreamString(StreamString source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public StreamString(streamType source)
            : base(source)
        {
            Ensure.Argument.AreEqual(stream_typeType.@string, source.type);
            this.@value = source.@Value;
        }

        public StreamString(string source)
            : this()
        {
            this.Value = source;
        }

        public override stream_typeType Type
        {
            get
            {
                return stream_typeType.@string;
            }
        }

        public override int Count
        {
            get
            {
                return 1;
            }
        }

        public string Value
        {
            get
            {
                return this.@value;
            }

            set
            {
                if (this.@value == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@value, value, () => this.CalcCRC());
                this.RaisePropertyChanged("Count");
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override streamType CreateSerializableData()
        {
            var writeData = base.CreateSerializableData();
            writeData.@type = this.Type;
            writeData.@count = this.Count;
            writeData.@Value = this.Value;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        StreamString IDeepCopyable<StreamString>.DeepCopy()
        {
            return new StreamString(this);
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        private protected override Stream DeepCopy()
        {
            return new StreamString(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(StreamString source)
        {
            base.DeepCopyFrom(source);
            this.@value = source.@value;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(base.CreateCRCInternal()));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Value)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }
    }
}
