﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class ShapeAnimInfo : ObservableEntity<shape_anim_infoType>, IDeepCopyable<ShapeAnimInfo>, IDeepCopyFrom<ShapeAnimInfo>
    {
        private int @frameCount;
        private bool @loop;
        private int @frameResolution = 1;
        private string @dccPreset = string.Empty;
        private int @dccStartFrame;
        private int @dccEndFrame;
        private float @dccFps = 1;
        private bool @bakeAll;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ShapeAnimInfo()
        {
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public ShapeAnimInfo(ShapeAnimInfo source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public ShapeAnimInfo(shape_anim_infoType source)
        {
            this.@frameCount = source.@frame_count;
            this.@loop = source.@loop;
            this.@frameResolution = source.@frame_resolution;
            this.@dccPreset = source.@dcc_preset;
            this.@dccStartFrame = source.@dcc_start_frame;
            this.@dccEndFrame = source.@dcc_end_frame;
            this.@dccFps = source.@dcc_fps;
            this.@bakeAll = source.@bake_all;
        }

        public int FrameCount
        {
            get
            {
                return this.@frameCount;
            }

            set
            {
                if (this.@frameCount == value)
                {
                    return;
                }

                this.SetProperty(ref this.@frameCount, value, () => this.CalcCRC());
            }
        }

        public bool Loop
        {
            get
            {
                return this.@loop;
            }

            set
            {
                if (this.@loop == value)
                {
                    return;
                }

                this.SetProperty(ref this.@loop, value, () => this.CalcCRC());
            }
        }

        public int FrameResolution
        {
            get
            {
                return this.@frameResolution;
            }

            set
            {
                if (this.@frameResolution == value)
                {
                    return;
                }

                this.SetProperty(ref this.@frameResolution, value, () => this.CalcCRC());
            }
        }

        public string DccPreset
        {
            get
            {
                return this.@dccPreset;
            }

            set
            {
                if (this.@dccPreset == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@dccPreset, value, () => this.CalcCRC());
            }
        }

        public int DccStartFrame
        {
            get
            {
                return this.@dccStartFrame;
            }

            set
            {
                if (this.@dccStartFrame == value)
                {
                    return;
                }

                this.SetProperty(ref this.@dccStartFrame, value, () => this.CalcCRC());
            }
        }

        public int DccEndFrame
        {
            get
            {
                return this.@dccEndFrame;
            }

            set
            {
                if (this.@dccEndFrame == value)
                {
                    return;
                }

                this.SetProperty(ref this.@dccEndFrame, value, () => this.CalcCRC());
            }
        }

        public float DccFps
        {
            get
            {
                return this.@dccFps;
            }

            set
            {
                if (this.@dccFps == value)
                {
                    return;
                }

                this.SetProperty(ref this.@dccFps, value, () => this.CalcCRC());
            }
        }

        public bool BakeAll
        {
            get
            {
                return this.@bakeAll;
            }

            set
            {
                if (this.@bakeAll == value)
                {
                    return;
                }

                this.SetProperty(ref this.@bakeAll, value, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override shape_anim_infoType CreateSerializableData()
        {
            var writeData = new shape_anim_infoType();
            writeData.@frame_count = this.FrameCount;
            writeData.@loop = this.Loop;
            writeData.@frame_resolution = this.FrameResolution;
            writeData.@dcc_preset = this.DccPreset;
            writeData.@dcc_start_frame = this.DccStartFrame;
            writeData.@dcc_end_frame = this.DccEndFrame;
            writeData.@dcc_fps = this.DccFps;
            writeData.@bake_all = this.BakeAll;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        ShapeAnimInfo IDeepCopyable<ShapeAnimInfo>.DeepCopy()
        {
            return new ShapeAnimInfo(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(ShapeAnimInfo source)
        {
            this.@frameCount = source.@frameCount;
            this.@loop = source.@loop;
            this.@frameResolution = source.@frameResolution;
            this.@dccPreset = source.@dccPreset;
            this.@dccStartFrame = source.@dccStartFrame;
            this.@dccEndFrame = source.@dccEndFrame;
            this.@dccFps = source.@dccFps;
            this.@bakeAll = source.@bakeAll;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.FrameCount)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Loop)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.FrameResolution)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.DccPreset)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.DccStartFrame)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.DccEndFrame)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.DccFps)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.BakeAll)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }
    }
}
