﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class ShaderDefinition : RootEntity<shader_definitionType>, IDeepCopyable<ShaderDefinition>, IDeepCopyFrom<ShaderDefinition>, IToolData
    {
        private readonly ToolData @toolData = new ToolData();
        private readonly UserToolData @userToolData = new UserToolData();
        private readonly ShaderDefinitionInfo @shaderDefinitionInfo = new ShaderDefinitionInfo();
        private readonly ObservableList<ForceInclude> @forceIncludes = new ObservableList<ForceInclude>();
        private readonly ObservableList<ShadingModel> @shadingModels = new ObservableList<ShadingModel>();
        private readonly ObservableList<ShaderSrc> @shaderSrcs = new ObservableList<ShaderSrc>();
        private readonly ObservableList<Stream> @streams = new ObservableList<Stream>();
        private shader_definition_versionType @version;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        internal ShaderDefinition()
        {
            this.@toolData.PropertyChanged += this.OnPropertyChanged;
            this.@userToolData.PropertyChanged += this.OnPropertyChanged;
            this.@shaderDefinitionInfo.PropertyChanged += this.OnPropertyChanged;
            this.@forceIncludes.CollectionChanged += OnCollectionChanged<ForceInclude>;
            this.@shadingModels.CollectionChanged += OnCollectionChanged<ShadingModel>;
            this.@shaderSrcs.CollectionChanged += OnCollectionChanged<ShaderSrc>;
            this.@streams.CollectionChanged += OnCollectionChanged<Stream>;
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public ShaderDefinition(ShaderDefinition source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public ShaderDefinition(shader_definitionType source)
        {
            this.@forceIncludes.CollectionChanged += OnCollectionChanged<ForceInclude>;
            this.@shadingModels.CollectionChanged += OnCollectionChanged<ShadingModel>;
            this.@shaderSrcs.CollectionChanged += OnCollectionChanged<ShaderSrc>;
            this.@streams.CollectionChanged += OnCollectionChanged<Stream>;
            if (source.@tool_data != null)
            {
                this.@toolData = new ToolData(source.@tool_data);
            }
            else
            {
                this.@toolData = new ToolData();
            }
            if (source.@user_tool_data != null)
            {
                this.@userToolData = new UserToolData(source.@user_tool_data);
            }
            else
            {
                this.@userToolData = new UserToolData();
            }
            if (source.@shader_definition_info != null)
            {
                this.@shaderDefinitionInfo = new ShaderDefinitionInfo(source.@shader_definition_info);
            }
            else
            {
                this.@shaderDefinitionInfo = new ShaderDefinitionInfo();
            }
            if (source.@force_include_array != null)
            {
                this.@forceIncludes.Clear();
                foreach (var elem in source.@force_include_array.Items)
                {
                    this.@forceIncludes.Add(new ForceInclude(elem));
                }
            }
            if (source.@shading_model_array != null)
            {
                this.@shadingModels.Clear();
                foreach (var elem in source.@shading_model_array.Items)
                {
                    this.@shadingModels.Add(new ShadingModel(elem));
                }
            }
            if (source.@shader_src_array != null)
            {
                this.@shaderSrcs.Clear();
                foreach (var elem in source.@shader_src_array.Items)
                {
                    this.@shaderSrcs.Add(new ShaderSrc(elem));
                }
            }
            if (source.@stream_array != null)
            {
                this.@streams.Clear();
                foreach (var elem in source.@stream_array.Items)
                {
                    this.@streams.Add(DataModelConverter.Convert(elem));
                }
            }
            this.@version = source.@version;
            this.@toolData.PropertyChanged += this.OnPropertyChanged;
            this.@userToolData.PropertyChanged += this.OnPropertyChanged;
            this.@shaderDefinitionInfo.PropertyChanged += this.OnPropertyChanged;
            this.ResolveInternalReferences();
        }

        public ToolData ToolData
        {
            get
            {
                return this.@toolData;
            }
        }

        public UserToolData UserToolData
        {
            get
            {
                return this.@userToolData;
            }
        }

        public ShaderDefinitionInfo ShaderDefinitionInfo
        {
            get
            {
                return this.@shaderDefinitionInfo;
            }
        }

        public ObservableList<ForceInclude> ForceIncludes
        {
            get
            {
                return this.@forceIncludes;
            }
        }

        public ObservableList<ShadingModel> ShadingModels
        {
            get
            {
                return this.@shadingModels;
            }
        }

        public ObservableList<ShaderSrc> ShaderSrcs
        {
            get
            {
                return this.@shaderSrcs;
            }
        }

        public override ObservableList<Stream> Streams
        {
            get
            {
                return this.@streams;
            }
        }

        public shader_definition_versionType Version
        {
            get
            {
                return this.@version;
            }

            set
            {
                if (this.@version == value)
                {
                    return;
                }

                this.SetProperty(ref this.@version, value, () => this.CalcCRC());
            }
        }

        public override void ResolveInternalReferences()
        {
            // インデックスで紐づけをしているデータに実データの割り当て
            foreach (var shadingModel in this.ShadingModels)
            {
                if (shadingModel.VertexStage != null)
                {
                    shadingModel.VertexStage.ShaderSrc = this.ShaderSrcs[shadingModel.VertexStage.SrcIndex];
                }
                if (shadingModel.GeometryStage != null)
                {
                    shadingModel.GeometryStage.ShaderSrc = this.ShaderSrcs[shadingModel.GeometryStage.SrcIndex];
                }
                if (shadingModel.FragmentStage != null)
                {
                    shadingModel.FragmentStage.ShaderSrc = this.ShaderSrcs[shadingModel.FragmentStage.SrcIndex];
                }
                if (shadingModel.ComputeStage != null)
                {
                    shadingModel.ComputeStage.ShaderSrc = this.ShaderSrcs[shadingModel.ComputeStage.SrcIndex];
                }
            }

            if (this.Streams.Count > 0)
            {
                foreach (var shaderSource in this.ShaderSrcs)
                {
                    shaderSource.Stream = this.Streams[(shaderSource as IStreamIndex).StreamIndex] as StreamWstring;
                }
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override shader_definitionType CreateSerializableData()
        {
            Ensure.Operation.Range(this.ShadingModels.Count, 1, int.MaxValue);
            Ensure.Operation.Range(this.ShaderSrcs.Count, 1, int.MaxValue);
            var writeData = new shader_definitionType();
            writeData.@tool_data = this.@toolData.CreateSerializableData() as tool_dataType;
            writeData.@user_tool_data = this.@userToolData.CreateSerializableData() as user_tool_dataType;
            writeData.@shader_definition_info = this.@shaderDefinitionInfo.CreateSerializableData() as shader_definition_infoType;
            if (this.@ForceIncludes.Count > 0)
            {
                writeData.@force_include_array = new force_include_arrayType();
                writeData.@force_include_array.Items = this.@forceIncludes.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@force_include_array.Items.Length == 0)
                {
                    writeData.@force_include_array = null;
                }
            }
            if (this.@ShadingModels.Count > 0)
            {
                writeData.@shading_model_array = new shading_model_arrayType();
                writeData.@shading_model_array.Items = this.@shadingModels.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@shading_model_array.Items.Length == 0)
                {
                    writeData.@shading_model_array = null;
                }
            }
            if (this.@ShaderSrcs.Count > 0)
            {
                writeData.@shader_src_array = new shader_src_arrayType();
                writeData.@shader_src_array.Items = this.@shaderSrcs.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@shader_src_array.Items.Length == 0)
                {
                    writeData.@shader_src_array = null;
                }
            }
            if (this.IsStreamSerializationEnabled)
            {
                if (this.@Streams.Count > 0)
                {
                    writeData.@stream_array = new stream_arrayType();
                    writeData.@stream_array.Items = this.@streams.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                    if (writeData.@stream_array.Items.Length == 0)
                    {
                        writeData.@stream_array = null;
                    }
                }
            }
            writeData.@version = this.Version;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        ShaderDefinition IDeepCopyable<ShaderDefinition>.DeepCopy()
        {
            return new ShaderDefinition(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(ShaderDefinition source)
        {
            this.@toolData.DeepCopyFrom(source.@toolData);
            this.@userToolData.DeepCopyFrom(source.@userToolData);
            this.@shaderDefinitionInfo.DeepCopyFrom(source.@shaderDefinitionInfo);
            CopyElements(source.@forceIncludes, this.@forceIncludes);
            CopyElements(source.@shadingModels, this.@shadingModels);
            CopyElements(source.@shaderSrcs, this.@shaderSrcs);
            CopyElements(source.@streams, this.@streams);
            this.@version = source.@version;
            this.ResolveInternalReferences();
        }

        private protected override IRootEntity DeepCopy()
        {
            return new ShaderDefinition(this);
        }

        private protected override void DeepCopyFrom(IRootEntity source)
        {
            Nintendo.ToolFoundation.Contracts.Ensure.Argument.True(source is ShaderDefinition);
            this.DeepCopyFrom(source as ShaderDefinition);
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.ToolData.HashValue)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.UserToolData.HashValue)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.ShaderDefinitionInfo.HashValue)));
            foreach (var elem in this.ForceIncludes)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            foreach (var elem in this.ShadingModels)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            foreach (var elem in this.ShaderSrcs)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            foreach (var elem in this.Streams)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Version)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            this.@toolData.AutoCalc = this.AutoCalc;
            this.@userToolData.AutoCalc = this.AutoCalc;
            this.@shaderDefinitionInfo.AutoCalc = this.AutoCalc;
            foreach (var elem in this.@forceIncludes)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            foreach (var elem in this.@shadingModels)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            foreach (var elem in this.@shaderSrcs)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            foreach (var elem in this.@streams)
            {
                elem.AutoCalc = this.AutoCalc;
            }
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            this.@toolData.Reset();
            this.@userToolData.Reset();
            this.@shaderDefinitionInfo.Reset();
            foreach (var elem in this.@forceIncludes)
            {
                elem.Reset();
            }
            foreach (var elem in this.@shadingModels)
            {
                elem.Reset();
            }
            foreach (var elem in this.@shaderSrcs)
            {
                elem.Reset();
            }
            foreach (var elem in this.@streams)
            {
                elem.Reset();
            }
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            this.@toolData.Refresh();
            this.@userToolData.Refresh();
            this.@shaderDefinitionInfo.Refresh();
            foreach (var elem in this.@forceIncludes)
            {
                elem.Refresh();
            }
            foreach (var elem in this.@shadingModels)
            {
                elem.Refresh();
            }
            foreach (var elem in this.@shaderSrcs)
            {
                elem.Refresh();
            }
            foreach (var elem in this.@streams)
            {
                elem.Refresh();
            }
        }

        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "HashValue")
            {
                this.CalcCRC();
            }
        }
    }
}
