﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class SceneAnim : RootEntity<scene_animType>, IDeepCopyable<SceneAnim>, IDeepCopyFrom<SceneAnim>, IToolData
    {
        private readonly ObservableList<ProcessLog> @processLogs = new ObservableList<ProcessLog>();
        private readonly SceneAnimInfo @sceneAnimInfo = new SceneAnimInfo();
        private readonly ObservableList<CameraAnim> @cameraAnims = new ObservableList<CameraAnim>();
        private readonly ObservableList<LightAnim> @lightAnims = new ObservableList<LightAnim>();
        private readonly ObservableList<FogAnim> @fogAnims = new ObservableList<FogAnim>();
        private readonly ObservableList<Stream> @streams = new ObservableList<Stream>();
        private readonly ToolData @toolData = new ToolData();
        private readonly UserToolData @userToolData = new UserToolData();
        private scene_anim_versionType @version;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        internal SceneAnim()
        {
            this.@processLogs.CollectionChanged += OnCollectionChanged<ProcessLog>;
            this.@sceneAnimInfo.PropertyChanged += this.OnPropertyChanged;
            this.@cameraAnims.CollectionChanged += OnCollectionChanged<CameraAnim>;
            this.@lightAnims.CollectionChanged += OnCollectionChanged<LightAnim>;
            this.@fogAnims.CollectionChanged += OnCollectionChanged<FogAnim>;
            this.@streams.CollectionChanged += OnCollectionChanged<Stream>;
            this.@toolData.PropertyChanged += this.OnPropertyChanged;
            this.@userToolData.PropertyChanged += this.OnPropertyChanged;
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public SceneAnim(SceneAnim source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public SceneAnim(scene_animType source)
        {
            this.@processLogs.CollectionChanged += OnCollectionChanged<ProcessLog>;
            this.@cameraAnims.CollectionChanged += OnCollectionChanged<CameraAnim>;
            this.@lightAnims.CollectionChanged += OnCollectionChanged<LightAnim>;
            this.@fogAnims.CollectionChanged += OnCollectionChanged<FogAnim>;
            this.@streams.CollectionChanged += OnCollectionChanged<Stream>;
            if (source.@process_log_array != null)
            {
                this.@processLogs.Clear();
                foreach (var elem in source.@process_log_array.Items)
                {
                    this.@processLogs.Add(new ProcessLog(elem));
                }
            }
            if (source.@scene_anim_info != null)
            {
                this.@sceneAnimInfo = new SceneAnimInfo(source.@scene_anim_info);
            }
            else
            {
                this.@sceneAnimInfo = new SceneAnimInfo();
            }
            if (source.@camera_anim_array != null)
            {
                this.@cameraAnims.Clear();
                foreach (var elem in source.@camera_anim_array.Items)
                {
                    this.@cameraAnims.Add(new CameraAnim(elem));
                }
            }
            if (source.@light_anim_array != null)
            {
                this.@lightAnims.Clear();
                foreach (var elem in source.@light_anim_array.Items)
                {
                    this.@lightAnims.Add(new LightAnim(elem));
                }
            }
            if (source.@fog_anim_array != null)
            {
                this.@fogAnims.Clear();
                foreach (var elem in source.@fog_anim_array.Items)
                {
                    this.@fogAnims.Add(new FogAnim(elem));
                }
            }
            if (source.@stream_array != null)
            {
                this.@streams.Clear();
                foreach (var elem in source.@stream_array.Items)
                {
                    this.@streams.Add(DataModelConverter.Convert(elem));
                }
            }
            if (source.@tool_data != null)
            {
                this.@toolData = new ToolData(source.@tool_data);
            }
            else
            {
                this.@toolData = new ToolData();
            }
            if (source.@user_tool_data != null)
            {
                this.@userToolData = new UserToolData(source.@user_tool_data);
            }
            else
            {
                this.@userToolData = new UserToolData();
            }
            this.@version = source.@version;
            this.@sceneAnimInfo.PropertyChanged += this.OnPropertyChanged;
            this.@toolData.PropertyChanged += this.OnPropertyChanged;
            this.@userToolData.PropertyChanged += this.OnPropertyChanged;
            this.ResolveInternalReferences();
        }

        public ObservableList<ProcessLog> ProcessLogs
        {
            get
            {
                return this.@processLogs;
            }
        }

        public SceneAnimInfo SceneAnimInfo
        {
            get
            {
                return this.@sceneAnimInfo;
            }
        }

        public ObservableList<CameraAnim> CameraAnims
        {
            get
            {
                return this.@cameraAnims;
            }
        }

        public ObservableList<LightAnim> LightAnims
        {
            get
            {
                return this.@lightAnims;
            }
        }

        public ObservableList<FogAnim> FogAnims
        {
            get
            {
                return this.@fogAnims;
            }
        }

        public override ObservableList<Stream> Streams
        {
            get
            {
                return this.@streams;
            }
        }

        public ToolData ToolData
        {
            get
            {
                return this.@toolData;
            }
        }

        public UserToolData UserToolData
        {
            get
            {
                return this.@userToolData;
            }
        }

        public scene_anim_versionType Version
        {
            get
            {
                return this.@version;
            }

            set
            {
                if (this.@version == value)
                {
                    return;
                }

                this.SetProperty(ref this.@version, value, () => this.CalcCRC());
            }
        }

        public override void ResolveInternalReferences()
        {
            // インデックスで紐づけをしているデータに実データの割り当て
            if (this.Streams.Count > 0)
            {
                foreach (var anim in this.CameraAnims)
                {
                    foreach (var target in anim.CameraAnimTargets.Values)
                    {
                        if (target.Curve != null)
                        {
                            target.Curve.Stream = this.Streams[(target.Curve as IStreamIndex).StreamIndex];
                        }
                    }
                }
                foreach (var anim in this.LightAnims)
                {
                    foreach (var target in anim.LightAnimTargets.Values)
                    {
                        if (target.Curve != null)
                        {
                            target.Curve.Stream = this.Streams[(target.Curve as IStreamIndex).StreamIndex];
                        }
                    }
                }
                foreach (var anim in this.FogAnims)
                {
                    foreach (var target in anim.FogAnimTargets.Values)
                    {
                        if (target.Curve != null)
                        {
                            target.Curve.Stream = this.Streams[(target.Curve as IStreamIndex).StreamIndex];
                        }
                    }
                }
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override scene_animType CreateSerializableData()
        {
            var writeData = new scene_animType();
            if (this.@ProcessLogs.Count > 0)
            {
                writeData.@process_log_array = new process_log_arrayType();
                writeData.@process_log_array.Items = this.@processLogs.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@process_log_array.Items.Length == 0)
                {
                    writeData.@process_log_array = null;
                }
            }
            writeData.@scene_anim_info = this.@sceneAnimInfo.CreateSerializableData() as scene_anim_infoType;
            if (this.@CameraAnims.Count > 0)
            {
                writeData.@camera_anim_array = new camera_anim_arrayType();
                writeData.@camera_anim_array.Items = this.@cameraAnims.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@camera_anim_array.Items.Length == 0)
                {
                    writeData.@camera_anim_array = null;
                }
            }
            if (this.@LightAnims.Count > 0)
            {
                writeData.@light_anim_array = new light_anim_arrayType();
                writeData.@light_anim_array.Items = this.@lightAnims.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@light_anim_array.Items.Length == 0)
                {
                    writeData.@light_anim_array = null;
                }
            }
            if (this.@FogAnims.Count > 0)
            {
                writeData.@fog_anim_array = new fog_anim_arrayType();
                writeData.@fog_anim_array.Items = this.@fogAnims.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@fog_anim_array.Items.Length == 0)
                {
                    writeData.@fog_anim_array = null;
                }
            }
            if (this.IsStreamSerializationEnabled)
            {
                if (this.@Streams.Count > 0)
                {
                    writeData.@stream_array = new stream_arrayType();
                    writeData.@stream_array.Items = this.@streams.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                    if (writeData.@stream_array.Items.Length == 0)
                    {
                        writeData.@stream_array = null;
                    }
                }
            }
            writeData.@tool_data = this.@toolData.CreateSerializableData() as tool_dataType;
            writeData.@user_tool_data = this.@userToolData.CreateSerializableData() as user_tool_dataType;
            writeData.@version = this.Version;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        SceneAnim IDeepCopyable<SceneAnim>.DeepCopy()
        {
            return new SceneAnim(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(SceneAnim source)
        {
            CopyElements(source.@processLogs, this.@processLogs);
            this.@sceneAnimInfo.DeepCopyFrom(source.@sceneAnimInfo);
            CopyElements(source.@cameraAnims, this.@cameraAnims);
            CopyElements(source.@lightAnims, this.@lightAnims);
            CopyElements(source.@fogAnims, this.@fogAnims);
            CopyElements(source.@streams, this.@streams);
            this.@toolData.DeepCopyFrom(source.@toolData);
            this.@userToolData.DeepCopyFrom(source.@userToolData);
            this.@version = source.@version;
            this.ResolveInternalReferences();
        }

        private protected override IRootEntity DeepCopy()
        {
            return new SceneAnim(this);
        }

        private protected override void DeepCopyFrom(IRootEntity source)
        {
            Nintendo.ToolFoundation.Contracts.Ensure.Argument.True(source is SceneAnim);
            this.DeepCopyFrom(source as SceneAnim);
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            foreach (var elem in this.ProcessLogs)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.SceneAnimInfo.HashValue)));
            foreach (var elem in this.CameraAnims)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            foreach (var elem in this.LightAnims)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            foreach (var elem in this.FogAnims)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            foreach (var elem in this.Streams)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.ToolData.HashValue)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.UserToolData.HashValue)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Version)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            foreach (var elem in this.@processLogs)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            this.@sceneAnimInfo.AutoCalc = this.AutoCalc;
            foreach (var elem in this.@cameraAnims)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            foreach (var elem in this.@lightAnims)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            foreach (var elem in this.@fogAnims)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            foreach (var elem in this.@streams)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            this.@toolData.AutoCalc = this.AutoCalc;
            this.@userToolData.AutoCalc = this.AutoCalc;
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            foreach (var elem in this.@processLogs)
            {
                elem.Reset();
            }
            this.@sceneAnimInfo.Reset();
            foreach (var elem in this.@cameraAnims)
            {
                elem.Reset();
            }
            foreach (var elem in this.@lightAnims)
            {
                elem.Reset();
            }
            foreach (var elem in this.@fogAnims)
            {
                elem.Reset();
            }
            foreach (var elem in this.@streams)
            {
                elem.Reset();
            }
            this.@toolData.Reset();
            this.@userToolData.Reset();
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            foreach (var elem in this.@processLogs)
            {
                elem.Refresh();
            }
            this.@sceneAnimInfo.Refresh();
            foreach (var elem in this.@cameraAnims)
            {
                elem.Refresh();
            }
            foreach (var elem in this.@lightAnims)
            {
                elem.Refresh();
            }
            foreach (var elem in this.@fogAnims)
            {
                elem.Refresh();
            }
            foreach (var elem in this.@streams)
            {
                elem.Refresh();
            }
            this.@toolData.Refresh();
            this.@userToolData.Refresh();
        }

        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "HashValue")
            {
                this.CalcCRC();
            }
        }
    }
}
