﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class RenderInfoFloat : RenderInfo, IDeepCopyable<RenderInfoFloat>, IDeepCopyFrom<RenderInfoFloat>
    {
        private readonly ObservableList<float> @values = new ObservableList<float>();

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public RenderInfoFloat()
        {
            this.@values.CollectionChanged += OnValueCollectionChanged;
            this.@values.CollectionChanged += OnValuesCollectionChanged;
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public RenderInfoFloat(RenderInfoFloat source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public RenderInfoFloat(render_infoType source)
            : base(source)
        {
            this.@values.CollectionChanged += OnValueCollectionChanged;
            this.@values.CollectionChanged += OnValuesCollectionChanged;
            Ensure.Argument.AreEqual(render_info_typeType.@float, source.type);
            if (source.count > 0)
            {
                this.Values.Add(StringUtility.ConvertTextToValueArray<float>(source.Value));
            }
        }

        public RenderInfoFloat(RenderInfoSlot definition)
            : this()
        {
            this.Name = definition.Name;
            this.SetToDefault(definition);
        }

        public override render_info_typeType Type
        {
            get
            {
                return render_info_typeType.@float;
            }
        }

        public override int Count
        {
            get
            {
                return this.values.Count;
            }
        }

        public ObservableList<float> Values
        {
            get
            {
                return this.@values;
            }
        }

        private protected override void SetToDefaultInternal(RenderInfoSlot definition)
        {
            Ensure.Argument.True(definition is RenderInfoSlotFloat);
            this.Values.Clear();
            this.Values.Add((definition as RenderInfoSlotFloat).Default);
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override render_infoType CreateSerializableData()
        {
            var writeData = base.CreateSerializableData();
            writeData.@type = this.Type;
            writeData.@count = this.Count;
            if (this.Count > 0)
            {
                StringBuilder valuesAsText = new StringBuilder();
                foreach (var value in this.Values)
                {
                    valuesAsText.Append($"{value} ");
                }
                writeData.Value = valuesAsText.ToString().Trim();
            }
            else
            {
                writeData.Value = null;
            }
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        RenderInfoFloat IDeepCopyable<RenderInfoFloat>.DeepCopy()
        {
            return new RenderInfoFloat(this);
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        private protected override RenderInfo DeepCopy()
        {
            return new RenderInfoFloat(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(RenderInfoFloat source)
        {
            base.DeepCopyFrom(source);
            CopyValues(source.@values, this.@values);
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(base.CreateCRCInternal()));
            foreach (var elem in this.Values)
            {
                buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(elem)));
            }
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        private void OnValuesCollectionChanged(object sender, System.Collections.Specialized.NotifyCollectionChangedEventArgs e)
        {
            this.RaisePropertyChanged("Count");
        }
    }
}
