﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class ParamAnim : ObservableEntity<param_animType>, IChildEntity, IDeepCopyable<ParamAnim>, IDeepCopyFrom<ParamAnim>
    {
        private readonly ObservableList<ParamAnimTarget> @paramAnimTargets = new ObservableList<ParamAnimTarget>();
        private string @id = string.Empty;
        private string @originalHint = string.Empty;
        private shader_param_typeType @type;
        private PerMaterialAnim parent = null;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ParamAnim()
        {
            this.@paramAnimTargets.CollectionChanged += OnCollectionChanged<ParamAnimTarget>;
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public ParamAnim(ParamAnim source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public ParamAnim(param_animType source)
        {
            this.@paramAnimTargets.CollectionChanged += OnCollectionChanged<ParamAnimTarget>;
            if (source.@param_anim_target != null)
            {
                this.@paramAnimTargets.Clear();
                foreach (var elem in source.@param_anim_target)
                {
                    this.@paramAnimTargets.Add(new ParamAnimTarget(elem));
                }
            }
            this.@id = source.@id;
            this.@originalHint = source.@original_hint;
            this.@type = source.@type;
        }

        public ObservableList<ParamAnimTarget> ParamAnimTargets
        {
            get
            {
                return this.@paramAnimTargets;
            }
        }

        public string Id
        {
            get
            {
                return this.@id;
            }

            set
            {
                if (this.@id == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@id, value, () => this.CalcCRC());
            }
        }

        public string OriginalHint
        {
            get
            {
                return this.@originalHint;
            }

            set
            {
                if (this.@originalHint == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@originalHint, value, () => this.CalcCRC());
            }
        }

        public shader_param_typeType Type
        {
            get
            {
                return this.@type;
            }

            set
            {
                if (this.@type == value)
                {
                    return;
                }

                this.SetProperty(ref this.@type, value, () => this.CalcCRC());
            }
        }

        IEntity IChildEntity.Parent
        {
            get
            {
                return this.parent as IEntity;
            }

            set
            {
                if (this.parent == value)
                {
                    return;
                }

                Ensure.Argument.True(value != null ? value is PerMaterialAnim : true);
                this.SetProperty(ref this.parent, value as PerMaterialAnim, () => this.CalcCRC());
            }
        }

        public PerMaterialAnim Parent
        {
            get
            {
                return this.parent;
            }
        }

        Type IChildEntity.ParentType
        {
            get
            {
                return typeof(PerMaterialAnim);
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override param_animType CreateSerializableData()
        {
            var writeData = new param_animType();
            if (this.@ParamAnimTargets.Count > 0)
            {
                writeData.@param_anim_target = this.@paramAnimTargets.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@param_anim_target.Length == 0)
                {
                    writeData.@param_anim_target = null;
                }
            }
            writeData.@id = this.Id;
            writeData.@original_hint = this.OriginalHint;
            writeData.@type = this.Type;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        ParamAnim IDeepCopyable<ParamAnim>.DeepCopy()
        {
            return new ParamAnim(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(ParamAnim source)
        {
            CopyElements(source.@paramAnimTargets, this.@paramAnimTargets);
            this.@id = source.@id;
            this.@originalHint = source.@originalHint;
            this.@type = source.@type;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            foreach (var elem in this.ParamAnimTargets)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Id)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.OriginalHint)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Type)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            foreach (var elem in this.@paramAnimTargets)
            {
                elem.AutoCalc = this.AutoCalc;
            }
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            foreach (var elem in this.@paramAnimTargets)
            {
                elem.Reset();
            }
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            foreach (var elem in this.@paramAnimTargets)
            {
                elem.Refresh();
            }
        }

        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "HashValue")
            {
                this.CalcCRC();
            }
        }
    }
}
