﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class OriginalTexsrt : ObservableEntity<original_texsrtType>, IDeepCopyable<OriginalTexsrt>, IDeepCopyFrom<OriginalTexsrt>
    {
        private string @hint = string.Empty;
        private string @uvHint = string.Empty;
        private original_texsrt_modeType @mode;
        private readonly Float2 @scale = new Float2(1, 1);
        private float @rotate;
        private readonly Float2 @translate = new Float2(0, 0);

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public OriginalTexsrt()
        {
            this.@scale.PropertyChanged += this.OnPropertyChanged;
            this.@translate.PropertyChanged += this.OnPropertyChanged;
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public OriginalTexsrt(OriginalTexsrt source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public OriginalTexsrt(original_texsrtType source)
        {
            this.@hint = source.@hint;
            this.@uvHint = source.@uv_hint;
            this.@mode = source.@mode;
            Utility.CopyArray(source.scale, this.@scale.Values);
            this.@rotate = source.@rotate;
            Utility.CopyArray(source.translate, this.@translate.Values);
            this.@scale.PropertyChanged += this.OnPropertyChanged;
            this.@translate.PropertyChanged += this.OnPropertyChanged;
        }

        public string Hint
        {
            get
            {
                return this.@hint;
            }

            set
            {
                if (this.@hint == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@hint, value, () => this.CalcCRC());
            }
        }

        public string UvHint
        {
            get
            {
                return this.@uvHint;
            }

            set
            {
                if (this.@uvHint == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@uvHint, value, () => this.CalcCRC());
            }
        }

        public original_texsrt_modeType Mode
        {
            get
            {
                return this.@mode;
            }

            set
            {
                if (this.@mode == value)
                {
                    return;
                }

                this.SetProperty(ref this.@mode, value, () => this.CalcCRC());
            }
        }

        public Float2 Scale
        {
            get
            {
                return this.@scale;
            }
        }

        public float Rotate
        {
            get
            {
                return this.@rotate;
            }

            set
            {
                if (this.@rotate == value)
                {
                    return;
                }

                this.SetProperty(ref this.@rotate, value, () => this.CalcCRC());
            }
        }

        public Float2 Translate
        {
            get
            {
                return this.@translate;
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override original_texsrtType CreateSerializableData()
        {
            var writeData = new original_texsrtType();
            writeData.@hint = this.Hint;
            writeData.@uv_hint = this.UvHint;
            writeData.@mode = this.Mode;
            writeData.scale = new float[2];
            Utility.CopyArray(this.Scale.Values, writeData.scale);
            writeData.@rotate = this.Rotate;
            writeData.translate = new float[2];
            Utility.CopyArray(this.Translate.Values, writeData.translate);
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        OriginalTexsrt IDeepCopyable<OriginalTexsrt>.DeepCopy()
        {
            return new OriginalTexsrt(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(OriginalTexsrt source)
        {
            this.@hint = source.@hint;
            this.@uvHint = source.@uvHint;
            this.@mode = source.@mode;
            this.@scale.DeepCopyFrom(source.@scale);
            this.@rotate = source.@rotate;
            this.@translate.DeepCopyFrom(source.@translate);
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Hint)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.UvHint)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Mode)));
            buffers.AddRange(BitConverter.GetBytes(this.Scale.HashValue));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Rotate)));
            buffers.AddRange(BitConverter.GetBytes(this.Translate.HashValue));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            this.@scale.AutoCalc = this.AutoCalc;
            this.@translate.AutoCalc = this.AutoCalc;
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            this.@scale.Reset();
            this.@translate.Reset();
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            this.@scale.Refresh();
            this.@translate.Refresh();
        }

        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "HashValue")
            {
                this.CalcCRC();
            }
        }
    }
}
