﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class MaterialInfo : ObservableEntity<material_infoType>, IDeepCopyable<MaterialInfo>, IDeepCopyFrom<MaterialInfo>
    {
        private bool @visibility = true;
        private bool @meshAdjacency = false;
        private bool @compressEnable = true;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public MaterialInfo()
        {
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public MaterialInfo(MaterialInfo source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public MaterialInfo(material_infoType source)
        {
            this.@visibility = source.@visibility;
            this.@meshAdjacency = source.@mesh_adjacency;
            this.@compressEnable = source.@compress_enable;
        }

        public bool Visibility
        {
            get
            {
                return this.@visibility;
            }

            set
            {
                if (this.@visibility == value)
                {
                    return;
                }

                this.SetProperty(ref this.@visibility, value, () => this.CalcCRC());
            }
        }

        public bool MeshAdjacency
        {
            get
            {
                return this.@meshAdjacency;
            }

            set
            {
                if (this.@meshAdjacency == value)
                {
                    return;
                }

                this.SetProperty(ref this.@meshAdjacency, value, () => this.CalcCRC());
            }
        }

        public bool CompressEnable
        {
            get
            {
                return this.@compressEnable;
            }

            set
            {
                if (this.@compressEnable == value)
                {
                    return;
                }

                this.SetProperty(ref this.@compressEnable, value, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override material_infoType CreateSerializableData()
        {
            var writeData = new material_infoType();
            writeData.@visibility = this.Visibility;
            writeData.@mesh_adjacency = this.MeshAdjacency;
            writeData.@compress_enable = this.CompressEnable;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        MaterialInfo IDeepCopyable<MaterialInfo>.DeepCopy()
        {
            return new MaterialInfo(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(MaterialInfo source)
        {
            this.@visibility = source.@visibility;
            this.@meshAdjacency = source.@meshAdjacency;
            this.@compressEnable = source.@compressEnable;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Visibility)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.MeshAdjacency)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.CompressEnable)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }
    }
}
