﻿using Nintendo.G3dTool.Entities;
using Nintendo.ToolFoundation.Contracts;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace Nintendo.G3dTool.Entities.Internal
{
    internal static class IMatrixExtensions
    {
        /// <summary>
        /// X軸、Y軸、Z軸の順に回転させた回転行列に設定します。
        /// 回転の値をradianとして行列を求めます。
        /// </summary>
        /// <param name="target">設定する行列です。</param>
        /// <param name="xRadian">x 軸を中心とした回転角度 (Radian単位)です。</param>
        /// <param name="yRadian">y 軸を中心とした回転角度 (Radian単位)です。</param>
        /// <param name="zRadian">z 軸を中心とした回転角度 (Radian単位)です。</param>
        internal static void SetRotationEulerXyz(
            this IMatrix target, float xRadian, float yRadian, float zRadian)
        {
            Ensure.Argument.True(target.RowCount >= 3);
            Ensure.Argument.True(target.ColumnCount >= 3);

            target.SetToIdentity();

            float sinx = (float)Math.Sin(xRadian);
            float cosx = (float)Math.Cos(xRadian);
            float siny = (float)Math.Sin(yRadian);
            float cosy = (float)Math.Cos(yRadian);
            float sinz = (float)Math.Sin(zRadian);
            float cosz = (float)Math.Cos(zRadian);
            float f1, f2;

            target[2, 0] = -siny;
            target[0, 0] = cosz * cosy;
            target[1, 0] = sinz * cosy;
            target[2, 1] = cosy * sinx;
            target[2, 2] = cosy * cosx;

            f1 = cosx * sinz;
            f2 = sinx * cosz;

            target[0, 1] = (f2 * siny) - f1;
            target[1, 2] = (f1 * siny) - f2;

            f1 = sinx * sinz;
            f2 = cosx * cosz;
            target[0, 2] = (f2 * siny) + f1;
            target[1, 1] = (f1 * siny) + f2;
        }

        internal static void SetRotationQuaternion(this IMatrix mtx, Float4 quaternion)
        {
            SetRotationQuaternion(mtx, quaternion.X, quaternion.Y, quaternion.Z, quaternion.W);
        }

        /// <summary>
        /// クォータニオンが表す回転行列に設定します。
        /// </summary>
        /// <param name="mtx">設定する行列です。</param>
        /// <param name="quat">クォータニオンです。</param>
        internal static void SetRotationQuaternion(this IMatrix mtx, float quatX, float quatY, float quatZ, float quatW)
        {
            Ensure.Argument.True(mtx.RowCount >= 3);
            Ensure.Argument.True(mtx.ColumnCount >= 3);

            float s, xs, ys, zs, wx, wy, wz, xx, xy, xz, yy, yz, zz;

            s = 2.0f / ((quatX * quatX) + (quatY * quatY) + (quatZ * quatZ) + (quatW * quatW));

            xs = quatX * s;
            ys = quatY * s;
            zs = quatZ * s;
            wx = quatW * xs;
            wy = quatW * ys;
            wz = quatW * zs;
            xx = quatX * xs;
            xy = quatX * ys;
            xz = quatX * zs;
            yy = quatY * ys;
            yz = quatY * zs;
            zz = quatZ * zs;

            mtx.SetToIdentity();

            mtx[0, 0] = 1.0f - (yy + zz);
            mtx[0, 1] = xy - wz;
            mtx[0, 2] = xz + wy;

            mtx[1, 0] = xy + wz;
            mtx[1, 1] = 1.0f - (xx + zz);
            mtx[1, 2] = yz - wx;

            mtx[2, 0] = xz - wy;
            mtx[2, 1] = yz + wx;
            mtx[2, 2] = 1.0f - (xx + yy);
        }

        internal static void SetTranslation(this IMatrix mtx, Float3 translation)
        {
            SetTranslation(mtx, translation.X, translation.Y, translation.Z);
        }

        /// <summary>
        /// x 軸、y 軸、z 軸に沿って平行移動する行列に設定します。
        /// </summary>
        /// <param name="mtx">設定する行列です。</param>
        /// <param name="x">x 軸に沿った移動量です。</param>
        /// <param name="y">y 軸に沿った移動量です。</param>
        /// <param name="z">z 軸に沿った移動量です。</param>
        internal static void SetTranslation(this IMatrix mtx, float x, float y, float z)
        {
            Ensure.Argument.True(mtx.RowCount >= 3);
            Ensure.Argument.True(mtx.ColumnCount >= 4);

            mtx.SetToIdentity();
            mtx[0, 3] = x;
            mtx[1, 3] = y;
            mtx[2, 3] = z;
        }

        internal static void SetScale(this IMatrix mtx, Float3 scale)
        {
            SetScale(mtx, scale.X, scale.Y, scale.Z);
        }

        /// <summary>
        /// x 軸、y 軸、z 軸に沿ってスケーリングする行列に設定します。
        /// </summary>
        /// <param name="mtx">設定する行列です。</param>
        /// <param name="x">x 軸に沿ったスケールです。</param>
        /// <param name="y">y 軸に沿ったスケールです。</param>
        /// <param name="z">z 軸に沿ったスケールです。</param>
        internal static void SetScale(this IMatrix mtx, float x, float y, float z)
        {
            Ensure.Argument.True(mtx.RowCount >= 3);
            Ensure.Argument.True(mtx.ColumnCount >= 3);

            mtx.SetToIdentity();
            mtx[0, 0] = x;
            mtx[1, 1] = y;
            mtx[2, 2] = z;
        }
    }
}
