﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class IntermediateFile : RootEntity<nw4f_3difType>, IDeepCopyable<IntermediateFile>, IDeepCopyFrom<IntermediateFile>, IToolData
    {
        private readonly FileInfo @fileInfo = new FileInfo();
        private readonly IRootEntity @rootEntity = null;
        private nw4f_3dif_versionType @version;
        private IntermediateFileKind @kind;
        private string @path = string.Empty;

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public IntermediateFile(IntermediateFile source)
        {
            this.kind = source.Kind;
            switch (source.Kind)
            {
                case IntermediateFileKind.Model:
                    this.rootEntity = new Model();
                    break;
                case IntermediateFileKind.Material:
                    this.rootEntity = new Material();
                    break;
                case IntermediateFileKind.Texture:
                    this.rootEntity = new Texture();
                    break;
                case IntermediateFileKind.SkeletalAnim:
                    this.rootEntity = new SkeletalAnim();
                    break;
                case IntermediateFileKind.MaterialAnim:
                    this.rootEntity = new MaterialAnim();
                    break;
                case IntermediateFileKind.BoneVisibilityAnim:
                    this.rootEntity = new BoneVisibilityAnim();
                    break;
                case IntermediateFileKind.ShapeAnim:
                    this.rootEntity = new ShapeAnim();
                    break;
                case IntermediateFileKind.SceneAnim:
                    this.rootEntity = new SceneAnim();
                    break;
                case IntermediateFileKind.ShaderConfig:
                    this.rootEntity = new ShaderConfig();
                    break;
                case IntermediateFileKind.ShaderDefinition:
                    this.rootEntity = new ShaderDefinition();
                    break;
                case IntermediateFileKind.ShaderVariation:
                    this.rootEntity = new ShaderVariation();
                    break;
                case IntermediateFileKind.CombinerShader:
                    this.rootEntity = new CombinerShader();
                    break;
                default:
                    throw new Exception("Unsupported file type");
            }
            this.DeepCopyFrom(source);
            this.@fileInfo.PropertyChanged += this.OnPropertyChanged;
            this.@rootEntity.PropertyChanged += this.OnPropertyChanged;
            (this.@rootEntity as IChildEntity).Parent = this;
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public IntermediateFile(nw4f_3difType source)
        {
            if (source.@file_info != null)
            {
                this.@fileInfo = new FileInfo(source.@file_info);
            }
            else
            {
                this.@fileInfo = new FileInfo();
            }
            Nintendo.ToolFoundation.Contracts.Ensure.Argument.NotNull(source.Item);
            if (source.Item is modelType)
            {
                this.kind = IntermediateFileKind.Model;
                this.@rootEntity = new Model(source.Item as modelType);
            }
            else if (source.Item is materialType)
            {
                this.kind = IntermediateFileKind.Material;
                this.@rootEntity = new Material(source.Item as materialType);
            }
            else if (source.Item is textureType)
            {
                this.kind = IntermediateFileKind.Texture;
                this.@rootEntity = new Texture(source.Item as textureType);
            }
            else if (source.Item is skeletal_animType)
            {
                this.kind = IntermediateFileKind.SkeletalAnim;
                this.@rootEntity = new SkeletalAnim(source.Item as skeletal_animType);
            }
            else if (source.Item is material_animType)
            {
                this.kind = IntermediateFileKind.MaterialAnim;
                this.@rootEntity = new MaterialAnim(source.Item as material_animType);
            }
            else if (source.Item is bone_visibility_animType)
            {
                this.kind = IntermediateFileKind.BoneVisibilityAnim;
                this.@rootEntity = new BoneVisibilityAnim(source.Item as bone_visibility_animType);
            }
            else if (source.Item is shape_animType)
            {
                this.kind = IntermediateFileKind.ShapeAnim;
                this.@rootEntity = new ShapeAnim(source.Item as shape_animType);
            }
            else if (source.Item is scene_animType)
            {
                this.kind = IntermediateFileKind.SceneAnim;
                this.@rootEntity = new SceneAnim(source.Item as scene_animType);
            }
            else if (source.Item is shader_configType)
            {
                this.kind = IntermediateFileKind.ShaderConfig;
                this.@rootEntity = new ShaderConfig(source.Item as shader_configType);
            }
            else if (source.Item is shader_definitionType)
            {
                this.kind = IntermediateFileKind.ShaderDefinition;
                this.@rootEntity = new ShaderDefinition(source.Item as shader_definitionType);
            }
            else if (source.Item is shader_variationType)
            {
                this.kind = IntermediateFileKind.ShaderVariation;
                this.@rootEntity = new ShaderVariation(source.Item as shader_variationType);
            }
            else if (source.Item is combiner_shaderType)
            {
                this.kind = IntermediateFileKind.CombinerShader;
                this.@rootEntity = new CombinerShader(source.Item as combiner_shaderType);
            }
            else
            {
                throw new Exception("Unsupported file type");
            }
            (this.@rootEntity as IChildEntity).Parent = this;
            this.@version = source.@version;
            this.@fileInfo.PropertyChanged += this.OnPropertyChanged;
            this.@rootEntity.PropertyChanged += this.OnPropertyChanged;
            if (this.Kind == IntermediateFileKind.Material && !string.IsNullOrEmpty(this.Name))
            {
                // マテリアル中間ファイルではファイル名がマテリアル名となるので同期する
                var material = GetRootEntity<Material>();
                material.Name = this.Name;
            }
            this.ResolveInternalReferences();
        }

        public IntermediateFile(IntermediateFileKind kind)
        {
            this.kind = kind;
            switch (kind)
            {
                case IntermediateFileKind.Model:
                    this.rootEntity = new Model();
                    break;
                case IntermediateFileKind.Material:
                    this.rootEntity = new Material();
                    break;
                case IntermediateFileKind.Texture:
                    this.rootEntity = new Texture();
                    break;
                case IntermediateFileKind.SkeletalAnim:
                    this.rootEntity = new SkeletalAnim();
                    break;
                case IntermediateFileKind.MaterialAnim:
                    this.rootEntity = new MaterialAnim();
                    break;
                case IntermediateFileKind.BoneVisibilityAnim:
                    this.rootEntity = new BoneVisibilityAnim();
                    break;
                case IntermediateFileKind.ShapeAnim:
                    this.rootEntity = new ShapeAnim();
                    break;
                case IntermediateFileKind.SceneAnim:
                    this.rootEntity = new SceneAnim();
                    break;
                case IntermediateFileKind.ShaderConfig:
                    this.rootEntity = new ShaderConfig();
                    break;
                case IntermediateFileKind.ShaderDefinition:
                    this.rootEntity = new ShaderDefinition();
                    break;
                case IntermediateFileKind.ShaderVariation:
                    this.rootEntity = new ShaderVariation();
                    break;
                case IntermediateFileKind.CombinerShader:
                    this.rootEntity = new CombinerShader();
                    break;
                default:
                    throw new Exception("Unsupported file type");
            }
            this.fileInfo.PropertyChanged += this.OnPropertyChanged;
            this.rootEntity.PropertyChanged += this.OnPropertyChanged;
            (this.@rootEntity as IChildEntity).Parent = this;
        }

        public FileInfo FileInfo
        {
            get
            {
                return this.@fileInfo;
            }
        }

        public IRootEntity RootEntity
        {
            get
            {
                return this.@rootEntity;
            }
        }

        public nw4f_3dif_versionType Version
        {
            get
            {
                return this.@version;
            }

            set
            {
                if (this.@version == value)
                {
                    return;
                }

                this.SetProperty(ref this.@version, value, () => this.CalcCRC());
            }
        }

        public IntermediateFileKind Kind
        {
            get
            {
                return this.@kind;
            }
        }

        public string Name
        {
            get
            {
                if (string.IsNullOrEmpty(this.Path))
                {
                    return string.Empty;
                }
                else
                {
                    return System.IO.Path.GetFileNameWithoutExtension(this.Path);
                }
            }
        }

        public string Path
        {
            get
            {
                return this.@path;
            }

            set
            {
                if (this.@path == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@path, value, () => this.CalcCRC());
                this.RaisePropertyChanged("Name");
            }
        }

        public ToolData ToolData
        {
            get
            {
                var toolDataHolder = this.RootEntity as IToolData;
                Ensure.Operation.NotNull(toolDataHolder);
                return toolDataHolder.ToolData;
            }
        }

        public T GetRootEntity<T>()
            where T : class, IRootEntity
        {
            return this.rootEntity as T;
        }

        public override ObservableList<Stream> Streams
        {
            get
            {
                return this.RootEntity.Streams;
            }
        }

        public override void ResolveInternalReferences()
        {
            this.rootEntity.ResolveInternalReferences();
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override nw4f_3difType CreateSerializableData()
        {
            var writeData = new nw4f_3difType();
            writeData.@file_info = this.@fileInfo.CreateSerializableData() as file_infoType;
            writeData.@Item = this.@rootEntity.CreateSerializableData() as Object;
            writeData.@version = this.Version;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        IntermediateFile IDeepCopyable<IntermediateFile>.DeepCopy()
        {
            return new IntermediateFile(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(IntermediateFile source)
        {
            this.@fileInfo.DeepCopyFrom(source.@fileInfo);
            this.@rootEntity.DeepCopyFrom(source.@rootEntity);
            this.@version = source.@version;
            this.@path = source.@path;
            this.ResolveInternalReferences();
        }

        private protected override IRootEntity DeepCopy()
        {
            return new IntermediateFile(this);
        }

        private protected override void DeepCopyFrom(IRootEntity source)
        {
            Nintendo.ToolFoundation.Contracts.Ensure.Argument.True(source is IntermediateFile);
            this.DeepCopyFrom(source as IntermediateFile);
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.FileInfo.HashValue)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.RootEntity.HashValue)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Version)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            this.@fileInfo.AutoCalc = this.AutoCalc;
            this.@rootEntity.AutoCalc = this.AutoCalc;
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            this.@fileInfo.Reset();
            this.@rootEntity.Reset();
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            this.@fileInfo.Refresh();
            this.@rootEntity.Refresh();
        }

        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "HashValue")
            {
                this.CalcCRC();
            }
        }
    }
}
