﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;
namespace Nintendo.G3dTool.Entities{
    public class Float2x2 : ObservableEntity<string>, IMatrix, IEquatable<Float2x2>
    {
        private readonly float[] values = new float[4];

        public Float2x2()
        {
            this.SetToIdentity();
        }

        public Float2x2(
            float elem00,
            float elem01,
            float elem10,
            float elem11)
            : this()
        {
            this[0] = elem00;
            this[1] = elem01;
            this[2] = elem10;
            this[3] = elem11;
        }

        internal float[] Values
        {
            get
            {
                return this.values;
            }
        }

        public float this[int index]
        {
            get
            {
                Nintendo.ToolFoundation.Contracts.Ensure.Argument.Range(index, 0, 4);
                return this.values[index];
            }

            set
            {
                Nintendo.ToolFoundation.Contracts.Ensure.Argument.Range(index, 0, 4);
                this.SetProperty(ref this.values[index], value, () => this.CalcCRC());
            }
        }


        private float Get(int rowIndex, int columnIndex)
        {
            Nintendo.ToolFoundation.Contracts.Ensure.Argument.Range(rowIndex, 0, 2);
            Nintendo.ToolFoundation.Contracts.Ensure.Argument.Range(columnIndex, 0, 2);

            return this[rowIndex * 2 + columnIndex];
        }

        public int RowCount
        {
            get
            {
                return 2;
            }
        }

        public int ColumnCount
        {
            get
            {
                return 2;
            }
        }

        public static Float2x2 Identity
        {
            get
            {
                return new Float2x2();
            }
        }

        public float this[int rowIndex, int columnIndex]
        {
            get
            {
                Nintendo.ToolFoundation.Contracts.Ensure.Argument.Range(rowIndex, 0, 2);
                Nintendo.ToolFoundation.Contracts.Ensure.Argument.Range(columnIndex, 0, 2);
                return this.values[rowIndex * 2 + columnIndex];
            }

            set
            {
                Nintendo.ToolFoundation.Contracts.Ensure.Argument.Range(rowIndex, 0, 2);
                Nintendo.ToolFoundation.Contracts.Ensure.Argument.Range(columnIndex, 0, 2);
                this.SetProperty(ref this.values[rowIndex * 2 + columnIndex], value, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// 行成分のベクトル列を取得します。
        /// </summary>
        public IVector[] Rows
        {
            get
            {
                IVector[] data =
                {
                    new Float2(this[0, 0], this[0, 1]),
                    new Float2(this[1, 0], this[1, 1]),
                };
                return data;
            }
        }

        /// <summary>
        /// 列成分のベクトル列を取得します。
        /// </summary>
        public IVector[] Columns
        {
            get
            {
                IVector[] data =
                {
                    new Float2(this[0, 0], this[1, 0]),
                    new Float2(this[0, 1], this[1, 1]),
                };

                return data;
            }
        }

        public void SetToIdentity()
        {
            Array.ForEach(this.values, x => x = 0.0f);
            this[0, 0] = 1.0f;
            this[1, 1] = 1.0f;
        }

        /// <summary>
        /// 行列とスカラーを乗算します。
        /// </summary>
        /// <param name="mtx">演算の左の行列です。</param>
        /// <param name="scalar">演算の右のスカラーです。</param>
        /// <returns>演算結果です。</returns>
        public static Float2x2 operator *(Float2x2 mtx, float scalar)
        {
            var result = new Float2x2();
            for (int i = 0; i < mtx.RowCount; ++i)
            {
                for (int j = 0; j < mtx.ColumnCount; ++j)
                {
                    result[i, j] = mtx[i, j] * scalar;
                }
            }

            return result;
        }


        /// <summary>
        /// 2 つの行列を乗算します。
        /// </summary>
        /// <param name="rhs">演算の右の行列です。</param>
        /// <returns>演算結果です。</returns>
        public static Float2x2 operator *(Float2x2 lhs, Float2x2 rhs)
        {
            int dimension = lhs.ColumnCount;
            var result = new Float2x2();
            for (int i = 0; i < lhs.RowCount; ++i)
            {
                for (int j = 0; j < lhs.ColumnCount; ++j)
                {
                    float sum = 0f;
                    for (int k = 0; k < dimension; ++k)
                    {
                        sum += lhs[i, k] * rhs[k, j];
                    }

                    result[i, j] = sum;
                }
            }

            return result;
        }

        public void SetToZero()
        {
            for (int elemIndex = 0; elemIndex < 4; ++elemIndex)
            {
                this[elemIndex] = 0;
            }
        }

        /// <summary>
        /// 比較演算子
        /// </summary>
        /// <param name="obj"></param>
        /// <returns>一致していれば true、そうでなければ false を返します。</returns>
        public bool Equals(Float2x2 obj)
        {
            if (obj == null)
            {
                return false;
            }

            bool isEqual = true;
            for (int elemIndex = 0; elemIndex < 4; ++elemIndex)
            {
                isEqual &= (obj[elemIndex] == this[elemIndex]);
            }

            return isEqual;
        }

        public override string CreateSerializableData()
        {
            return this.ToString();
        }

        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            return crc.ComputeHashUInt32(this.ToString());
        }

        public override string ToString()
        {
            string result = string.Empty;
            for (int elemIndex = 0; elemIndex < 4; ++elemIndex)
            {
                result += Convert.ToDouble(this[elemIndex]).ToString() + " ";
            }

            return result.TrimEnd();
        }

        public void DeepCopyFrom(Float2x2 source)
        {
            for (int elemIndex = 0; elemIndex < 4; ++elemIndex)
            {
                this[elemIndex] = source.Values[elemIndex];
            }
        }

        public static Float2x2 Parse(string text)
        {
            string[] valueTexts = StringUtility.SplitValueListText(text);
            Float2x2 result = new Float2x2();
            for (int elemIndex = 0; elemIndex < 4; ++elemIndex)
            {
                result[elemIndex] = float.Parse(valueTexts[elemIndex]);
            }

            return result;
        }
    }
}
