﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class Connection : ObservableEntity<connectionType>, IDeepCopyable<Connection>, IDeepCopyFrom<Connection>
    {
        private readonly Source @source = new Source();
        private readonly Target @target = new Target();

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public Connection()
        {
            this.@source.PropertyChanged += this.OnPropertyChanged;
            this.@target.PropertyChanged += this.OnPropertyChanged;
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public Connection(Connection source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public Connection(connectionType source)
        {
            if (source.@source != null)
            {
                this.@source = new Source(source.@source);
            }
            else
            {
                this.@source = new Source();
            }
            if (source.@target != null)
            {
                this.@target = new Target(source.@target);
            }
            else
            {
                this.@target = new Target();
            }
            this.@source.PropertyChanged += this.OnPropertyChanged;
            this.@target.PropertyChanged += this.OnPropertyChanged;
        }

        public Source Source
        {
            get
            {
                return this.@source;
            }
        }

        public Target Target
        {
            get
            {
                return this.@target;
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override connectionType CreateSerializableData()
        {
            var writeData = new connectionType();
            writeData.@source = this.@source.CreateSerializableData() as sourceType;
            writeData.@target = this.@target.CreateSerializableData() as targetType;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        Connection IDeepCopyable<Connection>.DeepCopy()
        {
            return new Connection(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(Connection source)
        {
            this.@source.DeepCopyFrom(source.@source);
            this.@target.DeepCopyFrom(source.@target);
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Source.HashValue)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Target.HashValue)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            this.@source.AutoCalc = this.AutoCalc;
            this.@target.AutoCalc = this.AutoCalc;
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            this.@source.Reset();
            this.@target.Reset();
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            this.@source.Refresh();
            this.@target.Refresh();
        }

        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "HashValue")
            {
                this.CalcCRC();
            }
        }
    }
}
