﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class ComputeStage : ObservableEntity<compute_stageType>, IChildEntity, IDeepCopyable<ComputeStage>, IDeepCopyFrom<ComputeStage>
    {
        private int @srcIndex;
        private ShadingModel parent = null;
        private ShaderSrc @shaderSrc = null;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ComputeStage()
        {
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public ComputeStage(ComputeStage source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public ComputeStage(compute_stageType source)
        {
            this.@srcIndex = source.@src_index;
        }

        internal int SrcIndex
        {
            get
            {
                if (this.ShaderSrc != null)
                {
                    // 親が見つからない場合はインデックスを求められないので -1 を返す
                    if (this.Parent == null)
                    {
                        return -1;
                    }
                    if (this.Parent.Parent == null)
                    {
                        return -1;
                    }
                    Ensure.Operation.True(this.Parent.Parent.ShaderSrcs.Contains(this.ShaderSrc));
                    return this.Parent.Parent.ShaderSrcs.IndexOf(this.ShaderSrc);
                }
                return this.@srcIndex;
            }
        }

        IEntity IChildEntity.Parent
        {
            get
            {
                return this.parent as IEntity;
            }

            set
            {
                if (this.parent == value)
                {
                    return;
                }

                Ensure.Argument.True(value != null ? value is ShadingModel : true);
                this.SetProperty(ref this.parent, value as ShadingModel, () => this.CalcCRC());
            }
        }

        public ShadingModel Parent
        {
            get
            {
                return this.parent;
            }
        }

        Type IChildEntity.ParentType
        {
            get
            {
                return typeof(ShadingModel);
            }
        }

        public ShaderSrc ShaderSrc
        {
            get
            {
                return this.@shaderSrc;
            }

            set
            {
                if (this.@shaderSrc == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@shaderSrc, value, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override compute_stageType CreateSerializableData()
        {
            var writeData = new compute_stageType();
            {
                Ensure.Operation.NotNull(this.ShaderSrc);
                writeData.src_index = this.SrcIndex;
            }
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        ComputeStage IDeepCopyable<ComputeStage>.DeepCopy()
        {
            return new ComputeStage(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(ComputeStage source)
        {
            this.@srcIndex = source.@srcIndex;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.SrcIndex)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }
    }
}
