﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;
using System.Text.RegularExpressions;

namespace Nintendo.G3dTool.Entities
{
    public abstract class ChoiceValue<TValue> : ObservableEntity<string>
    {
        private TValue value;
        private string label;

        public ChoiceValue()
        {
        }

        public ChoiceValue(TValue value, string label)
        {
            this.Value = value;
            this.Label = label;
        }

        public TValue Value
        {
            get
            {
                return this.value;
            }

            set
            {
                this.SetProperty<TValue>(ref this.value, value, () => CalcCRC());
            }
        }

        public string Label
        {
            get
            {
                return this.label;
            }

            set
            {
                this.SetProperty<string>(ref this.label, value, () => CalcCRC());
            }

        }

        public override string ToString()
        {
            if (string.IsNullOrEmpty(this.Label))
            {
                return this.Value.ToString();
            }
            else
            {
                return $"{this.Value.ToString()}:{this.Label}";
            }
        }

        public override string CreateSerializableData()
        {
            return this.ToString();
        }

        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.value)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.label)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        protected void ExtractValueAndLabel(out string value, out string label, string source)
        {
            int splitIndex = source.IndexOf(':');
            if (splitIndex >= 0)
            {
                value = source.Substring(0, splitIndex);
                label = source.Substring(splitIndex + 1, source.Length - splitIndex - 1);
            }
            else
            {
                value = source;
                label = null;
            }
        }
    }

    public class ChoiceValueString : ChoiceValue<string>, IEquatable<ChoiceValueString>, IDeepCopyable<ChoiceValueString>
    {
        public ChoiceValueString(string source)
        {
            string value, label;
            ExtractValueAndLabel(out value, out label, source);
            this.Value = value;
            this.Label = label;
        }

        public ChoiceValueString(string value, string label)
            : base(value, label)
        {
        }

        public bool Equals(ChoiceValueString obj)
        {
            if (obj == null)
            {
                return false;
            }

            if (this.Value != obj.Value)
            {
                return false;
            }

            if (this.Label != obj.Label)
            {
                return false;
            }

            return true;
        }

        public ChoiceValueString DeepCopy()
        {
            return new ChoiceValueString(this.Value, this.Label);
        }
    }

    public class ChoiceValueInt : ChoiceValue<int>, IEquatable<ChoiceValueInt>, IDeepCopyable<ChoiceValueInt>
    {
        public ChoiceValueInt(string source)
        {
            string value, label;
            ExtractValueAndLabel(out value, out label, source);
            this.Value = int.Parse(value);
            this.Label = label;
        }

        public ChoiceValueInt(int value, string label)
            : base(value, label)
        {
        }

        public bool Equals(ChoiceValueInt obj)
        {
            if (obj == null)
            {
                return false;
            }

            if (this.Value != obj.Value)
            {
                return false;
            }

            if (this.Label != obj.Label)
            {
                return false;
            }

            return true;
        }

        public ChoiceValueInt DeepCopy()
        {
            return new ChoiceValueInt(this.Value, this.Label);
        }
    }

    public class ChoiceValueFloat : ChoiceValue<float>, IEquatable<ChoiceValueFloat>, IDeepCopyable<ChoiceValueFloat>
    {
        public ChoiceValueFloat(string source)
        {
            string value, label;
            ExtractValueAndLabel(out value, out label, source);
            this.Value = float.Parse(value);
            this.Label = label;
        }

        public ChoiceValueFloat(float value, string label)
            : base(value, label)
        {
        }

        public bool Equals(ChoiceValueFloat obj)
        {
            if (obj == null)
            {
                return false;
            }

            if (this.Value != obj.Value)
            {
                return false;
            }

            if (this.Label != obj.Label)
            {
                return false;
            }

            return true;
        }

        public ChoiceValueFloat DeepCopy()
        {
            return new ChoiceValueFloat(this.Value, this.Label);
        }
    }
}
