﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Windows.Forms;

namespace App.Controls
{
    //--------------------------------------------------------------------------
    // ToolStripItem基本クラス階層
    //
    // Component
    // └ToolStripItem
    //   ├ToolStripSeparator
    //   ├ToolStripButton
    //   ├ToolStripLabel
    //   │└ToolStripStatusLabel
    //   ├ToolStripControlHost
    //   │├ToolStripComboBox
    //   │├ToolStripProgressBar
    //   │└ToolStripTextBox
    //   └ToolStripDropDownItem
    //     ├ToolStripMenuItem
    //     ├ToolStripSplitButton
    //     └ToolStripDropDownButton
    //       └(ToolStripOverflowButton)
    //
    //--------------------------------------------------------------------------

    /// <summary>
    /// ＵＩツールストリップセパレータクラス。
    /// </summary>
    [ToolboxBitmap(typeof(ToolStripSeparator))]
    public class UIToolStripSeparator : ToolStripSeparator
    {
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool CanRaiseEvents
        {
            get
            {
                if (UIControlEventSuppressBlock.Enabled)
                {
                    return false;
                }
                return base.CanRaiseEvents;
            }
        }
    }

    /// <summary>
    /// ＵＩツールストリップボタンクラス。
    /// </summary>
    [ToolboxBitmap(typeof(ToolStripButton))]
    public class UIToolStripButton : ToolStripButton
    {
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool CanRaiseEvents
        {
            get
            {
                if (UIControlEventSuppressBlock.Enabled)
                {
                    return false;
                }
                return base.CanRaiseEvents;
            }
        }
    }

    /// <summary>
    /// ＵＩツールストリップラベルクラス。
    /// </summary>
    [ToolboxBitmap(typeof(ToolStripLabel))]
    public class UIToolStripLabel : ToolStripLabel
    {
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool CanRaiseEvents
        {
            get
            {
                if (UIControlEventSuppressBlock.Enabled)
                {
                    return false;
                }
                return base.CanRaiseEvents;
            }
        }
    }

    /// <summary>
    /// ＵＩツールストリップステータスラベルクラス。
    /// </summary>
    [ToolboxBitmap(typeof(ToolStripStatusLabel))]
    public class UIToolStripStatusLabel : ToolStripStatusLabel
    {
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnTextChanged(EventArgs e)
        {
            // 規定の処理で Owner.Invalidate が呼ばれる
            base.OnTextChanged(e);

            // Idle まで再描画されないのを防ぐため強制的に描画させる
            if (this.Owner != null)
            {
                this.Owner.Update();
            }
        }
    }

    /// <summary>
    /// ＵＩツールストリップコンボボックスクラス。
    /// </summary>
    [ToolboxBitmap(typeof(ToolStripComboBox))]
    public class UIToolStripComboBox : ToolStripComboBox
    {
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool CanRaiseEvents
        {
            get
            {
                if (UIControlEventSuppressBlock.Enabled)
                {
                    return false;
                }
                return base.CanRaiseEvents;
            }
        }
    }

    /// <summary>
    /// ＵＩツールストリッププログレスバークラス。
    /// </summary>
    [ToolboxBitmap(typeof(ToolStripProgressBar))]
    public class UIToolStripProgressBar : ToolStripProgressBar
    {
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool CanRaiseEvents
        {
            get
            {
                if (UIControlEventSuppressBlock.Enabled)
                {
                    return false;
                }
                return base.CanRaiseEvents;
            }
        }
    }

    /// <summary>
    /// ＵＩツールストリップテキストボックスクラス。
    /// </summary>
    [ToolboxBitmap(typeof(ToolStripTextBox))]
    public class UIToolStripTextBox : ToolStripTextBox
    {
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool CanRaiseEvents
        {
            get
            {
                if (UIControlEventSuppressBlock.Enabled)
                {
                    return false;
                }
                return base.CanRaiseEvents;
            }
        }
    }

    /// <summary>
    /// ＵＩツールストリップメニュー項目クラス。
    /// </summary>
    [ToolboxBitmap(typeof(ToolStripMenuItem))]
    [DefaultEvent("MenuCommandHandler")]
    public class UIToolStripMenuItem : ToolStripMenuItem
    {
        /// <summary>ショートカットキー処理前イベント。</summary>
        public event EventHandler BeforeShortcutKeyProcessing = null;

        public event MenuCommandHandler MenuCommandHandler = null;

        public string CaptionText { get; set; }

        public void InvokeMenuCommand(MenuCommandArgs args)
        {
            if (MenuCommandHandler != null)
            {
                MenuCommandHandler(args);
            }
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UIToolStripMenuItem()
        {
            // 必要に応じてオーバーフローさせる
            base.Overflow = ToolStripItemOverflow.AsNeeded;
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        [DefaultValue(ToolStripItemOverflow.AsNeeded)]
        public new ToolStripItemOverflow Overflow
        {
            get { return base.Overflow; }
            set { base.Overflow = value; }
        }

        /// <summary>
        /// 複製を作成。
        /// </summary>
        public UIToolStripMenuItem CreateClone()
        {
            UIToolStripMenuItem item = new UIToolStripMenuItem();

            // プロパティのコピー
            item.Text                     = this.Text;
            item.Image                    = this.Image;
            item.ShortcutKeys             = this.ShortcutKeys;
            item.ShowShortcutKeys         = this.ShowShortcutKeys;
            item.ShortcutKeyDisplayString = this.ShortcutKeyDisplayString;
            item.Tag                      = this.Tag;
            item.MenuCommandHandler = this.MenuCommandHandler;

            // イベントのコピー
            item.Events.AddHandlers(this.Events);
            item.BeforeShortcutKeyProcessing += this.BeforeShortcutKeyProcessing;

            return item;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool ProcessCmdKey(ref Message m, Keys keyData)
        {
            // ショートカットキー実行前に更新の機会を与える
            if (this.ShortcutKeys == keyData)
            {
                if (this.BeforeShortcutKeyProcessing != null)
                {
                    this.BeforeShortcutKeyProcessing(this, EventArgs.Empty);
                }
            }
            return base.ProcessCmdKey(ref m, keyData);
        }
    }

    /// <summary>
    /// ＵＩツールストリップスプリットボタンクラス。
    /// </summary>
    [ToolboxBitmap(typeof(ToolStripSplitButton))]
    public class UIToolStripSplitButton : ToolStripSplitButton
    {
    }

    /// <summary>
    /// ＵＩツールストリップドロップダウンボタンクラス。
    /// </summary>
    [ToolboxBitmap(typeof(ToolStripDropDownButton))]
    public class UIToolStripDropDownButton : ToolStripDropDownButton
    {
    }
}
