﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Runtime.InteropServices;
using System.Windows.Forms;
using System.Windows.Forms.Design;

namespace App.Controls
{
     /// <summary>
    /// ＵＩテキストボックスクラス。
    /// </summary>
    public class UITextBox : TextBox
    {
        // ツールストリップ用オーナー
        private ToolStripItem _toolStripOwner = null;
        // マウスオーバーフラグ
        private bool _mouseIsOver = false;

        private UIToolTip _ttpHint = null;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UITextBox()
        {
        }

        private const int EM_SETRECT = 0x00B3;

        [StructLayout(LayoutKind.Sequential)]
        private struct Rect
        {
            public int Left;
            public int Top;
            public int Right;
            public int Bottom;
        }

        [DllImport("user32.dll", SetLastError=true)]
        private static extern int SendMessage(IntPtr hwnd, int msg, IntPtr wParam, ref Rect lParam);

        private Point padding;

        [DesignerSerializationVisibility(DesignerSerializationVisibility.Visible)]
        [DefaultValue(typeof(Point))]
        public new Point Padding
        {
            get
            {
                return padding;
            }
            set
            {
                padding = value;
                SetPaddingRect();
            }
        }

        private bool _AutoHint = false;
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Visible)]
        public bool AutoHint
        {
            get
            {
                return _AutoHint;
            }
            set
            {
                _AutoHint = value;
                UpdateHint();
            }
        }

        private bool _OverrideContextMenu = false;
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Visible)]
        public bool OverrideContextMenu
        {
            get
            {
                return _OverrideContextMenu;
            }
            set
            {
                _OverrideContextMenu = value;
            }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public override string Text
        {
            get
            {
                return base.Text;
            }
            set
            {
                base.Text = value;
                UpdateHint();
            }
        }

        public override Color ForeColor
        {
            get
            {
                return base.ForeColor;
            }
            set
            {
                // バックカラーを設定しないとフォアカラーが反映されないことがある。http://bytes.com/topic/c-sharp/answers/233961-read-only-textbox
                BackColor = BackColor;
                base.ForeColor = value;
            }
        }

        /// <summary>
        /// ツールストリップ用オーナー。
        /// </summary>
        internal ToolStripItem ToolStripOwner
        {
            get { return _toolStripOwner; }
            set { _toolStripOwner = value; }
        }

        /// <summary>
        /// マウスオーバーフラグ。
        /// </summary>
        private bool MouseIsOver
        {
            get { return _mouseIsOver; }
            set
            {
                if (_mouseIsOver != value)
                {
                    _mouseIsOver = value;
                    if (!this.Focused)
                    {
                        InvalidateNonClient();
                    }
                }
            }
        }

        /// <summary>
        /// 絶対クライアント領域。
        /// </summary>
        private Win32.RECT AbsoluteClientRect
        {
            get
            {
                Win32.RECT rect = new App.Win32.RECT();

                CreateParams cp = this.CreateParams;
                Win32.User32.AdjustWindowRectEx(ref rect, cp.Style, false, cp.ExStyle);
                int offsetH = -rect.left;
                int offsetV = -rect.top;

                Win32.User32.GetClientRect(this.Handle, ref rect);
                rect.left   += offsetH;
                rect.top    += offsetV;
                rect.right  += offsetH;
                rect.bottom += offsetV;

                return rect;
            }
        }

        /// <summary>
        /// 非クライアント領域を無効化。
        /// </summary>
        private void InvalidateNonClient()
        {
            if (_toolStripOwner != null)
            {
                Win32.RECT rcClient = this.AbsoluteClientRect;
                IntPtr rgnResult = IntPtr.Zero;
                IntPtr rgnClient = IntPtr.Zero;
                IntPtr rgnEntire = IntPtr.Zero;
                try
                {
                    rgnEntire = Win32.Gdi32.CreateRectRgn(0, 0, base.Width, base.Height);
                    rgnClient = Win32.Gdi32.CreateRectRgn(rcClient.left, rcClient.top, rcClient.right, rcClient.bottom);
                    rgnResult = Win32.Gdi32.CreateRectRgn(0, 0, 0, 0);
                    Win32.Gdi32.CombineRgn(rgnResult, rgnEntire, rgnClient, Win32.RGN.RGN_XOR);

                    // 0x705 =
                    // Win32.RDW.RDW_INVALIDATE |
                    // Win32.RDW.RDW_ERASE      |
                    // Win32.RDW.RDW_UPDATENOW  |
                    // Win32.RDW.RDW_ERASENOW   |
                    // Win32.RDW.RDW_FRAME
                    Win32.RECT rect = new App.Win32.RECT();
                    Win32.User32.RedrawWindow(base.Handle, ref rect, rgnResult, 0x705);
                }
                finally
                {
                    // REGON ハンドルを開放
                    try
                    {
                        if (rgnResult != IntPtr.Zero) { Win32.Gdi32.DeleteObject(rgnResult); }
                    }
                    finally
                    {
                        try
                        {
                            if (rgnClient != IntPtr.Zero) { Win32.Gdi32.DeleteObject(rgnClient); }
                        }
                        finally
                        {
                            if (rgnEntire != IntPtr.Zero) { Win32.Gdi32.DeleteObject(rgnEntire); }
                        }
                    }
                }
            }
        }

        /// <summary>
        /// WM_NCPAINT ハンドラ。
        /// </summary>
        private void WmNCPaint(ref Message m)
        {
            // 通常の描画
            if (_toolStripOwner == null)
            {
                base.WndProc(ref m);
            }
            // ツールストリップスタイル
            else
            {
                IntPtr hDC = Win32.User32.GetWindowDC(m.HWnd);
                if (hDC == IntPtr.Zero)
                {
                    throw new Win32Exception();
                }
                try
                {
                    // カラー
                    Color lineColor = this.BackColor;
                    Color backColor = this.BackColor;
                    if (this.MouseIsOver || this.Focused)
                    {
                        ToolStrip toolStrip = _toolStripOwner.Owner;
                        if (toolStrip != null)
                        {
                            ToolStripProfessionalRenderer renderer = toolStrip.Renderer as ToolStripProfessionalRenderer;
                            if (renderer != null)
                            {
                                // ProfessionalColorTable.TextBoxBorder（internal）と同じ
                                lineColor = renderer.ColorTable.ButtonSelectedHighlightBorder;
                            }
                        }
                    }
                    if (!this.Enabled)
                    {
                        lineColor = SystemColors.ControlDark;
                        backColor = SystemColors.Control;
                    }

                    // 描画
                    using (Graphics g = Graphics.FromHdcInternal(hDC))
                    {
                        Rectangle rcClient = this.AbsoluteClientRect.ToRectangle();
                        using (Brush brush = new SolidBrush(backColor))
                        {
                            g.FillRectangle(brush, 0, 0, this.Width, rcClient.Top);
                            g.FillRectangle(brush, 0, 0, rcClient.Left, this.Height);
                            g.FillRectangle(brush, 0, rcClient.Bottom, this.Width, this.Height - rcClient.Height);
                            g.FillRectangle(brush, rcClient.Right, 0, this.Width - rcClient.Right, this.Height);
                        }
                        using (Pen pen = new Pen(lineColor))
                        {
                            g.DrawRectangle(pen, 0, 0, this.Width - 1, this.Height - 1);
                        }
                    }
                }
                finally
                {
                    Win32.User32.ReleaseDC(this.Handle, hDC);
                }
                m.Result = IntPtr.Zero;
            }
        }

        #region オーバーライド
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool CanRaiseEvents
        {
            get
            {
                if (UIControlEventSuppressBlock.Enabled)
                {
                    return false;
                }
                return base.CanRaiseEvents;
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void WndProc(ref Message m)
        {
            switch (m.Msg)
            {
                // 非クライアント領域描画
                case Win32.WM.WM_NCPAINT:
                    //WmNCPaint(ref m);
                    //return;
                    break;

                default:
                    break;
            }
            base.WndProc(ref m);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnMouseMove(MouseEventArgs e)
        {
            if (_OverrideContextMenu)
            {
                if (e.Button == MouseButtons.Right)
                {
                    if (e.X < 0 || e.Y < 0 || ClientSize.Width >= e.X || ClientSize.Height >= e.Y)
                    {
                        Capture = false;
                    }
                }
            }

            base.OnMouseMove(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnGotFocus(EventArgs e)
        {
            base.OnGotFocus(e);
            InvalidateNonClient();
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnLostFocus(EventArgs e)
        {
            base.OnLostFocus(e);
            InvalidateNonClient();
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnMouseEnter(EventArgs e)
        {
            base.OnMouseEnter(e);
            this.MouseIsOver = true;

            ForceShowHint();
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnMouseLeave(EventArgs e)
        {
            base.OnMouseLeave(e);
            this.MouseIsOver = false;

            ForceHideHint();
        }

        protected override void OnResize(EventArgs e)
        {
            base.OnResize(e);

            SetPaddingRect();
        }

        protected override void OnVisibleChanged(EventArgs e)
        {
            base.OnVisibleChanged(e);

            SetPaddingRect();
        }

        #endregion

        #region デザイナ制御
        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public sealed override RightToLeft RightToLeft
        {
            get { return base.RightToLeft;  }
            set { base.RightToLeft = value; }
        }
        #endregion

        private void SetPaddingRect()
        {
            if (Multiline)
            {
                Rect rect = new Rect()
                {
                    Left = Padding.X,
                    Top = Padding.Y,
                    Right = ClientSize.Width - Padding.X,
                    Bottom = ClientSize.Height - Padding.Y,
                };

                SendMessage(Handle, EM_SETRECT, IntPtr.Zero, ref rect);
            }
        }

        private void UpdateHint()
        {
            if (AutoHint)
            {
                if (_ttpHint == null)
                {
                    _ttpHint = new UIToolTip();
                }

                _ttpHint.SetToolTip(this, Text);
            }
            else
            {
                if (_ttpHint != null)
                {
                    _ttpHint.RemoveAll();
                    _ttpHint = null;
                }
            }
        }

        private void ForceShowHint()
        {
            if (AutoHint)
            {
                if (_ttpHint == null)
                {
                    _ttpHint = new UIToolTip();
                }

                _ttpHint.SetToolTip(this, Text);
            }
        }

        private void ForceHideHint()
        {
            if (_ttpHint != null)
            {
                _ttpHint.RemoveAll();
                _ttpHint = null;
            }
        }
    }
}
