﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ShellExtension_PCH.h"
#include "ShellExtension_Utility.h"
#include "ShellExtension_Type.h"
#include "ShellExtension_Texture.h"
#include "ImageLoader/ShellExtension_ImageLoader.h"
#include "ImageLoader/ShellExtension_PSDLoader.h"
#include "ImageLoader/ShellExtension_CTexLoader.h"
#include <algorithm>
#include <vector>
#include <functional>

//==============================================================================
//
// Utility functions
//
//==============================================================================
namespace
{
_Check_return_
bool readDescriptionStructure(FILE* file, int* s = NULL);

_Check_return_
bool readDescription(int OSType, FILE* file, int* s = NULL);

//------------------------------------------------------------------------------
template <typename T> void swapData(T& v)
{
    if (sizeof(T) > 1)
    {
        std::reverse((char*)&v, (char*)&v + sizeof(T));
    } // End if
} // End swapData

//------------------------------------------------------------------------------
template <typename T> _Check_return_ bool readData(T& v, FILE* file, int* s = NULL)
{
    if (fread(&v, sizeof(T), 1, file) != 1)
    {
        return false;
    }
    if (s != NULL) *s += sizeof(T);
    swapData(v);
    return true;
} // End readData

//------------------------------------------------------------------------------
template <typename T> _Check_return_ bool readData(T* v, int c, FILE* file, int* s = NULL)
{
    if (fread(v, sizeof(T), c, file) != static_cast<size_t>(c))
    {
        return false;
    }
    if (s != NULL) *s += c;
    return true;
} // End readData

//------------------------------------------------------------------------------
_Check_return_
bool readUnicodeString(std::wstring& str, FILE* file, int* s = NULL)
{
    int length;
    if (!readData(length, file, s))
    {
        return false;
    }

    // 文字列の長さにはバイト数ではなく文字数が格納されている
    for (int i = 0; i < length; i++)
    {
        wchar_t ch;
        if (!readData(ch, file, s))
        {
            return false;
        }

        if (ch != 0)
        {
            str.append(1, ch);
        } // End if
    } // End for

    return true;
} // End readUnicodeString

//------------------------------------------------------------------------------
_Check_return_
bool readObjectID(std::vector<char>& id, FILE* file, int* s = NULL)
{
    int keylen;
    if (!readData(keylen, file, s))
    {
        return false;
    }
    id.resize((keylen == 0) ? 4 : keylen);
    return readData(&id[0], (int)id.size(), file ,s);
} // End readObjectID

//------------------------------------------------------------------------------
_Check_return_
bool readClassStructure(FILE* file, int* s = NULL)
{
    std::wstring name;

    std::vector<char> classID;
    return readUnicodeString(name, file, s) && readObjectID(classID, file, s);
} // End readClassStructure

//------------------------------------------------------------------------------
_Check_return_
bool readListStructure(FILE* file, int* s = NULL)
{
    int numItems;
    if (!readData(numItems, file, s))
    {
        return false;
    }

    for (int iItem = 0; iItem < numItems; iItem++)
    {
        unsigned int OSType;
        if (!readData(OSType, file, s))
        {
            return false;
        }

        if (!readDescription(OSType, file, s))
        {
            return false;
        } // End if
    } // End for
    return true;
} // End readListStructure

//------------------------------------------------------------------------------
_Check_return_
bool readDoubleStructure(FILE* file, int* s = NULL)
{
    double val;
    return readData(val, file, s);
} // End readDoubleStructure

//------------------------------------------------------------------------------
_Check_return_
bool readUnitFloatStructure(FILE* file, int* s = NULL)
{
    unsigned int units;
    double val;
    return readData(units, file, s) && readData(val, file, s);
} // End readUnitFloatStructure

//------------------------------------------------------------------------------
_Check_return_
bool readStringStructure(FILE* file, int* s = NULL)
{
    std::wstring str;
    return readUnicodeString(str, file, s);
} // End readStringStructure

//------------------------------------------------------------------------------
_Check_return_
bool readEnumeratedStructure(FILE* file, int* s = NULL)
{
    std::vector<char> typeID;
    std::vector<char> enumID;
    return readObjectID(typeID, file, s) && readObjectID(enumID, file, s);
} // End readEnumeratedStructure

//------------------------------------------------------------------------------
_Check_return_
bool readIntegerStructure(FILE* file, int* s = NULL)
{
    int i;
    return readData(i, file, s);
} // End readIntegerStructure

//------------------------------------------------------------------------------
_Check_return_
bool readBooleanStructure(FILE* file, int* s = NULL)
{
    int b;
    return readData(b, file, s);
} // End readBooleanStructure

//------------------------------------------------------------------------------
_Check_return_
bool readAliasStructure(FILE* file, int* s = NULL)
{
    int len;
    if (!readData(len, file, s))
    {
        return false;
    }

    if (len > 0)
    {
        std::vector<char> alias;
        alias.resize(len);
        if (!readData(&alias[0], (int)alias.size(), file, s))
        {
            return false;
        }
    } // End if

    return true;
} // End readAliasStructure

//------------------------------------------------------------------------------
_Check_return_
bool readReferenceStructure(FILE* file, int* s = NULL)
{
    int numItems;
    if (!readData(numItems, file, s))
    {
        return false;
    }

    for (int iItem = 0; iItem < numItems; iItem++)
    {
        unsigned int OSType;
        if (!readData(OSType, file, s))
        {
            return false;
        }

        switch (OSType)
        {
        case 'prop':
            {
                std::wstring name;
                if (!readUnicodeString(name, file, s))
                {
                    return false;
                }
                std::vector<char> classID;
                if (!readObjectID(classID, file, s))
                {
                    return false;
                }
                std::vector<char> keyID;
                if (!readObjectID(keyID, file, s))
                {
                    return false;
                }
                break;
            }
        case 'Clss':
            {
                if (!readClassStructure(file, s))
                {
                    return false;
                }
                break;
            }
        case 'Enmr':
            {
                std::wstring name;
                if (!readUnicodeString(name, file, s))
                {
                    return false;
                }
                std::vector<char> classID;
                if (!readObjectID(classID, file, s))
                {
                    return false;
                }
                std::vector<char> typeID;
                if (!readObjectID(typeID, file, s))
                {
                    return false;
                }
                std::vector<char> enumID;
                if (!readObjectID(enumID, file, s))
                {
                    return false;
                }
                break;
            }
        case 'rele':
            {
                std::wstring name;
                if (!readUnicodeString(name, file, s))
                {
                    return false;
                }
                std::vector<char> classID;
                if (!readObjectID(classID, file, s))
                {
                    return false;
                }
                int offset;
                if (!readData(offset, file, s))
                {
                    return false;
                }
                break;
            }
        case 'Idnt':
            {
                unsigned int id;
                if (!readData(id, file, s))
                {
                    return false;
                }
                break;
            }
        case 'indx':
            {
                unsigned int index;
                if (!readData(index, file, s))
                {
                    return false;
                }
                break;
            }
        case 'name':
            {
                std::wstring name;
                if (!readUnicodeString(name, file, s))
                {
                    return false;
                }
                break;
            }
        default:
            return false;
        } // End switch
    } // End for

    return true;
 } // End readReferenceStructure


//------------------------------------------------------------------------------
 _Check_return_
bool readDescription(int OSType, FILE* file, int* s)
{
    switch (OSType)
    {
    case 'obj ':
        return readReferenceStructure(file, s);
    case 'Objc':
        return readDescriptionStructure(file, s);
    case 'VlLs':
        return readListStructure(file, s);
    case 'doub':
        return readDoubleStructure(file, s);
    case 'UntF':
        return readUnitFloatStructure(file, s);
    case 'TEXT':
        return readStringStructure(file, s);
    case 'enum':
        return readEnumeratedStructure(file, s);
    case 'long':
        return readIntegerStructure(file, s);
    case 'bool':
        return readBooleanStructure(file, s);
    case 'GlbO':
        return readDescriptionStructure(file, s);
    case 'type':
    case 'GlbC':
        return readClassStructure(file, s);
    case 'alis':
        return readAliasStructure(file, s);
    default:
        return false;
    } // End switch
} // End readDescription

//------------------------------------------------------------------------------
 _Check_return_
bool readDescriptionStructure(FILE* file, int* s)
{
    std::wstring name;
    if (!readUnicodeString(name, file, s))
    {
        return false;
    }
    std::vector<char> classID;
    if (!readObjectID(classID, file, s))
    {
        return false;
    }

    int numItems;
    if (!readData(numItems, file, s))
    {
        return false;
    }
    for (int iItem = 0; iItem < numItems; iItem++)
    {
        std::vector<char> key;
        if (!readObjectID(key, file, s))
        {
            return false;
        }

        unsigned int OSType;
        if (!readData(OSType, file, s))
        {
            return false;
        }

        if (!readDescription(OSType, file, s))
        {
            return false;
        } // End if
    } // End for

    return true;
} // End readDescriptionStructure


} // End namespace


//==============================================================================
//
// Implementation of CNWPSDLoader
//
//==============================================================================
CNWPSDLoader::CNWPSDLoader() :
    CNWImageLoader()
{
} // End of Constructor for CNWPSDLoader


//------------------------------------------------------------------------------
CNWPSDLoader::~CNWPSDLoader()
{
} // End of Destructor for CNWPSDLoader


//------------------------------------------------------------------------------
// Load from image file onto pTexture
//
// - If bLoadPreview is true, create preview image for icon
//   If bLoadPreview is false, just load information ( format, size etc )
//------------------------------------------------------------------------------
_Check_return_
bool CNWPSDLoader::Load( CNWTexture *pTexture,
                         const WCHAR *szFilePath,
                         bool bLoadPreview )
{
    FILE* file = NULL;
    _wfopen_s(&file, szFilePath, L"rb");
    if (file == NULL)
        return false;

    std::unique_ptr<std::FILE, decltype(&std::fclose)> fp(file, &std::fclose);

    // read header
    PSDHeader header;
    if (!ReadPSDHeader(header, file))
    {
        return false;
    }

    if (header.signature != '8BPS')
    {
        return false;
    }

    // skip color mode data
    int sizeColorMode;
    if (!readData(sizeColorMode, file))
    {
        return false;
    }

    fseek(file, sizeColorMode, SEEK_CUR);

    // read image resources
    int sizeImageResources;
    if (!readData(sizeImageResources, file))
    {
        return false;
    }

    ThumbnailResourceHeader thumbnailHeader;
    std::vector<char> thumbnailData;
    std::vector<char> xmpData;
    std::list<std::wstring> layerComps;
    int readSize = 0;
    int lastApplied = -1;
    while(readSize < sizeImageResources)
    {
        ImageResourceBlocks block;
        if (!ReadResourceBlock(block, file, &readSize))
        {
            return false;
        }

        if (block.signature != '8BIM')
        {
            return false;
        } // End if

        if (block.resourceID == 0x0409 ||
            block.resourceID == 0x040C)
        {
            if (!ReadThumbnailHeader(thumbnailHeader, file, &readSize))
            {
                return false;
            }

            // xmp データを読み込む
            thumbnailData.resize(thumbnailHeader.compressedSize);
            size_t s = fread(&thumbnailData[0], 1, thumbnailHeader.compressedSize, file);
            if (s != (size_t)thumbnailHeader.compressedSize)
            {
                readSize += (int)s;
                thumbnailData.clear();
                continue;
            }

            readSize += thumbnailHeader.compressedSize;
            break;
        } // End if
        else if (block.resourceID == 0x0424)
        {
            if (block.resourceSize == 0) { continue; }

            // xmp データを読み込む
            xmpData.resize(block.resourceSize + 1);
            size_t s = fread(&xmpData[0], 1, block.resourceSize, file);
            if (s == (size_t)block.resourceSize)
            {
                xmpData.back() = '\0';
            } // End if
            else
            {
                xmpData.clear();
            } // End if

            readSize += (int)s;
        } // End if
        else if (block.resourceID == 0x0429)
        {
            // layer comps データを読み込む
            int fpos = ftell(file);
            int savedSize = readSize;
            if (!ReadLayerComps(layerComps, lastApplied, file, &readSize))
            {
                // レイヤーカンプの読み込みに失敗した場合はデータをスキップする
                fseek(file, fpos + block.resourceSize, SEEK_SET);
                layerComps.clear();
                readSize = savedSize + block.resourceSize;
            }
        } // End if
        else
        {
            // skip resource
            fseek(file, block.resourceSize, SEEK_CUR);
            readSize += block.resourceSize;
        } // End if
    } // End while

    // ファイルを閉じる
    fp.reset();

    NW_EXPORT_SETTING setting;
    FileExportSetting fileSetting;
    bool linearFlags[4] = { false, false, false, false };
    std::vector<PSD_LAYER_COMPS> layerCompsData;

    // Xmpにメタデータが存在しない時のために、事前に PSD の出力情報を設定しておく。
    // メタデータが存在する場合はこの情報は上書きされる。
    setting.format = NW_IMG_FMT_RGB8;
    setting.width = header.width;
    setting.height = header.height;
    setting.numMipmaps = 1;

    // サムネイルに表示する画像情報の決定方法
    // PSD を保存するときにレイヤーカンプが選択されていたら (lastApplied != -1)、
    // そのレイヤーカンプの情報をサムネイルに表示する。
    // レイヤーカンプが選択されていない時に、エクスポートプラグインの
    // 「Exports Layer Comp」(fileSetting.exportLayerComp) が true の場合は最初のレイヤーカンプの情報を表示する。
    // false の場合はレイヤーカンプを出力しない時の情報(ReadXmpMetadata関数の第一引数に渡した情報のまま)
    // を表示する。
    bool bReadSetting = false;
    if (!xmpData.empty())
    {
        bReadSetting =
            ReadXmpMetadata(setting, fileSetting, layerCompsData, layerComps, header, &xmpData[0],(int)xmpData.size());
    } // End if

    if (lastApplied >= 0 && lastApplied < (int)layerCompsData.size())
    {    // 最後にレイヤーカンプが選択されている
        setting = layerCompsData[lastApplied].setting;
    } // End if
    else if (fileSetting.exportLayerComp && !layerCompsData.empty())
    {    // 最後にレイヤーカンプが選択されおらず、レイヤーカンプを出力する
        setting = layerCompsData[0].setting;
    }// End if

    // 二リアフラグを更新する
    if (setting.setLinearFlag)
    {
        linearFlags[0] = setting.linearFlag[0];
        linearFlags[1] = setting.linearFlag[1];
        linearFlags[2] = setting.linearFlag[2];
        linearFlags[3] = setting.linearFlag[3];
    } // End if
    else if (NWTextureFormatIsLinear(setting.format))
    {
        // リニアフラグが PSD ファイルから取得できなかった場合は、
        // ピクセルフォーマットに従ってリニアフラグを設定する。
        linearFlags[0] = linearFlags[1] = linearFlags[2] = true;
        linearFlags[3] = false;
    } // End if
    else
    {
        linearFlags[0] = linearFlags[1] = linearFlags[2] = linearFlags[3] = false;
    } // End if

    // PSD ファイルかどうかの設定は SetDescription 内部のプレビュー画像サイズの
    // 決定に影響をあたえるため、SetDescription よりも先に呼び出す必要がある。
    pTexture->SetAsPSDFile(true, bReadSetting);

    if (pTexture->HasPSDExportSetting())
    {
        pTexture->SetShowMipmapCount(true);
        pTexture->SetForNW4F(true);
    }

    if (setting.useMipmap)
    {
        pTexture->SetShowMipmapCount(true);
    }

    if (!thumbnailData.empty())
    {
        // set description
        pTexture->SetDescription(NW_IMG_TYPE_2D,
                                 setting.format,
                                 NW_IMG_FMT_NONE,
                                 setting.width,
                                 setting.height,
                                 thumbnailHeader.width,
                                 thumbnailHeader.height,
                                 0, setting.numMipmaps);

        if (bLoadPreview)
        {
            if (!ReadThumbnailImage(pTexture,
                thumbnailHeader,
                &thumbnailData[0]))
            {
                return false;
            }
        } // End if
    }    // End if
    else
    {
        // サムネイルがロードできなかった場合は description のみ設定する
        pTexture->SetDescription(NW_IMG_TYPE_2D,
                                 setting.format,
                                 NW_IMG_FMT_NONE,
                                 setting.width,
                                 setting.height,
                                 0, setting.numMipmaps);
    } // End else

    if (!layerCompsData.empty())
    {
        pTexture->SetLayerComps(&layerCompsData[0], (int)layerCompsData.size());
    } // End if

    pTexture->SetOutputFolder(fileSetting.outputFolder.c_str());

    if (fileSetting.outputsTga)
    {
        pTexture->SetTgaFolder(fileSetting.tgaFolder.c_str());
    } // End if

    if (fileSetting.outputsCtex)
    {
        pTexture->SetCtexFolder(fileSetting.tgaFolder.c_str());
    } // End if

    pTexture->SetCompSel(setting.compSel);
    pTexture->SetLinearFlags(linearFlags[0], linearFlags[1], linearFlags[2], linearFlags[3], false);
    pTexture->SetHintNormalMap(setting.hintNormalMap);
    if (setting.hintNormalMap)
    {
        pTexture->SetHint(L"normal");
    } // End if
    else if (setting.hint.size()>0)
    {
        pTexture->SetHint(setting.hint.c_str());
    } // End else

    pTexture->SetCompressTypeText(setting.etcEncoding.c_str());

    pTexture->SetComment(setting.comment.c_str());
    pTexture->ComputeImageDataSize();

    return true;
} // End of Load for CNWPSDLoader


//------------------------------------------------------------------------------
_Check_return_
bool CNWPSDLoader::ReadPSDHeader(PSDHeader& header, FILE* file)
{
    char reserved[6];
    return
        readData(header.signature, file) &&
        readData(header.version, file) &&
        readData(reserved, 6, file) &&
        readData(header.numChannels, file) &&
        readData(header.height, file) &&
        readData(header.width, file) &&
        readData(header.depth, file) &&
        readData(header.colorMode, file);
} // End of Load for ReadPSDHeader


//------------------------------------------------------------------------------
_Check_return_
bool CNWPSDLoader::ReadResourceBlock(ImageResourceBlocks& block,
                                     FILE* file,
                                     int* s)
{
    int p = ftell(file);
    // ImageResourceBlocks は 2 バイト境界に配置されている
    if ((p & 1) == 1)
    {
        fseek(file, 1, SEEK_CUR);
        if (s) { *s += 1; }
    } // End if

    if (!readData(block.signature, file, s))
    {
        return false;
    }
    if (!readData(block.resourceID, file, s))
    {
        return false;
    }
    if (!readData(block.nameLen, file, s))
    {
        return false;
    }
    if (block.nameLen > 0)
    {
        if (block.nameLen >= std::size(block.name))
        {
            return false;
        }

        if (!readData(block.name, block.nameLen, file, s))
        {
            return false;
        }

        block.name[block.nameLen] = '\0';
    } // End if
    else
    {
        fseek(file, 1, SEEK_CUR);
        block.name[0] = '\0';
        if (s) { *s += 1; }
    } // End if
    if (!readData(block.resourceSize, file, s))
    {
        return false;
    }

    return true;
} // End of Load for ReadResourceBlock


//------------------------------------------------------------------------------
_Check_return_
bool CNWPSDLoader::ReadThumbnailHeader(ThumbnailResourceHeader& header,
                                       FILE* file,
                                       int* s)
{
    return
        readData(header.format, file, s) &&
        readData(header.width, file, s) &&
        readData(header.height, file, s) &&
        readData(header.widthBytes, file, s) &&
        readData(header.totalSize, file, s) &&
        readData(header.compressedSize, file, s) &&
        readData(header.pixelBits, file, s) &&
        readData(header.numPlanes, file, s);
} // End of Load for ReadThumbnailHeader


//------------------------------------------------------------------------------
_Check_return_
bool CNWPSDLoader::ReadThumbnailImage(CNWTexture *pTexture,
                                      const ThumbnailResourceHeader& header,
                                      const void* thumbnailData)
{
    // read jpg data
    IStream *iStream;
    void* pData;
    std::unique_ptr<void, decltype(&::GlobalFree)> hGlobal(GlobalAlloc(GPTR, header.compressedSize), &::GlobalFree);
    if (hGlobal == NULL)
        return false;

    pData = GlobalLock(hGlobal.get());
    if (pData==NULL)
        return false;

    memcpy(pData, thumbnailData, header.compressedSize);
    GlobalUnlock(hGlobal.get());

    if (CreateStreamOnHGlobal(hGlobal.get(), TRUE, &iStream) != S_OK)
    {
        return false;
    }

    // 所有権を開放する
    hGlobal.release();

    auto deleter = [](IStream* x) { x->Release(); };
    std::unique_ptr<IStream, decltype(deleter)> pS(iStream, deleter);

    // create bitmap
    std::unique_ptr<Gdiplus::Bitmap> psdBitmap(
        Gdiplus::Bitmap::FromStream(iStream));
    if (psdBitmap.get() == NULL)
    {
        return false;
    } // End if

    // Lock bits
    Gdiplus::Rect srcLockRect(0, 0, psdBitmap->GetWidth(), psdBitmap->GetHeight());
    Gdiplus::BitmapData psdData;
    if (psdBitmap->LockBits(&srcLockRect,Gdiplus::ImageLockModeRead,
        psdBitmap->GetPixelFormat(),
        &psdData)!=Gdiplus::Ok)
    {
        return false;
    } // End if

    // Create Texture
    if (pTexture->CreatePreviewImage()==false)
    {
        // psdBitmap Finish locking
        psdBitmap->UnlockBits(&psdData);
        return false;
    }

    // サムネイルの Bitmap にコピする
    Gdiplus::Bitmap *pPreviewBitmap = pTexture->GetPreviewBitmap();
    if (pPreviewBitmap==NULL)
    {
        // psdBitmap Finish locking
        psdBitmap->UnlockBits(&psdData);
        return false;
    }

    // Lock bits to prepare preview image
    Gdiplus::Rect lockRect(0,0,pPreviewBitmap->GetWidth(),pPreviewBitmap->GetHeight());
    Gdiplus::BitmapData bmpData;
    if (pPreviewBitmap->LockBits(&lockRect,Gdiplus::ImageLockModeWrite,
                                 pPreviewBitmap->GetPixelFormat(),
                                 &bmpData)!=Gdiplus::Ok)
    {
        // psdBitmap Finish locking
        psdBitmap->UnlockBits(&psdData);
        return false;
    } // End if

    // Read pixels
    unsigned char *pSrcBuffer = (unsigned char*)psdData.Scan0;
    unsigned char *pDestBuffer = (unsigned char*)bmpData.Scan0;

    unsigned int wd = bmpData.Width;
    unsigned int ht = bmpData.Height;

    unsigned int xStart = (wd - psdData.Width) / 2;
    unsigned int yStart = (ht - psdData.Height) / 2;
    for (unsigned int y = 0; y < psdData.Height; y++)
    {
        unsigned char *pSrcPixel  = &pSrcBuffer[y * psdData.Stride];
        unsigned char *pDestPixel = &pDestBuffer[(yStart + y) * bmpData.Stride];
        for (unsigned int x = 0; x < psdData.Width; x++)
        {
            unsigned char* pSrcPixel2 = &pSrcPixel[x * 3];
            unsigned char* pDestPixel2 = &pDestPixel[(xStart + x) * 4];
            pDestPixel2[0] = pSrcPixel2[0];
            pDestPixel2[1] = pSrcPixel2[1];
            pDestPixel2[2] = pSrcPixel2[2];
            pDestPixel2[3] = 0xFF;
        } // End for
    } // End for

    // psdBitmap Finish locking
    psdBitmap->UnlockBits(&psdData);

    Gdiplus::Rect drawRect(xStart, yStart, psdData.Width, psdData.Height);

    // Finalize
    pTexture->CreateColorMap(bmpData, ORIGINAL_FORMAT_NOT_USED, drawRect);
    if (pTexture->HasAlpha())
    {
        pTexture->CreateAlphaMap(bmpData, ORIGINAL_FORMAT_NOT_USED, drawRect);
    } // End if

    // pPreviewBitmap Finish locking
    pPreviewBitmap->UnlockBits(&bmpData);

    return true;
} // End of Load for ReadThumbnail

//------------------------------------------------------------------------------
// Read XML section to buffer for parsing
//------------------------------------------------------------------------------
_Check_return_
bool CNWPSDLoader::ReadXMLSectionToBuffer( const char *pRawBuffer,
                                           int iBufferSize,
                                           VARIANT *pXMLBuffer )
{
    const char *pSrc = pRawBuffer;

    int iSize = 0;

    // Skip the BOM bytes
    if ( pSrc[0]==0xEF &&
         pSrc[1]==0xBB &&
         pSrc[2]==0xBF )
    {
        iSize += 3;
        pSrc  += 3;
    } // End if

    // Count the buffer size
    while ( pSrc<(pRawBuffer + iBufferSize) )
    {
        // Is this byte NULL?
        if ( *pSrc==0x00 )
        {
            break;
        } // End if

        if ( (*pSrc & 0x80)==0x00 ) // 0x0xxxxxxx : 1 Byte
        {
            ++iSize;
            ++pSrc;
        } // End if
        else if ( (*pSrc & 0xE0)==0xC0 ) // 0x110xxxxx : 2 Bytes
        {
            iSize += 2;
            pSrc  += 2;
        } // End else if
        else if ( (*pSrc & 0xF0)==0xE0 ) // 0x1110xxxx : 3 Bytes
        {
            iSize += 3;
            pSrc  += 3;
        } // End else if
        else if ( (*pSrc & 0xF8)==0xF0 ) // 0x11110xxx : 4 Bytes
        {
            iSize += 4;
            pSrc  += 4;
        } // End else if
        else if ( (*pSrc & 0xFC)==0xF8 ) // 0x111110xx : 5 Bytes
        {
            iSize += 5;
            pSrc  += 5;
        } // End else if
        else if ( (*pSrc & 0xFE)==0xFC ) // 0x1111110x : 6 Bytes
        {
            iSize += 6;
            pSrc  += 6;
        } // End else if
        else
        {
            // Invalid file
            return false;
        } // End else
    }

    if ( iSize<=0 )
        return false;

    SAFEARRAYBOUND bound;
    bound.cElements = iSize;
    bound.lLbound   = 0;

    // Create the buffer.
    pXMLBuffer->vt     = VT_ARRAY | VT_UI1;
    pXMLBuffer->parray = SafeArrayCreate( VT_UI1, 1, &bound );

    unsigned char *pBuffer = NULL;
    SafeArrayAccessData( pXMLBuffer->parray, (void**)&pBuffer );

    memcpy_s( pBuffer, iSize, pRawBuffer, iSize );

    SafeArrayUnaccessData( pXMLBuffer->parray );

    return true;
} // End of ReadXMLSectionToBuffer for CNWPSDLoader

//------------------------------------------------------------------------------
_Check_return_
bool CNWPSDLoader::ReadXmpMetadata(NW_EXPORT_SETTING& setting,
                                   FileExportSetting& fileExportSetting,
                                   std::vector<PSD_LAYER_COMPS>& layerCompsData,
                                   const std::list<std::wstring>& layerComps,
                                   const PSDHeader& psdHeader,
                                   const char* xmpData,
                                   int xmpDataSize)
{
    fileExportSetting.exportLayerComp = false;

    VARIANT xmlChunk;
    if (ReadXMLSectionToBuffer(xmpData,xmpDataSize,&xmlChunk) == false)
    {
        return false;
    } // End if

    auto destroy = [](SAFEARRAY* p) { SafeArrayDestroy(p); };
    std::unique_ptr<SAFEARRAY, decltype(destroy)> p(xmlChunk.parray, destroy);

    //--------------------------------------------------------------------------
    // Create document
    //--------------------------------------------------------------------------
    MSXML2::IXMLDOMDocument2Ptr pDoc;
    HRESULT hr = pDoc.CreateInstance(__uuidof(MSXML2::DOMDocument30));
    if (FAILED(hr))
    {
        return false;
    } // End if

    pDoc->async = VARIANT_FALSE; // (default = TRUE)
    pDoc->validateOnParse = VARIANT_FALSE; // disable schema check (default = TRUE)
    pDoc->resolveExternals = VARIANT_FALSE; // (default = TRUE)
    VARIANT_BOOL result = pDoc->load(xmlChunk);
    if (result != VARIANT_TRUE)
    {
        return false;
    } // End if

    MSXML2::IXMLDOMNodePtr settingNode = NULL;
    MSXML2::IXMLDOMNodePtr xmpMetaNode = NWXMLFindChildNode(pDoc,"x:xmpmeta");
    if( xmpMetaNode == NULL)
    {
        return false;
    }

    MSXML2::IXMLDOMNodePtr rdfNode = NWXMLFindChildNode(xmpMetaNode,"rdf:RDF");
    if( rdfNode == NULL)
    {
        return false;
    }

    MSXML2::IXMLDOMNodeListPtr rdfChildren = rdfNode->childNodes;
    for( MSXML2::IXMLDOMNodePtr rdfChild = rdfChildren->nextNode();
        rdfChild != NULL;
        rdfChild = rdfChildren->nextNode())
    {
        if (strcmp(rdfChild->nodeName, "rdf:Description") != 0)
            continue;

        MSXML2::IXMLDOMNodeListPtr children = rdfChild->childNodes;
        for (MSXML2::IXMLDOMNodePtr child = children->nextNode();
             child != NULL;
             child = children->nextNode())
        {
            if (strcmp(child->nodeName, "dc:subject") == 0)
            {
                MSXML2::IXMLDOMNodePtr bagNode = NWXMLFindChildNode(child,"rdf:Bag");
                if(bagNode != NULL)
                {
                    settingNode = bagNode;
                    rdfChild = NULL;    // キーワードがあれば検索を打ち切る
                    break;
                } //End if
            } // End if
            else if (strcmp(child->nodeName, "photoshop:SupplementalCategories") == 0)
            {
                // 追加のカテゴリに設定を発見してもキーワードが存在していたら
                // そちらを優先するため検索を続ける
                if (settingNode == NULL)
                {
                    settingNode = NWXMLFindChildNode(child,"rdf:Bag");
                } // End if
            } // End if
        } // End for
    }

    // 設定を読み込む
    if (settingNode != NULL)
    {
        //NW4Fの設定が出力されていたかどうか
        bool bExportSetting = false;

        //定義されているパラメータを列挙する。
        MSXML2::IXMLDOMNodeListPtr nodeList = settingNode->childNodes;
        for(MSXML2::IXMLDOMNodePtr node = nodeList->nextNode();
            node != NULL;
            node = nodeList->nextNode())
        {
            // 念のためにタグ名を確認する。
            if (strcmp(node->nodeName, "rdf:li") != 0)
            {
                continue;
            }

            // ※ ファイル中には UTF-8 で格納されているが MSXML がマルチバイトストリングに変換している。
            const std::wstring textValue(node->text);
            std::wstring::size_type eqPos = textValue.find_first_of('=');
            if (eqPos == std::wstring::npos)
                continue;

            std::wstring value = textValue.substr(eqPos + 1);
            std::wstring paramStr = textValue.substr(0, eqPos);

            std::wstring paramName; // レイヤーカンプ名を除いた純粋な出力設定のパラメータ名
            NW_EXPORT_SETTING* exportSetting = NULL; // 設定を記録する構造体のポインタ

            // NW4R の出力設定が存在すると photoshop:SupplementalCategories データが出力される。
            // そのときに NW4F のパラメータが存在していると誤認識しないように NW4F のパラメータが
            // 存在していたか確認する。
            if (paramStr.find(L"nw4f_") != std::wstring::npos)
            {
                bExportSetting = true;
            }

            // 出力設定のパラメータ名の後ろに '_'に続いてレイヤーカンプ名が指定されているか確認する
            std::wstring::size_type ubPos = paramStr.find_first_of(L'_', 5);
            if (ubPos != std::wstring::npos)
            {
                // レイヤーカンプ名の前のパラメータ名のみを抽出する
                paramName = paramStr.substr(0, ubPos);

                // レイヤーカンプが指定されていたら、対応するレイヤーカンプのデータを検索する
                std::wstring layerCompsName = paramStr.substr(ubPos + 1);
                std::list<std::wstring>::const_iterator itComps =
                    std::find(layerComps.begin(), layerComps.end(), layerCompsName);
                if (itComps != layerComps.end())
                {
                    struct LayerCompsComparer
                    {
                        const std::wstring& m_Name;
                        LayerCompsComparer(const std::wstring& name) : m_Name(name) {}
                        bool operator () (const PSD_LAYER_COMPS& data)
                        {
                            return data.name == m_Name;
                        } // End operator ()
                    }; // End struct LayerCompsComparer

                    std::vector<PSD_LAYER_COMPS>::iterator itData = std::find_if(
                        layerCompsData.begin(),
                        layerCompsData.end(),
                        LayerCompsComparer(layerCompsName));
                    if (itData == layerCompsData.end())
                    {
                        itData = layerCompsData.insert(layerCompsData.end(), PSD_LAYER_COMPS());
                        itData->name = layerCompsName;
                    } // End if

                    exportSetting = &(itData->setting);
                } // End if
                else
                {
                    continue;
                } // End if
            } // End if
            else
            {
                paramName = paramStr;
                exportSetting = &setting;
            } // End if

            std::function<void(const std::wstring&, const std::wstring& value)>
                apply = [&](const std::wstring& paramName, const std::wstring& value)
            {
                //------------------------------------------------------------------
                // NW4F
                //------------------------------------------------------------------
                if (paramName == L"nw4f_CompOpts")
                {
                    // 古い仕様に従った形式に変換して読み込む
                    size_t pos = 0;
                    while (pos < value.size())
                    {
                        auto subParamNameEnd = value.find('=', pos);
                        if (subParamNameEnd == std::wstring::npos)
                        {
                            break;
                        }

                        auto subValueStart = subParamNameEnd + 1;
                        auto subValueEnd = value.find('?', subValueStart);
                        if (subValueEnd == std::wstring::npos)
                        {
                            subValueEnd = value.size();
                        }

                        auto subParamName = value.substr(pos, subParamNameEnd - pos);
                        auto subValue = value.substr(subValueStart, subValueEnd - subValueStart);

                        // エスケープされた ? を元に戻します。
                        auto key = L"&#x3f!";
                        auto keyLength = wcslen(key);
                        while (true)
                        {
                            auto replacePos = subValue.find(key);
                            if (replacePos != std::wstring::npos)
                            {
                                subValue.replace(replacePos, keyLength, L"?");
                            }
                            else
                            {
                                break;
                            }
                        }

                        apply(L"nw4f_" + subParamName, subValue);

                        pos = subValueEnd + 1;
                    }
                }
                else if (paramName == L"nw4f_ExportsLayerComp")
                {
                    fileExportSetting.exportLayerComp = _wcsnicmp(value.c_str(), L"false", 5) != 0;
                } // End if
                else if (paramName == L"nw4f_OutputFolder")
                {
                    fileExportSetting.outputFolder = value.c_str();
                } // End if
                else if (paramName == L"nw4f_OutputsTga")
                {
                    fileExportSetting.outputsTga = _wcsnicmp(value.c_str(), L"false", 5) != 0;
                } // End if
                else if (paramName == L"nw4f_TgaFolder")
                {
                    fileExportSetting.tgaFolder = value.c_str();
                } // End if
                else if (paramName == L"nw4f_ResizeMode")
                {
                    if (value == L"1/2")
                    {
                        exportSetting->width = psdHeader.width / 2;
                        exportSetting->height = psdHeader.height / 2;
                    } // End if
                    else if (value == L"1/4")
                    {
                        exportSetting->width = psdHeader.width / 4;
                        exportSetting->height = psdHeader.height / 4;
                    } // End if
                    else
                    {
                        exportSetting->width = psdHeader.width;
                        exportSetting->height = psdHeader.height;
                    } // End if
                }
                else if (paramName == L"nw4f_ResizeW")
                {
                    exportSetting->width = _wtoi(value.c_str());
                } // End if
                else if (paramName == L"nw4f_ResizeH")
                {
                    exportSetting->height = _wtoi(value.c_str());
                } // End if
                else if (paramName == L"nw4f_ResizePercentW")
                {
                    float ratio = (float)_wtoi(value.c_str()) / 100.0f;
                    if (ratio > 0.0f)
                    {
                        exportSetting->width = (int)((float)psdHeader.width * ratio);
                    }
                } // End if
                else if (paramName == L"nw4f_ResizePercentH")
                {
                    float ratio = (float)_wtoi(value.c_str()) / 100.0f;
                    if (ratio > 0.0f)
                    {
                        exportSetting->height = (int)((float)psdHeader.height * ratio);
                    }
                } // End if

                else if (paramName == L"nw4f_Format")
                {
                    exportSetting->format = NWGetTextureFormatFromString(NWWStringToString(value).c_str());
                } // End if
                else if (paramName == L"nw4f_Mipmap")
                {
                    exportSetting->useMipmap = _wcsnicmp(value.c_str(), L"false", 5) != 0;
                } // End if
                else if (paramName == L"nw4f_CompSel")
                {
                    exportSetting->compSel = NWStringToCompSel(NWWStringToString(value).c_str());
                } // End if
                else if (paramName == L"nw4f_Hint")
                {
                    exportSetting->hintNormalMap = (value == L"normal");
                    exportSetting->hint = value;
                } // End if
                else if (paramName == L"nw4f_LinearFlag")
                {
                    exportSetting->setLinearFlag = true;
                    for (int i = 0; i < 4; i++)
                    {
                        exportSetting->linearFlag[i] = (value[i] != '0');
                    }
                } // End if
                else if (paramName == L"nw4f_CommentText")
                {
                    exportSetting->comment = value;
                } // End if

                //------------------------------------------------------------------
                // NW4C
                //------------------------------------------------------------------
                else if (paramName == L"nw4c_ExportsLayerComp")
                {
                    fileExportSetting.exportLayerComp = _wcsnicmp(value.c_str(), L"false", 5) != 0;
                } // End if
                else if (paramName == L"nw4c_OutputFolder")
                {
                    fileExportSetting.outputFolder = value;
                } // End if
                else if (paramName == L"nw4c_OutputsTga")
                {
                    fileExportSetting.outputsTga = _wcsnicmp(value.c_str(), L"false", 5) != 0;
                } // End if
                else if (paramName == L"nw4c_TgaFolder")
                {
                    fileExportSetting.tgaFolder = value.c_str();
                } // End if
                else if (paramName == L"nw4c_OutputsCtex")
                {
                    fileExportSetting.outputsCtex = _wcsnicmp(value.c_str(), L"false", 5) != 0;
                } // End if
                else if (paramName == L"nw4c_CtexFolder")
                {
                    fileExportSetting.ctexFolder = value.c_str();
                } // End if
                else if (paramName == L"nw4c_Format")
                {
                    exportSetting->format = NWGetTextureFormatFromString(NWWStringToString(value).c_str());
                } // End if
                else if (paramName == L"nw4c_Mipmap")
                {
                    exportSetting->useMipmap = _wcsnicmp(value.c_str(), L"false", 5) != 0;
                } // End if
                else if (paramName == L"nw4c_GenMip")
                {
                    exportSetting->useMipmap = true;
                } // End if
                else if (paramName == L"nw4c_ResizeW")
                {
                    exportSetting->width = _wtoi(value.c_str());
                } // End if
                else if (paramName == L"nw4c_ResizeH")
                {
                    exportSetting->height = _wtoi(value.c_str());
                } // End if
                else if (paramName == L"nw4c_ResizeMode")
                {
                    if (value == L"1/2")
                    {
                        exportSetting->width = psdHeader.width / 2;
                        exportSetting->height = psdHeader.height / 2;
                    } // End if
                    else if (value == L"1/4")
                    {
                        exportSetting->width = psdHeader.width / 4;
                        exportSetting->height = psdHeader.height / 4;
                    } // End if
                    else
                    {
                        exportSetting->width = psdHeader.width;
                        exportSetting->height = psdHeader.height;
                    } // End if
                }
                else if (paramName == L"nw4c_ResizePercentW")
                {
                    float ratio = (float)_wtoi(value.c_str()) / 100.0f;
                    if (ratio > 0.0f)
                    {
                        exportSetting->width = (int)((float)psdHeader.width * ratio);
                    }
                } // End if
                else if (paramName == L"nw4c_ResizePercentH")
                {
                    float ratio = (float)_wtoi(value.c_str()) / 100.0f;
                    if (ratio > 0.0f)
                    {
                        exportSetting->height = (int)((float)psdHeader.height * ratio);
                    }
                } // End if
                else if (paramName == L"nw4c_EtcEncoding")
                {
                    exportSetting->etcEncoding = value;
                } // End if
            };

            apply(paramName, value);
        } // End for

        // ミップマップレベルを計算する
        setting.CalculateMipmapLevel();
        for (std::vector<PSD_LAYER_COMPS>::iterator itComps = layerCompsData.begin();
             itComps != layerCompsData.end(); ++itComps)
        {
            itComps->setting.CalculateMipmapLevel();
        } // End for

        return bExportSetting;
    }
    else
    {
        return false;
    } // End if

} // End of Load for ReadXmpMetadata

//------------------------------------------------------------------------------
_Check_return_
bool CNWPSDLoader::ReadLayerComps(std::list<std::wstring>& layerComps,
                                  int& lastApplied, FILE* file, int* s)
{
    lastApplied = -1;

    int version;
    if (!readData(version, file, s))
    {
        return false;
    }
    if (version != 16)
        return false;

    int lastAppliedLayerCompID = 0;
    std::vector<unsigned int> classIDs;

    std::wstring classIDName;
    if (!readUnicodeString(classIDName, file, s))
    {
        return false;
    }

    std::vector<char> classID;
    if (!readObjectID(classID, file, s))
    {
        return false;
    }

    int numItems;
    if (!readData(numItems, file, s))
    {
        return false;
    }

    for (int iItem = 0; iItem < numItems; iItem++)
    {
        std::vector<char> compKey;
        if (!readObjectID(compKey, file, s))
        {
            return false;
        }

        unsigned int OSType;
        if (!readData(OSType, file, s))
        {
            return false;
        }

        if (OSType == 'long')
        {
            if (memcmp(&compKey[0], "lastAppliedComp", 15/*15 == strlen("lastAppliedComp")*/) == 0)
            {
                if (!readData(lastAppliedLayerCompID, file, s))
                {
                    return false;
                }
                continue;
            } // End if
            else
            {
                if (!readDescription(OSType, file, s))
                {
                    return false;
                }
                continue;
            } // End if
        } // End if
        else if (OSType != 'VlLs')
        {
            if (!readDescription(OSType, file, s))
            {
                return false;
            }

            continue;
        } // End if

        int numData;
        if (!readData(numData, file, s))
        {
            return false;
        }
        for (int iData = 0; iData < numData; iData++)
        {
            if (!readData(OSType, file, s))
            {
                return false;
            }

            if (OSType != 'Objc' && OSType != 'GlbO')
            {
                if (!readDescription(OSType, file, s))
                {
                    return false;
                }

                continue;
            } // End if

            std::wstring name;
            if (!readUnicodeString(name, file, s))
            {
                return false;
            }

            std::vector<char> id;
            if (!readObjectID(id, file, s))
            {
                return false;
            }

            if (memcmp(&id[0], "Comp", 4) != 0)
                return false;

            int num;
            if (!readData(num, file, s))
            {
                return false;
            }

            for (int iComp = 0; iComp < num; iComp++)
            {
                std::vector<char> key;
                if (!readObjectID(key, file, s))
                {
                    return false;
                }

                if (!readData(OSType, file, s))
                {
                    return false;
                }

                if (OSType == 'TEXT')
                {
                    std::wstring nm;
                    if (!readUnicodeString(nm, file, s))
                    {
                        return false;
                    }
                    if (memcmp(&key[0], "Nm  ", 4) == 0)
                    {
                        layerComps.push_back(nm);
                    } // End if
                } // End if
                else if (OSType == 'long')
                {
                    int integer;
                    if (!readData(integer, file, s))
                    {
                        return false;
                    }
                    if (memcmp(&key[0], "compID", 6) == 0)
                    {
                        classIDs.push_back((unsigned int)integer);
                    } // End if
                } // End if
                else if (OSType == 'bool')
                {
                    bool boolean;
                    if (!readData(boolean, file, s))
                    {
                        return false;
                    }
                } // End if
                else
                {
                    if (!readDescription(OSType, file, s))
                    {
                        return false;
                    } // End if
                } // End if
            } // End for
        } // End for
    } // End for

    // レイヤーカンプ名の数とクラス ID の数が一致するときのみ
    // 最後に適用されたレイヤーカンプのインデックスを設定する
    if (layerComps.size() == classIDs.size() && lastAppliedLayerCompID != 0)
    {
        std::vector<unsigned int>::iterator itFoundID =
            std::find(classIDs.begin(), classIDs.end(), lastAppliedLayerCompID);
        if (itFoundID != classIDs.end())
        {
            lastApplied = (int)std::distance(classIDs.begin(), itFoundID);
        } // End if
    } // End if

    return true;
} // End of Load for ReadLayerComps



