﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using nw.g3d.iflib;
using nw.g3d.toollib;
using nw.g3d.nw4f_3dif;
using System.Windows.Forms;
using System.Diagnostics;
using System.Xml;
using System.Reflection;

namespace nw.g3d.bifedit
{
    /// <summary>
    /// Document
    /// </summary>
    public class Document : IDisposable
    {
        /// <summary>
        /// Constructor
        /// </summary>
        public Document()
        {
        }

        /// <summary>
        /// Dispose
        /// </summary>
        public void Dispose()
        {
            using (new MethodCallDump("Dispose"))
            {
                Cleanup();
            }
        } // End of Dispose

        #region Document
        /// <summary>
        /// Load from file
        /// </summary>
        /// <param name="path">Path to intermediate file</param>
        /// <returns>true on success</returns>
        public bool LoadFromFile(String path, out bool isIFtextFile )
        {
            isIFtextFile = false;

            using (var mcd = new MethodCallDump("LoadFromFile", "path: '{0}'", path))
            {
                m_filePath = path;

                try
                {
                    // Is this ASCII?
                    if (nw.g3d.nw4f_3dif.G3dPath.IsTextPath(path))
                    {
                        mcd.Log("inside G3dPath.IsTextPath(path: '{0}')", path);
                        isIFtextFile = true;
                        return mcd.LogReturnValue(true);

                        /*
                        m_xmlDocument = new XmlDocument();
                        m_xmlDocument.Load(path);

                        // ドキュメントから stream を取り出す
                        m_binaryStreams = new List<G3dStream>();
                        nw.g3d.nw4f_3dif.G3dStreamUtility.GetStreams(m_binaryStreams, m_xmlDocument);
                        */
                    }
                    else
                    {
                        mcd.Log("File.ReadAllBytes(path: '{0}')", path);
                        byte[] data = System.IO.File.ReadAllBytes(path);
                        mcd.Log("    OK, read {0} bytes", data.Length);

                        mcd.Log("IfBinaryReadUtility.Separate(data, out m_textImage, out m_binaryImage)");
                        // テキスト部とバイナリ部の分割
                        nw.g3d.iflib.IfBinaryReadUtility.Separate(data, out m_textImage, out m_binaryImage);
                        mcd.Log("    OK");
                        mcd.Log("        m_textImage: {0} bytes", m_textImage.Length);
                        mcd.Log("        m_binaryImage: {0} bytes", m_binaryImage == null ? 0 : m_binaryImage.Length);
                    }
                }

                catch (System.Exception ex)
                {
                    mcd.LogException(ex);

                    Debug.Print(ex.Message);
                    return mcd.LogReturnValue(false);
                }

                return mcd.LogReturnValue(true);
            }
        }

        /*
        /// <summary>
        /// XML 中間ファイルにストリーム列を設定します。
        /// </summary>
        /// <param name="document">設定先の XML 中間ファイルです。</param>
        /// <param name="streams">設定するストリーム列です。</param>
        private void SetStreamsToXMLDocument( XmlDocument document, List<G3dStream> streams )
        {
            if (streams.Count == 0) { return; }

            XmlNode node = document.SelectSingleNode("/nw4f_3dif/*[last()]/*[last()]");
            XmlElement element = node as XmlElement;
            Debug.Assert(element != null);

            if (element.Name == "user_tool_data")
            {
                element = element.PreviousSibling as XmlElement;
            }
            if (element.Name == "tool_data")
            {
                element = element.PreviousSibling as XmlElement;
            }

            XmlElement stream_array = document.CreateElement("stream_array");
            stream_array.SetAttribute("length", streams.Count.ToString());
            for (int i = 0; i < streams.Count; i++)
            {
                stream_array.AppendChild(streams[i].GetElement(document, i));
            }

            element.ParentNode.InsertAfter(stream_array, element);
        }
        */

        private class CaseInsensitiveStringEqualityComparer : IEqualityComparer<string>
        {
            public bool Equals(string x, string y)
            {
                return string.Equals(x, y, StringComparison.OrdinalIgnoreCase);
            }

            public int GetHashCode(string obj)
            {
                return obj != null ? obj.GetHashCode() : 0;
            }
        }

        private HashSet<string> eventedFiles = new HashSet<string>(new CaseInsensitiveStringEqualityComparer());

        /// <summary>
        /// Save to temporary file
        /// </summary>
        /// <returns></returns>
        public bool SaveToTempFile()
        {
            using (var mcd = new MethodCallDump("SaveToTempFile"))
            {

                //if (m_xmlDocument == null)
                //    return false;

                String tempDir = System.IO.Path.GetTempPath();
                String tempFileName = System.IO.Path.GetFileNameWithoutExtension(System.IO.Path.GetTempFileName());
                String thisFileName = System.IO.Path.GetFileNameWithoutExtension(m_filePath);
                String thisFileExt = System.IO.Path.GetExtension(m_filePath);
                String fileNameToWatch = thisFileName + "_" + tempFileName + thisFileExt;
                fileNameToWatch = G3dPath.ToTextPath(fileNameToWatch);

                m_tempFilePath = tempDir + "\\" + fileNameToWatch;
                mcd.Log("Creation of m_tempFilePath (tempDir + \"\\\" + fileNameToWatch)");
                mcd.Log("    tempDir: '{0}'", tempDir);
                mcd.Log("    fileNameToWatch: '{0}'", fileNameToWatch);
                mcd.Log("    m_tempFilePath: '{0}'", m_tempFilePath);

                try
                {
                    /*
                    String xmlStr = Utility.ConvertXMLDocumentToString(m_xmlDocument);

                    System.IO.StringReader rd = new System.IO.StringReader(xmlStr);
                    System.IO.StreamWriter wt = new System.IO.StreamWriter(m_tempFilePath);

                    nw.g3d.iflib.IfTextFormatter.Format(rd, wt);
                    wt.Flush();
                    */

                    mcd.Log("File.WriteAllBytes(m_tempFilePath: '{0}', m_textImage: {1} bytes)", m_tempFilePath, m_textImage.Length);
                    System.IO.File.WriteAllBytes(m_tempFilePath, m_textImage);
                    System.IO.File.SetAttributes(m_tempFilePath, System.IO.File.GetAttributes(m_filePath));
                    mcd.Log("    OK");
                }
                catch (System.Exception ex)
                {
                    mcd.LogException(ex);
                    Debug.Print(ex.Message);
                }

                // Add modification watcher
                m_fileWatcher = new System.IO.FileSystemWatcher();
                m_fileWatcher.Path = tempDir;
                m_fileWatcher.Filter = fileNameToWatch;
                m_fileWatcher.IncludeSubdirectories = false;
                m_fileWatcher.Changed += new System.IO.FileSystemEventHandler(OnFileChange);
                m_fileWatcher.EnableRaisingEvents = true;
                mcd.Log("FileWatcher creation OK");

                return mcd.LogReturnValue(true);
            }
        }

        /// <summary>
        /// XSD File path
        /// </summary>
        public String XSDFilePath
        {
            set { m_xsdFilePath = value; }
            get { return m_xsdFilePath; }
        }
        private String m_xsdFilePath = "";

        /// <summary>
        /// File path
        /// </summary>
        public String FilePath
        {
            get { return m_filePath; }
        }
        private String          m_filePath = "";

        /// <summary>
        /// Temp File path
        /// </summary>
        public String TempFilePath
        {
            get { return m_tempFilePath; }
        }
        private String m_tempFilePath = "";

        /*
        /// <summary>
        /// XML document
        /// </summary>
        public XmlDocument XMLDoc
        {
            get { return m_xmlDocument; }
        }
        */
        private XmlDocument     m_xmlDocument;

        private byte[] m_textImage;
        private byte[] m_binaryImage;

        /// <summary>
        /// Binary streams
        /// </summary>
        public List<nw.g3d.nw4f_3dif.G3dStream> BinaryStreams
        {
            get { return m_binaryStreams; }
        }
        private List<nw.g3d.nw4f_3dif.G3dStream> m_binaryStreams;

        /// <summary>
        /// File watcher
        /// </summary>
        private System.IO.FileSystemWatcher m_fileWatcher;

        /// <summary>
        /// File watcher event handler
        /// </summary>
        private void OnFileChange(object source, System.IO.FileSystemEventArgs e)
        {
            using (var mcd = new MethodCallDump("OnFileChange", "e.FullPath: '{0}'", e.FullPath))
            {
                if (eventedFiles.Add(e.FullPath)) // double call protection
                {
                    mcd.Log("Skipped by double call protection");
                    return;
                }

                /*
                // We want to make sure that we can get the exclusive rights
                System.IO.Stream stream = null;
                try
                {
                    stream = System.IO.File.Open(m_tempFilePath,System.IO.FileMode.Open,
                                                 System.IO.FileAccess.Read,
                                                 System.IO.FileShare.Read);
                }
                catch (System.Exception ex)
                {
                    Debug.Print(ex.Message);
                    return;
                }

                // Try to read file
                XmlDocument xmlDoc = new XmlDocument();
                try
                {
                    xmlDoc.Load(stream);
                }
                catch (System.Exception ex)
                {
                    Debug.Print(ex.Message);
                    return;
                }

                // ドキュメントから stream を取り出す
                m_xmlDocument   = xmlDoc;
                m_binaryStreams = new List<G3dStream>();
                nw.g3d.nw4f_3dif.G3dStreamUtility.GetStreams(m_binaryStreams, m_xmlDocument);
                */

                // Save to original file
                SaveBackToOriginalFile();

                bool removed = eventedFiles.Remove(e.FullPath);
                mcd.Log("eventedFiles.Remove(e.FullPath: '{0}') -> {1}", e.FullPath, removed);
            }
        }

        /// <summary>
        /// Save back to original file
        /// </summary>
        private bool SaveBackToOriginalFile()
        {
            using (var mcd = new MethodCallDump("SaveBackToOriginalFile"))
            {
                if (m_filePath.Length <= 0)
                {
                    mcd.Log("m_filePath.Length <= 0, return false (m_filePath: '{0}')", m_filePath);
                    return mcd.LogReturnValue(false);
                }

                if (m_tempFilePath.Length <= 0)
                {
                    mcd.Log("m_tempFilePath.Length <= 0, return false (m_tempFilePath: '{0}')", m_tempFilePath);
                    return mcd.LogReturnValue(false);
                }

                //if (m_xmlDocument==null)
                //    return false;

                if (nw.g3d.nw4f_3dif.G3dPath.IsTextPath(m_filePath))
                {
                    mcd.Log("inside G3dPath.IsTextPath(m_filePath: '{0}') block", m_filePath);

                    // If ascii, just save normally
                    try
                    {
                        m_xmlDocument.Save(m_filePath);
                    }
                    catch (System.Exception ex)
                    {
                        Debug.Print(ex.Message);
                        return false;
                    }
                }
                else
                {
                    try
                    {
                        // check XML validity
                        try
                        {
                            mcd.Log("IfTextReadUtility.Read(m_tempFilePath: '{0}', m_xsdFilePath: '{1}')", m_tempFilePath, m_xsdFilePath);
                            IfTextReadUtility.Read(m_tempFilePath, m_xsdFilePath);
                            mcd.Log("    OK");
                        }
                        catch (Exception ex)
                        {
                            mcd.Log("Trying to show dialog");
                            CustomMessageBox.Show(ex.Message, Properties.Resources.NW4F_FORMAT_ERROR,
                                MessageBoxButtons.OK, MessageBoxIcon.Error);
                            mcd.LogException(ex);
                            mcd.Log("End Trying to show dialog");
                            return mcd.LogReturnValue(false);
                        }

                        byte[] newTextImage = System.IO.File.ReadAllBytes(m_tempFilePath);
                        mcd.Log("File.ReadAllBytes(m_tempFilePath: '{0}')", m_tempFilePath);

                        mcd.Log("new BinaryWriter(new FileStream(m_filePath: '{0}')) for write", m_filePath);
                        using (var writer = new System.IO.BinaryWriter(new System.IO.FileStream(m_filePath, System.IO.FileMode.OpenOrCreate, System.IO.FileAccess.Write, System.IO.FileShare.ReadWrite)))
                        {
                            mcd.Log("Write ascii part ({0} bytes)", newTextImage.Length);
                            writer.Write(newTextImage, 0, newTextImage.Length);
                            mcd.Log("    OK");

                            mcd.Log("Write separating zero");
                            writer.Write((byte)0); // ensure at least one separating zero
                            mcd.Log("    OK");

                            mcd.Log("Write padding zeros");
                            IfBinaryFormatter.WritePadding(writer);
                            mcd.Log("    OK");

                            var streams = new List<G3dStream>();
                            mcd.Log("Convert binary data to streams");
                            IfBinaryReadUtility.ReadStreamArray(streams, m_binaryImage);
                            mcd.Log("    OK");

                            mcd.Log("Write binary part (streams)");
                            writer.Write(IfBinaryWriteUtility.WriteStreamArray(streams));
                            mcd.Log("    OK");
                        }
                    }
                    catch (Exception ex)
                    {
                        mcd.LogException(ex);

                        Debug.Print(ex.Message);
                        return mcd.LogReturnValue(false);
                    }
                }

                return mcd.LogReturnValue(true);
            }
        }

        /// <summary>
        /// Clean up
        /// </summary>
        public void Cleanup()
        {
            using (var mcd = new MethodCallDump("Cleanup"))
            {
                if (m_fileWatcher != null)
                {
                    m_fileWatcher.Dispose();
                    m_fileWatcher = null;
                }

                if (m_xmlDocument != null)
                {
                    m_xmlDocument = null;
                }

                if (m_binaryStreams != null)
                {
                    m_binaryStreams.Clear();
                    m_binaryStreams = null;
                }

                mcd.Log("Before GC.WaitFor*");
                System.GC.WaitForFullGCComplete();
                System.GC.WaitForPendingFinalizers();
                mcd.Log("After GC.WaitFor*");

                if (m_tempFilePath.Length > 0)
                {
                    try
                    {
                        mcd.Log("File.Delete(m_tempFilePath: '{0}')", m_tempFilePath);
                        System.IO.File.Delete(m_tempFilePath);
                        mcd.Log("    OK");
                        m_tempFilePath = "";
                    }
                    catch (System.Exception ex)
                    {
                        mcd.LogException(ex);
                        Debug.Print(ex.Message);
                    }
                }
            }
        }
        #endregion
    }

    internal class MethodCallDump : IDisposable
    {
        #if DEBUG || LOG_IN_RELEASE
            private readonly string methodName;
            private static int callStackLevel = 0;
            private static readonly object syncRoot = new object();
        #endif // DEBUG || LOG_IN_RELEASE

        public MethodCallDump(string methodName, string parametersFormat = null, params object[] parametersArgs)
        {
            #if DEBUG || LOG_IN_RELEASE
                this.methodName = methodName;

                Log("{0}({1}) [Begin]", methodName, string.Format(parametersFormat ?? "", parametersArgs));
                callStackLevel++;
            #endif // DEBUG || LOG_IN_RELEASE
        }

        public void Dispose()
        {
            #if DEBUG || LOG_IN_RELEASE
                callStackLevel--;
                Log("{0} [End]", methodName);
            #endif // DEBUG || LOG_IN_RELEASE
        }

        public void Log(string format, params object[] parameters)
        {
            #if DEBUG || LOG_IN_RELEASE
                Dump(string.Format("{0}{1}", new string(' ', callStackLevel * 4), format), parameters);
            #endif // DEBUG || LOG_IN_RELEASE
        }

        public T LogReturnValue<T>(T value)
        {
            #if DEBUG || LOG_IN_RELEASE
                Log("return {0} [Type: {1}]", value, typeof(T).FullName);
            #endif // DEBUG || LOG_IN_RELEASE
            return value;
        }

        public void LogException(Exception ex)
        {
            #if DEBUG || LOG_IN_RELEASE
                if (ex == null)
                    return;

                lock (syncRoot)
                {
                    Log("    Failed ({0})", ex.GetType().FullName);
                    Log("        Message: {0}", ex.Message);
                    Log("        Call Stack:");
                    Log("-------------------\r\n{0}", ex.StackTrace);
                    Log("-------------------");
                }
            #endif // DEBUG || LOG_IN_RELEASE
        }

        public static void Dump(string format, params object[] parameters)
        {
            #if DEBUG || LOG_IN_RELEASE
                try
                {
                    lock (syncRoot)
                    {
                        string path = System.IO.Path.GetDirectoryName(Assembly.GetEntryAssembly().Location);
                        string file = string.Format("{0}\\3dBinaryIntermediateFile.dump.txt", path);
                        using (var stream = new System.IO.FileStream(file, System.IO.FileMode.Append, System.IO.FileAccess.Write, System.IO.FileShare.ReadWrite))
                        {
                            string message = string.Format(format, parameters);
                            string line = string.Format("{0}> {1}\r\n", DateTime.Now.ToString("yyyy/MM/dd HH:mm:ss.fff"), message);
                            byte[] data = Encoding.UTF8.GetBytes(line);
                            stream.Write(data, 0, data.Length);
                        }
                    }
                }
                catch { }
#endif // DEBUG || LOG_IN_RELEASE
        }
    }
} // End of nw.g3d.bifedit
