﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Diagnostics;

namespace NW4F.LayoutBinaryConverter
{

    public class WriterUtil
    {
        static public TDstFlag ConvertBackFlagEnumToArray<TDstFlag, TSrc>(TSrc[] flags)
        {
            int dst = 0;

            foreach (TSrc flag in flags)
            {
                dst |= (int)Enum.Parse(typeof(TDstFlag), flag.ToString());
            }

            return (TDstFlag)Enum.ToObject(typeof(TDstFlag), dst);
        }

        public WriterUtil(int fileBasePosition)
        {
            FileBasePosition = fileBasePosition;
        }

        public class RewriteOffsetValue
        {
            public Type Type { get; set; }
            public Stream Stream { get; set; }
            public int Position { get; set; }

            public int FinalPosition { get; set; }

            public MemoryStream Content { get; set; }
        }

        public void AddOffset(Type type, Stream stream, int position, MemoryStream content)
        {
            var val = new RewriteOffsetValue();
            val.Type = type;
            val.Stream = stream;
            val.Position = position;
            val.Content = content;

            RewriteOffsetValues.Add(val);
        }

        public int FileBasePosition { get; private set; }

        public List<RewriteOffsetValue> RewriteOffsetValues { get; } = new List<RewriteOffsetValue>();
    }

    public class RvlBinWriter
    {
        readonly Stream _fs;

        public RvlBinWriter(Stream fs)
        {
            _fs = fs;
        }

        public Stream BaseStream { get { return _fs; } }

        int
        WriteSub(byte[] bytes)
        {
#if !NW
            // TODO: バイトオーダーを正しく扱う
            Array.Reverse(bytes);
#endif
            _fs.Write(bytes, 0, bytes.Length);
            return bytes.Length;
        }

        public int Write(byte value, int numBytes)
        {
            Debug.Assert(numBytes == 1);
            _fs.WriteByte(value);
            return numBytes;
        }

        public int Write(short value, int numBytes)
        {
            Debug.Assert(numBytes == 2);

            return WriteSub(BitConverter.GetBytes(value));
        }

        public int Write(ushort value, int numBytes)
        {
            Debug.Assert(numBytes == 2);

            return WriteSub(BitConverter.GetBytes(value));
        }

        public int Write(int value, int numBytes)
        {
            Debug.Assert(numBytes == 4);

            return WriteSub(BitConverter.GetBytes(value));
        }

        public int Write(uint value, int numBytes)
        {
            Debug.Assert(numBytes == 4);

            return WriteSub(BitConverter.GetBytes(value));
        }

        public int Write(double value, int numBytes)
        {
            Debug.Assert(numBytes == 8);

            return WriteSub(BitConverter.GetBytes(value));
        }

        public int Write(byte[] block)
        {
            return Write(block, block.Length);
        }

        public int Write(byte[] block, int numBytes)
        {
            _fs.Write(block, 0, numBytes);

            return numBytes;
        }

        public void Flush()
        {
            _fs.Flush();
        }

        public int RoundUp32()
        {
            int seekVal = 0;
            int mod = (int)(_fs.Position % 32);
            if (mod != 0)
            {
                seekVal = 32 - mod;
                _fs.Seek(seekVal, SeekOrigin.Current);
            }
            return seekVal;
        }

    }
}
