﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Diagnostics;
using System.Threading;

namespace NW4F.LayoutBinaryConverter
{
    public class ShaderFileUtil
    {
        // シェーダープリプロセッサマクロの数
        const int ShaderPreprocessorMacroCount = 12;

        /// <summary>
        /// シェーダファイルをコピーする
        /// </summary>
        /// <param name="srcBaseDir"></param>
        /// <param name="fontFiles"></param>
        /// <param name="dstBaseDir"></param>
        /// <param name="resType"></param>
        public static void MakeArchiveShaders(ICollection<Schema.Flyt.ArchiveShaderFile> archiveShaderFileEnum, string dstBaseDir, string outFileName)
        {
            // ユーザーが、カスタムシェーダーをコピーする場合にフォルダがないと困るので、出力するものが無くても、常に空フォルダだけは作成するようにする。
            string dstShaderDir = FileUtil.MakeResourceDirectory(dstBaseDir, BinaryLytWriter.TypeOfShaderResource);

            if (archiveShaderFileEnum.Count <= 0)
            {
                return;
            }

            System.Xml.Serialization.XmlSerializer xmlSerializer = new System.Xml.Serialization.XmlSerializer(typeof(GfxShaderVariation));

            // シェーダバリエーションの XML の構築
            GfxShaderVariation variation = new GfxShaderVariation();
            variation.version = "0.0.0";
            variation.ShaderVariationDefinition = new GfxShaderVariation.ShaderVariationDefinitionClass();
            variation.ShaderVariationDefinition.VertexShaderVariationDefinition = new GfxShaderVariation.ShaderVariationDefinitionClass.VertexShaderVariationDefinitionClass();
            variation.ShaderVariationDefinition.PixelShaderVariationDefinition = new GfxShaderVariation.ShaderVariationDefinitionClass.PixelShaderVariationDefinitionClass();
            {
                GfxShaderVariation.ShaderVariationDefinitionClass.PreprocessorDefinitionDefinitionArrayClass definitionArray = new GfxShaderVariation.ShaderVariationDefinitionClass.PreprocessorDefinitionDefinitionArrayClass();
                definitionArray.length = ShaderPreprocessorMacroCount;
                definitionArray.PreprocessorDefinitionDefinition = new GfxShaderVariation.ShaderVariationDefinitionClass.PreprocessorDefinitionDefinitionArrayClass.PreprocessorDefinitionDefinitionClass[ShaderPreprocessorMacroCount];
                for (int i = 0; i < ShaderPreprocessorMacroCount; i++)
                {
                    definitionArray.PreprocessorDefinitionDefinition[i] = new GfxShaderVariation.ShaderVariationDefinitionClass.PreprocessorDefinitionDefinitionArrayClass.PreprocessorDefinitionDefinitionClass();
                    definitionArray.PreprocessorDefinitionDefinition[i].index = i;
                }
                definitionArray.PreprocessorDefinitionDefinition[0].name = "NW_VERTEX_COLOR_ENABLED";
                definitionArray.PreprocessorDefinitionDefinition[1].name = "NW_MESH_ENABLED";
                definitionArray.PreprocessorDefinitionDefinition[2].name = "NW_MULTI_TEXTURE_QUANTITY";
                definitionArray.PreprocessorDefinitionDefinition[3].name = "NW_ADVANCED_MULTI_TEXTURE_ENABLED";
                definitionArray.PreprocessorDefinitionDefinition[4].name = "NW_TEXTURE_COMBINE_TYPE";
                definitionArray.PreprocessorDefinitionDefinition[5].name = "NW_TEXTURE_COMBINE_TYPE2";
                definitionArray.PreprocessorDefinitionDefinition[6].name = "NW_DETAILED_COMBINER_ENABLED";
                definitionArray.PreprocessorDefinitionDefinition[7].name = "NW_COMBINERUSERSHADER_TYPE";
                definitionArray.PreprocessorDefinitionDefinition[8].name = "NW_TEXTURE_PROJECTION_IN_PIXEL_SHADER";
                definitionArray.PreprocessorDefinitionDefinition[9].name = "NW_PROCEDURAL_SHAPE";
                definitionArray.PreprocessorDefinitionDefinition[10].name = "_INTERNAL_SHADER_VARIATION_KEY0";
                definitionArray.PreprocessorDefinitionDefinition[11].name = "_INTERNAL_SHADER_VARIATION_KEY1";
                variation.ShaderVariationDefinition.VertexShaderVariationDefinition.PreprocessorDefinitionDefinitionArray = definitionArray;
                variation.ShaderVariationDefinition.PixelShaderVariationDefinition.PreprocessorDefinitionDefinitionArray = definitionArray;
            }

            // 詳細コンバイナのコンスタント値
            {
                GfxShaderVariation.ShaderVariationDefinitionClass.VariationConstantBufferClass constanBuffer = new GfxShaderVariation.ShaderVariationDefinitionClass.VariationConstantBufferClass();
                constanBuffer.name = "VariationConstant";

                GfxShaderVariation.ShaderVariationDefinitionClass.VariationConstantDefinitionArrayClass definitionArray = new GfxShaderVariation.ShaderVariationDefinitionClass.VariationConstantDefinitionArrayClass();
                const int num = 6 + 1;  // ステージ分の ivec4 の数 ＋ ステージ数
                definitionArray.length = num;
                definitionArray.VariationConstantDefinition = new GfxShaderVariation.ShaderVariationDefinitionClass.VariationConstantDefinitionArrayClass.VariationConstantDefinitionClass[num];

                // ステージ数
                definitionArray.VariationConstantDefinition[0] = new GfxShaderVariation.ShaderVariationDefinitionClass.VariationConstantDefinitionArrayClass.VariationConstantDefinitionClass();
                definitionArray.VariationConstantDefinition[0].index = 0;
                definitionArray.VariationConstantDefinition[0].name = "stageCount";
                definitionArray.VariationConstantDefinition[0].type = "uint";

                // ステージ
                for (int i = 1; i < num; i++)
                {
                    definitionArray.VariationConstantDefinition[i] = new GfxShaderVariation.ShaderVariationDefinitionClass.VariationConstantDefinitionArrayClass.VariationConstantDefinitionClass();
                    definitionArray.VariationConstantDefinition[i].index = i;
                    definitionArray.VariationConstantDefinition[i].name = "bit" + (i - 1).ToString();
                    definitionArray.VariationConstantDefinition[i].type = "int4";
                }

                variation.ShaderVariationDefinition.PixelShaderVariationDefinition.VariationConstantBuffer = constanBuffer;
                variation.ShaderVariationDefinition.PixelShaderVariationDefinition.VariationConstantDefinitionArray = definitionArray;
            }

            {
                int num = archiveShaderFileEnum.Count * 3; // VertexColor 有効と無効、メッシュ用があるので 3 倍になる
                variation.ShaderVariationValueArray = new GfxShaderVariation.ShaderVariationValueArrayClass();
                variation.ShaderVariationValueArray.length = num;
                variation.ShaderVariationValueArray.ShaderVariationValue = new GfxShaderVariation.ShaderVariationValueArrayClass.ShaderVariationValueClass[num];
                int idx = 0;
                foreach (Schema.Flyt.ArchiveShaderFile archiveShaderFile in archiveShaderFileEnum)
                {
                    // 0 ... 頂点カラーあり
                    // 1 ... 頂点カラーなし
                    // 2 ... 3D メッシュ用頂点シェーダー
                    for (int vertexShaderSelection = 0; vertexShaderSelection < 3; vertexShaderSelection++)
                    {
                        int[] definitionValues = new int[]
                        {
                            vertexShaderSelection == 0 ? 1 : 0,
                            vertexShaderSelection == 2 ? 1 : 0,
                            archiveShaderFile.texMapCount,
                            1,
                            archiveShaderFile.firstBlend,
                            archiveShaderFile.secondBlend,
                            archiveShaderFile.useDetailedCombiner,
                            0,                  // NW_COMBINERUSERSHADER_TYPE はまだ使われるシェーダ用の値が判断できないので、 '0' が入る
                            archiveShaderFile.perspectiveTextureProjection ? 1 : 0,
                            archiveShaderFile.proceduralShape ? 1 : 0,
                            archiveShaderFile.key0,
                            archiveShaderFile.key1
                        };
                        variation.ShaderVariationValueArray.ShaderVariationValue[idx] = new GfxShaderVariation.ShaderVariationValueArrayClass.ShaderVariationValueClass();
                        variation.ShaderVariationValueArray.ShaderVariationValue[idx].index = idx;
                        variation.ShaderVariationValueArray.ShaderVariationValue[idx].VertexShaderVariationValue = new GfxShaderVariation.ShaderVariationValueArrayClass.ShaderVariationValueClass.VertexShaderVariationValueClass();
                        variation.ShaderVariationValueArray.ShaderVariationValue[idx].PixelShaderVariationValue = new GfxShaderVariation.ShaderVariationValueArrayClass.ShaderVariationValueClass.PixelShaderVariationValueClass();
                        {
                            GfxShaderVariation.ShaderVariationValueArrayClass.ShaderVariationValueClass.PreprocessorDefinitionValueArrayClass valueArray = new GfxShaderVariation.ShaderVariationValueArrayClass.ShaderVariationValueClass.PreprocessorDefinitionValueArrayClass();
                            valueArray.length = definitionValues.Length;
                            valueArray.PreprocessorDefinitionValue = new GfxShaderVariation.ShaderVariationValueArrayClass.ShaderVariationValueClass.PreprocessorDefinitionValueArrayClass.PreprocessorDefinitionValueClass[definitionValues.Length];
                            for (int i = 0; i < definitionValues.Length; i++)
                            {
                                valueArray.PreprocessorDefinitionValue[i] = new GfxShaderVariation.ShaderVariationValueArrayClass.ShaderVariationValueClass.PreprocessorDefinitionValueArrayClass.PreprocessorDefinitionValueClass();
                                valueArray.PreprocessorDefinitionValue[i].index = i;
                                valueArray.PreprocessorDefinitionValue[i].value = definitionValues[i];
                            }
                            variation.ShaderVariationValueArray.ShaderVariationValue[idx].VertexShaderVariationValue.PreprocessorDefinitionValueArray = valueArray;
                            variation.ShaderVariationValueArray.ShaderVariationValue[idx].PixelShaderVariationValue.PreprocessorDefinitionValueArray = valueArray;

                            // 詳細コンバイナ用バリエーション変数定義
                            // VariationConstantDefinitionArray で定義した場合は、VariationConstantValueArray を利用していなければ ShaderConverter.exe でエラーとなります。
                            const int variationValueLength = 7;
                            GfxShaderVariation.ShaderVariationValueArrayClass.ShaderVariationValueClass.VariationConstantValueArrayClass detailedCombinerValueArray = new GfxShaderVariation.ShaderVariationValueArrayClass.ShaderVariationValueClass.VariationConstantValueArrayClass();
                            detailedCombinerValueArray.length = variationValueLength;
                            detailedCombinerValueArray.VariationConstantValueArray = new GfxShaderVariation.ShaderVariationValueArrayClass.ShaderVariationValueClass.VariationConstantValueArrayClass.VariationConstantValueClass[detailedCombinerValueArray.length];
                            detailedCombinerValueArray.VariationConstantValueArray[0] = new GfxShaderVariation.ShaderVariationValueArrayClass.ShaderVariationValueClass.VariationConstantValueArrayClass.VariationConstantValueClass();
                            detailedCombinerValueArray.VariationConstantValueArray[0].index = 0;

                            if (archiveShaderFile.useDetailedCombiner != 1 &&
                                archiveShaderFile.useCombinerUserShader != 1)
                            {
                                detailedCombinerValueArray.VariationConstantValueArray[0].value = String.Format("0");
                                for (int i = 0; i < variationValueLength - 1; i++)
                                {
                                    detailedCombinerValueArray.VariationConstantValueArray[1 + i] = new GfxShaderVariation.ShaderVariationValueArrayClass.ShaderVariationValueClass.VariationConstantValueArrayClass.VariationConstantValueClass();
                                    detailedCombinerValueArray.VariationConstantValueArray[1 + i].index = i + 1;
                                    detailedCombinerValueArray.VariationConstantValueArray[1 + i].value = String.Format("0 0 0 0");
                                }
                            }
                            else
                            {
                                detailedCombinerValueArray.VariationConstantValueArray[0].value = String.Format("{0}", archiveShaderFile.stageCount);
                                for (int i = 0; i < variationValueLength - 1; i++)
                                {
                                    detailedCombinerValueArray.VariationConstantValueArray[1 + i] = new GfxShaderVariation.ShaderVariationValueArrayClass.ShaderVariationValueClass.VariationConstantValueArrayClass.VariationConstantValueClass();
                                    detailedCombinerValueArray.VariationConstantValueArray[1 + i].index = i + 1;
                                    detailedCombinerValueArray.VariationConstantValueArray[1 + i].value = String.Format("{0} {1} {2} {3}",
                                        archiveShaderFile.stageBits[i * 4 + 0].ToString(),
                                        archiveShaderFile.stageBits[i * 4 + 1].ToString(),
                                        archiveShaderFile.stageBits[i * 4 + 2].ToString(),
                                        archiveShaderFile.stageBits[i * 4 + 3].ToString());
                                }
                            }
                            variation.ShaderVariationValueArray.ShaderVariationValue[idx].PixelShaderVariationValue.VariationConstantValueArray = detailedCombinerValueArray;
                        }
                        idx++;
                    }
                }
            }
            string variationPath = dstShaderDir + "\\ArchiveShaderVariation_" + outFileName + ".xml";
            {
                System.IO.StreamWriter file = new StreamWriter(variationPath, false, new UTF8Encoding(true));
                xmlSerializer.Serialize(file, variation);
                file.Close();
            }
        }

        static void CopyFile(string srcPath, string dstPath)
        {
            /*
             * File.Copy() は内部で、Win32 の CopyFile() を呼び出すが、
             * このAPIは コピー先ファイルに Hidden, ReadOnly属性が付いていると、ERROR_ACCESS_DENIED となり、
             * 結果として、UnauthorizedAccessException が throw される。
             * そのため、あらかじめ属性をチェックして、これら属性が付いている場合は外しておく。
             */
            FileInfo dstFile = new FileInfo(dstPath);
            if (dstFile.Exists)
            {
                try
                {
                    RemoveReadOnlyAndHiddenAttribute(dstFile);
                }
                catch (IOException)
                {
                    // 属性を落とすだけなので、ここでIOException例外が発生しても
                    // コピー操作を実行させてみる。
                }
            }

            File.Copy(srcPath, dstPath, true);  // 上書きする

            /*
             * srcPathのファイルにHidden, ReadOnly属性が付いていた場合はこれもコピーされる。
             * レイアウトを構成するバイナリファイルのうち、コピーだけですむバイナリファイルのみが
             * 属性を受け継いでしまうのは構成上望ましくないので、これらの属性を外す。
             */
            dstFile = new FileInfo(dstPath);
            try
            {
                RemoveReadOnlyAndHiddenAttribute(dstFile);
            }
            catch (IOException)
            {
                // 属性を落とすだけなので、ここでIOException例外が発生しても
                // コンバート処理は継続させる
            }
        }

        /// <summary>
        /// Hidden, ReadOnly属性を取り除く
        /// </summary>
        /// <param name="dstFile"></param>
        static void RemoveReadOnlyAndHiddenAttribute(FileInfo dstFile)
        {
            FileAttributes attributes = dstFile.Attributes;
            if (0 != (attributes & FileAttributes.Hidden)
              || 0 != (attributes & FileAttributes.ReadOnly))
            {
                attributes &= ~FileAttributes.Hidden;
                attributes &= ~FileAttributes.ReadOnly;
                dstFile.Attributes = attributes;
            }
        }

        public static void MakeCombinerUserShader(string outputPath, List<string> ecbmPathLists, string shaderEnvDirectories, string combinerEditorPath, string fileDirectoryPath)
        {
            if (ecbmPathLists.Count <= 0)
            {
                return;
            }

            if (string.IsNullOrEmpty(shaderEnvDirectories))
            {
                throw new LayoutConverterException(
                    Properties.Resources.ErrorShaderEnvDirectoriesEmpty
                );
            }
            if (!Directory.Exists(shaderEnvDirectories))
            {
                throw new LayoutConverterException(
                    string.Format(
                        Properties.Resources.ErrorShaderEvnPathNotExist,
                        shaderEnvDirectories
                    )
                );
            }

            DirectoryInfo di = new DirectoryInfo(Path.Combine(outputPath, DataUtil.GetResourceTypeString(BinaryLytWriter.TypeOfShaderResource)));
            foreach(var ecmbFilepath in ecbmPathLists)
            {
                string glslFilePath = System.IO.Path.Combine(di.FullName, Path.GetFileNameWithoutExtension (ecmbFilepath) + ".glsl");
                string ecmbFilePath = Path.Combine(fileDirectoryPath, ecmbFilepath);
                // ファイルが存在しない場合は通知を行い、コンバートを止める
                if (!File.Exists(ecmbFilePath))
                {
                    throw new LayoutConverterException(
                        string.Format(
                            Properties.Resources.ErrorEcmbFileNameEmpty,
                            ecmbFilePath
                        )
                    );
                }


                CallCombinerEditor(glslFilePath, ecmbFilePath, shaderEnvDirectories, combinerEditorPath);
            }
        }

        /// <summary>
        /// 外部プロセスのエラーを読む
        /// </summary>
        public class ProccesErrorReader
        {
            public readonly StreamReader stderr;
            public readonly EventWaitHandle waitHandle = new EventWaitHandle(false, EventResetMode.ManualReset);
            public volatile string errStr;

            /// <summary>
            /// 構築
            /// </summary>
            public ProccesErrorReader(StreamReader procStdError)
            {
                this.stderr = procStdError;
                ThreadPool.QueueUserWorkItem(new WaitCallback(GenerateStdErrString_), null);
            }

            /// <summary>
            ///
            /// </summary>
            public string ReadErrorMsg()
            {
                bool isErrMsg = this.waitHandle.WaitOne(5000, false);
                return isErrMsg ? this.errStr : "";
            }

            /// <summary>
            /// コンバータの標準エラー出力を文字列として取得。
            /// </summary>
            void GenerateStdErrString_(object state)
            {
                StringBuilder sb = new StringBuilder();
                try
                {
                    string line;
                    while (null != (line = this.stderr.ReadLine()))
                    {
                        sb.AppendLine(line);
                    }
                }
                catch (Exception ex)
                {
                    // ログの書き出し失敗
                    Debug.WriteLine(ex.ToString());
                }

                this.errStr = sb.ToString();

                if (!this.waitHandle.Set())
                {
                    Debug.WriteLine("StdErr EventHandle fail \"Set()\"");
                }
            }
        }

        static bool CallCombinerEditor(string outputPath, string inputPath, string shaderEnvDirectories, string combinerEditorPath)
        {
            System.Diagnostics.Process process = new System.Diagnostics.Process();
            System.Diagnostics.ProcessStartInfo info = new System.Diagnostics.ProcessStartInfo();
            string path = System.IO.Path.GetDirectoryName(System.Reflection.Assembly.GetEntryAssembly().Location);

            if (!string.IsNullOrWhiteSpace(combinerEditorPath))
            {
                // 絶対パスか相対パスか調べます。
                // 先頭の文字が '\'(または '/')か、二文字目が ':' の時に True になります。
                if (Path.IsPathRooted(combinerEditorPath))
                {
                    info.FileName = combinerEditorPath;
                }
                else
                {
                    info.FileName = Path.Combine(path, combinerEditorPath);
                }

                // 設定されパスに CombinerEditor.exe が存在しない場合は変換を止める。
                if (!File.Exists(info.FileName))
                {
                    throw new LayoutConverterException(
                        string.Format(
                            Properties.Resources.ErrorCombinerEditorNotExist,
                            info.FileName
                        )
                    );
                }
            }
            else
            {
                // CombinerEditor.exe へのパスが未設定な場合は --combiner-editor-path を案内する。
                throw new LayoutConverterException(Properties.Resources.ErrorCombinerEditorUnset);
            }

            info.Arguments =
                " -i " + inputPath +
                " -o " + outputPath +
                " --silent" +
                " --disable-alloc-console";

            if(shaderEnvDirectories != null && !shaderEnvDirectories.Equals(""))
            {
                info.Arguments += " --shader-env-path " + shaderEnvDirectories;
            }

            info.UseShellExecute = false;
            info.CreateNoWindow = false;
            info.RedirectStandardError = true;
            info.StandardErrorEncoding = Encoding.Default;
            info.RedirectStandardOutput = true;
            info.StandardOutputEncoding = Encoding.Default;
            process.StartInfo = info;
            StringBuilder errorBuilder = new StringBuilder();
            process.OutputDataReceived += new DataReceivedEventHandler((s, e) => { errorBuilder.Append((e != null && e.Data != null) ? e.Data : ""); });
            process.Start();
            ProccesErrorReader procErrorReader = new ProccesErrorReader(process.StandardError);
            process.WaitForExit();

            string error = procErrorReader.ReadErrorMsg();
            if (error != "")
            {
                throw new LayoutConverterException(error);
            }
            return true;
        }
    }
}
