﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using System.Diagnostics;

namespace NW4F.LayoutBinaryConverter
{
    using Lyt = Schema.Flyt;
    using Lan = Schema.Flan;
    using NW4F.LayoutBinaryConverter.Schema.Flan;

    /// <summary>
    /// AnimContentをマージするして保持するクラス。
    /// </summary>
    class MergeAnimData
    {
        MergeAnimContent[] _animContAry;

        /// <summary>
        ///
        /// </summary>
        public static bool IsEndWithWindowMaterialSuffix(string name)
        {
            if(name.Length <= 2)
            {
                return false;
            }

            string suffix = name.Substring(name.Length - 2);
            return
                suffix == "LT" ||
                suffix == "RT" ||
                suffix == "LB" ||
                suffix == "RB" ||
                suffix == "L " ||
                suffix == "R " ||
                suffix == "T " ||
                suffix == "B " ||
                suffix == "C ";
        }

        /// <summary>
        ///
        /// </summary>
        public static string GetNameWithoutWindowMaterialSuffix(string name)
        {
            return name.Substring(0, name.Length - 2);
        }

        private bool IsContentTargetedByGroup_(AnimSection animSec, Lyt::GroupSet groupSet, Lan::AnimContent content)
        {
            // 対処グループすべてについて。。。
            foreach (GroupRef groupRef in animSec.Tag.group)
            {
                Lyt::Group group = Array.Find(groupSet.group.group, (item) => item.name == groupRef.name);
                if (group != null && group.paneRef != null)
                {
                    // すべてのペインについて。。。
                    foreach (Lyt::GroupPaneRef paneRef in group.paneRef)
                    {
                        // content を対象としているかを調べる。
                        if (paneRef.name == content.name)
                        {
                            return true;
                        }
                        else if(IsEndWithWindowMaterialSuffix(content.name))
                        {
                            // ウィンドウペインのマテリアルアニメーションの場合、ペイン名に二文字追加されているため、
                            // コンテント名から二文字削った名前でもチェックが必要
                            if (paneRef.name == GetNameWithoutWindowMaterialSuffix(content.name))
                            {
                                return true;
                            }
                        }
                    }
                }
            }

            return false;
        }

        public MergeAnimData(LanInfo lanInfo, AnimSection animSec, Lyt::GroupSet groupSet, bool bUseAnimSecTag, bool bOmitSameKey, bool bOmitSameKeyAll, bool bOmitNoKeyAll, bool bBakeInfinityAreaKey, bool bConvertAllAnimContent, bool bAllowedNoGroupAnimTag)
        {
            Debug.Assert(lanInfo.LanArray != null);

            Dictionary<string, MergeAnimContent>[] mergeContentDicAry = new Dictionary<string, MergeAnimContent>[(int)AnimContentType.MAX];
            for (int i = 0; i < mergeContentDicAry.Length; ++i)
            {
                mergeContentDicAry[i] = new Dictionary<string, MergeAnimContent>();
            }
            // 拡張ユーザーデータを別途収集するためのコンテナです。
            List<MergeAnimContent> extUserDataAnimContentList = new List<MergeAnimContent>();

            foreach (Lan::LAN lan in lanInfo.LanArray)
            {
                Debug.Assert(lan.animContent != null);

                if (! IsEnableOutput(lan.animType, animSec.Tag))
                {
                    continue;
                }

                AnimContentType contentType;
                int animIdx = GetContentType(lan.animType, out contentType);

                Dictionary<string, MergeAnimContent> mgContentDic = mergeContentDicAry[(int)contentType];

                // アニメーション対処を限定する group が存在しているのかどうか？
                bool animTargetGroupExist = (animSec.Tag.group != null && animSec.Tag.group.Length > 0) && groupSet.group.group != null;

                // 対象グループの省略を許さない場合はエラーとする
                if (!bConvertAllAnimContent && !animTargetGroupExist && bUseAnimSecTag)
                {
                    if (!bAllowedNoGroupAnimTag)
                    {
                        throw new LayoutDataException(string.Format(Properties.Resources.ErrorNoTargetAnimTag, animSec.Tag.name, System.IO.Path.GetFileName(lanInfo.InputFileName),  0));
                    }
                }

                foreach (Lan::AnimContent content in lan.animContent)
                {
                    // グループに含まれないペインについてはアニメーションを出力しない設定で、グループが指定されているアニメーションのときは、
                    // contentのターゲットがグループに含まれているかをチェックする。
                    // 含まれているときのみアニメーションを出力する。
                    if (!bConvertAllAnimContent && animTargetGroupExist && bUseAnimSecTag)
                    {
                        bool isContentTargeted = IsContentTargetedByGroup_(animSec, groupSet, content);
                        if (!isContentTargeted)
                        {
                            continue;
                        }
                    }

                    // 一文字アニメーションの開始フレームのアニメーションの場合は、Lyt側でテキストボックスの一文字アニメーションが
                    // 有効になっているかどうかのチェックも必要
                    if (lan.animType == AnimationType.PerCharacterTransform && lanInfo.GetLytInfo() != null)
                    {
                        Lyt::Pane pane = lanInfo.GetLytInfo().PaneDic[content.name];
                        if (pane == null)
                        {
                            continue;
                        }
                        Lyt::TextBox textBox = pane.Item as Lyt::TextBox;
                        if (textBox == null || ! textBox.perCharacterTransformEnabled)
                        {
                            continue;
                        }
                    }

                    // 各AnimTarget の出力範囲のキーのリストを抽出
                    List<Lan::AnimTarget> targetList = new List<Lan::AnimTarget>(content.Items.Length);
                    foreach (Lan::AnimTarget animTarget in content.Items)
                    {
                        Lan::AnimTarget subAnimTarget = animTarget;
                        if (subAnimTarget.parameterizedAnimParameter != null)
                        {
                            // パラメタライズドアニメーションの場合は無条件に抽出する
                            targetList.Add(subAnimTarget);
                            continue;
                        }
                        if (bBakeInfinityAreaKey)
                        {
                            subAnimTarget = BakeInfinityAreaKeyFrame(subAnimTarget, animSec.Tag.startFrame, animSec.Tag.endFrame);
                        }
                        subAnimTarget = GetSubAnimTarget(subAnimTarget, animSec.Tag.startFrame, animSec.Tag.endFrame);
                        if (bUseAnimSecTag && subAnimTarget.key.Length == 1)
                        {
                            if (   (bOmitSameKey && animSec.TagOrder > 0)
                                ||  bOmitSameKeyAll
                            )
                            {
                                continue;
                            }
                        }
                        if (bUseAnimSecTag && bOmitNoKeyAll)
                        {
                            // タグ区間中にキーが打たれていないときはカーブを出力しない。
                            bool bKeyExist = Array.Exists(subAnimTarget.key, (x) => animSec.Tag.startFrame <= x.frame && x.frame <= animSec.Tag.endFrame);
                            if (!bKeyExist)
                            {
                                continue;
                            }
                        }
                        targetList.Add(subAnimTarget);
                    }

                    if (targetList.Count > 0)
                    {
                        MergeAnimContent mgContent;

                        if (contentType != AnimContentType.ExtUserData)
                        {
                            if (! mgContentDic.TryGetValue(content.name, out mgContent))
                            {
                                mgContent = new MergeAnimContent(contentType, content);
                                mgContentDic.Add(mgContent.Name, mgContent);
                            }

                            // 詳細コンバイナ設定の場合、マテリアルカラーアニメーションは
                            // 白黒補間カラーとコンスタントカラーで２つの<animContent>に分かれるので
                            // 内容をマージします。
                            if (lan.animType == AnimationType.MaterialColor)
                            {
                                if (mgContent.TargetLists[animIdx] == null)
                                {
                                    mgContent.TargetLists[animIdx] = targetList;
                                }
                                else
                                {
                                    mgContent.TargetLists[animIdx].AddRange(targetList);
                                }
                            }
                            else
                            {
                                Debug.Assert(mgContent.TargetLists[animIdx] == null);
                                mgContent.TargetLists[animIdx] = targetList;
                            }
                        }
                        else
                        {
                            // 同じペイン内の拡張ユーザーデータに対するアニメーションは name が同じで extUserDataTargetName が違うデータとなる。
                            // 通常のアニメーションは同じ name のものは処理しない(ASSERT & 上書き)ようになっているが
                            // 拡張ユーザーデータではすべて出力したいため別途リストにまとめてすべて書き出す。
                            mgContent = new MergeAnimContent(contentType, content);
                            mgContent.TargetLists[animIdx] = targetList;
                            extUserDataAnimContentList.Add(mgContent);
                        }
                    }
                }
            }

            // AnimContentの総数を計算
            int animContCount = 0;
            foreach (Dictionary<string, MergeAnimContent> mergeContentDic in mergeContentDicAry)
            {
                animContCount += mergeContentDic.Count;
            }
            animContCount += extUserDataAnimContentList.Count;

            // 1次配列にコピー
            if (animContCount > 0)
            {
                _animContAry = new MergeAnimContent[animContCount];

                int insertIdx = 0;
                foreach (Dictionary<string, MergeAnimContent> mergeContentDic in mergeContentDicAry)
                {
                    mergeContentDic.Values.CopyTo(_animContAry, insertIdx);
                    insertIdx += mergeContentDic.Count;
                }
                foreach (MergeAnimContent content in extUserDataAnimContentList)
                {
                    _animContAry[insertIdx++] = content;
                }
            }
        }

        public MergeAnimContent[] AnimContentAry { get { return _animContAry; } }

        public SortedList<string, Lan::RefRes> MakeRefTextureList(Lan::AnimTag animTag, List<Lyt::MaterialInfo> materialList)
        {
            SortedList<string, Lan::RefRes> refTexList = new SortedList<string, Lan::RefRes>(StringComparer.InvariantCultureIgnoreCase);

            AnimContentType contType;
            int animIdx = GetContentType(Lan::AnimationType.TexturePattern, out contType);  // ファイルリストを持っているのはテクスチャパターンのみ

            foreach (MergeAnimContent content in _animContAry)
            {
                if (content.ContentType != contType || content.TargetLists[animIdx] == null)
                {
                    continue;
                }
                // アニメーションが適用されるマテリアルを取得しておく(nullの場合もある)
                Lyt::MaterialInfo matInfo = materialList.Find((item) => item.name == content.Name);

                foreach (Lan::AnimTarget target in content.TargetLists[animIdx])
                {
                    if (target.refRes == null)  // テクスチャパターンリストが空
                    {
                        throw new LayoutDataException(string.Format(Properties.Resources.ErrorOutOfRangeAnimTexPatternTarget, content.Name, 0));
                    }

                    try
                    {
                        foreach (Lan::StepU16 stepKey in target.key)
                        {
                            Lan::RefRes res = null;
                            try
                            {
                                UInt16 stepValue = checked((UInt16)stepKey.value);
                                res = target.refRes[stepValue];
                            }
                            catch (OverflowException)   // uint16_t で表現できない値
                            {
                                throw new LayoutDataException(string.Format(Properties.Resources.ErrorOutOfRangeAnimTexPatternTarget, content.Name, stepKey.value));
                            }
                            catch (IndexOutOfRangeException)    // キーの値がテクスチャパターンリストを超えている
                            {
                                throw new LayoutDataException(string.Format(Properties.Resources.ErrorOutOfRangeAnimTexPatternTarget, content.Name, stepKey.value));
                            }

                            if (! refTexList.ContainsKey(res.name))
                            {
                                refTexList.Add(res.name, res);
                                // このリソースがインダイレクトとして使われているか調べる
                                if (matInfo != null) {
                                    res.isIndirect = LytInfo.IsIndirect(matInfo, target.id);
                                }
                            }
                        }
                    }
                    catch (InvalidCastException)    // キーの型が正しくない(StepU16にキャストできない)
                    {
                        throw new LayoutDataException(Properties.Resources.ErrorInvalidTexPatternKeyType);
                    }
                }
            }

            return refTexList;
        }

        static Lan::AnimTarget GetSubAnimTarget(Lan::AnimTarget target, int convertStartFrame, int convertEndFrame)
        {
            Lan::Hermite[] keys = target.key;

            // convertStartFrame を超えるフレーム値を持つキーの1つ手前を探す
            int stKeyIdx = 0;
            for (; stKeyIdx < keys.Length; ++stKeyIdx)
            {
                if (keys[stKeyIdx].frame > convertStartFrame)
                {
                    break;
                }
            }

            if (stKeyIdx > 0)
            {
                --stKeyIdx;
            }

            // convertEndFrame を下回るフレーム値を持つキーの1つ次を探す
            int edKeyIdx = keys.Length - 1;
            for (; edKeyIdx >= 0; --edKeyIdx)
            {
                if (keys[edKeyIdx].frame < convertEndFrame)
                {
                    break;
                }
            }

            if (edKeyIdx < keys.Length - 1)
            {
                ++edKeyIdx;
            }

            // 最終フレームと同一フレームの最後のキーを探す。
            for (; edKeyIdx + 1 < keys.Length; ++edKeyIdx)
            {
                // 処理をステップ型に限定。
                // 本来は必要無いはずだが修正の影響範囲を限定しておく。
                if (keys[edKeyIdx].slopeType != Lan.SlopeType.Step)
                {
                    break;
                }

                if (keys[edKeyIdx].frame != keys[edKeyIdx + 1].frame)
                {
                    break;
                }

            }

            if (stKeyIdx > edKeyIdx && convertStartFrame == convertEndFrame) {
                // 長さが0のアニメーションの場合、edKeyIdxがstKeyIdxを超えてしまう場合がある。これは、
                // convertStartFrame及びconvertEndFrameのフレームにキーが二つある場合に発生する。
                // この場合、keys[edKeyIdx].frame == keys[stKeyIdx].frameのはずなので、どちらに
                // なっても同じ。edKeyIdxをstKeyIdxと同じ値にしておく。
                Debug.Assert(keys[stKeyIdx].frame == keys[edKeyIdx].frame);
                edKeyIdx = stKeyIdx;
            }

            Debug.Assert(stKeyIdx <= edKeyIdx);

            // 終了フレームから同じ値でスロープが０のキーを省く
            for (int i = edKeyIdx - 1; i >= stKeyIdx; --i)
            {
                if (keys[i].frame < convertStartFrame && keys[edKeyIdx].frame <= convertEndFrame)
                {
                    break;
                }

                if (EqualKeyValue(keys[edKeyIdx], keys[i]))
                {
                    edKeyIdx = i;
                }
                else
                {
                    break;
                }
            }

            int stBaseKeyIdx = stKeyIdx;
            // 開始フレームから同じ値、スロープを持つキーを省く
            for (int i = stKeyIdx + 1; i <= edKeyIdx; ++i)
            {
                if (EqualKeyValue(keys[stKeyIdx], keys[i]))
                {
                    stKeyIdx = i;
                }
                else
                {
                    break;
                }
            }
            // キーを省いた後、先頭に同一フレームに2つのキーがある場合は、1つ戻す。
            // (詳細については、このソースの下の方にある"Column1"を参照)
            if (stBaseKeyIdx < stKeyIdx)
            {
                if (stKeyIdx + 1 <= edKeyIdx && keys[stKeyIdx].frame == keys[stKeyIdx + 1].frame)
                {
                    stKeyIdx--;
                }
            }

            // 要素数が変化しない場合はそのままキー配列を返す。
            if (stKeyIdx == 0 && edKeyIdx == keys.Length - 1)
            {
                return target;
            }

            // 必要なサイズの配列を確保してその配列にコピーする。
            Lan::Hermite[] subKeys = new Lan::Hermite[edKeyIdx - stKeyIdx + 1];
            Array.Copy(keys, stKeyIdx, subKeys, 0, subKeys.Length);
            return target.Duplicate(subKeys);
        }

        /// <summary>
        /// Infinity領域のキーフレーム焼付け処理を行います。
        ///
        /// キーフレームが時間順に格納されていることを前提としています。
        /// </summary>
        public static Lan::AnimTarget BakeInfinityAreaKeyFrame(Lan::AnimTarget animTarget, int startFrame, int endFrame)
        {
            float firstKeyFrame = animTarget.key[0].frame;
            float lastKeyFrame = animTarget.key[animTarget.key.Length - 1].frame;
            Debug.Assert(firstKeyFrame <= lastKeyFrame);
            float allKeyFrameRange = lastKeyFrame - firstKeyFrame;

            int preRepeatCnt = 0;
            int postRepeatCnt = 0;

            // PreInfinity処理
            if ( startFrame < firstKeyFrame
              && animTarget.preInfinityType == Lan::InfinityType.Cycle
              && allKeyFrameRange > 0
            )
            {
                preRepeatCnt = (int)Math.Ceiling(((double)firstKeyFrame - startFrame) / allKeyFrameRange);
            }

            // PostInfinity処理
            if ( lastKeyFrame < endFrame
              && animTarget.postInfinityType == Lan::InfinityType.Cycle
              && allKeyFrameRange > 0
            )
            {
                postRepeatCnt = (int)Math.Ceiling(((double)endFrame - lastKeyFrame) / allKeyFrameRange);
            }

            List<Lan::Hermite> newKeySet = new List<Lan::Hermite>(animTarget.key.Length * (preRepeatCnt + 1 + postRepeatCnt));

            // キーのコピー
            for (int i = -preRepeatCnt; i <= postRepeatCnt; ++i)
            {
                CopyKeyFrameRange(newKeySet, animTarget.key, i * allKeyFrameRange);
            }

            return animTarget.Duplicate(newKeySet.ToArray());
        }

        static void CopyKeyFrameRange(List<Lan::Hermite> keyList, Lan::Hermite[] copyKeys, float offsetFrame)
        {
            if (copyKeys.Length == 0)
            {
                return;
            }

            // コピーするキーレンジの最初のキーのフレーム値と同じ値を持つフレームがコピー先に存在する場合は、
            // コピー先のキーのフレーム値を少し前にする
            float firstCopyKeyFrame = copyKeys[0].frame + offsetFrame;

            for (int i = keyList.Count; i > 0; )    // 末尾から
            {
                --i;
                if (keyList[i].frame != firstCopyKeyFrame)
                {
                    break;
                }

                keyList[i] = keyList[i].Duplicate(keyList[i].frame - 0.001f);
            }

            foreach (Lan::Hermite copyKey in copyKeys)
            {
                Lan::Hermite newKey = copyKey;
                if (offsetFrame != 0)
                {
                    newKey = copyKey.Duplicate(copyKey.frame + offsetFrame);
                }
                keyList.Add(newKey);
            }
        }

        static bool EqualKeyValue(Lan::Hermite key1, Lan::Hermite key2)
        {
            return key1.value == key2.value && key1.slope == 0 && key2.slope == 0;
        }

        static bool IsEnableOutput(Lan::AnimationType animType, Lan::AnimTag animTag)
        {
            switch (animType)
            {
            case Lan::AnimationType.PaneSRT: return animTag.outputPaneSRT;
            case Lan::AnimationType.Visibility: return animTag.outputVisibility;
            case Lan::AnimationType.VertexColor: return animTag.outputVertexColor;
            case Lan::AnimationType.MaterialColor: return animTag.outputMaterialColor;
            case Lan::AnimationType.TextureSRT: return animTag.outputTextureSRT;
            case Lan::AnimationType.TexturePattern: return animTag.outputTexturePattern;
            case Lan::AnimationType.IndTextureSRT: return animTag.outputIndTextureSRT;
            case Lan::AnimationType.AlphaTest: return animTag.outputAlphaTest;
            case Lan::AnimationType.FontShadow: return animTag.outputMaterialColor;
            case Lan::AnimationType.PerCharacterTransform: return animTag.outputPerCharacterTransform;
            case Lan::AnimationType.PerCharacterTransformCurve: return false;
            case Lan::AnimationType.Window: return animTag.outputWindow;
            case Lan::AnimationType.ExtUserData: return animTag.outputExtUserData;
            case Lan::AnimationType.MaskTextureSRT: return animTag.outputMaskTextureSRT;
            case Lan::AnimationType.DropShadow: return animTag.outputDropShadow;
            case Lan::AnimationType.ProceduralShape: return animTag.outputProceduralShape;
            default: return false;
            }
        }

        static int GetContentType(Lan::AnimationType animType, out AnimContentType contType)
        {
            switch (animType)
            {
            case Lan::AnimationType.PaneSRT:        contType = AnimContentType.Pane;        return (int)PaneAnimType.PaneSRT;
            case Lan::AnimationType.Visibility:     contType = AnimContentType.Pane;        return (int)PaneAnimType.Visibility;
            case Lan::AnimationType.VertexColor:    contType = AnimContentType.Pane;        return (int)PaneAnimType.VertexColor;
            case Lan::AnimationType.PerCharacterTransform: contType = AnimContentType.Pane; return (int)PaneAnimType.PerCharacterTransform;
            case Lan::AnimationType.PerCharacterTransformCurve: contType = AnimContentType.Pane; return (int)PaneAnimType.PerCharacterTransformCurve;
            case Lan::AnimationType.Window: contType = AnimContentType.Pane; return (int)PaneAnimType.Window;
            case Lan::AnimationType.MaskTextureSRT: contType = AnimContentType.Pane;        return (int)PaneAnimType.Mask;
            case Lan::AnimationType.DropShadow: contType = AnimContentType.Pane;        return (int)PaneAnimType.DropShadow;
            case Lan::AnimationType.ProceduralShape: contType = AnimContentType.Pane;        return (int)PaneAnimType.ProceduralShape;

            case Lan::AnimationType.MaterialColor:  contType = AnimContentType.Material;    return (int)MaterialAnimType.MaterialColor;
            case Lan::AnimationType.TextureSRT:     contType = AnimContentType.Material;    return (int)MaterialAnimType.TextureSRT;
            case Lan::AnimationType.TexturePattern: contType = AnimContentType.Material;    return (int)MaterialAnimType.TexturePattern;
            case Lan::AnimationType.IndTextureSRT:  contType = AnimContentType.Material;    return (int)MaterialAnimType.IndTextureSRT;
            case Lan::AnimationType.AlphaTest:      contType = AnimContentType.Material;    return (int)MaterialAnimType.AlphaTest;
            case Lan::AnimationType.FontShadow:     contType = AnimContentType.Material;    return (int)MaterialAnimType.FontShadowColor;

            case Lan::AnimationType.ExtUserData:    contType = AnimContentType.ExtUserData; return (int)PaneAnimType.ExtUserData;

            default:
                throw new LayoutDataException(string.Format(Properties.Resources.ErrorUnknownAnimationType, animType));
            }
        }

        public static Lan::AnimationType GetAnimationType(AnimContentType contType, int idx)
        {
            switch (contType)
            {
            case AnimContentType.Pane:
                switch ((PaneAnimType)idx)
                {
                case PaneAnimType.PaneSRT:      return Lan::AnimationType.PaneSRT;
                case PaneAnimType.Visibility:   return Lan::AnimationType.Visibility;
                case PaneAnimType.VertexColor:  return Lan::AnimationType.VertexColor;
                case PaneAnimType.PerCharacterTransform: return Lan::AnimationType.PerCharacterTransform;
                case PaneAnimType.PerCharacterTransformCurve: return Lan::AnimationType.PerCharacterTransformCurve;
                case PaneAnimType.Window: return Lan::AnimationType.Window;
                case PaneAnimType.Mask: return Lan::AnimationType.MaskTextureSRT;
                case PaneAnimType.DropShadow: return Lan::AnimationType.DropShadow;
                case PaneAnimType.ProceduralShape: return Lan::AnimationType.ProceduralShape;
                }
                    break;

            case AnimContentType.Material:
                switch ((MaterialAnimType)idx)
                {
                case MaterialAnimType.MaterialColor:    return Lan::AnimationType.MaterialColor;
                case MaterialAnimType.TextureSRT:       return Lan::AnimationType.TextureSRT;
                case MaterialAnimType.TexturePattern:   return Lan::AnimationType.TexturePattern;
                case MaterialAnimType.IndTextureSRT:    return Lan::AnimationType.IndTextureSRT;
                case MaterialAnimType.AlphaTest:        return Lan::AnimationType.AlphaTest;
                case MaterialAnimType.FontShadowColor:  return Lan::AnimationType.FontShadow;
                }
                break;
            case AnimContentType.ExtUserData:
                return Lan::AnimationType.ExtUserData;
            }

            throw new ArgumentException("unknown AnimContentType.");  // ここに到達することはない
        }
    }
}
/*
 * Column1:
 *
 * アニメーションのエルミート補間について
 *
 * 仕様では、同一フレームに2つのキーを持つことが出来る。
 * この場合、そのフレームの値を取得する際は2つ目(右)の値を採用する。
 *
 * ライブラリ側では、補間の対象となる左右のキーを決定する際、
 * 仮に次のようなデータがあり、フレーム2の値を求めようとしたとすると、
 * LとRは次の位置を指す。
 *
 *     frame  0  1  2  2  3  ...
 *               L  R
 *                     +
 *
 * そして、補間処理に入る前の判定で、求めるフレームがRと一致している場合、
 * R のキーのフレームと R+1 のキーのフレームが同じなら R+1 を採用し、(上記のパターン)
 * そうでない場合は、R のキーを採用するようになっている。
 * ('+'の位置が採用されるキー)
 *
 * 初めのキーから同一フレームに値を持つ場合の L と R の関係は次のようになる。
 * この場合でも、上記のルールによれば、R のキーが採用されるはずである。
 *
 *     frame  2  2  3  ...
 *            L  R
 *
 * しかし、現状のライブラリはそうはならない。
 * なぜなら、L と R を求める前に、求めるフレームが先頭のキーのフレーム値以下である場合、
 *  (求めるフレーム <= キー配列[0].フレーム)
 * 先頭のフレームを採用するようになっているからである。
 * (つまり、L が採用される)
 *
 * これは、同一フレームに2つのキーがある場合に2つ目を採用するという仕様に反してしまう。
 *
 * ただ、同一フレームにあるキーの値が同じであれば、特に結果として問題にはならない。(スロープだけが異なる場合)
 * しかし、値がステップで変化する場合は、異なる値を持っているため問題が起こる。
 * しかし、値がステップで変化する場合ケースが、先頭から始まることはないはずである。
 *
 * 問題が発生するのは、このコンバータ処理の中で、前後のキーが同一の値とスロープを持つキーをまとめる処理を通すときにある。
 * この処理を通すと、同一フレームに2つのキーがあるデータが先頭にきてしまう可能性がある。
 * そのため、まとめる処理を行う際、先頭に同一フレームを持つキーが来ないようにした。
 *
 * ライブラリ側を直す方法もあるが、この場合既に作成済みのレイアウトのバイナリデータの挙動が変わってしまうため、
 * バナー表示利用されている今となってはこの対処は不可能といえる。
 *
 */
