﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Linq;
using System.Diagnostics;
using System.Runtime.InteropServices;

namespace NW4F.LayoutBinaryConverter
{
    using Lyt = Schema.Flyt;

    partial class LytWriter
    {
        private enum ArchiveShaderId
        {
            // \Programs\Iris\Include\nn\ui2d\ui2d_GraphicsResource.h ArchiveShaderid と同じとなります。
            // また、\Programs\Iris\Sources\Resources\Ui2dShaders にある NW_TEXTURE_COMBINE_TYPE、NW_TEXTURE_COMBINE_TYPE2 と一致しています。
            ArchiveShaderId_DetailedCombinerNonSource = 100,
            ArchiveShaderId_DetailedCombinerSingleSource,
            ArchiveShaderId_DetailedCombinerDoubleSource,
            ArchiveShaderId_DetailedCombinerTripleSource,

            ArchiveShaderId_DetailedCombinerBaseIdx = ArchiveShaderId_DetailedCombinerNonSource,
        };

        // ドロップシャドウのアルファ付き乗算ブレンドシェーダー
        const int DropShadowColorMulValue = 10;
        // ドロップシャドウのノックアウトシェーダー
        const int DropShadowColorKnockoutValue = 100;

        private enum DropShadowShaderId
        {
            // ドロップシャドウ横ブラーシェーダー
            HorizontalBlur0 = 0,
            HorizontalBlur1,
            HorizontalBlur2,
            HorizontalBlur3,

            // ドロップシャドウ縦ブラーシェーダー
            VerticalBlur0,
            VerticalBlur1,
            VerticalBlur2,
            VerticalBlur3,

            // ドロップシャドウ縦ブラー＆アルファ付き乗算ブレンドシェーダー
            VerticalBlur0WithColorMul = VerticalBlur0 + DropShadowColorMulValue,
            VerticalBlur1WithColorMul,
            VerticalBlur2WithColorMul,
            VerticalBlur3WithColorMul,

            // ドロップシャドウ縦ブラー＆ノックアウトシェーダー
            VerticalBlur0WithKnockout = VerticalBlur0 + DropShadowColorKnockoutValue,
            VerticalBlur1WithKnockout,
            VerticalBlur2WithKnockout,
            VerticalBlur3WithKnockout,

            // ドロップシャドウ縦ブラー＆ノックアウト＆アルファ付き乗算ブレンドシェーダー
            VerticalBlur0WithKnockoutColorMul = VerticalBlur0 + DropShadowColorMulValue + DropShadowColorKnockoutValue,
            VerticalBlur1WithKnockoutColorMul,
            VerticalBlur2WithKnockoutColorMul,
            VerticalBlur3WithKnockoutColorMul,

            // ペイン本体の最終描画用コピーシェーダー
            Copy = 8,
            CopyColorMul = Copy + DropShadowColorMulValue,
        };

        // 静的分岐版詳細コンバイナ用統合アーカイブシェーダーを判別するためのシェーダーキー0 の値です
        const int DetailedCombinerKey0 = 110;
        // コンバイナユーザーシェーダー用統合アーカイブシェーダーを判別するためのシェーダーキー0 の値です
        const int CombinerUserShaderKey0 = 111;
        // マスク用シェーダーを表すシェーダーキー0 の値です
        const int MaskShader = 120;
        // ドロップシャドウ用シェーダーを表すシェーダーキー0 の値です
        const int DropShadowShader = 130;
        // 統合済みアーカイブシェーダで透視投影テクスチャ用の標準シェーダーを表すシェーダーキー0の値です。シェーダーキー1 は ShaderId に対応しています。
        const int PerspectiveTextureProjectionDefaultShader = 250;
        // 統合済みアーカイブシェーダで透視投影テクスチャ用の3枚ブレンドシェーダーを表すシェーダーキー0のオフセットです。
        const int PerspectiveTextureProjectionBlendShaderOffset = 200;
        // 統合済みアーカイブシェーダで角丸用の標準シェーダーを表すシェーダーキー0の値です。シェーダーキー1 は ShaderId に対応しています。
        const int ProceduralShapeDefaultShader = 251;
        // 統合済みアーカイブシェーダで角丸の3枚ブレンドシェーダーを表すシェーダーキー0のオフセットです。
        const int ProceduralShapeBlendShaderOffset = 220;

        private struct ComplexFontPath
        {
            public string path;
            public bool isBitmapFont;
        }

        readonly bool _bImageFileWriteTimeSameOriginal;
        readonly bool _bConvertTagChar;
        readonly bool _bDiscardTextBoxText;
        readonly bool _bBannerFormat;
        readonly bool _bConvertRefTexOnly;
        readonly bool _bNoCopyFont;
        readonly bool _bDegammaTexture;
        readonly bool _bDegammaParameter;
        readonly bool _bViewerHidden;
        readonly bool _bNoCopyFfnt;
        readonly string _ftxbCacheDirectory;
        readonly bool _bCompatible_0_12_x;
        readonly string _shaderEnvDirectories;
        readonly string _combinerEditorPath;
        readonly bool _isEcmbEmptyErr;
        readonly bool _isKeepingFontScaleEnabled;

        public LytWriter(bool bImageFileWriteTimeSameOriginal, bool bConvertTagChar, bool bDiscardTextBoxText, bool bBannerFormat, bool bConvertRefTexOnly, bool bNoCopyFont, bool bDegammaTexture, bool bDegammaParameter, bool bViewerHidden, bool bNoCopyFfnt, string ftxbCacheDirectory, bool bCompatible_0_12_x, string shaderEnvDirectories, string combinerEditorPath, bool isEcmbEmptyErr, bool isKeepingFontScaleEnabled)
        {
            _bImageFileWriteTimeSameOriginal = bImageFileWriteTimeSameOriginal;
            _bConvertTagChar = bConvertTagChar;
            _bDiscardTextBoxText = bDiscardTextBoxText;
            _bBannerFormat = bBannerFormat;
            _bConvertRefTexOnly = bConvertRefTexOnly;
            _bNoCopyFont = bNoCopyFont;
            _bDegammaTexture = bDegammaTexture;
            _bDegammaParameter = bDegammaParameter;
            _bViewerHidden = bViewerHidden;
            _bNoCopyFfnt = bNoCopyFfnt;
            _ftxbCacheDirectory = ftxbCacheDirectory;
            _bCompatible_0_12_x = bCompatible_0_12_x;
            _shaderEnvDirectories = shaderEnvDirectories;
            _combinerEditorPath = combinerEditorPath;
            _isEcmbEmptyErr = isEcmbEmptyErr;
            _isKeepingFontScaleEnabled = isKeepingFontScaleEnabled;
        }

        //------------------------------------------------------------------------

        private void GenBfcpx(ComprexFontLoader comprexFontLoader, string outDirName, CpxInfo cpxInfo)
        {
            // バイナリライターで、出力する
            string dstDir = FileUtil.MakeResourceDirectory(outDirName, BinaryCpxWriter.TypeOfComplexFontResource);
            CpxWriter cpxWriter = new CpxWriter();
            cpxWriter.Write(outDirName, cpxInfo);
        }

        /// <summary>
        /// 複合フォントファイルの変換
        /// </summary>
        public void WriteComplexFont(string outDirName, string inFileName, IEnumerable<NW4F.LayoutBinaryConverter.Schema.Flyt.FontFile> complexFontSet, string tileMode, bool isQuiet)
        {
            if (!complexFontSet.Any())
            {
                return;
            }

            List<string> scalableFontPathSet = new List<string>();

            ComprexFontLoader comprexFontLoader = new ComprexFontLoader(false);
            foreach (Schema.Flyt.FontFile fontFile in complexFontSet)
            {
                string fontFilePath = FileUtil.GetAbsolutePath(inFileName, fontFile.path);
                // ファイルを読み込みデシリアライズして
                CpxInfo cpxInfo = comprexFontLoader.LoadCPX(fontFilePath);
                // バイナリに書き出します。
                GenBfcpx(comprexFontLoader, outDirName, cpxInfo);

                // 縁取りの情報が不一致ではないかをチェック
                CheckBorderEffect(cpxInfo.ComplexFontDescription.subFont.Item);

                // 設定に応じて、フォントファイルをコピーしておきます。
                if (!_bNoCopyFont)
                {
                    List<ComplexFontPath> fontPaths = new List<ComplexFontPath>();
                    SearchFontPaths(cpxInfo.ComplexFontDescription.subFont.Item, fontPaths, fontFilePath, tileMode, isQuiet);
                    string dstFontDir = FileUtil.MakeResourceDirectory(outDirName, BinaryCpxWriter.TypeOfFontResource);
                    string dstScalableFontDir = FileUtil.MakeResourceDirectory(outDirName, BinaryCpxWriter.TypeOfScalableFontResource);

                    foreach (ComplexFontPath fontPath in fontPaths)
                    {
                        string scFontPath = Path.Combine(Path.GetDirectoryName(cpxInfo.DocFilePath), fontPath.path);
                        if (!File.Exists(scFontPath))
                        {
                            // フォントが発見できなかった。エラー終了します。
                            throw new FileNotFoundException(string.Format(Properties.Resources.ErrorFontFileNameNotFound, scFontPath));
                        }

                        if (scalableFontPathSet.Contains(scFontPath))
                        {
                            continue;
                        }

                        string dstDir = fontPath.isBitmapFont ? dstFontDir : dstScalableFontDir;
                        string dstFilePath = Path.Combine(dstDir, Path.GetFileName(scFontPath));
                        File.Copy(scFontPath, dstFilePath, true);

                        // ファイル属性から読み取り専用を削除
                        FileAttributes fas = File.GetAttributes(dstFilePath) & ~FileAttributes.ReadOnly;
                        File.SetAttributes(dstFilePath, fas);

                        scalableFontPathSet.Add(scFontPath);
                    }
                }
            }
        }

        /// <summary>
        /// 縁取りの情報が不一致ではないかをチェック
        /// </summary>
        bool CheckBorderEffect(object font)
        {
            // ビットマップフォント
            var bitmapFont = font as Schema.Fcpx.BitmapFont;
            if (bitmapFont != null)
            {
                return false;
            }

            // スケーラブルフォント
            var scalableFont = font as Schema.Fcpx.ScalableFont;
            if (scalableFont != null)
            {
                return scalableFont.scalableFontDescription.borderWidth > 0;
            }

            // マルチスケーラブルフォント
            var multiScalableFont = font as Schema.Fcpx.MultiScalableFont;
            if (multiScalableFont != null)
            {
                bool first = true;
                bool borderEffect = false;
                foreach (var desc in multiScalableFont.scalableFontDescriptionSet)
                {
                    if (first)
                    {
                        borderEffect = desc.borderWidth > 0;
                        first = false;
                    }
                    else
                    {
                        if (borderEffect != (desc.borderWidth > 0))
                        {
                            throw new LayoutDataException(Properties.Resources.ErrorInvalidBorderEffect);
                        }
                    }
                }
                return borderEffect;
            }

            // ペアフォント
            var pairFont = font as Schema.Fcpx.PairFont;
            if (pairFont != null)
            {
                bool first = CheckBorderEffect(pairFont.firstFont.Item);
                bool second = CheckBorderEffect(pairFont.secondFont.Item);
                if (first != second)
                {
                    throw new LayoutDataException(Properties.Resources.ErrorInvalidBorderEffect);
                }
                return first;
            }

            throw new LayoutDataException(Properties.Resources.ErrorInvalidFcpxFormat);
        }

        /// <summary>
        /// 複合フォントファイルのフォントパスを探索します
        /// </summary>
        void SearchFontPaths(object font, List<ComplexFontPath> paths, string fcpxPath, string tileMode, bool isQuiet)
        {
            // ビットマップフォント
            var bitmapFont = font as Schema.Fcpx.BitmapFont;
            if (bitmapFont != null)
            {
                // .ffnt の場合は .bffnt に読み替える
                string path = bitmapFont.path;
                if (Path.GetExtension(path).ToLowerInvariant() == ".ffnt")
                {
                    if (_bNoCopyFfnt)
                    {
                        return;
                    }

                    // .ffnt を .bffnt にコンバートする
                    string bffntPath = Path.GetDirectoryName(path) + "\\" + Path.GetFileNameWithoutExtension(path) + ".bffnt";
                    ConvertFfnt(path, bffntPath, fcpxPath, tileMode, isQuiet);
                    path = bffntPath;
                }

                ComplexFontPath fontPath;
                fontPath.path = path;
                fontPath.isBitmapFont = true;
                paths.Add(fontPath);
                return;
            }

            // スケーラブルフォント
            var scalableFont = font as Schema.Fcpx.ScalableFont;
            if (scalableFont != null)
            {
                ComplexFontPath fontPath;
                fontPath.path = scalableFont.scalableFontDescription.path;
                fontPath.isBitmapFont = false;
                paths.Add(fontPath);
                return;
            }

            // マルチスケーラブルフォント
            var multiScalableFont = font as Schema.Fcpx.MultiScalableFont;
            if (multiScalableFont != null)
            {
                foreach (var desc in multiScalableFont.scalableFontDescriptionSet)
                {
                    ComplexFontPath fontPath;
                    fontPath.path = desc.path;
                    fontPath.isBitmapFont = false;
                    paths.Add(fontPath);
                }
                return;
            }

            // ペアフォント
            var pairFont = font as Schema.Fcpx.PairFont;
            if (pairFont != null)
            {
                SearchFontPaths(pairFont.firstFont.Item, paths, fcpxPath, tileMode, isQuiet);
                SearchFontPaths(pairFont.secondFont.Item, paths, fcpxPath, tileMode, isQuiet);
                return;
            }

            throw new LayoutDataException(Properties.Resources.ErrorInvalidFcpxFormat);
        }

        /// <summary>
        /// ffnt をコンバートする関数
        /// </summary>
        void ConvertFfnt(string ffntPath, string bffntPath, string fcpxPath, string tileMode, bool isQuiet)
        {
            string fontConverterPath = Path.Combine(FileUtil.GetNwToolPath(), @"FontConverter\FontConverterConsole.exe");

            ProcessStartInfo startInfo = new ProcessStartInfo();
            startInfo.UseShellExecute = false;
            startInfo.CreateNoWindow = true;
            startInfo.RedirectStandardOutput = true;
            startInfo.RedirectStandardError = true;
            startInfo.FileName = fontConverterPath;
            startInfo.WorkingDirectory = Path.GetDirectoryName(fcpxPath);
            startInfo.Arguments = "-i ffnt" +
                " -if " + ffntPath +
                " -o bffnt" +
                " -of " + bffntPath;
            if (tileMode != null)
            {
                startInfo.Arguments += " -tile-mode " + tileMode;
            }
            else
            {
                throw new LayoutConverterException(string.Format("--tile-mode option must be set when converting FFNT. [{0}]", ffntPath));
            }
            if (isQuiet)
            {
                startInfo.Arguments += " -silent";
            }

            using (Process process = Process.Start(startInfo))
            {
                const int timeOutMillisec = 60 * 1000;
                if (!process.WaitForExit(timeOutMillisec))
                {
                    // タイムアウト
                    process.Kill();
                    throw new LayoutConverterException(string.Format("FFNT conversion timed out. [{0}]", ffntPath));
                }
                if (process.ExitCode != 0)
                {
                    // 失敗
                    throw new LayoutConverterException(string.Format("FFNT conversion failed. [{0}]", ffntPath));
                }
            }
        }

        /// <summary>
        /// シェーダーキー0 の値がコンバイナ関係のシェーダーかどうか判定します。
        /// </summary>
        /// <param name="key0"></param>
        /// <returns></returns>
        private bool IsCombinerShaderKey(int key0)
        {
            return key0 == DetailedCombinerKey0 ||
                key0 == CombinerUserShaderKey0;
        }

        /// <summary>
        /// ペインエフェクト用のシェーダーをバリエーションに追加します。
        /// </summary>
        /// <param name="rlyt"></param>
        void AddPaneEffectShaders_(List<Lyt.ArchiveShaderFile> archiveShaderFileList, LytInfo rlyt)
        {
            bool maskUsed = false;
            bool dropShadowUsed = false;

            // ペインのマスク機能などの専用シェーダーを追加する
            foreach (var paneIter in rlyt.PaneDic)
            {
                // 使われているマスクテクスチャの種類を抜き出す。
                if (paneIter.Value.IsMaskEnabled())
                {
                    maskUsed = true;
                }

                if (paneIter.Value.IsDropShadowEnabled())
                {
                    dropShadowUsed = true;
                }
            }

            // 使用されているマスク用シェーダーを追加する
            if (maskUsed)
            {
                Lyt.ArchiveShaderFile shader = new Lyt.ArchiveShaderFile();
                shader.key0 = MaskShader;
                shader.key1 = 0;
                shader.firstBlend = shader.key0;
                shader.secondBlend = shader.key1;
                shader.texMapCount = 2;
                shader.useDetailedCombiner = 0;
                shader.useCombinerUserShader = 0;
                shader.stageBits = new UInt32[6 * 4];
                shader.stageCount = 0;
                archiveShaderFileList.Add(shader);
            }

            // ドロップシャドウ用シェーダーを追加する
            if (dropShadowUsed)
            {
                foreach(var id in Enum.GetValues(typeof(DropShadowShaderId)))
                {
                    Lyt.ArchiveShaderFile shader = new Lyt.ArchiveShaderFile();
                    shader.key0 = DropShadowShader;
                    shader.key1 = (ushort)((int)id);
                    shader.firstBlend = shader.key0;
                    shader.secondBlend = shader.key1;
                    // ノックアウトシェーダーの時のみテクスチャを 2 枚使用する。
                    if ((int)id / DropShadowColorKnockoutValue > 0)
                    {
                        shader.texMapCount = 2;
                    }
                    else
                    {
                        shader.texMapCount = 1;
                    }
                    shader.useDetailedCombiner = 0;
                    shader.useCombinerUserShader = 0;
                    shader.stageBits = new UInt32[6 * 4];
                    shader.stageCount = 0;
                    archiveShaderFileList.Add(shader);
                }
            }
        }

        //------------------------------------------------------------------------

        private bool CheckShaderKeyExsit_(List<Lyt.ArchiveShaderFile> archiveShaderFileList, int[] shaderKeys, bool perspectiveTextureProjection, bool proceduralShape, int texMapCount, int stageCount, UInt32[] stageBits)
        {
            bool exist = false;
            foreach (Lyt.ArchiveShaderFile shader in archiveShaderFileList.OfType<Lyt.ArchiveShaderFile>())
            {
                if (!IsCombinerShaderKey(shaderKeys[0]))
                {
                    if (shader.key0 == shaderKeys[0] &&
                        shader.key1 == shaderKeys[1] &&
                        shader.perspectiveTextureProjection == perspectiveTextureProjection &&
                        shader.proceduralShape == proceduralShape)
                    {
                        exist = true;
                        break;
                    }
                }
                else
                {
                    // 詳細コンバイナ用シェーダーの重複チェック
                    if (shader.key0 == shaderKeys[0]
                    && shader.texMapCount == texMapCount
                    && shader.stageCount == stageCount
                    && shader.stageBits.SequenceEqual(stageBits))
                    {
                        exist = true;
                        break;
                    }
                }
            }

            return exist;
        }

        //------------------------------------------------------------------------

        private void AddToArchiveShaderFileList_(List<Lyt.ArchiveShaderFile> archiveShaderFileList, int[] shaderKeys, bool perspectiveTextureProjection, bool proceduralShape, int texMapCount, ref UInt32[] stageBits, int stageCount)
        {
            Lyt.ArchiveShaderFile shader = new Lyt.ArchiveShaderFile();

            shader.key0 = (ushort)shaderKeys[0];
            shader.key1 = (ushort)shaderKeys[1];

            if (!IsCombinerShaderKey(shaderKeys[0]))
            {
                // 通常のブレンドシェーダーで透視投影テクスチャを使用している場合は専用のシェーダーが必要となるためルールにのっとってシェーダーキーを操作する。
                // 2 枚以下の場合は shaderKey0 が 250, shaderKey1 が ShaderId に対応した値でバリエーションが定義されている。
                // 3 枚ブレンドの場合はブレンド設定から作成した Key0 と Key1 に対して、 Key0 に 200 を加算した場所にバリエーションが定義されている。
                // 詳細コンバイナシェーダーでは対応していないためキーを操作しません。
                if (perspectiveTextureProjection)
                {
                    if (texMapCount < 3)
                    {
                        shader.key0 = PerspectiveTextureProjectionDefaultShader;
                        if (texMapCount < 2)
                        {
                            shader.key1 = (ushort)texMapCount;
                        }
                        else
                        {
                            shader.key1 = (ushort)(2 + shaderKeys[0]);
                        }
                    }
                    else
                    {
                        shader.key0 += PerspectiveTextureProjectionBlendShaderOffset;
                    }
                }

                // 通常のブレンドシェーダーで角丸機能を使用している場合は専用のシェーダーが必要となるためルールにのっとってシェーダーキーを操作する。
                // 2 枚以下の場合は shaderKey0 が 251, shaderKey1 が ShaderId に対応した値でバリエーションが定義されている。
                // 3 枚ブレンドの場合はブレンド設定から作成した Key0 と Key1 に対して、 Key0 に 220 を加算した場所にバリエーションが定義されている。
                // 詳細コンバイナシェーダーでは対応していないためキーを操作しません。
                if (proceduralShape)
                {
                    if (texMapCount < 3)
                    {
                        shader.key0 = ProceduralShapeDefaultShader;
                        if (texMapCount < 2)
                        {
                            shader.key1 = (ushort)texMapCount;
                        }
                        else
                        {
                            shader.key1 = (ushort)(2 + shaderKeys[0]);
                        }
                    }
                    else
                    {
                        shader.key0 += ProceduralShapeBlendShaderOffset;
                    }
                }
            }

            shader.firstBlend = (ushort)shaderKeys[0];
            shader.secondBlend = (ushort)shaderKeys[1];
            shader.texMapCount = texMapCount;
            shader.useDetailedCombiner = shaderKeys[0] == DetailedCombinerKey0 ? 1 : 0;
            shader.useCombinerUserShader = shaderKeys[0] == CombinerUserShaderKey0 ? 1 : 0;
            shader.stageBits = new UInt32[6 * 4];
            stageBits.CopyTo(shader.stageBits, 0);
            shader.stageCount = stageCount;
            shader.perspectiveTextureProjection = perspectiveTextureProjection;
            shader.proceduralShape = proceduralShape;

            archiveShaderFileList.Add(shader);
        }

        //------------------------------------------------------------------------

        private bool IsUsePerspectiveTextureProjection_(Lyt.MaterialInfo materialInfo)
        {
            foreach (var texCoordGen in materialInfo.texCoordGen)
            {
                if (texCoordGen.srcParam == Lyt.TexGenSrc.PerspectiveProjection ||
                    texCoordGen.srcParam == Lyt.TexGenSrc.PaneBasedPerspectiveProjection)
                {
                    return true;
                }
            }

            return false;
        }

        //------------------------------------------------------------------------

        private bool IsBasicCombiner_(Lyt.MaterialInfo materialInfo)
        {
            return !materialInfo.UseLowLevelCombinerSettings && !materialInfo.UseCombinerUserShaderSettings;
        }

        //------------------------------------------------------------------------

        private bool IsDetailedCombiner_(Lyt.MaterialInfo materialInfo)
        {
            return materialInfo.UseLowLevelCombinerSettings && !materialInfo.UseCombinerUserShaderSettings;
        }

        //------------------------------------------------------------------------

        private bool IsCombinerUserShaderCombiner_(Lyt.MaterialInfo materialInfo)
        {
            return !materialInfo.UseLowLevelCombinerSettings &&
                        materialInfo.UseCombinerUserShaderSettings;
        }

        //------------------------------------------------------------------------
        private void DoShaderStuff_(LytInfo rlyt, string inFileName, string outDirName)
        {
            // コンバイナユーザーシェーダー用 ECMBファイル
            List<string> _combinerUserShaderFileList = new List<string>();

            // シェーダファイルのコピー
            foreach (Lyt.MaterialInfo materialInfo in rlyt.MaterialList)
            {
                int[] shaderKeys = { 0, 0 };
                int texMapCount = 0;
                bool perspectiveTextureProjection = false;
                bool proceduralShape = materialInfo.Pane.IsProceduralShapeEnabled();
                int stageCount = 0;
                const int stageBistsCount = 6 * 4;
                UInt32[] stageBits = new UInt32[stageBistsCount];     // ※最大段数は LayoutEditor 内の LECore.LayoutEditorCore.PlatformDetail.MaxTevStageCount に依存します

                //標準コンバイナ
                if (IsBasicCombiner_(materialInfo))
                {
                    perspectiveTextureProjection = IsUsePerspectiveTextureProjection_(materialInfo);

                    // 透視投影テクスチャの機能を使用する場合は 2 枚ブレンド以下でもアーカイブシェーダーに含まれる。
                    if (materialInfo.tevStage.Length < 2 &&
                        !perspectiveTextureProjection &&
                        !proceduralShape)
                    {
                        continue;
                    }

                    // ブレンド設定がシェーダーキーとなっている
                    for (int i = 0; i < materialInfo.tevStage.Length; ++i)
                    {
                        shaderKeys[i] = (int)materialInfo.tevStage[i].rgb.mode;
                    }

                    if (!proceduralShape)
                    {
                        texMapCount = 1 + materialInfo.tevStage.Length;
                    }
                    else
                    {
                        // 角丸機能シェーダーではテクスチャ 0 枚の時にシェーダーを切り替えるためテクスチャ数を直接設定する。
                        texMapCount = materialInfo.GetTextureCount();
                    }
                }
                // 詳細コンバイナ
                else if (IsDetailedCombiner_(materialInfo))
                {
                    // ソース段数
                    Debug.Assert(materialInfo.tevStage.Count() <= 6, "number of stages can be used up to 6");

                    stageCount = materialInfo.tevStage.Count();
                    for (int i = 0; i < stageCount; i++)
                    {
                        var stage = materialInfo.tevStage[i];
                        stageBits[i * 4 + 0] = BinaryLytWriter.GetDetailedCombinerStageBit0(stage);
                        stageBits[i * 4 + 1] = BinaryLytWriter.GetDetailedCombinerStageBit1(stage);
                        stageBits[i * 4 + 2] = BinaryLytWriter.GetDetailedCombinerStageBit2(stage);
                        stageBits[i * 4 + 3] = BinaryLytWriter.GetDetailedCombinerStageBit3(stage);
                    }
                    // 静的分岐版詳細コンバイナ用統合アーカイブシェーダーの場合は専用のシェーダーキーの値が設定される
                    shaderKeys[0] = DetailedCombinerKey0;
                    shaderKeys[1] = 0;     // 利用しません。
                    texMapCount = materialInfo.texMap.Count();
                }
                // コンバイナユーザーシェーダー
                else if (IsCombinerUserShaderCombiner_(materialInfo))
                {
                    Debug.Assert(materialInfo.combinerUserShader != null);
                    string fileName = materialInfo.combinerUserShader.fileName;
                    string fileNameWithoutExtension = Path.GetFileNameWithoutExtension(materialInfo.combinerUserShader.fileName);

                    // オプションに --ecmb-empty-error が無い、又は false の場合は、ファイル名が設定されていない場合にエラーにせず、登録を行わない。
                    if (!_isEcmbEmptyErr && string.IsNullOrWhiteSpace(fileNameWithoutExtension))
                    {
                        continue;
                    }

                    if (string.IsNullOrWhiteSpace(fileNameWithoutExtension))
                    {
                        throw new LayoutConverterException(
                            string.Format(
                                Properties.Resources.ErrorEcmbFileNameEmpty,
                                inFileName,
                                materialInfo.name
                            )
                        );
                    }

                    Debug.Assert(fileNameWithoutExtension.Count() <= stageBistsCount * sizeof(UInt32), "fileName is over " + (stageBistsCount * sizeof(UInt32)).ToString() + "-character");
                    if (_combinerUserShaderFileList.Contains(fileName))
                    {
                        continue;
                    }
                    // 重複しなければ追加
                    _combinerUserShaderFileList.Add(fileName);

                    // コンバイナユーザーシェーダーの場合は専用のシェーダーキーの値が設定される
                    shaderKeys[0] = CombinerUserShaderKey0;
                    shaderKeys[1] = 0;     // 利用しません。
                    texMapCount = materialInfo.texMap.Count();

                    byte[] codeArray = new byte[stageBits.Count() * sizeof(UInt32)];
                    fileNameWithoutExtension.Select(s => Convert.ToByte(s)).ToArray().CopyTo(codeArray, 0);    // char => byte 配列の変換

                    for (int i = 0; i < stageBits.Count(); i++)
                    {
                        stageBits[i] = BitConverter.ToUInt32(codeArray, i * 4);
                    }
                }
                else
                {
                    // 両方のフラグが同時に設定される事はない。
                    Debug.Assert(false, "UseLowLevelCombinerSettings and UseCombinerUserShaderSettings are set at the same time");
                    continue;
                }

                // 重複しなければコピー対象に追加
                if (CheckShaderKeyExsit_(rlyt.ArchiveShaderFileList, shaderKeys, perspectiveTextureProjection, proceduralShape, texMapCount, stageCount, stageBits))
                {
                    continue;
                }

                AddToArchiveShaderFileList_(rlyt.ArchiveShaderFileList, shaderKeys, perspectiveTextureProjection, proceduralShape, texMapCount, ref stageBits, stageCount);
            }

            // ペインエフェクト用シェーダーを ArchiveShaderFileList に追加します。
            AddPaneEffectShaders_(rlyt.ArchiveShaderFileList, rlyt);

            // ArchiveShaderFileList の内容を生成します。
            ShaderFileUtil.MakeArchiveShaders(rlyt.ArchiveShaderFileList, outDirName, System.IO.Path.GetFileNameWithoutExtension(inFileName));

            // コンバイナユーザーシェーダ用のチェックを行う
            if (_combinerUserShaderFileList.Count != 0)
            {
                CheckCombinerUserShader(rlyt, _shaderEnvDirectories, _combinerEditorPath);
            }

            // コンバイナユーザーシェーダ用の GLSL を作成する
            ShaderFileUtil.MakeCombinerUserShader(outDirName, _combinerUserShaderFileList, _shaderEnvDirectories, _combinerEditorPath, Path.GetDirectoryName(rlyt.FilePath));
        }

        //----------------------------------------------------------

        public void Write(string outDirName, string inFileName, LytInfo rlyt, string tileMode, bool isQuiet)
        {
            // fontファイルのコピー
            if (!_bNoCopyFont)
            {
                FontFileUtil.CopyFontFile(
                    inFileName,
                    rlyt.RefFontFileList.Values,
                    outDirName);
            }

            // 複合フォントファイルの変換
            {
                WriteComplexFont(outDirName, inFileName, rlyt.ComplexFontSet, tileMode, isQuiet);
            }

            // シェーダー関連の処理
            DoShaderStuff_(rlyt, inFileName, outDirName);

            string blytDirName = FileUtil.MakeResourceDirectory(outDirName, BinaryLytWriter.TypeOfLayoutResource);
            string outFileName = FileUtil.MakeOutFileNameStr(blytDirName, null, inFileName, null);
            using (FileStream fs = new FileStream(outFileName, FileMode.Create, FileAccess.Write, FileShare.None))
            {
                // アニメーションの方にPerCharacterTransformCurveがあるかを調べる。もしあったらBinaryLanWriterを作成してRlytSubWriterに与える必要あり。
                BinaryLanWriter lanWriter = null;
                if (rlyt.LanInfo != null && rlyt.LanInfo.LanArray != null)
                {
                    foreach (Schema.Flan.LAN lan in rlyt.LanInfo.LanArray)
                    {
                        if (lan.animType == Schema.Flan.AnimationType.PerCharacterTransformCurve)
                        {
                            lanWriter = new BinaryLanWriter(fs);
                            break;
                        }
                    }
                }

                RlytSubWriter subWriter = new RlytSubWriter(
                    new BinaryLytWriter(fs, _bDegammaParameter, _bViewerHidden),
                    rlyt, lanWriter, _bConvertTagChar, _bDiscardTextBoxText, _bBannerFormat, _bCompatible_0_12_x, _bDegammaParameter, _isKeepingFontScaleEnabled);

                subWriter.Write();
            }
        }
    }

    /// <summary>
    /// 拡張ユーザー情報を書き出すクラスです。
    /// </summary>
    class UserDataSubWriter
    {
        // システム拡張ユーザーデータのフォーマットバージョン
        public const ushort SystemExtUserDataVersion = 0;
        // システム拡張ユーザーデータのデータ名
        public const string SystemExtUserDataName = "ui2dsys";

        /// <summary>
        /// 拡張ユーザー情報の種類。
        /// </summary>
        public enum ExUserDataType
        {
            None,
            String,
            IntList,
            FloatList,
            SystemData
        }

        /// <summary>
        /// シリアライズオブジェクトを解釈して、out パラメーターに書き出します。
        /// </summary>
        public delegate void ReadUserDataElement(object userData, out string nameStr, out string valueStr, out ExUserDataType type);

        /// <summary>
        /// システム用拡張ユーザーデータ領域に書き込むバイナリデータを作成します。
        /// </summary>
        /// <param name="systemData">バイナリ化するシステムデータ</param>
        /// <returns>書き出すバイナリデータ</returns>
        public static byte[] CreateSystemExtUserDataBlock(Lyt::SystemExtData systemData)
        {
            MemoryStream ms = new MemoryStream();
            BinaryWriter bw = EndianAwareBinaryWriter.CreateResouceEndianBinaryWriter(ms);

            var binaries = systemData.BinaryDataList;

            if (binaries.Count >= UInt16.MaxValue)
            {
                // バイナリデータの要素数が多すぎる。
                // 現在の仕様ではバイナリデータがひとつより大きくなることはないが、
                // 将来的に可変長のデータを追加した際に適切なエラーメッセージを出力すること。
                return null;
            }

            // バイナリバージョン
            bw.Write(SystemExtUserDataVersion);
            // 要素数
            bw.Write((ushort)binaries.Count);
            // バイナリデータへのオフセットテーブル
            int offset = sizeof(ushort) + sizeof(ushort) + binaries.Count * sizeof(uint);
            foreach (var bin in binaries)
            {
                bw.Write(offset);
                offset += bin.Length;
                // アラインメント調整
                if (offset % 4 != 0)
                {
                    offset += (4 - (offset % 4));
                }
            }
            // バイナリデータ
            foreach (var bin in binaries)
            {
                bw.Write(bin);
                FileUtil.RoundUpFileSize(ms, 4);
            }

            return ms.ToArray();
        }

        /// <summary>
        /// 拡張ユーザー情報ブロックを書き出す。
        /// </summary>
        public static void WriteUserDataBlock(BinaryWriterBase tgtBinWriter, string userDataName, object[] userData, int startIndex, ReadUserDataElement readUserDataElement, bool bDegammaParameter)
        {
            // ユーザデータの名前の重複チェック用
            var userDataNameDic = new Dictionary<string, string>(StringComparer.InvariantCulture);
            var userDataList = new List<object>(userData.Length);

            for (int i = startIndex; i < userData.Length; ++i)
            {
                object userDataObj = userData[i];

                string nameStr = null;
                string valStr = null;
                ExUserDataType type;
                readUserDataElement(userDataObj, out nameStr, out valStr, out type);

                if (valStr == null)
                {
                    continue;
                }

                valStr = valStr.Trim();
                if (valStr == string.Empty)
                {
                    continue;
                }

                userDataList.Add(userDataObj);

                try
                {
                    userDataNameDic.Add(nameStr, nameStr);
                }
                catch (ArgumentException)
                {
                    throw new LayoutDataException(string.Format(Properties.Resources.ErrorSameUserDataNameExist, nameStr));
                }
            }

            if (userDataList.Count == 0)
            {
                return;
            }
            else if (userDataList.Count > UInt16.MaxValue)     // 要素の数が多すぎる?
            {
                throw new LayoutDataException(
                    string.Format(
                        Properties.Resources.ErrorUserDataNum,
                        userDataName,
                        UInt16.MaxValue                             // 最大値
                    )
                );
            }


            MemoryStream nameSm = new MemoryStream();
            MemoryStream intLstMs = new MemoryStream();
            BinaryWriter intLstBw = EndianAwareBinaryWriter.CreateResouceEndianBinaryWriter(intLstMs);

            List<UserDataHead> userDataHeadList = new List<UserDataHead>(userDataList.Count);

            foreach (object userDataObj in userDataList)
            {
                UserDataHead userDataInfo = new UserDataHead();

                string nameStr = null;
                string valStr = null;
                ExUserDataType type;
                readUserDataElement(userDataObj, out nameStr, out valStr, out type);

                bool bUserDataFormatError = false;

                if (type == ExUserDataType.String)
                {

                    userDataInfo.dataOffset = (int)nameSm.Position;
                    valStr = valStr.Replace("\r\n", "\n"); // 改行コードの CRLF を LF に統一する
                    byte[] valBytes = Encoding.ASCII.GetBytes(valStr + "\0"); // Cの終端文字を付加しておく
                    nameSm.Write(valBytes, 0, valBytes.Length);
                    userDataInfo.num = valStr.Length;
                    userDataInfo.type = UserDataType.String;
                }
                else if (type == ExUserDataType.IntList)
                {
                    userDataInfo.dataOffset = (int)intLstBw.BaseStream.Position;
                    string[] intDataList = GetTokens(valStr);

                    Lyt.UserDataIntList userDataIntList = userDataObj as Lyt.UserDataIntList;
                    if (bDegammaParameter && userDataIntList != null && userDataIntList.kind == Lyt.UserDataElementType.AnmByteRGBA4Degamma)
                    {
                        // ガンマ特性を与える、0～255になっていることに注意が必要
                        // 0.00129465f -> (1 / Math::Pow(255.f, 2.2f)) * 255.f
                        // アルファ成分にはガンマ特性を与えない
                        Debug.Assert(intDataList.Length == 4);
                        for(int i = 0; i < intDataList.Length - 1; i++)
                        {
                            int numValue;
                            bool parsed = Int32.TryParse(intDataList[i], out numValue);
                            if (parsed)
                            {
                                intDataList[i] = ((int)(Math.Pow(numValue, 2.2f) * 0.00129465)).ToString();
                            }
                        }
                    }

                    try
                    {
                        Array.ForEach(
                            intDataList,
                            delegate (string val) { intLstBw.Write(int.Parse(val)); }
                        );
                    }
                    catch (FormatException) { bUserDataFormatError = true; }
                    catch (OverflowException) { bUserDataFormatError = true; }

                    intLstBw.Flush();
                    userDataInfo.num = intDataList.Length;
                    userDataInfo.type = UserDataType.IntList;
                }
                else if (type == ExUserDataType.FloatList)
                {
                    userDataInfo.dataOffset = (int)intLstBw.BaseStream.Position;
                    string[] fltDataList = GetTokens(valStr);

                    try
                    {
                        Array.ForEach(
                            fltDataList,
                            delegate (string val) { intLstBw.Write(float.Parse(val)); }
                        );
                    }
                    catch (FormatException) { bUserDataFormatError = true; }
                    catch (OverflowException) { bUserDataFormatError = true; }

                    intLstBw.Flush();
                    userDataInfo.num = fltDataList.Length;
                    userDataInfo.type = UserDataType.FloatList;
                }
                else if (type == ExUserDataType.SystemData)
                {
                    // readUserDataElement 内で型が確定しているためキャストしても問題ない。
                    var systemData = (Lyt::SystemExtData)userDataObj;

                    // システムデータはバイナリデータを扱うため、データを文字列で扱う既存のユーザーデータの仕組みは使用できない。
                    // SysteExtData から直接バイナリデータを取得してデータを書き出す。

                    userDataInfo.dataOffset = (int)intLstBw.BaseStream.Position;

                    byte[] binaryData = CreateSystemExtUserDataBlock(systemData);
                    if (binaryData != null)
                    {
                        intLstBw.Write(binaryData);
                        intLstBw.Flush();
                        userDataInfo.num = 1;
                        userDataInfo.type = UserDataType.SystemData;
                    }

                }
                else if (type == ExUserDataType.None)
                {
                    continue;
                }

                if (bUserDataFormatError)
                {
                    throw new LayoutDataException(
                        string.Format(
                            Properties.Resources.ErrorUserDataItemFormat,
                            userDataName,                                  // ペイン名
                            string.IsNullOrEmpty(nameStr) ?             // ユーザデータ名
                                userDataHeadList.Count.ToString() :          // 名前が無いならインデックス値
                                nameStr
                        )
                    );
                }

                if (userDataInfo.num > UInt16.MaxValue)     // 要素の数が多すぎる?
                {
                    throw new LayoutDataException(
                        string.Format(
                            Properties.Resources.ErrorUserDataItemNum,
                            userDataName,                                  // ペイン名
                            string.IsNullOrEmpty(nameStr) ?             // ユーザデータ名
                                userDataHeadList.Count.ToString() :          // 名前が無いならインデックス値
                                nameStr,
                            UInt16.MaxValue                             // 最大値
                        )
                    );
                }

                if (string.IsNullOrEmpty(nameStr))
                {
                    userDataInfo.nameStrOffset = 0;  // 名前なし
                }
                else
                {
                    userDataInfo.nameStrOffset = (int)nameSm.Position;
                    byte[] nameBytes = Encoding.ASCII.GetBytes(nameStr + "\0"); // Cの終端文字を付加しておく
                    nameSm.Write(nameBytes, 0, nameBytes.Length);
                }

                userDataHeadList.Add(userDataInfo);
            }

            FileUtil.RoundUpFileSize(nameSm, 4);

            int baseOffset = (int)(userDataHeadList.Count * BinaryLytWriter.SizeOfUserDataStruct);
            int strOffset = (int)intLstBw.BaseStream.Position;

            foreach (UserDataHead userDataInfo in userDataHeadList)
            {
                userDataInfo.nameStrOffset += baseOffset + strOffset;
                userDataInfo.dataOffset += baseOffset;

                switch (userDataInfo.type)
                {
                    case UserDataType.String:
                        userDataInfo.dataOffset += strOffset;
                        break;
                }

                baseOffset -= (int)BinaryLytWriter.SizeOfUserDataStruct;
            }

            long blockHeadFilePos = tgtBinWriter.Stream.Position;

            ushort userDataNum = (ushort)userDataHeadList.Count;
            tgtBinWriter.WriteUserDataList(userDataNum, 0);

            userDataHeadList.ForEach(
                delegate(UserDataHead data) { tgtBinWriter.WriteStruct(data); }
            );

            intLstMs.WriteTo(tgtBinWriter.Stream);
            nameSm.WriteTo(tgtBinWriter.Stream);

            FileUtil.WriteDataBlock(
                tgtBinWriter.Stream,
                delegate(uint blockSize)
                {
                    tgtBinWriter.WriteUserDataList(userDataNum, blockSize);
                },
                blockHeadFilePos);
        }


        /// <summary>
        /// 文字列を、文字配列に分割する。
        /// </summary>
        static string[] GetTokens(string value)
        {
            return value.Split(new char[] { ' ' }, StringSplitOptions.RemoveEmptyEntries);
        }
    }

    class RlytSubWriter
    {
        readonly BinaryLytWriter _tgtBinWriter;
        readonly LytInfo _rlyt;
        readonly bool _bConvertTagChar;
        readonly bool _bDiscardTextBoxText;
        readonly bool _bBannerFormat;
        BinaryLanWriter _lanWriter;
        readonly bool _bCompatible_0_12_x;
        readonly bool _DegammaParameter;
        readonly bool _isKeepingFontScaleEnabled;

        ushort _dataBlockNum = 0;

        Stream _stream { get { return _tgtBinWriter.Stream; } }

        public RlytSubWriter(BinaryLytWriter tgtBinWriter, LytInfo rlyt, BinaryLanWriter lanWriter, bool bConvertTagChar, bool bDiscardTextBoxText, bool bBannerFormat, bool bCompatible_0_12_x, bool bDegammaParameter, bool isKeepingFontScaleEnabled)
        {
            _tgtBinWriter = tgtBinWriter;
            _rlyt = rlyt;
            _bConvertTagChar = bConvertTagChar;
            _bDiscardTextBoxText = bDiscardTextBoxText;
            _bBannerFormat = bBannerFormat;
            _lanWriter = lanWriter;
            _bCompatible_0_12_x = bCompatible_0_12_x;
            _DegammaParameter = bDegammaParameter;
            _isKeepingFontScaleEnabled = isKeepingFontScaleEnabled;
        }

        private bool HasAnyUserData_()
        {
            return (_rlyt.UserData != null && _rlyt.UserData.Length > 0) || (_rlyt.SystemExtData != null);
        }

        private object[] BuildUserDataArray_()
        {
            List<object> userDataSet = new List<object>();

            // SystemExtData は必ず 0 番目 に
            if (_rlyt.SystemExtData != null)
            {
                userDataSet.Add(_rlyt.SystemExtData);
            }

            if(_rlyt.UserData != null)
            {
                userDataSet.AddRange(_rlyt.UserData);
            }

            return userDataSet.ToArray();
        }

        public void Write()
        {
            _tgtBinWriter.WriteBinaryFileHeader(0, 0, false, _bCompatible_0_12_x);      // バイナリファイルヘッダ

            // Layoutのデータ
            string layoutName = Path.GetFileNameWithoutExtension(_rlyt.FilePath);
            Write(_rlyt.ScreenSetting, _rlyt.PartsSize, layoutName);
            ++_dataBlockNum;

            // Layoutの拡張ユーザーデーター(Layout の次のブロックとして書き出す)
            if (HasAnyUserData_())
            {
                UserDataSubWriter.WriteUserDataBlock(_tgtBinWriter, layoutName, BuildUserDataArray_(), 0, ReadUserDataElement_, _DegammaParameter);
                ++_dataBlockNum;
            }

            // TextureListのデータ
            // テクスチャファイルを1つも参照していないときは、このブロック自体出力しない
            if (_rlyt.RefTextureFileList.Count > 0)
            {
                WriteBlock(_rlyt.RefTextureFileList);
                ++_dataBlockNum;
            }

            // FontListのデータ
            // フォントファイルを1つも参照していないときは、このブロック自体出力しない
            if (_rlyt.RefFontFileList.Count > 0)
            {
                WriteBlock(_rlyt.RefFontFileList);
                ++_dataBlockNum;
            }

            // MaterialListのデータ
            // マテリアルを1つも参照していないときは、このブロック自体出力しない
            if (_rlyt.MaterialList.Count > 0)
            {
                WriteBlock(_rlyt.MaterialList);
                ++_dataBlockNum;
            }

            // ShapeInfoList のデータ
            // シェイプを1つも参照していないときは、このブロック自体出力しない
            if (_rlyt.ShapeInfoList.Count > 0)
            {
                WriteBlock(_rlyt.ShapeInfoList);
                ++_dataBlockNum;
            }

            // CaptureTextureList のデータ
            // キャプチャテクスチャをひとつも使用していないときは、このブロック自体出力しない
            if (_rlyt.RefCaptureTextureList.Count > 0)
            {
                WriteBlock(_rlyt.RefCaptureTextureList);
                ++_dataBlockNum;
            }

            // Paneのデータ
            //   ルートのペインより、再帰的にペインのデータを書き込む
            WritePaneHierarchy(_rlyt.RootPane, null);

            // グループ情報
            WriteGroupTree(_rlyt.GroupSet.group);

            // ステートの情報
            if (_rlyt.StateMachine != null)
            {
                WriteState(_rlyt.StateMachine);
                ++_dataBlockNum;
            }

            // コントロールのデータ(定義されていて、コントロール名が空文字でないときのみ出力する)
            if (_rlyt.Control != null)
            {
                foreach (Lyt::Control control in _rlyt.Control)
                {
                    if (control.name.Length > 0)
                    {
                        WriteControl(control);
                        ++_dataBlockNum;
                        if (control.userData != null && control.userData.Length > 0)
                        {
                            UserDataSubWriter.WriteUserDataBlock(_tgtBinWriter, control.name, control.userData, 0, ReadUserDataElement_, _DegammaParameter);
                            ++_dataBlockNum;
                        }
                    }
                }
            }

            long fileSize = _stream.Position;   // ファイルポインタの現在位置がファイルサイズ
            _stream.Position = 0;
            // この時点でファイル書き込みが全て終了したので、バイナリファイルヘッダを更新
            _tgtBinWriter.WriteBinaryFileHeader((uint)fileSize, _dataBlockNum, _bBannerFormat, _bCompatible_0_12_x);
        }

        /// <summary>
        /// ユーザー情報を out パラメーターに読み込みます。
        /// </summary>
        public static void ReadUserDataElement_(object userDataObj, out string nameStr, out string valueStr, out UserDataSubWriter.ExUserDataType type)
        {

            if (userDataObj is Lyt::UserDataString)
            {
                var src = userDataObj as Lyt::UserDataString;
                nameStr = src.name;
                valueStr = src.Value;
                type = UserDataSubWriter.ExUserDataType.String;
            }
            else if (userDataObj is Lyt::UserDataIntList)
            {
                var src = userDataObj as Lyt::UserDataIntList;
                nameStr = src.name;
                valueStr = src.Value;
                type = UserDataSubWriter.ExUserDataType.IntList;
            }
            else if (userDataObj is Lyt::UserDataFloatList)
            {
                var src = userDataObj as Lyt::UserDataFloatList;
                nameStr = src.name;
                valueStr = src.Value;
                type = UserDataSubWriter.ExUserDataType.FloatList;
            }
            else if (userDataObj is Lyt::SystemExtData)
            {
                var src = userDataObj as Lyt::SystemExtData;
                //valueStr = src.Value;
                // システムデータ用拡張ユーザーデータの名前は固定(バイナリデータに含まれるので極力短いほうが良い)。
                nameStr = UserDataSubWriter.SystemExtUserDataName;
                // バイナリデータを扱っているため、文字列に変換する仕組み上にそのまま乗っかることができない。
                // ダミーのテキストを返して上位実装で直接バイナリデータを取得する。
                valueStr = "dummy";
                type = UserDataSubWriter.ExUserDataType.SystemData;
            }
            else if (null != (userDataObj as Lyt::UserDataNone))
            {
                nameStr = null;
                valueStr = null;
                type = UserDataSubWriter.ExUserDataType.None;
            }
            else
            {
                throw new LayoutDataException(string.Format(Properties.Resources.ErrorUnknownUserDataType, userDataObj.GetType()));
            }
        }

        /// <summary>
        /// ユーザー情報の型を out パラメータに読み込みます。
        /// </summary>
        /// <param name="userDataObj"></param>
        /// <param name="kind"></param>
        /// <remarks> 型は Lyt.UserDataElementType と同様に Lan.UserDataElementType が存在します。</remarks>
        public static void ReadUserDataKind(object userDataObj, out Lyt.UserDataElementType kind)
        {
            if (userDataObj is Lyt::UserDataString)
            {
                var src = userDataObj as Lyt::UserDataString;
                kind = src.kind;
            }
            else if (userDataObj is Lyt::UserDataIntList)
            {
                var src = userDataObj as Lyt::UserDataIntList;
                kind = src.kind;
            }
            else if (userDataObj is Lyt::UserDataFloatList)
            {
                var src = userDataObj as Lyt::UserDataFloatList;
                kind = src.kind;
            }
            else if (userDataObj is Lyt::SystemExtData)
            {
                var src = userDataObj as Lyt::SystemExtData;
                kind = Lyt.UserDataElementType.None;
            }
            else if (null != (userDataObj as Lyt::UserDataNone))
            {
                kind = Lyt.UserDataElementType.None;
            }
            else
            {
                throw new LayoutDataException(string.Format(Properties.Resources.ErrorUnknownUserDataType, userDataObj.GetType()));
            }
        }

        void WriteBlock(SortedList<string, Lyt::TextureFile> texFiles)
        {
            long blockHeadFilePos = _stream.Position;

            _tgtBinWriter.WriteStruct(texFiles, 0);

            // 文字列プールの作成
            NameTable nameTable = DataUtil.GenNameTable(
                GetTextureFileNameEnumerable(texFiles),
                Encoding.ASCII,
                BinaryLytWriter.SizeOfTextureStruct,
                0);

            // Textureの書き込み
            foreach (uint offset in nameTable.offsets)
            {
                _tgtBinWriter.WriteTextureStruct(offset);
            }

            FileUtil.WriteStringPool(_stream, nameTable);     // 文字列プールの書き込み

            FileUtil.WriteDataBlock(
                _stream,
                delegate(uint blockSize)
                {
                    _tgtBinWriter.WriteStruct(texFiles, blockSize);
                },
                blockHeadFilePos);
        }

        /// <summary>
        /// テクスチャファイル名のIEnumerableを取得します。
        /// </summary>
        /// <param name="texFiles"></param>
        /// <returns></returns>
        static IEnumerable<string> GetTextureFileNameEnumerable(SortedList<string, Lyt::TextureFile> texFiles)
        {
            bool isUseBntxTextureFormat = true;
            foreach (Lyt::TextureFile texFile in texFiles.Values)
            {
                // キャプチャテクスチャの場合はフォーマットの変換などは必要ないため設定されているテクスチャの名称をそのまま返す。
                if (texFile.IsCaptureTexture)
                {
                    yield return texFile.imagePath;
                }
                else
                {
                    if (texFile.IsNoIndirect)
                    {
                        yield return texFile.GetConvertedFileName(false, isUseBntxTextureFormat);
                    }
                    if (texFile.IsIndirect)
                    {
                        yield return texFile.GetConvertedFileName(true, isUseBntxTextureFormat);
                    }
                }
            }
        }

        void WriteBlock(SortedList<string, Lyt::FontFile> fontFiles)
        {
            long blockHeadFilePos = _stream.Position;

            _tgtBinWriter.WriteStruct(fontFiles, 0);

            // 文字列プールの作成
            NameTable nameTable = DataUtil.GenNameTable(
                GetFontRegisterNameEnumerable(fontFiles),
                Encoding.ASCII,
                BinaryLytWriter.SizeOfFontStruct,
                0);

            // Fontの書き込み
            for (int i = 0; i < fontFiles.Count; ++i)
            {
                _tgtBinWriter.WriteFontStruct(nameTable.offsets[i]);
            }

            FileUtil.WriteStringPool(_stream, nameTable);     // 文字列プールの書き込み

            FileUtil.WriteDataBlock(
                _stream,
                delegate(uint blockSize)
                {
                    _tgtBinWriter.WriteStruct(fontFiles, blockSize);
                },
                blockHeadFilePos);
        }

        /// <summary>
        /// フォントの登録名のIEnumerableを取得します。
        /// </summary>
        /// <param name="fontFiles"></param>
        /// <returns></returns>
        static IEnumerable<string> GetFontRegisterNameEnumerable(SortedList<string, Lyt::FontFile> fontFiles)
        {
            foreach (Lyt::FontFile fontFile in fontFiles.Values)
            {
                yield return fontFile.GetRegisterName();
            }
        }

        void WriteBlock(ICollection<Lyt::MaterialInfo> matInfos)
        {
            long blockHeadFilePos = _stream.Position;

            _tgtBinWriter.WriteStruct(matInfos, 0);

            FileUtil.WriteArray(WriteMaterialStruct, _stream, matInfos, blockHeadFilePos);

            FileUtil.WriteDataBlock(
                _stream,
                delegate(uint blockSize)
                {
                    _tgtBinWriter.WriteStruct(matInfos, blockSize);
                },
                blockHeadFilePos);
        }

        void WriteMaterialStruct(Lyt::MaterialInfo matInfo)
        {
            _tgtBinWriter.WriteStruct(matInfo);

            bool captureTextureEnabled = false;
            bool[] captureTextureFlags = new bool[matInfo.texMap.Length];

            // TexMap
            int count = 0;
            foreach (Lyt::TexMap texMap in matInfo.texMap)
            {
                // バナーモードオプション指定時に、テクスチャのフィルタモードでLinear以外が設定されているときは、
                // 変換を中止する。
                if (_bBannerFormat)
                {
                    if (texMap.minFilter != Lyt::TexFilter.Linear || texMap.magFilter != Lyt::TexFilter.Linear)
                    {
                        throw new BannerConvertException(string.Format(Properties.Resources.ErrorCantUseTexFilter, matInfo.Pane.name));
                    }
                }

                if (texMap.textureResourceType == Schema.Flyt.TextureResourceType.LocalCaptured ||
                    texMap.textureResourceType == Schema.Flyt.TextureResourceType.OverrideCaptured)
                {
                    captureTextureFlags[count] = true;
                    captureTextureEnabled = true;
                }
                else
                {
                    captureTextureFlags[count] = false;
                }

                _tgtBinWriter.WriteStruct(texMap, _rlyt.RefTextureFileList, LytInfo.IsIndirect(matInfo, count));

                count += 1;
            }

            if (captureTextureEnabled)
            {
                foreach (var flag in captureTextureFlags)
                {
                    var info = new Lyt.TexMapAdditionalInfo();
                    if (flag)
                    {
                        info.Enable(Schema.Flyt.TexMapAdditionalInfo.InfoType.CaptureTexture);
                    }
                    _tgtBinWriter.WriteStruct(info);
                }
            }

            // テクスチャ行列
            foreach (Lyt::TexMatrix texMatrix in matInfo.texMatrix)
            {
                _tgtBinWriter.WriteStruct(texMatrix);
            }

            // テクスチャ座標変換
            foreach (Lyt::TexCoordGen texCoordGen in matInfo.texCoordGen)
            {
                _tgtBinWriter.WriteStruct(texCoordGen);
            }

            // TEVステージ
            foreach (Lyt::Material_CTRTevStage tevStage in matInfo.tevStage)
            {
                _tgtBinWriter.WriteStruct(tevStage);
            }

            // アルファコンペア
            if (matInfo.alphaCompare != null)
            {
                _tgtBinWriter.WriteStruct(matInfo.alphaCompare);
            }

            // ブレンドモード
            if (matInfo.blendMode != null)
            {
                _tgtBinWriter.WriteStruct(matInfo.blendMode);

                // アルファを別に設定する場合
                if (matInfo.blendModeAlpha != null && matInfo.blendModeAlpha.type == Lyt::BlendMode.Blend)
                {
                    _tgtBinWriter.WriteStruct(matInfo.blendModeAlpha);
                }
            }

            if (!matInfo.UseLowLevelCombinerSettings)
            {
                // インダイレクトのパラメータ
                foreach (Lyt::Material_CTRTevStage tevStage in matInfo.tevStage)
                {
                    if (tevStage.rgb.mode == Lyt::TevMode.Indirect || tevStage.rgb.mode == Lyt::TevMode.BlendIndirect || tevStage.rgb.mode == Lyt::TevMode.EachIndirect)
                    {
                        _tgtBinWriter.WriteIndirectParameter(tevStage);
                        break;
                    }
                }
            }

            if (matInfo.UseLowLevelCombinerSettings)
            {
                _tgtBinWriter.WriteRawTevStageInfo(matInfo);

                // 詳細なコンバイナのパラメータ
                foreach (Lyt::Material_CTRTevStage tevStage in matInfo.tevStage)
                {
                    _tgtBinWriter.WriteRawTevStage(tevStage);
                }
            }

            // 投影テクスチャのパラメータ
            if (matInfo.GetProjectionTexGenParameterNum() > 0)
            {
                foreach (Lyt::TexCoordGen texGen in matInfo.texCoordGen)
                {
                    if (texGen.IsProjection())
                    {
                        _tgtBinWriter.WriteProjectionTexGenParameter(texGen);
                    }
                }
            }

            // フォント影カラー補完のパラメータ
            if (matInfo.HasFontShadowParameter())
            {
                _tgtBinWriter.WriteFontShadowParameter(matInfo.FontShadowBlackColor, matInfo.FontShadowWhiteColor);
            }

            // コンバイナユーザーシェーダのパラメータ
            if (matInfo.UseCombinerUserShaderSettings && matInfo.combinerUserShader.ExistFileName())
            {
                _tgtBinWriter.WriteCombinerUserShader(matInfo);
            }
        }

        void WriteBlock(ICollection<Lyt::ShapeInfo> shapeInfos)
        {
            long blockHeadFilePos = _stream.Position;

            _tgtBinWriter.WriteStruct(shapeInfos, 0);

            FileUtil.WriteArray(WriteShapeStruct, _stream, shapeInfos, blockHeadFilePos);

            FileUtil.WriteDataBlock(
                _stream,
                delegate(uint blockSize)
                {
                    _tgtBinWriter.WriteStruct(shapeInfos, blockSize);
                },
                blockHeadFilePos);
        }

        void WriteShapeStruct(Lyt::ShapeInfo shapeInfo)
        {
            _tgtBinWriter.WriteStruct(shapeInfo);
        }

        void WriteBlock(SortedList<string, Lyt::CaptureTexture> captureTextures)
        {
            long blockHeadFilePos = _stream.Position;

            _tgtBinWriter.WriteStruct(captureTextures, 0);

            // キャプチャテクスチャ関連のデータは設定によってデータサイズが変わるため
            // 固定オフセットは 0 で文字列プールを作成して、実際に書き込まれるキャプチャテクスチャデータサイズ分を計算して加える
            var texAndPaneNames = GetCaptureTextureNameEnumerable(captureTextures).ToArray();
            NameTable nameTable = DataUtil.GenNameTable(
                texAndPaneNames,
                Encoding.ASCII,
                0,
                BinaryLytWriter.SizeOfResCaptureTexutreList);

            uint captureTextureDataSize = 0;
            foreach (var iter in captureTextures)
            {
                captureTextureDataSize += (uint)_tgtBinWriter.GetCaptureTextureStructSize(iter.Value);
            }

            // 文字列テーブルのオフセット調整
            for (int i = 0; i < nameTable.offsets.Count(); ++i)
            {
                nameTable.offsets[i] += captureTextureDataSize;
            }
            nameTable.dataSize += (int)captureTextureDataSize;

            // Textureの書き込み
            foreach (var iter in captureTextures)
            {
                int texNameIndex = Array.IndexOf(texAndPaneNames, iter.Key);
                int paneNameIndex = Array.IndexOf(texAndPaneNames, iter.Value.paneName);
                _tgtBinWriter.WriteCaptureTextureStruct(nameTable.offsets[texNameIndex], nameTable.offsets[paneNameIndex], iter.Value);
            }

            FileUtil.WriteStringPool(_stream, nameTable);     // 文字列プールの書き込み

            FileUtil.WriteDataBlock(
                _stream,
                delegate (uint blockSize)
                {
                    _tgtBinWriter.WriteStruct(captureTextures, blockSize);
                },
                blockHeadFilePos);
        }

        /// <summary>
        /// キャプチャテクスチャ名と参照ペイン名をまとめた IEnumerable を作成します。
        /// </summary>
        /// <param name="captureTextures"></param>
        /// <returns></returns>
        static IEnumerable<string> GetCaptureTextureNameEnumerable(SortedList<string, Lyt::CaptureTexture> captureTextures)
        {
            var list = new List<string>();
            foreach (var iter in captureTextures)
            {
                if (!list.Contains(iter.Key))
                {
                    list.Add(iter.Key);
                }
                if (!list.Contains(iter.Value.paneName))
                {
                    list.Add(iter.Value.paneName);
                }
            }

            return list;
        }

        void WritePaneHierarchy(Lyt::Pane pane, Lyt::Alignment alignment)
        {
            // 自分のデータを書き込む
            WritePane(pane, alignment);

            // 整列ペインの場合、以降のペインは拡張ユーザーデータのシステム利用領域に固有パラメータを持つ
            if (pane.kind == Lyt::PaneKind.Alignment)
            {
                alignment = pane.Item as Lyt::Alignment;
            }

            // 子供のペインを再帰的に処理する
            if (pane.ChildList != null)
            {
                // 開始ヘッダ
                _tgtBinWriter.WriteBlock(BinaryLytWriter.SignatureOfPaneBeginBlock);
                ++_dataBlockNum;

                foreach (Lyt::Pane childPane in pane.ChildList)
                {
                    WritePaneHierarchy(childPane, alignment);
                }

                // 終了ヘッダ
                _tgtBinWriter.WriteBlock(BinaryLytWriter.SignatureOfPaneEndBlock);
                ++_dataBlockNum;
            }
        }

        void AppendAlignmentPaneSystemExtData(Lyt::Pane pane, Lyt::Alignment alignment)
        {
            MemoryStream ms = new MemoryStream();
            BinaryWriter bw = EndianAwareBinaryWriter.CreateResouceEndianBinaryWriter(ms);

            UInt32 flags = 0;
            flags |= (UInt32)(pane.alignmentIgnore            ? 1 << 1 : 0);    // 整列ペインの処理を除外
            flags |= (UInt32)(pane.kind == Lyt::PaneKind.Null ? 1 << 2 : 0);    // 整列ペイン利用時用の Null ペイン判定(ランタイム側で Nullペインかの判定が出来ない為)

            // 整列（水平）と整列（垂直）が同時に設定される事はない。
            Debug.Assert(!(alignment.isAlignmentHorizontalEnabled && alignment.isAlignmentVerticalEnabled), "horizontalAlignment and verticalAlignment are set at the same time");

            float alignmentMargin = 0.0f;                                       // 整列が利用されていない場合や、フラグで有効になっていない場合は '0' とする
            if (alignment.isAlignmentHorizontalEnabled && pane.alignmentMarginEnabled)
            {
                flags |= (UInt32)(pane.alignmentMarginEnabled     ? 1 << 0 : 0);    // 整列マージンの有効無効
                alignmentMargin = pane.alignmentMargin;
            }
            else if (alignment.isAlignmentVerticalEnabled && pane.alignmentVerticalMarginEnabled)
            {
                flags |= (UInt32)(pane.alignmentVerticalMarginEnabled ? 1 << 0 : 0);    // 整列マージンの有効無効
                alignmentMargin = pane.alignmentVerticalMargin;
            }

            bw.Write((int)NW4F.LayoutBinaryConverter.Schema.Flyt.SystemExtData.PaneSystemDataType.AlignmentInfo);
            bw.Write((UInt32)flags);
            bw.Write(alignmentMargin);

            pane.AddSystemExtDataBlock(ms.ToArray());
        }

        void WritePane(Lyt::Pane pane, Lyt::Alignment alignment)
        {
            Debug.WriteLine(pane.name);

            // 親に整列ペインを持つペインは拡張ユーザーデータのシステム利用領域に固有パラメータを持つ
            if (alignment != null)
            {
                AppendAlignmentPaneSystemExtData(pane, alignment);
            }

            switch (pane.kind)
            {
            case Lyt::PaneKind.Null:
            default:
                _tgtBinWriter.WriteNullPane(pane);
                break;
            case Lyt::PaneKind.Picture:
                Write((Lyt::Picture)pane.Item, pane);
                break;
            case Lyt::PaneKind.TextBox:
                Write((Lyt::TextBox)pane.Item, pane);
                break;
            case Lyt::PaneKind.Window:
                Write((Lyt::Window)pane.Item, pane, true);
                break;
            case Lyt::PaneKind.Bounding:
                _tgtBinWriter.WriteBlock((Lyt::Bounding)pane.Item, pane);
                break;
            case Lyt::PaneKind.Capture:
                Write((Lyt::Capture)pane.Item, null, pane);
                break;
            case Lyt::PaneKind.Parts:
                Write((Lyt::Parts)pane.Item, pane);
                break;
            case Lyt::PaneKind.Alignment:
                Write((Lyt::Alignment)pane.Item, pane);
                break;
            case Lyt::PaneKind.Scissor:
                Write((Lyt::Scissor)pane.Item, pane);
                break;
            }
            ++_dataBlockNum;

            // ユーザデータブロック

            // システム用ユーザーデータと通常ユーザーデータを一つの配列にまとめる。
            // システム用がある場合は 0 番固定。
            int combinedCount = pane.userData != null ? pane.userData.Length - 1 : 0;
            combinedCount += pane.SystemExtData != null ? 1 : 0;

            if (combinedCount > 0)
            {
                object[] combinedUserData = new object[combinedCount];

                int writeIndex = 0;
                if (pane.SystemExtData != null)
                {
                    combinedUserData[writeIndex++] = pane.SystemExtData;
                }

                bool basicUserData = true;
                foreach (var userData in pane.userData)
                {
                    // "__BasicUserDataString" を書き出さないようにスキップする。
                    // ユーザーデータの 0 番に必ず入っている。
                    if (basicUserData)
                    {
                        basicUserData = false;
                    }
                    else
                    {
                        combinedUserData[writeIndex++] = userData;
                    }
                }

                if (_bBannerFormat)
                {
                    throw new LayoutDataException(Properties.Resources.ErrorCantUserExtUserData);
                }

                UserDataSubWriter.WriteUserDataBlock(_tgtBinWriter, pane.name, combinedUserData, 0, ReadUserDataElement_, _DegammaParameter);
                ++_dataBlockNum;
            }
        }

        void Write(Lyt::ScreenSetting screenSetting, Lyt::Vec2 partsSize, string layoutName)
        {
            long blockHeadFilePos = _stream.Position;

            _tgtBinWriter.WriteBlock(_rlyt.ScreenSetting, partsSize, 0);

            {
                byte[] bytePartsName = Encoding.ASCII.GetBytes(layoutName + "\0");
                _stream.Write(bytePartsName, 0, bytePartsName.Length);
                FileUtil.RoundUpFileSize(_stream, 4);
            }

            FileUtil.WriteDataBlock(
                _stream,
                delegate(uint blockSize)
                {
                    _tgtBinWriter.WriteBlock(_rlyt.ScreenSetting, partsSize, blockSize);
                },
                blockHeadFilePos);
        }

        void WriteControl(Lyt::Control control)
        {
            long blockHeadFilePos = _stream.Position;

            _tgtBinWriter.WriteControlBlock(control, 0, 0, 0, 0, 0);

            // コントロール名の書き出し
            {
                byte[] byteControlName = Encoding.ASCII.GetBytes(control.name + "\0");
                _stream.Write(byteControlName, 0, byteControlName.Length);
                FileUtil.RoundUpFileSize(_stream, 4);
            }
            long controlUserNameOffset = _stream.Position - blockHeadFilePos;

            // コントロールのユーザー名の書き出し
            {
                byte[] byteControlUserName = Encoding.ASCII.GetBytes(control.userName + "\0");
                _stream.Write(byteControlUserName, 0, byteControlUserName.Length);
                FileUtil.RoundUpFileSize(_stream, 4);
            }
            long controlParameterPaneNamesOffset = _stream.Position - blockHeadFilePos;

            // パラメータペイン名の書き出し
            if (control.parameterPane != null)
            {
                foreach (Lyt::ControlParameterPane parameterPane in control.parameterPane)
                {
                    byte[] byteTagName = Encoding.ASCII.GetBytes(parameterPane.paneName + "\0");
                    Array.Resize(ref byteTagName, BinaryLytWriter.ResourceNameStrMax);
                    _stream.Write(byteTagName, 0, byteTagName.Length);
                }
            }

            // パラメータアニメーションタグ名を書き出すためのNameTableを作成
            if (control.parameterAnimation != null && control.parameterAnimation.Length > 0)
            {
                NameTable nameTable = DataUtil.GenNameTable(
                    GetControlParameterAnimationEnumerable(control.parameterAnimation),
                    Encoding.ASCII,
                    (uint)Marshal.SizeOf(typeof(uint)),
                    0 /* ブロックの先頭ではなく、controlParameterAnimNameOffsetsからのオフセットなので0でよい */);

                FileUtil.WriteOffsetTable(_stream, nameTable.offsets);  // オフセットテーブルの書き込み
                FileUtil.WriteStringPool(_stream, nameTable);           // 文字列プールの書き込み
            }
            long controlFunctionalPaneParameterNameOffsetsOffset = _stream.Position - blockHeadFilePos;

            // 機能ペインのパラメータ名を書き出すためのNameTableを作成
            if (control.parameterPane != null && control.parameterPane.Length > 0)
            {
                NameTable nameTable = DataUtil.GenNameTable(
                    GetControlFunctionalPaneParameterNameEnumerable(control.parameterPane),
                    Encoding.ASCII,
                    (uint)Marshal.SizeOf(typeof(uint)),
                    0 /* ブロックの先頭ではなく、controlFunctionalPaneParameterNameOffsetsOffsetからのオフセットなので0でよい */);

                FileUtil.WriteOffsetTable(_stream, nameTable.offsets);  // オフセットテーブルの書き込み
                FileUtil.WriteStringPool(_stream, nameTable);           // 文字列プールの書き込み
            }
            long controlFunctionalAnimParameterNameOffsetsOffset = _stream.Position - blockHeadFilePos;

            // 機能アニメーションのパラメータ名を書き出すためのNameTableを作成
            if (control.parameterAnimation != null && control.parameterAnimation.Length > 0)
            {
                NameTable nameTable = DataUtil.GenNameTable(
                    GetControlFunctionalAnimParameterNameEnumerable(control.parameterAnimation),
                    Encoding.ASCII,
                    (uint)Marshal.SizeOf(typeof(uint)),
                    0 /* ブロックの先頭ではなく、controlFunctionalAnimParameterNameOffsetsOffsetからのオフセットなので0でよい */);

                FileUtil.WriteOffsetTable(_stream, nameTable.offsets);  // オフセットテーブルの書き込み
                FileUtil.WriteStringPool(_stream, nameTable);           // 文字列プールの書き込み
            }

            FileUtil.WriteDataBlock(
                _stream,
                delegate(uint blockSize)
                {
                    _tgtBinWriter.WriteControlBlock(control, blockSize, (uint)controlUserNameOffset, (uint)controlParameterPaneNamesOffset,
                        (uint)controlFunctionalPaneParameterNameOffsetsOffset, (uint)controlFunctionalAnimParameterNameOffsetsOffset);
                },
                blockHeadFilePos);
        }

        static IEnumerable<string> GetControlParameterAnimationEnumerable(Lyt::ControlParameterAnimation[] parameterAnimations)
        {
            foreach (Lyt::ControlParameterAnimation parameterAnimation in parameterAnimations)
            {
                yield return parameterAnimation.tagName;
            }
        }

        static IEnumerable<string> GetControlFunctionalPaneParameterNameEnumerable(Lyt::ControlParameterPane[] parameterPanes)
        {
            foreach (Lyt::ControlParameterPane parameterPane in parameterPanes)
            {
                yield return parameterPane.name;
            }
        }

        static IEnumerable<string> GetControlFunctionalAnimParameterNameEnumerable(Lyt::ControlParameterAnimation[] parameterAnimations)
        {
            foreach (Lyt::ControlParameterAnimation parameterAnimation in parameterAnimations)
            {
                yield return parameterAnimation.name;
            }
        }

        void Write(Lyt::Picture picture, Lyt::Pane pane)
        {
            long blockHeadFilePos = _stream.Position;

            DataUtil.ResizeArray(ref picture.texCoord, BinaryLytWriter.CapOfTexCoord);

            _tgtBinWriter.WriteBlock(picture, pane, 0, 0, false);

            // テクスチャ座標
            foreach (Lyt::TexCoord texCoord in picture.texCoord)
            {
                _tgtBinWriter.WriteStruct(texCoord);
            }

            // ここから下のデータは ResPicture の flags に対応フラグがセットされているときの出力される。

            // バイナリ互換性維持のため追加情報領域へシェイプバイナリ情報のインデックスを書き込む。
            bool shapeInfoEnabled = picture.ShapeInfo != null;
            if (shapeInfoEnabled)
            {
                _tgtBinWriter.WriteU32(picture.ShapeInfo.BinaryIndex);
            }

            // ここまで

            ushort materialIdx = (ushort)picture.MaterialInfo.BinaryIndex;
            FileUtil.WriteDataBlock(
                _stream,
                delegate(uint blockSize)
                {
                    _tgtBinWriter.WriteBlock(picture, pane, blockSize, materialIdx, shapeInfoEnabled);
                },
                blockHeadFilePos);
        }

        void Write(Lyt::TextBox textBox, Lyt::Pane pane)
        {
            long blockHeadFilePos = _stream.Position;

            // バナーモードオプション指定時に、テキスト配置が既定以外が設定されているときは、
            // 変換を中止する。
            if (_bBannerFormat)
            {
                if (textBox.textAlignment != Lyt::TextAlignment.Synchronous)
                {
                    throw new BannerConvertException(string.Format(Properties.Resources.ErrorCanUseTextAlignment, pane.name));
                }
            }

            _tgtBinWriter.WriteBlock(textBox, pane, 0, 0, 0, 0, 0, 0, 0, 0, 0, false);

            FileUtil.RoundUpFileSize(_stream, 2);

            long textStrOffset = _stream.Position - blockHeadFilePos;

            const int TextBytesMax = 0xFFFE;

            int textStrBytes = 0;
            if (TextBytesMax > 0 && textBox.text.Length > 0)
            {
                string textBoxStr;
                if(!_bDiscardTextBoxText)
                {
                    textBoxStr = _bConvertTagChar? DataUtil.ConvertTagString(textBox.text) : textBox.text;
                    textBoxStr = textBoxStr.Replace("\r\n", "\n"); // 改行コードの CRLF を LF に統一する
                }
                else
                {
                    textBoxStr = string.Empty;
                }

                // テキスト文字列の書き出し
                byte[] byteText = null;
                if (EndianEnvironment.ResouceEndian == Endian.Little)
                {
                    byteText = Encoding.Unicode.GetBytes(String.Concat(textBoxStr, "\0"));
                }
                else
                {
                    byteText = Encoding.BigEndianUnicode.GetBytes(String.Concat(textBoxStr, "\0"));
                }

                if (byteText.Length > TextBytesMax)
                {
                    byteText[TextBytesMax - 2] = byteText[TextBytesMax - 1] = 0;  // 終端文字
                    textStrBytes = TextBytesMax;
                }
                else
                {
                    textStrBytes = byteText.Length;
                }

                _stream.Write(byteText, 0, textStrBytes);
                FileUtil.RoundUpFileSize(_stream, 4);
            }

            long textIDOffset = 0;
            if (textBox.textID != null && textBox.textID.Length > 0)
            {
                textIDOffset = _stream.Position - blockHeadFilePos;

                // テキストIDの書き出し
                byte[] textIDBytes = Encoding.ASCII.GetBytes(textBox.textID + "\0");
                _stream.Write(textIDBytes, 0, textIDBytes.Length);
                FileUtil.RoundUpFileSize(_stream, 4);
            }

            // 行ごとの幅とオフセット
            long lineWidthOffsetOffset = 0;
            if (textBox.lineWidthOffsetEnabled)
            {
                lineWidthOffsetOffset = _stream.Position - blockHeadFilePos;
                _tgtBinWriter.WriteLineWidthOffset(textBox);
                FileUtil.RoundUpFileSize(_stream, 4);
            }

            long perCharacterTransformOffset = 0;
            if (textBox.perCharacterTransformEnabled)
            {
                perCharacterTransformOffset = _stream.Position - blockHeadFilePos;
                Schema.Flan.AnimContent foundAnimContent = null;
                // LanInfoの中を調べて、自分の一文字アニメーションカーブがあったら書き出す
                if (_lanWriter != null && _rlyt.LanInfo != null)
                {
                    foreach (Schema.Flan.LAN lan in _rlyt.LanInfo.LanArray)
                    {
                        if (lan.animType == Schema.Flan.AnimationType.PerCharacterTransformCurve)
                        {
                            foreach (Schema.Flan.AnimContent animContent in lan.animContent)
                            {
                                if (animContent.name == pane.name)
                                {
                                    foundAnimContent = animContent;
                                    break;
                                }
                            }
                        }
                    }
                }
                _tgtBinWriter.WritePerCharacterTransformStruct(textBox, foundAnimContent != null);
                if (foundAnimContent != null)
                {
                    List<Schema.Flan.AnimTarget> list = new List<Schema.Flan.AnimTarget>(foundAnimContent.Items);
                    long animInfoFilePos = _stream.Position;
                    _lanWriter.WriteBlock(Schema.Flan.AnimationType.PerCharacterTransformCurve, list);
                    FileUtil.WriteArray(
                        delegate(Schema.Flan.AnimTarget animTarget)
                        {
                            _lanWriter.WriteBlock(animTarget, Schema.Flan.AnimationType.PerCharacterTransformCurve);
                            _lanWriter.WriteKeys(animTarget.key, 0, false);
                        },
                        _stream,
                        list,
                        animInfoFilePos);
                }
            }

            if (_isKeepingFontScaleEnabled && !isSetFontSizeOriginal(textBox))
            {
                throw new LayoutDataException(string.Format("The --keep-font-scale option requires a valid <fontSizeOriginal> tag in the TextBox[{0}].", pane.name));
            }

            int textBufBytes = 0;
            if (textBox.allocateStringLengthSpecified)
            {
                textBufBytes = Math.Min(TextBytesMax, textBox.allocateStringLength == 0 ? 0 : (int)((textBox.allocateStringLength + 1) * 2));
            }
            else
            {
                textBufBytes = textStrBytes;
            }
            ushort materialIdx = (ushort)textBox.MaterialInfo.BinaryIndex;
            ushort fontIdx = (ushort)_rlyt.RefFontFileList.IndexOfKey(textBox.font);
            FileUtil.WriteDataBlock(
                _stream,
                delegate(uint blockSize)
                {
                    _tgtBinWriter.WriteBlock(textBox, pane, blockSize, (ushort)textBufBytes, (ushort)textStrBytes, (uint)textStrOffset, materialIdx, fontIdx, (uint)textIDOffset, (uint)lineWidthOffsetOffset, (uint)perCharacterTransformOffset, _isKeepingFontScaleEnabled);
                },
                blockHeadFilePos);
        }

        // FontSizeOriginal が設定されているかどうかを取得します。
        private bool isSetFontSizeOriginal(Lyt.TextBox textBox)
        {
            return (textBox.fontSize != null &&
                textBox.fontSizeOriginal != null &&
                textBox.fontSizeOriginal.x != 0.0f &&
                textBox.fontSizeOriginal.y != 0.0f);
        }

        /// <summary>
        /// ウィンドウフレームのテクスチャサイズを取得します。
        /// </summary>
        private Lyt::Vec2 GetWindowFrameTextureSize_(Lyt::Window window, Lyt::WindowFrameType wndFrmType)
        {
            byte index = _tgtBinWriter.GetWindowFrameType(wndFrmType);
            if (window.frame.Length <= index)
            {
                return null;
            }

            Lyt::WindowFrame frame = window.frame[index];
            if (frame == null)
            {
                return null;
            }

            if (frame.material.texMap == null || frame.material.texMap.Length <= 0)
            {
                return null;
            }

            string imgName = frame.material.texMap[0].imageName;
            Lyt::TextureFile textureFile;
            if (!_rlyt.TextureFileList.TryGetValue(imgName, out textureFile))
            {
                if (!_rlyt.RefTextureFileList.TryGetValue(imgName, out textureFile))
                {
                    return null;
                }
            }

            return textureFile.ActualImageSize;
        }

        void Write(Lyt::Window window, Lyt::Pane pane, bool useFrameSize)
        {
            long blockHeadFilePos = _stream.Position;

            // 利用されていないマテリアルが不正な状態になっており、本来不要であるコンバートエラーを引き起こすため、
            // 部品ペインの上書きプロパティでカラー補完のみ上書きをする場合など、useFrameSize が false になりスキップされます。
            if (useFrameSize)
            {
                // フレームサイズが指定されていなければ、テクスチャからサイズを計算を試みます。
                if (window.frameSize == null || window.frameSize.IsZero)
                {
                    window.frameSize = new Lyt.WindowFrameSize();

                    if (window.kind == Lyt.WindowKind.Around)
                    {
                        Lyt::Vec2 textureSizeLT = GetWindowFrameTextureSize_(window, Lyt::WindowFrameType.CornerLT);
                        Lyt::Vec2 textureSizeRB = GetWindowFrameTextureSize_(window, window.frame.Length > 1 ? Lyt::WindowFrameType.CornerRB : Lyt::WindowFrameType.CornerLT);
                        if (textureSizeLT != null && textureSizeRB != null)
                        {
                            window.frameSize.l = textureSizeLT.x;
                            window.frameSize.t = textureSizeLT.y;
                            window.frameSize.r = textureSizeRB.x;
                            window.frameSize.b = textureSizeRB.y;
                        }
                        else
                        {
                            throw new LayoutDataException(string.Format(Properties.Resources.ErrorCantCalculateFrameSizeLTRB, pane.name));
                        }
                    }
                    else
                    {
                        Lyt::Vec2 textureSizeLT = GetWindowFrameTextureSize_(window, Lyt::WindowFrameType.CornerLT);
                        Lyt::Vec2 textureSizeRT = GetWindowFrameTextureSize_(window, window.frame.Length > 1 ? Lyt::WindowFrameType.CornerRT : Lyt::WindowFrameType.CornerLT);
                        if (textureSizeLT != null && textureSizeRT != null)
                        {
                            window.frameSize.l = textureSizeLT.x;
                            window.frameSize.t = textureSizeLT.y;
                            window.frameSize.r = textureSizeRT.x;
                            window.frameSize.b = textureSizeRT.y;
                        }
                        else
                        {
                            throw new LayoutDataException(string.Format(Properties.Resources.ErrorCantCalculateFrameSizeLTRT, pane.name));
                        }
                    }
                }
            }
            else
            {
                // 適当な初期値を設定しておく。
                window.frameSize = new Lyt.WindowFrameSize();
            }

            _tgtBinWriter.WriteBlock(window, pane, 0, 0, 0);

            long contentOffset = _stream.Position - blockHeadFilePos;

            DataUtil.ResizeArray(ref window.content.texCoord, BinaryLytWriter.CapOfTexCoord);

            _tgtBinWriter.WriteBlock(
                window.content,
                (ushort)window.content.MaterialInfo.BinaryIndex);

            foreach (Lyt::TexCoord texCoord in window.content.texCoord)
            {
                _tgtBinWriter.WriteStruct(texCoord);
            }

            long frameOffsetTableOffset = _stream.Position - blockHeadFilePos;

            // フレーム情報の書き込み
            if (window.frame.Length != 1 && window.frame.Length != 2 && window.frame.Length != 4 && window.frame.Length != 8)
            {
                throw new LayoutDataException(string.Format("window[{0}] - invalid frame num {1}", pane.name, window.frame.Length));
            }

            Lyt::WindowFrame[] frames = new Lyt::WindowFrame[window.frame.Length];
            foreach (Lyt::WindowFrame frame in window.frame)
            {
                int index = _tgtBinWriter.GetWindowFrameType(frame.frameType);
                if (index >= frames.Length)
                {
                    throw new LayoutDataException(string.Format("window[{0}] - invalid frame type", pane.name));
                }
                frames[index] = frame;
            }

            foreach (Lyt::WindowFrame frame in frames)
            {
                if (frame == null)
                {
                    throw new LayoutDataException(string.Format("window[{0}] - invalid frame type", pane.name));
                }
            }

            FileUtil.WriteArray(
                delegate (Lyt::WindowFrame frame)
                {
                    _tgtBinWriter.WriteBlock(frame, (ushort)frame.MaterialInfo.BinaryIndex);
                },
                _stream,
                (ICollection<Lyt::WindowFrame>)frames,
                blockHeadFilePos);

            FileUtil.WriteDataBlock(
                _stream,
                delegate(uint blockSize)
                {
                    _tgtBinWriter.WriteBlock(window, pane, blockSize, (uint)contentOffset, (uint)frameOffsetTableOffset);
                },
                blockHeadFilePos);
        }

        void Write(Lyt::Capture capture, Lyt::CaptureTexture tex, Lyt::Pane pane)
        {
            long blockHeadFilePos = _stream.Position;

            // キャプチャペインのサイズチェック
            // 0 以下の設定はランタイムで再現できないため不可。
            if (pane.size.x <= 0.0f ||
                pane.size.y <= 0.0f)
            {
                throw new LayoutDataException(
                    string.Format(
                        Properties.Resources.ErrorCapturePaneSize,
                        pane.name
                    )
                );
            }

            _tgtBinWriter.WriteBlock(capture, pane, 0);

            if (tex != null)
            {
                // 実際のキャプチャテクスチャのデータを書き込む。
                // 上書きで名前は変わることがなく、ランタイムでも参照しないため名前文字列オフセットには固定で 0 を指定する。
                _tgtBinWriter.WriteCaptureTextureStruct(0, 0, tex);
            }

            FileUtil.WriteDataBlock(
                _stream,
                delegate (uint blockSize)
                {
                    _tgtBinWriter.WriteBlock(capture, pane, blockSize);
                },
                blockHeadFilePos);
        }

        void Write(Lyt::Parts parts, Lyt::Pane pane)
        {
            long blockHeadFilePos = _stream.Position;

            _tgtBinWriter.WriteBlock(parts, pane, 0);

            UInt32[] propertyOffsets = null;
            UInt32[] userDataOffsets = null;
            UInt32[] paneBasicInfoOffsets = null;

            // ランタイムでは PartsBuildDataSet でプロパティデータが上書きしているかどうかを
            // propertyTable の内容(名前)に 0 以外の値が設定されているかどうかで判断しています。
            // 以下の Position を進めているコードで propertyTable 分 0 で埋められていることが前提となる判定のため
            // コードを変更する際はその仕様に影響しないかどうか注意が必要です。
            if (parts.property != null)
            {
                propertyOffsets = new UInt32[parts.property.Length];
                userDataOffsets = new UInt32[parts.property.Length];
                paneBasicInfoOffsets = new UInt32[parts.property.Length];

                // propertyTableのぶんだけstreamを進める
                _stream.Position = _stream.Position + BinaryLytWriter.SizeOfPartsPropertyStruct * parts.property.Length;
            }

            {
                byte[] bytePartsName = Encoding.ASCII.GetBytes(Path.GetFileNameWithoutExtension(parts.path) + "\0");
                _stream.Write(bytePartsName, 0, bytePartsName.Length);
                FileUtil.RoundUpFileSize(_stream, 4);
            }

            // プロパティがある場合は書き出す。Paneの共通部分には、名前以外は仮のデータを入れる。
            if (parts.property != null && parts.property.Length > 0)
            {
                // 部品ペインのためのダミーのペインデータ。名前のみ設定する
                Lyt::Pane dummy = new Lyt::Pane(null, pane.size);
                int propertyCount = 0;
                int userDataCount = 0;
                int paneBasicInfoCount = 0;
                foreach (Lyt::Property property in parts.property)
                {
                    dummy.name = property.target;
                    dummy.userData = null;
                    // 上書き設定はしてあるが実際に上書きは行われていないときに、Itemがnullになる
                    // ことがある。このような場合は出力しない。
                    if (property.Item != null)
                    {
                        // プロパティのブロックのオフセット値を取得しておく
                        propertyOffsets[propertyCount] = (uint)(_stream.Position - blockHeadFilePos);
                        switch (property.kind)
                        {
                            case Lyt::PaneKind.Picture:
                                Write((Lyt::Picture)property.Item, dummy);
                                break;
                            case Lyt::PaneKind.TextBox:
                                Write((Lyt::TextBox)property.Item, dummy);
                                break;
                            case Lyt::PaneKind.Window:
                                bool useFrameSize = property.usageOptions == null || !property.usageOptions.Contains(Lyt::PartsPropertyUsageOptions.UseMaterialColorBlend);
                                Write((Lyt::Window)property.Item, dummy, useFrameSize);
                                break;
                            case Lyt::PaneKind.Capture:
                                // Lyt::Capture(ResPaneを継承) をパーツのデータ構造に合わせるためにダミーで用意して書き込む。
                                // キャプチャテクスチャ上書き時のデータは ResCapture に続けて記載される。
                                Write(new Lyt::Capture(), (Lyt::CaptureTexture)property.Item, dummy);
                                break;
                            case Lyt::PaneKind.Parts:
                                Write((Lyt::Parts)property.Item, dummy);
                                break;
                            case Lyt::PaneKind.Alignment:
                                Write((Lyt::Alignment)property.Item, dummy);
                                break;
                            case Lyt::PaneKind.Scissor:
                                Write((Lyt::Scissor)property.Item, dummy);
                                break;

                        }

                        propertyCount++;
                    }
                    if (property.userData != null && property.userData.Length > 0)
                    {
                        // ユーザーデータのブロックのオフセット値を取得しておく
                        userDataOffsets[userDataCount] = (uint)(_stream.Position - blockHeadFilePos);
                        UserDataSubWriter.WriteUserDataBlock(_tgtBinWriter, property.target, property.userData, 0, ReadUserDataElement_, _DegammaParameter);

                        userDataCount++;
                    }
                    if (NeedPaneBasicInfoOverride(property))
                    {
                        // ペイン基本情報の上書きデータのブロックのオフセット値を取得しておく
                        paneBasicInfoOffsets[paneBasicInfoCount] = (uint)(_stream.Position - blockHeadFilePos);
                        _tgtBinWriter.WritePartsPaneBasicInfoStruct(property);

                        paneBasicInfoCount++;
                    }
                }
            }

            FileUtil.WriteDataBlock(
                _stream,
                delegate(uint blockSize)
                {
                    _tgtBinWriter.WriteBlock(parts, pane, blockSize);
                    // propertyTableを書き出す
                    if (parts.property != null)
                    {
                        int propertyCount = 0;
                        int userDataCount = 0;
                        int paneBasicInfoCount = 0;
                        foreach (Lyt::Property property in parts.property)
                        {
                            if (property.Item != null || property.visible != null || property.userData != null || NeedPaneBasicInfoOverride(property))
                            {
                                uint propertyOffset = 0;
                                uint userDataOffset = 0;
                                uint paneBasicInfoOffset = 0;
                                if (property.Item != null)
                                {
                                    propertyOffset = propertyOffsets[propertyCount];
                                    propertyCount++;
                                }
                                if (property.userData != null)
                                {
                                    userDataOffset = userDataOffsets[userDataCount];
                                    userDataCount++;
                                }
                                if (NeedPaneBasicInfoOverride(property))
                                {
                                    paneBasicInfoOffset = paneBasicInfoOffsets[paneBasicInfoCount];
                                    paneBasicInfoCount++;
                                }
                                _tgtBinWriter.WritePartsPropertyStruct(property, propertyOffset, userDataOffset, paneBasicInfoOffset);
                            }
                        }
                    }
                },
                blockHeadFilePos);
        }

        void Write(Lyt::Alignment alignment, Lyt::Pane pane)
        {
            _tgtBinWriter.WriteBlock(alignment, pane);
        }

        void Write(Lyt::Scissor scissor, Lyt::Pane pane)
        {
            _tgtBinWriter.WriteBlock(scissor, pane);
        }

        void WriteState(Lyt::StateMachine stateMachine)
        {
            _tgtBinWriter.WriteBlockStateMachine(stateMachine);
        }

        void WriteGroupTree(Lyt::Group group)
        {
            Debug.WriteLine(string.Format("Group name {0}", group.name));

            // 自分のデータを書き込む
            _tgtBinWriter.WriteStruct(group);
            if (group.paneRef != null)
            {
                foreach (Lyt::GroupPaneRef paneRef in group.paneRef)
                {
                    _tgtBinWriter.WriteStruct(paneRef);
                }
            }
            ++_dataBlockNum;

            // 子供のグループを再帰的に処理する
            if (group.group != null)
            {

                // 開始ヘッダ
                _tgtBinWriter.WriteBlock(BinaryLytWriter.SignatureOfGroupBeginBlock);
                ++_dataBlockNum;

                for (int i = 0; i < group.group.Length; ++i)
                {
                    WriteGroupTree(group.group[i]);
                }

                // 終了ヘッダ
                _tgtBinWriter.WriteBlock(BinaryLytWriter.SignatureOfGroupEndBlock);
                ++_dataBlockNum;
            }
        }

        bool NeedPaneBasicInfoOverride(Lyt::Property property)
        {
            return (property.basicUserData != null || property.translate != null || property.rotate != null || property.scale != null || property.size != null || property.alpha != null);
        }
    }
}
