﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Linq;
using System.Diagnostics;

namespace NW4F.LayoutBinaryConverter
{
    using Schema.Flyt;

    class LytInfo
    {
        static readonly StringComparer PaneNameComparer = StringComparer.InvariantCulture;
        static readonly StringComparer MaterialNameComparer = StringComparer.InvariantCulture;
        static readonly StringComparer TagNameComparer = StringComparer.InvariantCultureIgnoreCase;

        /// <summary>
        /// フィルレート最適化適用モード。
        /// </summary>
        public enum FillrateOptimizationMode{
            Auto,
            AABB,
            OBB,
            Max
        }

        GroupSet _groupSet;
        ScreenSetting _screenSetting;
        Vec2 _partsSize;
        Control[] _control;
        TextureFile[] _textureFileAry;
        FontFile[] _fontFileAry;
        CaptureTexture[] _captureTexture;
        StateMachine _stateMachine;

        LanInfo _lanInfo;

        string _filePath;
        Pane _rootPane;
        Dictionary<string, Pane> _paneDic;
        List<MaterialInfo> _materialList;
        SortedList<string, TextureFile> _textureFileList;
        SortedList<string, TextureFile> _refTextureFileList;
        SortedList<string, FontFile> _fontFileList;
        SortedList<string, FontFile> _refFontFileList;
        List<ArchiveShaderFile> _archiveShaderFileList;
        List<ShapeInfo> _shapeInfoList;
        SortedList<string, CaptureTexture> _captureTextureList;
        SortedList<string, CaptureTexture> _refCaptureTextureList;

        public SystemExtData SystemExtData { get; private set; } = null;

        public void AddSystemExtDataBlock(byte[] bytes)
        {
            if (this.SystemExtData == null)
            {
                this.SystemExtData = new SystemExtData();
            }

            this.SystemExtData.Add(bytes);
        }

        public LytInfo(Document doc, string docFileName, bool bBannerLibBugFix, bool bNoCheckVer)
        {
            VerifyDoc(doc, bNoCheckVer);

            SetFileName(docFileName);

            if (doc.body.lyt.userData != null)
            {
                List<object> temp = new List<object>(doc.body.lyt.userData);
                this.UserData = temp.ToArray();
            }
            else
            {
                this.UserData = new object[0];
            }

            BuildPaneHierarchy(doc);

            if (bBannerLibBugFix)
            {
                // バナーデータ用にバグ対策を行う
                BugBuster bugBuster = new BugBuster(this);
                bugBuster.InsertDummyPicturePane();
            }
        }

        public static void AddPaneDic(Dictionary<string, Pane> paneDic, Pane pane)
        {
            try
            {
                paneDic.Add(pane.name, pane);
            }
            catch (ArgumentException)
            {
                throw new LayoutDataException(string.Format(Properties.Resources.ErrorSamePaneNameExist, pane.name));
            }
        }

        void SetFileName(string filePath)
        {
            if (! FileUtil.IsValidStringForFileName(Path.GetFileNameWithoutExtension(filePath)))
            {
                throw new LayoutDataException(string.Format(Properties.Resources.ErrorInvalidLayoutFileName, filePath));
            }

            _filePath = filePath;
        }

        public string FilePath { get { return _filePath; } }

        public Pane RootPane { get { return _rootPane; } }

        public Dictionary<string, Pane> PaneDic { get { return _paneDic; } }

        public List<MaterialInfo> MaterialList
        {
            get
            {
                if (_materialList == null)
                {
                    _materialList = MakeMaterialList(_paneDic);
                }

                return _materialList;
            }
        }

        public SortedList<string, TextureFile> TextureFileList
        {
            get
            {
                if (_textureFileList == null)
                {
                    _textureFileList = MakeTextureFileList(_filePath, _paneDic, MaterialList, _textureFileAry);
                }

                return _textureFileList;
            }
        }

        public SortedList<string, TextureFile> RefTextureFileList
        {
            get
            {
                if (_refTextureFileList == null)
                {
                    _refTextureFileList = MakeRefTextureFileList(_paneDic, MaterialList, TextureFileList);
                }

                return _refTextureFileList;
            }
        }

        public SortedList<string, FontFile> FontFileList
        {
            get
            {
                if (_fontFileList == null)
                {
                    _fontFileList = MakeFontFileList(_fontFileAry);
                }

                return _fontFileList;
            }
        }

        public SortedList<string, FontFile> RefFontFileList
        {
            get
            {
                if (_refFontFileList == null)
                {
                    _refFontFileList = MakeRefFontFileList(_paneDic.Values, FontFileList);
                }

                return _refFontFileList;
            }
        }

        public IEnumerable<FontFile> ComplexFontSet
        {
            get
            {
                foreach (FontFile ff in FontFileList.Values)
                {
                    if (Path.GetExtension(ff.path) != ".fcpx")
                    {
                        continue;
                    }
                    yield return ff;
                }
            }
        }

        public List<ArchiveShaderFile> ArchiveShaderFileList
        {
            get
            {
                if (_archiveShaderFileList == null)
                {
                    _archiveShaderFileList = new List<ArchiveShaderFile>();
                }

                return _archiveShaderFileList;
            }
        }

        public List<ShapeInfo> ShapeInfoList
        {
            get
            {
                if (_shapeInfoList == null)
                {
                    _shapeInfoList = MakeShapeInfoList(_paneDic);
                }

                return _shapeInfoList;
            }
        }

        public SortedList<string, CaptureTexture> CaptureTextureList
        {
            get
            {
                if (_captureTextureList == null)
                {
                    _captureTextureList = MakeCaptureTextureList(_paneDic, _captureTexture);
                }

                return _captureTextureList;
            }
        }



        public SortedList<string, CaptureTexture> RefCaptureTextureList
        {
            get
            {
                if (_refCaptureTextureList == null)
                {
                    _refCaptureTextureList = MakeRefCaptureTextureList(_paneDic, MaterialList, CaptureTextureList);
                }

                return _refCaptureTextureList;
            }
        }


        public GroupSet GroupSet { get { return _groupSet; } }

        public ScreenSetting ScreenSetting { get { return _screenSetting; } }

        public Vec2 PartsSize { get { return _partsSize; } }

        public StateMachine StateMachine { get { return _stateMachine; } }

        public Control[] Control { get { return _control; } }

        public LanInfo LanInfo { get { return _lanInfo; } set { _lanInfo = value; } }

        public object[] UserData { get; private set; }

        /// <summary>
        /// xmlドキュメント共通要素のチェックを行います。
        /// </summary>
        /// <param name="doc"></param>
        static void VerifyDoc(Document doc, bool bNoCheckVer)
        {
            if (doc == null)
            {
                throw new LayoutDataException(Properties.Resources.ErrorEmptyData);
            }

            if (!bNoCheckVer)
            {
                DataUtil.VerifyDocumentVersion(doc.version);
            }

            if (doc.body == null || doc.body.lyt == null)
            {
                throw new LayoutDataException(Properties.Resources.ErrorEmptyData);
            }

            LYT lyt = doc.body.lyt;

            if (lyt.paneHierarchy == null || lyt.paneHierarchy.paneTree == null || lyt.paneSet == null)
            {
                throw new LayoutDataException(Properties.Resources.ErrorEmptyData);
            }

            VerifyGroup(lyt.groupSet.group);
        }

        static void VerifyGroup(Group group)
        {
            VerifyGroupName(group.name);

            if (group.paneRef != null)
            {
                foreach (GroupPaneRef paneRef in group.paneRef)
                {
                    VerifyPaneName(paneRef.name);
                }
            }

            if (group.group != null)
            {
                foreach (Group childGroup in group.group)
                {
                    VerifyGroup(childGroup);
                }
            }
        }

        /// <summary>
        /// ペイン階層を構築します。
        /// </summary>
        void BuildPaneHierarchy(Document doc)
        {
            LYT lyt = doc.body.lyt;

            // RLYTのメンバのコピー
            _groupSet = lyt.groupSet;
            _screenSetting = lyt.screenSetting;
            _partsSize = lyt.partsSize;
            _control = lyt.control;
            _textureFileAry = lyt.textureFile;
            _fontFileAry = lyt.fontFile;
            _captureTexture = lyt.captureTexture;
            _stateMachine = lyt.stateMachine;


            _paneDic = new Dictionary<string, Pane>(PaneNameComparer);

            _rootPane = MakeRootPane();

            AddPaneDic(_paneDic, _rootPane);

            // Paneの名前による辞書を作る。
            foreach (Pane pane in lyt.paneSet)
            {
                SetupPane(pane);

                AddPaneDic(_paneDic, pane);
            }

            AssignChildPane(lyt.paneHierarchy, _rootPane, _paneDic);
        }

        /// <summary>
        /// ルートペインのインスタンスを作る
        /// </summary>
        /// <returns>作成したルートペイン</returns>
        Pane MakeRootPane()
        {
            Pane rootPane = new Pane("RootPane", _screenSetting.layoutSize);

            if (_screenSetting.origin == ScreenOriginType.Classic)
            {
                rootPane.basePositionType.x = HorizontalPosition.Left;
                rootPane.basePositionType.y = VerticalPosition.Top;
            }

            return rootPane;
        }

        static void SetupPane(Pane pane)
        {
            VerifyPaneName(pane.name);
            VerifyUserData(pane);
        }

        /// <summary>
        /// フィルレート最適化メッシュの生成実装。
        /// </summary>
        /// <param name="optInfo">最適化結果のログ</param>
        /// <param name="texSrcFiles">マテリアルの TextureFile から元画像を引いてい来るためのテーブル</param>
        /// <param name="margin">フィットする矩形の幅、高さマージンをピクセル単位で指定する</param>
        void FillrateOptimizationImpl(ref FillrateOptimizeInformation optInfo, Dictionary<Schema.Flyt.TextureFile, TexConv.SrcImage> texSrcFiles, FillrateOptimizationMode mode, int margin)
        {
            // 最適化が適用可能なピクチャペインに対して最適化をかける。
            foreach (var pair in PaneDic)
            {
                if (pair.Value.kind == PaneKind.Picture)
                {
                    var pane = pair.Value;
                    var picture = (Picture)pair.Value.Item;

                    // 最適化フラグが設定されているものだけ適用する。
                    if (picture.FillOptimizationEnabled)
                    {
                        var pictInfo = optInfo.PictInfo[pair.Key];

                        // ファイルを参照するテクスチャが設定されていることが条件
                        if (picture.material.texMap != null &&
                            picture.material.texMap[0].textureResourceType == TextureResourceType.LocalFile)
                        {
                            // 0 番のテクスチャで最適化をかける。
                            // 現状ではマルチテクスチャを最適化不可としているが、解像度が同じの場合はアルファチャンネルを統合して判断することも可能。
                            var tex = TextureFileList[picture.material.texMap[0].imageName];
                            var srcImage = texSrcFiles[tex];

                            // ピクチャペンごとの最適化情報を設定する。
                            pictInfo.ImagePath = Path.Combine(Path.GetDirectoryName(_filePath), tex.imagePath);

                            pictInfo.TexWidth = srcImage.lyColor.Width;
                            pictInfo.TexHeight = srcImage.lyColor.Height;
                            pictInfo.OptimizedTexWidth = pictInfo.TexWidth;
                            pictInfo.OptimizedTexHeight = pictInfo.TexHeight;

                            // AABB と OBB の最適化を実行して、結果が良いほうの最適化結果を採用する。
                            ImageFitMesh.Vertex2D[] aabbVertices;
                            ImageFitMesh.Vertex2D[] obbVertices;
                            ImageFitMesh.Vertex2D[] obbAxis;
                            int aabbOptimizedWidth, aabbOptimizedHeight;
                            int obbOptimizedWidth, obbOptimizedHeight;

                            int minimumAreaSize = pictInfo.TexWidth * pictInfo.TexHeight;
                            var optimizedMesh = new OptimizedMesh();

                            ImageFitMesh.CreateMeshResult result = ImageFitMesh.CreateMeshResult.Succeeded;
                            try
                            {

                                if (mode == FillrateOptimizationMode.Auto ||
                                    mode == FillrateOptimizationMode.AABB)
                                {
                                    // AABB で最適化。
                                    result = ImageFitMesh.Generator.CreateAABBMesh(out aabbVertices, out aabbOptimizedWidth, out aabbOptimizedHeight, srcImage.lyAlpha.Data, srcImage.lyAlpha.Width, srcImage.lyAlpha.Height, margin);

                                    int aabbAreaSize = aabbOptimizedWidth * aabbOptimizedHeight;

                                    if (aabbAreaSize < minimumAreaSize)
                                    {
                                        // AABB が一番小さい
                                        pictInfo.OptimizationType = ImageFitMesh.OptimizationType.AABB;
                                        pictInfo.OptimizedTexWidth = aabbOptimizedWidth;
                                        pictInfo.OptimizedTexHeight = aabbOptimizedHeight;

                                        optimizedMesh.FillrateOptimizationType = ImageFitMesh.OptimizationType.AABB;
                                        optimizedMesh.OptimizedVertexList = aabbVertices;
                                        minimumAreaSize = aabbAreaSize;
                                    }
                                }

                                if (mode == FillrateOptimizationMode.Auto ||
                                    mode == FillrateOptimizationMode.OBB)
                                {
                                    // OBB で最適化。
                                    result = ImageFitMesh.Generator.CreateOBBMesh(
                                                    out obbVertices,
                                                    out obbAxis,
                                                    out obbOptimizedWidth,
                                                    out obbOptimizedHeight,
                                                    srcImage.lyAlpha.Data,
                                                    srcImage.lyAlpha.Width,
                                                    srcImage.lyAlpha.Height,
                                                    margin);
                                    int obbAreaSize = obbOptimizedWidth * obbOptimizedHeight;

                                    if (obbAreaSize < minimumAreaSize)
                                    {
                                        // OBB が一番小さい
                                        pictInfo.OptimizationType = ImageFitMesh.OptimizationType.OBB;
                                        pictInfo.OptimizedTexWidth = obbOptimizedWidth;
                                        pictInfo.OptimizedTexHeight = obbOptimizedHeight;

                                        optimizedMesh.FillrateOptimizationType = ImageFitMesh.OptimizationType.OBB;
                                        optimizedMesh.OptimizedVertexList = obbVertices;
                                        optimizedMesh.Axis = obbAxis;
                                        minimumAreaSize = obbAreaSize;
                                    }
                                }

                                // 最適化できなかった場合は原因をエラーとして出力する。
                                if (result != ImageFitMesh.CreateMeshResult.Succeeded)
                                {
                                    switch (result)
                                    {
                                        case ImageFitMesh.CreateMeshResult.TransparentImage:
                                            pictInfo.Reasons.Add(PictureFillrateOptimizeInformation.UnoptimizedReason.TransparentImage);
                                            break;
                                        case ImageFitMesh.CreateMeshResult.OpaqueEdgeRepeat:
                                            pictInfo.Reasons.Add(PictureFillrateOptimizeInformation.UnoptimizedReason.OpaquePixelInEdgeForOBB);
                                            break;
                                    }
                                }

                                if (minimumAreaSize == (pictInfo.TexWidth * pictInfo.TexHeight))
                                {
                                    // 最適化がかからなかった
                                    pictInfo.OptimizationType = ImageFitMesh.OptimizationType.None;
                                    optimizedMesh = null;
                                }
                            }
                            catch (Exception ex)
                            {
                                // 最適化がかからなかった
                                Report.Err.WriteLine(ex.Message);
                                pictInfo.Reasons.Add(PictureFillrateOptimizeInformation.UnoptimizedReason.InternalError);
                                pictInfo.OptimizationType = ImageFitMesh.OptimizationType.None;
                                optimizedMesh = null;
                            }

                            // 生成された最適化情報に合わせて頂点情報を調整する。
                            if (optimizedMesh != null)
                            {
                                ArrangeVertexElementForFillrateOptimization(picture, optimizedMesh);

                                // 最適化メッシュの情報をシステム用拡張ユーザーデータへ追加する。
                                AddFillrateOptimizedMeshDataToSystemExtData(pane, picture, optimizedMesh);

                                // メモリの増加量を計算
                                // 初めての最適化データの場合は、システム用拡張ユーザーデータのヘッダ分増加。
                                if (optInfo.FillRateOptimizedPicturePaneCount == 0)
                                {
                                    // ushort バイナリバージョン
                                    optInfo.IncreasedMemorySize += sizeof(ushort);
                                    // ushort 要素数
                                    optInfo.IncreasedMemorySize += sizeof(ushort);
                                }
                                // バイナリオフセットテーブル
                                optInfo.IncreasedMemorySize += sizeof(int);

                                switch (pictInfo.OptimizationType)
                                {
                                    case ImageFitMesh.OptimizationType.AABB:
                                        // int データ種別
                                        // float x 4(offsetX, offsetY, width, height)
                                        optInfo.IncreasedMemorySize += sizeof(int);
                                        optInfo.IncreasedMemorySize += (sizeof(float) * 4);
                                        break;
                                    case ImageFitMesh.OptimizationType.OBB:
                                        // int データ種別
                                        // float x 4(offsetX, offsetY, width, height)
                                        // float (RotZ)
                                        optInfo.IncreasedMemorySize += sizeof(int);
                                        optInfo.IncreasedMemorySize += (sizeof(float) * 5);
                                        break;
                                }

                                ++optInfo.FillRateOptimizedPicturePaneCount;
                            }
                        }
                    }
                }
            }
        }


        double LinearInterpolate(double begin, double end, double rate)
        {
            return (begin * rate) + (end * (1.0 - rate));
        }

        double[] LinearInterpolateColor(double[] begin, double[] end, double rate)
        {
            return new double[4]
            {
                LinearInterpolate(begin[0], end[0], rate),
                LinearInterpolate(begin[1], end[1], rate),
                LinearInterpolate(begin[2], end[2], rate),
                LinearInterpolate(begin[3], end[3], rate)
            };
        }

        /// <summary>
        /// 頂点カラーの補間処理。
        /// </summary>
        /// <param name="picture">頂点カラー計算を行うピクチャペイン</param>
        /// <param name="u">頂点カラーの値を取得する横方向のテクスチャ座標</param>
        /// <param name="v">頂点カラーの値を取得する縦方向のテクスチャ座標</param>
        /// <returns></returns>
        Color4 InterpolateVertexColor(Picture picture, double u, double v)
        {
            // 精度を落とさないように浮動小数点数で計算して、最終的に byte に戻す。
            double[][] vtxColor = new double[][]
            {
                new double[] {picture.vtxColLT.r, picture.vtxColLT.g, picture.vtxColLT.b, picture.vtxColLT.a},
                new double[] {picture.vtxColRT.r, picture.vtxColRT.g, picture.vtxColRT.b, picture.vtxColRT.a},
                new double[] {picture.vtxColLB.r, picture.vtxColLB.g, picture.vtxColLB.b, picture.vtxColLB.a},
                new double[] {picture.vtxColRB.r, picture.vtxColRB.g, picture.vtxColRB.b, picture.vtxColRB.a},
            };

            double inverse = 1.0 / 255.0;

            for(int i = 0; i < vtxColor.Length;++i)
            {
                for(int j = 0; j < vtxColor[i].Length;++j)
                {
                    vtxColor[i][j] *= inverse;
                }
            }

            if (v <= u)
            {
                // 矩形の右上三角形
                // 指定された UV 座標から v(1.0f, 1.0f) の直線と各エッジの交点。
                // 上の辺との交点 (u - v, 0.0f)
                // 右の辺との交点 (1.0f, v - u + 1.0f)

                double fIntersectTopU = u - v;
                double fIntersectRightV = v - u + 1.0;

                // 上の辺と右の辺との交点の色から UV でバイリニア補間
                double[] colorU = LinearInterpolateColor(vtxColor[0], vtxColor[1], 1.0 - fIntersectTopU);
                double[] colorV = LinearInterpolateColor(vtxColor[1], vtxColor[3], 1.0 - fIntersectRightV);

                var unit = new ImageFitMesh.Vertex2D((1.0 - fIntersectTopU), fIntersectRightV);
                double unitLength = unit.Length();
                var uv = new ImageFitMesh.Vertex2D(u - fIntersectTopU, v);

                if (unitLength == 0.0)
                {
                    // 右上頂点で基準ベクトルの長さがゼロになり以下の処理でゼロ除算になるため特別処理。
                    return picture.vtxColRT;
                }

                double[] color = LinearInterpolateColor(colorU, colorV, 1.0 - (uv.Length() / unitLength));
                return new Color4(
                    (byte)Math.Round(color[0] * 255.0),
                    (byte)Math.Round(color[1] * 255.0),
                    (byte)Math.Round(color[2] * 255.0),
                    (byte)Math.Round(color[3] * 255.0));
            }
            else
            {
                // 矩形の左下三角形
                // 指定された UV 座標から v(1.0f, 1.0f) の直線と各エッジの交点。
                // 左の辺との交点 (0.0f, v - u)
                // 下の辺との交点 (u - v + 1.0f, 1.0f)

                double fIntersectBottomU = u - v + 1.0;
                double fIntersectLeftV = v - u;

                double[] colorU = LinearInterpolateColor(vtxColor[2], vtxColor[3], 1.0 - fIntersectBottomU);
                double[] colorV = LinearInterpolateColor(vtxColor[0], vtxColor[2], 1.0 - fIntersectLeftV);

                var unit = new ImageFitMesh.Vertex2D(fIntersectBottomU, 1.0f - fIntersectLeftV);
                double unitLength = unit.Length();
                var uv = new ImageFitMesh.Vertex2D(u - fIntersectBottomU, v - 1.0);

                if (unitLength == 0.0f)
                {
                    // 左下頂点で基準ベクトルの長さがゼロになり以下の処理でゼロ除算になるため特別処理。
                    return picture.vtxColRT;
                }

                double[] color = LinearInterpolateColor(colorU, colorV, 1.0f - (uv.Length() / unitLength));
                return new Color4(
                    (byte)Math.Round(color[0] * 255.0),
                    (byte)Math.Round(color[1] * 255.0),
                    (byte)Math.Round(color[2] * 255.0),
                    (byte)Math.Round(color[3] * 255.0));
            }
        }

        /// <summary>
        /// 最適化によって変形したメッシュに合わせて頂点属性を調整する。
        /// </summary>
        /// <param name="picture">調整する Picture データ</param>
        /// <param name="optimizedMesh">最適化メッシュの情報</param>
        public void ArrangeVertexElementForFillrateOptimization(Picture picture, OptimizedMesh optimizedMesh)
        {
            // レイアウトデータとして出力するテクスチャ座標をメッシュに沿った値に書き換える。
            if (optimizedMesh != null)
            {
                switch (optimizedMesh.FillrateOptimizationType)
                {
                    case ImageFitMesh.OptimizationType.AABB:
                    case ImageFitMesh.OptimizationType.OBB:
                        // テクスチャ座標
                        if (picture.texCoord != null)
                        {
                            for (int i = 0; i < picture.texCoord.Length; ++i)
                            {
                                picture.texCoord[i].texLT.s = (float)optimizedMesh.OptimizedVertexList[0].x;
                                picture.texCoord[i].texLT.t = (float)optimizedMesh.OptimizedVertexList[0].y;

                                picture.texCoord[i].texRT.s = (float)optimizedMesh.OptimizedVertexList[1].x;
                                picture.texCoord[i].texRT.t = (float)optimizedMesh.OptimizedVertexList[1].y;

                                picture.texCoord[i].texLB.s = (float)optimizedMesh.OptimizedVertexList[2].x;
                                picture.texCoord[i].texLB.t = (float)optimizedMesh.OptimizedVertexList[2].y;

                                picture.texCoord[i].texRB.s = (float)optimizedMesh.OptimizedVertexList[3].x;
                                picture.texCoord[i].texRB.t = (float)optimizedMesh.OptimizedVertexList[3].y;
                            }
                        }

                        /*
                        // 頂点カラー
                        var newLTColor = InterpolateVertexColor(picture, optimizedMesh.OptimizedVertexList[0].x, optimizedMesh.OptimizedVertexList[0].y);
                        var newRTColor = InterpolateVertexColor(picture, optimizedMesh.OptimizedVertexList[1].x, optimizedMesh.OptimizedVertexList[1].y);
                        var newLBColor = InterpolateVertexColor(picture, optimizedMesh.OptimizedVertexList[2].x, optimizedMesh.OptimizedVertexList[2].y);
                        var newRBColor = InterpolateVertexColor(picture, optimizedMesh.OptimizedVertexList[3].x, optimizedMesh.OptimizedVertexList[3].y);

                        picture.vtxColLT = newLTColor;
                        picture.vtxColRT = newRTColor;
                        picture.vtxColLB = newLBColor;
                        picture.vtxColRB = newRBColor;
                        */

                        break;
                }
            }
        }

        /// <summary>
        /// 最適化メッシュの情報をシステム用拡張ユーザーデータへ追加する。
        /// </summary>
        /// <param name="pane">拡張ユーザーデータを追加するペイン</param>
        /// <param name="picture">最適化されたピクチャペイン</param>
        /// <param name="optimizedMesh">最適化されたメッシュの情報</param>
        void AddFillrateOptimizedMeshDataToSystemExtData(Pane pane, Picture picture, OptimizedMesh optimizedMesh)
        {
            MemoryStream ms = new MemoryStream();
            BinaryWriter bw = EndianAwareBinaryWriter.CreateResouceEndianBinaryWriter(ms);

            // その後はタイプごとに違うデータ。
            switch (optimizedMesh.FillrateOptimizationType)
            {
                case ImageFitMesh.OptimizationType.AABB:
                    {
                        // 4 頂点の AABB
                        // スケールとオフセットを書き出してシェーダーの sizeAndPosition に反映する。
                        Debug.Assert(optimizedMesh.OptimizedVertexList.Length == 4);

                        float width = (float)(optimizedMesh.OptimizedVertexList[3].x - optimizedMesh.OptimizedVertexList[0].x);
                        float height = (float)(optimizedMesh.OptimizedVertexList[3].y - optimizedMesh.OptimizedVertexList[0].y);
                        float offsetX = (float)optimizedMesh.OptimizedVertexList[0].x;
                        float offsetY = (float)optimizedMesh.OptimizedVertexList[0].y;

                        bw.Write((int)SystemExtData.PaneSystemDataType.AABBMesh);
                        bw.Write(width);
                        bw.Write(height);
                        bw.Write(offsetX);
                        bw.Write(offsetY);

                        pane.AddSystemExtDataBlock(ms.ToArray());
                    }
                    break;
                case ImageFitMesh.OptimizationType.OBB:
                    {
                        Debug.Assert(optimizedMesh.OptimizedVertexList.Length == 4);

                        // OBB のメッシュをシェーダーの sizeAndPosition と回転行列で再現できるようにパラメータを計算する。

                        double tmpX = optimizedMesh.OptimizedVertexList[1].x - optimizedMesh.OptimizedVertexList[0].x;
                        double tmpY = optimizedMesh.OptimizedVertexList[1].y - optimizedMesh.OptimizedVertexList[0].y;

                        double width = Math.Sqrt(tmpX * tmpX + tmpY * tmpY);

                        tmpX = optimizedMesh.OptimizedVertexList[2].x - optimizedMesh.OptimizedVertexList[0].x;
                        tmpY = optimizedMesh.OptimizedVertexList[2].y - optimizedMesh.OptimizedVertexList[0].y;

                        double height = (float)Math.Sqrt(tmpX * tmpX + tmpY * tmpY);

                        // OBB の場合は軸に沿った大きさに sizeAndPosition のスケールで調整して offset で中心座標をメッシュの形状に合わせて配置する。
                        // シェーダーの sizeAndPosition では 0, 0 を左上とした大きさ 1 のメッシュを基本としているため、
                        // スケールされたメッシュの中心位置から表示したいメッシュの中心位置へのオフセットとして使用する。
                        double offsetX = optimizedMesh.OptimizedVertexList[0].x +
                                        (optimizedMesh.OptimizedVertexList[3].x - optimizedMesh.OptimizedVertexList[0].x) * 0.5f - width * 0.5f;
                        double offsetY = optimizedMesh.OptimizedVertexList[0].y +
                                        (optimizedMesh.OptimizedVertexList[3].y - optimizedMesh.OptimizedVertexList[0].y) * 0.5f - height * 0.5f;

                        bw.Write((int)SystemExtData.PaneSystemDataType.OBBMesh);
                        bw.Write((float)width);
                        bw.Write((float)height);
                        bw.Write((float)offsetX);
                        bw.Write((float)offsetY);

                        // OBB の軸から Z 軸の回転角度を計算する。
                        bw.Write((float)Math.Asin(-optimizedMesh.Axis[1].y));

                        pane.AddSystemExtDataBlock(ms.ToArray());
                    }

                    break;
            }
        }

        /// <summary>
        /// AnimContent の中に黒カラーアニメーションが存在するかどうか確認します。
        /// </summary>
        /// <param name="contents"></param>
        /// <returns></returns>
        bool IsBlackColorAnimContained(Schema.Flan.AnimContent[] contents)
        {
            if (contents != null)
            {
                foreach (var content in contents)
                {
                    if (content.Items != null)
                    {
                        foreach (var item in content.Items)
                        {
                            if (item.target == Schema.Flan.AnimTargetType.BlackColor_a)
                            {
                                return true;
                            }
                        }
                    }
                }
            }

            return false;
        }

        /// <summary>
        /// レイアウトに関連づく、アニメーション区間タグの名前用の拡張ユーザーデータを作成します。
        /// </summary>
        public void MakeAnimationTagNameSystemExtUserData()
        {
            if(this.LanInfo == null)
            {
                return;
            }

            if(this.LanInfo.AnimTagArray == null || this.LanInfo.AnimTagArray.Length == 0)
            {
                return;
            }

            using (MemoryStream ms = new MemoryStream())
            {
                using (BinaryWriter bw = EndianAwareBinaryWriter.CreateResouceEndianBinaryWriter(ms))
                {
                    bw.Write((int)SystemExtData.LayoutSystemDataType.AnimTag);
                    bw.Write((int)this.LanInfo.AnimTagArray.Count());

                    List<UInt32> nameOffsets = new List<uint>(LanInfo.AnimTagArray.Count());
                    List<byte> nameStringBuffer = new List<byte>();
                    {
                        // ブロック先頭からの文字列へのオフセット
                        int offset = sizeof(int) + sizeof(int) + sizeof(UInt32) * LanInfo.AnimTagArray.Length;
                        foreach (var animTag in this.LanInfo.AnimTagArray)
                        {
                            nameOffsets.Add((UInt32)offset);
                            byte[] animTagName = Encoding.ASCII.GetBytes(animTag.name + "\0");
                            nameStringBuffer.AddRange(animTagName);

                            offset += animTagName.Length;
                        }

                        // 終端を64バイト整列します。
                        int count = (nameStringBuffer.Count + (64 - 1)) & ~(64 - 1);
                        for (int i = 0; nameStringBuffer.Count < count; i++)
                        {
                            nameStringBuffer.Add((byte)0);
                        }
                    }

                    // オフセットの書き出し
                    foreach(var nameOffset in nameOffsets)
                    {
                        bw.Write(nameOffset);
                    }

                    // 文字列の書き出し
                    bw.Write(nameStringBuffer.ToArray());

                    this.AddSystemExtDataBlock(ms.ToArray());
                }
            }
        }

        /// <summary>
        /// マスクのシステム用拡張ユーザーデータを作成して追加します。
        /// </summary>
        /// <param name="pane"></param>
        void MakeMaskSystemExtUserData(Pane pane)
        {
            MemoryStream ms = new MemoryStream();
            BinaryWriter bw = EndianAwareBinaryWriter.CreateResouceEndianBinaryWriter(ms);

            // SystemDataMaskTexture
            bw.Write((int)SystemExtData.PaneSystemDataType.Mask);

            // maskWithPane 設定
            bw.Write((byte)(pane.mask.maskWithPane ? 1 : 0));

            // padding
            bw.Write((byte)0);
            bw.Write((byte)0);
            bw.Write((byte)0);

            TexMap[] maskTexMap =
            {
                pane.mask.maskTexMap,
                pane.mask.captureTexMap,
            };

            foreach (var texMap in maskTexMap)
            {
                int index = RefTextureFileList.IndexOfKey(texMap.imageName);
                if (index < 0)
                {
                    throw new LayoutDataException(string.Format(Properties.Resources.ErrorInvalidPaneName, pane.name));
                }

                // ResTexMap と同じ構造
                bw.Write((UInt16)index);
                bw.Write((byte)((int)texMap.wrap_s | ((int)texMap.minFilter << 2)));
                bw.Write((byte)((int)texMap.wrap_t | ((int)texMap.magFilter << 2)));

                // AdditionalInfo
                var info = new TexMapAdditionalInfo();
                if (texMap.textureResourceType != TextureResourceType.LocalFile)
                {
                    info.Enable(Schema.Flyt.TexMapAdditionalInfo.InfoType.CaptureTexture);
                }
                bw.Write((UInt32)info.GetFlags());
            }

            // マスクテクスチャ行列
            // ResTexSrt
            bw.Write((float)pane.mask.maskTexMatrix.translate.x);
            bw.Write((float)pane.mask.maskTexMatrix.translate.y);
            bw.Write((float)pane.mask.maskTexMatrix.rotate);
            bw.Write((float)pane.mask.maskTexMatrix.scale.x);
            bw.Write((float)pane.mask.maskTexMatrix.scale.y);

            pane.AddSystemExtDataBlock(ms.ToArray());
        }

        /// <summary>
        /// ドロップシャドウのシステム用拡張ユーザーデータを作成して追加します。
        /// </summary>
        /// <param name="pane"></param>
        /// <param name="isDegamma">出力するカラーデータをデガンマするかどうか</param>
        void MakeDropShadowSystemExtUserData(Pane pane, bool isDegamma)
        {
            MemoryStream ms = new MemoryStream();
            BinaryWriter bw = EndianAwareBinaryWriter.CreateResouceEndianBinaryWriter(ms);

            // SystemDataMaskTexture
            bw.Write((int)SystemExtData.PaneSystemDataType.DropShadow);

            DropShadow dropShadow = pane.dropShadow;
            TexMap texMap = dropShadow.captureTexMap;

            int index = RefTextureFileList.IndexOfKey(texMap.imageName);
            if (index < 0)
            {
                throw new LayoutDataException(string.Format(Properties.Resources.ErrorInvalidPaneName, pane.name));
            }

            // ResTexMap と同じ構造
            bw.Write((UInt16)index);
            bw.Write((byte)((int)texMap.wrap_s | ((int)texMap.minFilter << 2)));
            bw.Write((byte)((int)texMap.wrap_t | ((int)texMap.magFilter << 2)));

            byte flags = 0;
            if (dropShadow.strokeEnabled)
            {
                flags |= 1 << (int)DropShadowFlag.StrokeEnabled;
            }
            if (dropShadow.outerGlowEnabled)
            {
                flags |= 1 << (int)DropShadowFlag.OuterGlowEnabled;
            }
            if (dropShadow.dropShadowEnabled)
            {
                flags |= 1 << (int)DropShadowFlag.DropShadowEnabled;
            }
            if (dropShadow.knockout)
            {
                flags |= 1 << (int)DropShadowFlag.KnockoutEnabled;
            }
            if (dropShadow.onlyEffect)
            {
                flags |= 1 << (int)DropShadowFlag.DrawEffectOnlyEnabled;
            }
            if (dropShadow.staticRendering)
            {
                flags |= 1 << (int)DropShadowFlag.StaticRenderingEnabled;
            }

            bw.Write((byte)flags);
            // padding
            bw.Write((byte)0);
            bw.Write((byte)0);
            bw.Write((byte)0);

            bw.Write((byte)dropShadow.maxSize);
            bw.Write((byte)dropShadow.strokeBlendMode);
            bw.Write((byte)dropShadow.outerGlowBlendMode);
            bw.Write((byte)dropShadow.dropShadowBlendMode);

            // 拡張用のパディング
            bw.Write((UInt32)0);
            bw.Write((UInt32)0);
            bw.Write((UInt32)0);
            bw.Write((UInt32)0);

            // 以下はアニメーションする際に先頭から AnimTargetDropShadow の並びに合わせて float のポインタとしてアクセスされるため並びを合わせておく

            // 境界線
            bw.Write((float)dropShadow.strokeSize);
            WriteColor4(bw, dropShadow.strokeColor, isDegamma);

            // 光彩
            WriteColor4(bw, dropShadow.outerGlowColor, isDegamma);

            bw.Write((float)dropShadow.outerGlowSpread);
            bw.Write((float)dropShadow.outerGlowSize);

            //  ドロップシャドウ
            WriteColor4(bw, dropShadow.dropShadowColor, isDegamma);

            bw.Write((float)dropShadow.dropShadowAngle);
            bw.Write((float)dropShadow.dropShadowDistance);
            bw.Write((float)dropShadow.dropShadowSpread);
            bw.Write((float)dropShadow.dropShadowSize);

            // 拡張用のパディング
            bw.Write((UInt32)0);
            bw.Write((UInt32)0);
            bw.Write((UInt32)0);
            bw.Write((UInt32)0);

            pane.AddSystemExtDataBlock(ms.ToArray());
        }

        /// <summary>
        /// 任意形状のシステム用拡張ユーザーデータを出力します。
        /// </summary>
        /// <param name="pane"></param>
        /// <param name="pict"></param>
        /// <param name="isDegamma"></param>
        public void MakeProceduralShapeSystemExtUserData(Pane pane, Picture pict, bool isDegamma)
        {
            MemoryStream ms = new MemoryStream();
            BinaryWriter bw = EndianAwareBinaryWriter.CreateResouceEndianBinaryWriter(ms);
            ProceduralShapeParam ps = pict.proceduralShapeParam;

            // SystemDataMaskTexture
            bw.Write((int)SystemExtData.PaneSystemDataType.ProceduralShape);

            // アニメーションしない静的なパラメータ
            byte flags = 0;
            if (ps.innerStrokeEnabled)
            {
                flags |= 1 << (int)ProceduralShapeFlag.InnerStrokeEnabled;
            }
            if (ps.innerShadowEnabled)
            {
                flags |= 1 << (int)ProceduralShapeFlag.InnerShadowEnabled;
            }
            if (ps.colorOverlayEnabled)
            {
                flags |= 1 << (int)ProceduralShapeFlag.ColorOverlayEnabled;
            }
            if (ps.gradationOverlayEnabled)
            {
                flags |= 1 << (int)ProceduralShapeFlag.GradationOverlayEnabled;
            }
            if (ps.outerShadowEnabled)
            {
                flags |= 1 << (int)ProceduralShapeFlag.OuterShadowEnabled;
            }

            // ProceduralShape 基本
            // なし

            bw.Write((byte)flags);
            // InnerStroke
            bw.Write((byte)ps.innerStrokeBlendMode);
            // InnerShadow
            bw.Write((byte)ps.innerShadowBlendMode);
            bw.Write((byte)ps.innerShadowType);
            // colorOverlay
            bw.Write((byte)ps.colorOverlayBlendMode);
            // gradationOverlay
            bw.Write((byte)ps.gradationOverlayBlendMode);

            // OuterShadow
            bw.Write((byte)ps.outerShadowBlendMode);
            bw.Write((byte)ps.outerShadowType);

            // 拡張用のパディング
            bw.Write((UInt32)0);
            bw.Write((UInt32)0);
            bw.Write((UInt32)0);
            bw.Write((UInt32)0);

            // 以下はアニメーションする際に先頭から AnimTargetProceduralShape の並びに合わせて float のポインタとしてアクセスされるため並びを合わせておく

            // ProceduralShape 基本
            bw.Write((float)ps.exp);
            bw.Write((float)ps.radius);

            // InnerStroke
            bw.Write((float)ps.innerStrokeSize);
            WriteColor4(bw, ps.innerStrokeColor, isDegamma);

            // InnerShadow
            WriteColor4(bw, ps.innerShadowColor, isDegamma);
            bw.Write((float)ps.innerShadowAngle);
            bw.Write((float)ps.innerShadowDistance);
            bw.Write((float)ps.innerShadowSize);

            // ColorOverlay
            WriteColor4(bw, ps.colorOverlayColor, isDegamma);

            // gradationOverlay
            // グラデーションオーバーレイのコントロールポイントとカラーはデータが設定されている数だけ出力される。
            // バイナリのデータ構造としては最大数の 4 つ固定で確保されているため、不足分はコントロールポイントが 1.0 で最後に出力したカラーを出力する。
            int gradationWriteCount = 0;
            foreach (var value in ps.gradationOverlayControlPoint)
            {
                bw.Write((float)value);
                ++gradationWriteCount;
            }
            for (int i = 0; i < 4 - gradationWriteCount; ++i)
            {
                bw.Write(1.0f);
            }
            Color4 lastWriteColor = new Color4();
            foreach (var color in ps.gradationOverlayColor)
            {
                WriteColor4(bw, color, isDegamma);
                lastWriteColor = color;
            }
            for (int i = 0; i < 4 - gradationWriteCount; ++i)
            {
                WriteColor4(bw, lastWriteColor, isDegamma);
            }
            bw.Write((float)ps.gradationOverlayAngle);

            // OuterShadow
            WriteColor4(bw, ps.outerShadowColor, isDegamma);
            bw.Write((float)ps.outerShadowAngle);
            bw.Write((float)ps.outerShadowDistance);
            bw.Write((float)ps.outerShadowSize);

            // 拡張用のパディング
            bw.Write((UInt32)0);
            bw.Write((UInt32)0);
            bw.Write((UInt32)0);
            bw.Write((UInt32)0);

            pane.AddSystemExtDataBlock(ms.ToArray());
        }

        /// <summary>
        /// 拡張ユーザーデータバイナリにガンマを考慮したカラーを出力します。
        /// </summary>
        /// <param name="bw"></param>
        /// <param name="color"></param>
        /// <param name="isDegamma"></param>
        void WriteColor4(BinaryWriter bw, Color4 color, bool isDegamma)
        {
            bw.Write(isDegamma ? BinaryWriterBase.DegammaValue(color.r) : (float)color.r);
            bw.Write(isDegamma ? BinaryWriterBase.DegammaValue(color.g) : (float)color.g);
            bw.Write(isDegamma ? BinaryWriterBase.DegammaValue(color.b) : (float)color.b);
            bw.Write((float)color.a);
        }

        /// <summary>
        /// システム用拡張ユーザーデータへ出力するデータを作成します。
        /// 参照するテクスチャの TextureList 中のインデックスを決定する必要があるため、MakeRefTextureList を呼び出した後に呼び出す必要があります。
        /// </summary>
        public void MakeSystemExtUserData(bool isDegamma)
        {
            foreach (var iter in _paneDic)
            {
                // ピクチャペインの任意形状
                if (iter.Value.IsProceduralShapeEnabled())
                {
                    Picture pict = (Picture)iter.Value.Item;
                    MakeProceduralShapeSystemExtUserData(iter.Value, pict, isDegamma);
                }

                // ペインエフェクトのマスク
                if (iter.Value.IsMaskEnabled())
                {
                    MakeMaskSystemExtUserData(iter.Value);
                }

                // ペインエフェクトのドロップシャドウのシステム用拡張ユーザーデータを作成する
                if (iter.Value.IsDropShadowEnabled())
                {
                    MakeDropShadowSystemExtUserData(iter.Value, isDegamma);
                }
            }
        }

        /// <summary>
        /// フィルレート最適化を適用する。
        /// </summary>
        /// <param name="texSrcFiles">マテリアルの TextureFile から元画像を引いてい来るためのテーブル</param>
        /// <param name="margin">フィットする矩形の幅、高さマージンをピクセル単位で指定する</param>
        /// <returns>フィルレート最適化結果の情報</returns>
        public FillrateOptimizeInformation FillrateOptimization(Dictionary<Schema.Flyt.TextureFile, TexConv.SrcImage> texSrcFiles, FillrateOptimizationMode mode, int margin)
        {
            // 出力する最適化情報を生成
            var optInfo = new FillrateOptimizeInformation();

            optInfo.FileName = _filePath;
            optInfo.TotalPaneCount = PaneDic.Count;
            optInfo.PictInfo = new Dictionary<string, PictureFillrateOptimizeInformation>();

            // ペインごとにフィルレート最適化が適用可能か判断する。
            foreach (var pair in PaneDic)
            {
                try
                {
                    switch (pair.Value.kind)
                    {
                        case PaneKind.Picture:
                            {
                                var picture = (Picture)pair.Value.Item;
                                picture.FillOptimizationEnabled = true;

                                var pictInfo = new PictureFillrateOptimizeInformation();

                                // 最適化結果情報を作成する
                                pictInfo.Name = pair.Value.name;
                                pictInfo.TexWidth = 0;
                                pictInfo.TexHeight = 0;

                                ++optInfo.PicturePaneCount;
                                //  形状に「四角形」以外
                                if (picture.shapeType != ShapeType.NormalQuad)
                                {
                                    pictInfo.Reasons.Add(PictureFillrateOptimizeInformation.UnoptimizedReason.InvalidShapeType);
                                    picture.FillOptimizationEnabled = false;
                                }

                                bool vertexColorEnabled = false;
                                var vertexColors = new Color4[]
                                {
                                    picture.vtxColLT,
                                    picture.vtxColLB,
                                    picture.vtxColRT,
                                    picture.vtxColRB
                                };

                                foreach (var color in vertexColors)
                                {
                                    if (color.r != 255 ||
                                        color.g != 255 ||
                                        color.b != 255 ||
                                        color.a != 255)
                                    {
                                        vertexColorEnabled = true;
                                        break;
                                    }
                                }

                                // 頂点カラーが使用されている。
                                // 4 頂点では元の補間状態を再現できないため最適化適用外。
                                if (vertexColorEnabled)
                                {
                                    pictInfo.Reasons.Add(PictureFillrateOptimizeInformation.UnoptimizedReason.VertexColor);
                                    picture.FillOptimizationEnabled = false;
                                }

                                if (picture.material.texMap != null)
                                {
                                    var tex = TextureFileList[picture.material.texMap[0].imageName];
                                    var srcImage = texSrcFiles[tex];

                                    // テクスチャにアルファチャンネルがない
                                    if (srcImage.lyAlpha == null)
                                    {
                                        pictInfo.Reasons.Add(PictureFillrateOptimizeInformation.UnoptimizedReason.NoAlphaChannel);
                                        picture.FillOptimizationEnabled = false;
                                    }
                                    //  インダイレクトでの歪み(ひとまずマルチテクスチャを対象外とする)
                                    if (picture.material.texMap.Length > 1)
                                    {
                                        pictInfo.Reasons.Add(PictureFillrateOptimizeInformation.UnoptimizedReason.MultiTexture);
                                        picture.FillOptimizationEnabled = false;
                                    }
                                }
                                else
                                {
                                    // テクスチャがない
                                    pictInfo.Reasons.Add(PictureFillrateOptimizeInformation.UnoptimizedReason.NoTexture);
                                    picture.FillOptimizationEnabled = false;
                                }

                                // 黒カラーのアルファで最低値が持ち上げられて不透明領域が変わるため最適化をかけない
                                if ((picture.material.blackColorFloat != null &&
                                     picture.material.blackColorFloat.a != 0.0f) ||
                                    picture.material.blackColor.a != 0)
                                {
                                    pictInfo.Reasons.Add(PictureFillrateOptimizeInformation.UnoptimizedReason.BlackColorAlphaIsNotZero);
                                    picture.FillOptimizationEnabled = false;
                                }

                                // ブレンドの設定によって不透明領域が変わることがあるため最適化をかけない
                                if (!picture.materialCtr.useDefaultBlendSettings)
                                {
                                    pictInfo.Reasons.Add(PictureFillrateOptimizeInformation.UnoptimizedReason.FragmentBlendSettingIsNotDefault);
                                    picture.FillOptimizationEnabled = false;
                                }

                                //  詳細コンバイナでのアルファ操作
                                if (picture.materialCtr.useLowLevelCombinerSettings)
                                {
                                    pictInfo.Reasons.Add(PictureFillrateOptimizeInformation.UnoptimizedReason.DetailedCombiner);
                                    picture.FillOptimizationEnabled = false;
                                }
                                //  CombinerEditor 使用

                                //  「テクスチャマッピング」
                                //  SRT にデフォルト値以外、投影マッピングが設定されている
                                if (picture.material.texCoordGen != null)
                                {
                                    foreach (var gen in picture.material.texCoordGen)
                                    {
                                        if (gen.srcParam != TexGenSrc.Tex0 &&
                                            gen.srcParam != TexGenSrc.Tex1 &&
                                            gen.srcParam != TexGenSrc.Tex2)
                                        {
                                            pictInfo.Reasons.Add(PictureFillrateOptimizeInformation.UnoptimizedReason.TextureProjection);
                                            picture.FillOptimizationEnabled = false;
                                            break;
                                        }
                                    }
                                }

                                // テクスチャ行列に値が設定されている。
                                if (picture.material.texMatrix != null)
                                {
                                    foreach (var mat in picture.material.texMatrix)
                                    {
                                        if (mat.translate.x != 0.0f ||
                                            mat.translate.y != 0.0f ||
                                            mat.scale.x != 1.0f ||
                                            mat.scale.y != 1.0f ||
                                            mat.rotate != 0.0f)
                                        {
                                            pictInfo.Reasons.Add(PictureFillrateOptimizeInformation.UnoptimizedReason.TextureMatrix);
                                            picture.FillOptimizationEnabled = false;
                                            break;
                                        }
                                    }
                                }

                                // テクスチャ座標が 0 - 1 になっていない。
                                foreach (var texCoord in picture.texCoord)
                                {
                                    if (texCoord.texLT.s != 0.0f || texCoord.texLT.t != 0.0f ||
                                        texCoord.texRT.s != 1.0f || texCoord.texRT.t != 0.0f ||
                                        texCoord.texLB.s != 0.0f || texCoord.texLB.t != 1.0f ||
                                        texCoord.texRB.s != 1.0f || texCoord.texRB.t != 1.0f)
                                    {
                                        pictInfo.Reasons.Add(PictureFillrateOptimizeInformation.UnoptimizedReason.ComplexTextureCoordinate);
                                        picture.FillOptimizationEnabled = false;
                                        break;
                                    }
                                }

                                optInfo.PictInfo[pair.Key] = pictInfo;
                            }
                            break;
                        case PaneKind.TextBox:
                            ++optInfo.TextBoxPaneCount;
                            break;
                        case PaneKind.Window:
                            ++optInfo.WindowPaneCount;
                            break;
                        case PaneKind.Parts:
                            ++optInfo.PartsPaneCount;
                            break;
                        case PaneKind.Bounding:
                            ++optInfo.BoundingPaneCount;
                            break;
                        case PaneKind.Null:
                            ++optInfo.NullPaneCount;
                            break;
                        case PaneKind.Capture:
                            ++optInfo.CapturePaneCount;
                            break;
                        case PaneKind.Alignment:
                            ++optInfo.AlignmentPaneCount;
                            break;
                        case PaneKind.Scissor:
                            ++optInfo.ScissorPaneCount;
                            break;
                    }
                }
                catch (Exception e)
                {
                    Report.Err.WriteLine(pair.Value.name + ":" + e.Message);
                }
            }

            // アニメーションが適用されるペインごとにフィルレート最適化が適用可能か判断する。
            if (_lanInfo != null && _lanInfo.LanArray != null)
            {
                foreach (var anim in _lanInfo.LanArray)
                {
                    try
                    {
                        string paneName = null;
                        Pane pane = null;

                        switch (anim.animType)
                        {
                            case Schema.Flan.AnimationType.VertexColor:
                                {
                                    paneName = anim.animContent[0].name;
                                    pane = PaneDic[paneName];

                                    if (pane == null)
                                    {
                                        Report.Err.WriteLine(paneName + " isn't found in PaneDic.");
                                        continue;
                                    }
                                }
                                break;
                            case Schema.Flan.AnimationType.TextureSRT:
                            case Schema.Flan.AnimationType.IndTextureSRT:
                            case Schema.Flan.AnimationType.TexturePattern:
                            case Schema.Flan.AnimationType.MaterialColor:
                            case Schema.Flan.AnimationType.AlphaTest:
                                {
                                    string materialName = anim.animContent[0].name;
                                    foreach (var pair in PaneDic)
                                    {
                                        if (pair.Value.kind == PaneKind.Picture)
                                        {
                                            Picture pict = (Picture)pair.Value.Item;
                                            if (pict.material.name == materialName ||
                                                pict.materialCtr.name == materialName)
                                            {
                                                paneName = pair.Value.name;
                                                pane = pair.Value;
                                                break;
                                            }
                                        }
                                    }
                                }
                                break;
                            default:
                                continue;
                        }

                        if (pane == null ||
                            pane.kind != PaneKind.Picture)
                        {
                            continue;
                        }

                        Picture picture = (Picture)pane.Item;
                        var pictInfo = optInfo.PictInfo[paneName];

                        if (pictInfo == null)
                        {
                            Report.Err.WriteLine(paneName + " isn't found in optInfo.PictInfo.");
                            continue;
                        }

                        //  頂点カラーアニメーション
                        if (anim.animType == Schema.Flan.AnimationType.VertexColor)
                        {
                            pictInfo.Reasons.Add(PictureFillrateOptimizeInformation.UnoptimizedReason.VertexColorAnimation);
                            picture.FillOptimizationEnabled = false;
                        }
                        //  TexSRT アニメーション
                        if (anim.animType == Schema.Flan.AnimationType.TextureSRT)
                        {
                            pictInfo.Reasons.Add(PictureFillrateOptimizeInformation.UnoptimizedReason.TextureSRTAnimation);
                            picture.FillOptimizationEnabled = false;
                        }
                        //  黒カラーアニメーション
                        if (anim.animType == Schema.Flan.AnimationType.MaterialColor &&
                            IsBlackColorAnimContained(anim.animContent))
                        {
                            pictInfo.Reasons.Add(PictureFillrateOptimizeInformation.UnoptimizedReason.BlackColorAlphaAnimation);
                            picture.FillOptimizationEnabled = false;
                        }
                        //  IndirectTexSRT アニメーション
                        if (anim.animType == Schema.Flan.AnimationType.IndTextureSRT)
                        {
                            pictInfo.Reasons.Add(PictureFillrateOptimizeInformation.UnoptimizedReason.IndTextureSRTAnimation);
                            picture.FillOptimizationEnabled = false;
                        }
                        //  TexturePattern アニメーション
                        if (anim.animType == Schema.Flan.AnimationType.TexturePattern)
                        {
                            pictInfo.Reasons.Add(PictureFillrateOptimizeInformation.UnoptimizedReason.TexturePatternAnimation);
                            picture.FillOptimizationEnabled = false;
                        }
                    }
                    catch (Exception e)
                    {
                        Report.Err.WriteLine(anim.animType + ":" + e.Message);
                    }
                }
            }

            // 最適化をかける。
            FillrateOptimizationImpl(ref optInfo, texSrcFiles, mode, margin);

            return optInfo;
        }

        /// <summary>
        /// ルートペイン以下に子ペインをセットします。
        /// </summary>
        /// <param name="rootPane"></param>
        /// <param name="paneDic"></param>
        static void AssignChildPane(PaneHierarchy paneHierarchy, Pane rootPane, Dictionary<string, Pane> paneDic)
        {
            PaneTree root = paneHierarchy.paneTree;
            if (root.paneTree == null)
            {
                // root.paneTreeがnullのときは、RootPaneのみのレイアウト。
                // このときは子ペインの登録は必要ない。
                return;
            }
            rootPane.ChildList = new List<Pane>(root.paneTree.Length);
            foreach (PaneTree child in root.paneTree)
            {
                AssignChildPane(child, rootPane, paneDic);
            }
        }

        /// <summary>
        /// 親ペインに自分を追加します。
        /// また、自分の子供に対して再帰的に実行します。
        /// </summary>
        /// <param name="parent"></param>
        /// <param name="paneDic"></param>
        static void AssignChildPane(PaneTree self, Pane parentPane, Dictionary<string, Pane> paneDic)
        {
            Pane selfPane = paneDic[self.name];
            parentPane.ChildList.Add(selfPane);

            if (self.paneTree != null)   // 再帰的に処理を行う
            {
                selfPane.ChildList = new List<Pane>(self.paneTree.Length);
                foreach (PaneTree child in self.paneTree)
                {
                    AssignChildPane(child, selfPane, paneDic);
                }
            }
        }

        /// <summary>
        /// ペイン名のチェックを行う。
        /// 問題がある場合は、LayoutDataException例外がスローされる。
        /// </summary>
        /// <param name="paneName">チェックするペイン名</param>
        static void VerifyPaneName(string paneName)
        {
            if (string.IsNullOrEmpty(paneName) || paneName.Length > BinaryLytWriter.ResourceNameStrMax)
            {
                throw new LayoutDataException(string.Format(Properties.Resources.ErrorInvalidPaneName, paneName));
            }
        }

        /// <summary>
        /// ペインのユーザデータのチェックを行う。
        /// 問題がある場合は、LayoutDataException例外がスローされる。
        /// </summary>
        /// <param name="pane">チェックするペイン</param>
        static void VerifyUserData(Pane pane)
        {
            if (pane.userData == null)
            {
                return;
            }

            UserDataString userDataStr = pane.userData[0] as UserDataString;
            if (userDataStr != null)
            {
                // nullか空文字列
                // あるいは、文字列の長さが制限以内なら OK
                if ( string.IsNullOrEmpty(userDataStr.Value)
                  || userDataStr.Value.Length <= BinaryLytWriter.UserDataStrMax
                )
                {
                    return;
                }
            }

            throw new LayoutDataException(string.Format(Properties.Resources.ErrorInvalidUserData, userDataStr.Value));
        }

        /// <summary>
        /// グループ名のチェックを行う。
        /// 問題がある場合は、LayoutDataException例外がスローされる。
        /// </summary>
        /// <param name="name"></param>
        public static void VerifyGroupName(string name)
        {
            if (string.IsNullOrEmpty(name) || name.Length > BinaryLytWriter.GroupNameStrMax)
            {
                throw new LayoutDataException(string.Format(Properties.Resources.ErrorInvalidGroupName, name));
            }
        }

        static void RegisterMaterialItem(MaterialRegister materialRegister, Pane pane, PaneKind kind, object item, bool isOutTexture = true)
        {
            switch (kind)
            {
            case PaneKind.Picture:
                {
                    Picture picture = (Picture)item;
                    picture.MaterialInfo = materialRegister.Add(MaterialInfoHelper.Create(pane, picture.material, picture.materialCtr, isOutTexture, picture.detailSetting));
                }
                break;

            case PaneKind.TextBox:
                {
                    TextBox textBox = (TextBox)item;
                    textBox.MaterialInfo = materialRegister.Add(MaterialInfoHelper.Create(pane, textBox.material, textBox.materialCtr, false, textBox.detailSetting));
                    textBox.MaterialInfo.SetFontShadowBlackWhiteColor(textBox.shadowBlackColor, textBox.shadowWhiteColor);
                }
                break;

            case PaneKind.Window:
                    {
                        Window window = (Window)item;
                        window.content.MaterialInfo = materialRegister.Add(MaterialInfoHelper.Create(pane, window.content.material, window.content.materialCtr, isOutTexture, window.content.detailSetting));

                        foreach (WindowFrame frame in window.frame)
                        {
                            frame.MaterialInfo = materialRegister.Add(MaterialInfoHelper.Create(pane, frame.material, frame.materialCtr, isOutTexture, frame.detailSetting));
                        }

                        // 左上フレームのマテリアル設定を全フレームで利用する場合。
                        if (window.useLTMaterial != null && window.useLTMaterial.value == true)
                        {
                            Debug.Assert(window.frame != null);
                            WindowFrame ltFrame = Array.Find(window.frame, (f) => f.frameType == WindowFrameType.CornerLT);
                            Debug.Assert(ltFrame != null);
                            int ltTexNum = ltFrame.MaterialInfo.GetTextureCount();

                            // テクスチャ枚数が一致しているかチェックします。
                            bool is_horizontal = (window.kind == WindowKind.Horizontal || window.kind == WindowKind.HorizontalNoContent);
                            if (is_horizontal)
                            {
                                // 水平ウィンドウ
                                // RT フレーム
                                WindowFrame rtFrame = Array.Find(window.frame, (f) => f.frameType == WindowFrameType.CornerRT);
                                if (rtFrame != null)
                                {
                                    if (rtFrame.MaterialInfo.GetTextureCount() != ltTexNum)
                                    {
                                        throw new LayoutDataException(string.Format(Properties.Resources.ErrorMissingFrameTexture, pane.name, rtFrame.frameType.ToString()));
                                    }
                                    rtFrame.MaterialInfo.UseTextureOnly = true;
                                }

                                // コンテント(コンテンツ無しはチェックしない)
                                if (window.kind != WindowKind.HorizontalNoContent)
                                {
                                    if (window.content.MaterialInfo.GetTextureCount() != ltTexNum)
                                    {
                                        throw new LayoutDataException(string.Format(Properties.Resources.ErrorMissingFrameTexture, pane.name, "Contents"));
                                    }
                                    window.content.MaterialInfo.UseTextureOnly = true;
                                }
                            }
                            else
                            {
                                Debug.Assert(window.kind == WindowKind.Around);
                                foreach (var frame in window.frame)
                                {
                                    if(frame == ltFrame)
                                    {
                                        continue;
                                    }

                                    if (frame.MaterialInfo.GetTextureCount() != ltTexNum)
                                    {
                                        throw new LayoutDataException(string.Format(Properties.Resources.ErrorMissingFrameTexture, pane.name, frame.frameType.ToString()));
                                    }

                                    frame.MaterialInfo.UseTextureOnly = true;
                                }
                            }
                        }
                    }
                break;
            case PaneKind.Parts:
                {
                    Parts parts = (Parts)item;
                    if (parts.property != null)
                    {
                        foreach (Property property in parts.property)
                        {
                            // 上書き設定はしてあるが実際に上書きは行われていないときに、Itemがnullになる
                            // ことがある。このような場合は登録しない。
                            if (property.Item != null)
                            {
                                // 実際には利用されないテクスチャーを区別しないと、テクスチャが発見できないというエラーや例外がコンバーター無いで起こってしまうため
                                // 上書きペインについては、上書きの種類によって、テクスチャを利用するかどうかを指定します。
                                bool notUseTexture = property.usageOptions != null && property.usageOptions.Any((opt) => opt == PartsPropertyUsageOptions.UseMaterialColorBlend) && !property.usageOptions.Any((opt) => opt == PartsPropertyUsageOptions.UseMaterialTexture);
                                RegisterMaterialItem(materialRegister, pane, property.kind, property.Item, !notUseTexture);
                            }
                        }
                    }
                }
                break;
            }
        }

        static List<MaterialInfo> MakeMaterialList(Dictionary<string, Pane> paneDic)
        {
            List<MaterialInfo> matList = new List<MaterialInfo>();

            MaterialRegister materialRegister = new MaterialRegister(matList);

            foreach (Pane pane in paneDic.Values)
            {
                RegisterMaterialItem(materialRegister, pane, pane.kind, pane.Item);
            }

            return matList;
        }

        public static bool IsIndirect(MaterialInfo matInfo, int stage_num)
        {
            if (!matInfo.UseLowLevelCombinerSettings)
            {
                if (stage_num > 0)
                {
                    TevMode tevMode = matInfo.tevStage[stage_num - 1].rgb.mode;
                    if (tevMode == TevMode.Indirect || tevMode == TevMode.BlendIndirect || tevMode == TevMode.EachIndirect || (stage_num == 2 && matInfo.tevStage[0].rgb.mode == TevMode.BlendIndirect))
                    {
                        // 自分がインダイレクトかブレンドインダイレクト、あるいは前段がブレンドインダイレクトの場合に、
                        // インダイレクトで使われるテクスチャと見なす
                        return true;
                    }
                }
            }
            return false;
        }

        private static void SetIndirectFlagTo(TextureFile texFile, bool isIndirect)
        {
            if (isIndirect)
            {
                texFile.IsIndirect = true;
            }
            else
            {
                texFile.IsNoIndirect = true;
            }
        }

        /// <summary>
        /// キャプチャテクスチャを TextureList へ加えるためのダミーの TextureFile インスタンスを作成します。
        /// </summary>
        /// <param name="texMap"></param>
        /// <returns>作成されたダミーの TextureFile のインスタンス</returns>
        static TextureFile CreateDummyTextureFileInstanceForCaptureTexture_(TexMap texMap)
        {
            TextureFile tmpCaptureTexture = new TextureFile();
            tmpCaptureTexture.imagePath = texMap.imageName;
            tmpCaptureTexture.format = TexelFormat.RGBA8;
            tmpCaptureTexture.IsCaptureTexture = true;

            return tmpCaptureTexture;
        }

        /// <summary>
        /// ペインのマスク機能などのエフェクトで参照されるテクスチャを使用テクスチャリストへ追加します。
        /// </summary>
        /// <param name="dic"></param>
        /// <param name="paneDic"></param>
        static void MakePaneEffectTextureList_(SortedDictionary<string, TextureFile> dic, Dictionary<string, Pane> paneDic)
        {
            // ペインの設定からテクスチャが参照されているかどうかチェック
            foreach (var pane in paneDic)
            {
                // マスク機能関連で参照されているかどうかチェック
                if (pane.Value.IsMaskEnabled())
                {
                    TexMap[] maskTexMap =
                    {
                        pane.Value.mask.maskTexMap,
                        pane.Value.mask.captureTexMap,
                    };

                    foreach (var texMap in maskTexMap)
                    {
                        if (texMap.textureResourceType == TextureResourceType.LocalFile)
                        {
                            TextureFile texFile = dic[texMap.imageName];

                            SetIndirectFlagTo(texFile, false);
                        }
                        else
                        {
                            // キャプチャテクスチャの場合はファイルを参照していないが、参照する際の名前情報だけ出力したい。
                            // そのためダミーの TextureFile を作成して dic へ登録する。
                            if (!dic.ContainsKey(texMap.imageName))
                            {
                                dic.Add(texMap.imageName, CreateDummyTextureFileInstanceForCaptureTexture_(texMap));
                            }
                        }
                    }
                }

                // ドロップシャドウで参照されているかどうかチェック
                if (pane.Value.IsDropShadowEnabled())
                {
                    string imageName = pane.Value.dropShadow.captureTexMap.imageName;
                    if (!dic.ContainsKey(imageName))
                    {
                        TexMap texMap = new TexMap();
                        texMap.imageName = imageName;
                        dic.Add(imageName, CreateDummyTextureFileInstanceForCaptureTexture_(texMap));
                    }
                }
            }
        }

        /// <summary>
        /// マテリアルから参照されるテクスチャを使用テクスチャリストへ追加します。
        /// </summary>
        /// <param name="dic"></param>
        /// <param name="layoutFilePath"></param>
        /// <param name="matInfoEnum"></param>
        static void MakeMaterialTextureList_(SortedDictionary<string, TextureFile> dic, string layoutFilePath, IEnumerable<MaterialInfo> matInfoEnum)
        {
            foreach (MaterialInfo matInfo in matInfoEnum)
            {
                int count = 0;
                foreach (TexMap texMap in matInfo.texMap)
                {
                    switch (texMap.textureResourceType)
                    {
                        case TextureResourceType.LocalFile:
                            if (!dic.ContainsKey(texMap.imageName))
                            {
                                throw new LayoutDataException(string.Format(Properties.Resources.ErrorTexFileNotExistForMaterial, matInfo.name, texMap.imageName), layoutFilePath);
                            }

                            TextureFile texFile = dic[texMap.imageName];

                            SetIndirectFlagTo(texFile, IsIndirect(matInfo, count));
                            break;
                        case TextureResourceType.LocalCaptured:
                        case TextureResourceType.OverrideCaptured:
                            // キャプチャテクスチャの場合はファイルを参照していないが、参照する際の名前情報だけ出力したい。
                            // そのためダミーの TextureFile を作成して dic へ登録する。
                            if (!dic.ContainsKey(texMap.imageName))
                            {
                                dic.Add(texMap.imageName, CreateDummyTextureFileInstanceForCaptureTexture_(texMap));
                            }
                            break;
                        default:
                            break;
                    }
                    count += 1;
                }
            }
        }

        /// <summary>
        /// テクスチャファイルのファイル名をキーとしたリストを作成する。
        /// </summary>
        /// <param name="files"></param>
        /// <returns></returns>
        static SortedList<string, TextureFile> MakeTextureFileList(string layoutFilePath, Dictionary<string, Pane> paneDic, IEnumerable<MaterialInfo> matInfoEnum, TextureFile[] files)
        {
            IComparer<string> comparer = StringComparer.InvariantCultureIgnoreCase;

            SortedDictionary<string, TextureFile> dic = new SortedDictionary<string, TextureFile>(comparer);

            if (files != null)
            {
                foreach (TextureFile file in files)
                {
                    string name = file.GetName();
                    if (name == string.Empty)
                    {
                        throw new LayoutDataException(string.Format(Properties.Resources.ErrorInvalidFileName, file.imagePath));
                    }
                    if (!FileUtil.IsValidStringForFileName(name))
                    {
                        throw new LayoutDataException(string.Format(Properties.Resources.ErrorInvalidTextureFileName, file.imagePath));
                    }

                    try
                    {
                        dic.Add(name, file);
                    }
                    catch (ArgumentException)
                    {
                        throw new LayoutDataException(string.Format(Properties.Resources.ErrorSameTexFileNameExist, name));
                    }
                }
            }

            // ペインの設定から参照されているテクスチャをリストアップ
            MakePaneEffectTextureList_(dic, paneDic);

            // マテリアルで参照されているテクスチャをリストアップ
            MakeMaterialTextureList_(dic, layoutFilePath, matInfoEnum);

            if (dic.Count == 0)
            {
                return new SortedList<string, TextureFile>(0, comparer);
            }

            return new SortedList<string, TextureFile>(dic, comparer);
        }

        /// <summary>
        /// 実際にマテリアルからから参照されているテクスチャファイルのソートされたリストを作る。
        /// </summary>
        /// <param name="rlyt"></param>
        /// <param name="isHiddenNull"></param>
        /// <returns></returns>
        static SortedList<string, TextureFile> MakeRefTextureFileList(Dictionary<string, Pane> paneDic, IEnumerable<MaterialInfo> matInfoEnum, SortedList<string, TextureFile> texFileList)
        {
            SortedDictionary<string, TextureFile> refTexFileDic = new SortedDictionary<string, TextureFile>(texFileList.Comparer);

            // ペインの設定からテクスチャが参照されているかどうかチェック
            foreach (var pane in paneDic)
            {
                // マスク機能関連で参照されているかどうかチェック
                if (pane.Value.IsMaskEnabled())
                {
                    TexMap[] maskTexMap =
                    {
                        pane.Value.mask.maskTexMap,
                        pane.Value.mask.captureTexMap,
                    };

                    foreach (var texMap in maskTexMap)
                    {
                        string fileName = texMap.imageName;
                        if (!refTexFileDic.ContainsKey(fileName))
                        {
                            TextureFile texFile = texFileList[fileName];
                            SetIndirectFlagTo(texFile, false);
                            refTexFileDic.Add(fileName, texFile);
                        }
                    }
                }

                // ドロップシャドウで参照されているかどうかチェック
                if (pane.Value.IsDropShadowEnabled())
                {
                    string fileName = pane.Value.dropShadow.captureTexMap.imageName;
                    if (!refTexFileDic.ContainsKey(fileName))
                    {
                        TextureFile texFile = texFileList[fileName];
                        SetIndirectFlagTo(texFile, false);
                        refTexFileDic.Add(fileName, texFile);
                    }
                }

            }

            // マテリアルから実際に参照されているテクスチャを参照テクスチャリストへ加える
            foreach (MaterialInfo matInfo in matInfoEnum)
            {
                int count = 0;
                foreach (TexMap texMap in matInfo.texMap)
                {
                    bool is_indirect = IsIndirect(matInfo, count);
                    string fileName = texMap.imageName;

                    if (!refTexFileDic.ContainsKey(fileName))
                    {
                        try
                        {
                            TextureFile texFile = texFileList[fileName];
                            SetIndirectFlagTo(texFile, is_indirect);
                            refTexFileDic.Add(fileName, texFile);
                        }
                        catch (KeyNotFoundException)    // fileNameDicにTexMapが参照するファイル名がない
                        {
                            throw new LayoutDataException(string.Format(Properties.Resources.ErrorTexFileNameNotFound, fileName));
                        }
                    }
                    else
                    {
                        // 既に登録されている場合、用途に関する情報のみを更新する
                        // これにより、両方の用途で使われている場合は両方のフラグが立つことになる
                        TextureFile texFile = refTexFileDic[fileName];
                        SetIndirectFlagTo(texFile, is_indirect);
                    }
                    count += 1;
                }
            }

            return new SortedList<string, TextureFile>(refTexFileDic, refTexFileDic.Comparer);
        }

        /// <summary>
        /// フォントファイルのファイル名をキーとしたリストを作成する。
        /// </summary>
        /// <param name="files"></param>
        /// <returns></returns>
        static SortedList<string, FontFile> MakeFontFileList(FontFile[] files)
        {
            IComparer<string> comparer = StringComparer.InvariantCultureIgnoreCase;

            if (files == null)
            {
                return new SortedList<string, FontFile>(0, comparer);
            }

            SortedDictionary<string, FontFile> dic = new SortedDictionary<string, FontFile>(comparer);
            foreach (FontFile file in files)
            {
                string name = file.GetName();
                if (name == string.Empty)
                {
                    throw new LayoutDataException(string.Format(Properties.Resources.ErrorInvalidFileName, file.path));
                }
                if (! FileUtil.IsValidStringForFileName(name))
                {
                    throw new LayoutDataException(string.Format(Properties.Resources.ErrorInvalidFontFileName, file.path));
                }

                try
                {
                    dic.Add(name, file);
                }
                catch (ArgumentException)
                {
                    throw new LayoutDataException(string.Format(Properties.Resources.ErrorSameFontFileNameExist, name));
                }
            }

            return new SortedList<string,FontFile>(dic, comparer);
        }

        /// <summary>
        /// refFontFileDicにfontNameで指定されたフォントをfontFileListから追加する。
        /// </summary>
        static void AddFontToFontFileDic(SortedDictionary<string, FontFile> refFontFileDic, string fontName, SortedList<string, FontFile> fontFileList)
        {
            if (!refFontFileDic.ContainsKey(fontName))
            {
                try
                {
                    refFontFileDic.Add(fontName, fontFileList[fontName]);
                }
                catch (KeyNotFoundException)    // fontFileListにテキストボックスが参照するフォント名がない
                {
                    throw new LayoutDataException(string.Format(Properties.Resources.ErrorFontFileNameNotFound, fontName));
                }
            }
        }

        static void AddPartsPropertyFontToFontFileDic(SortedDictionary<string, FontFile> refFontFileDic, SortedList<string, FontFile> fontFileList, Parts parts)
        {
            if (parts.property != null)
            {
                foreach (Property property in parts.property)
                {
                    if (property.Item != null)
                    {
                        // 上書き設定はしてあるが実際に上書きは行われていないときに、Itemがnullになる
                        // ことがある。このような場合は登録しない。
                        switch (property.kind)
                        {
                            case PaneKind.TextBox:
                                TextBox textBox = property.Item as TextBox;

                                if (property.usageOptions != null) {
                                    if (Array.Exists(property.usageOptions, x => x == Schema.Flyt.PartsPropertyUsageOptions.UseTextBoxText))
                                    {
                                        // 部分上書きで、文字列とテキストIDだけ上書きが指定されているので、フォントを登録する必要はない。
                                        continue;
                                    }
                                }

                                AddFontToFontFileDic(refFontFileDic, textBox.font, fontFileList);
                                break;
                            case PaneKind.Parts:
                                Parts propertyParts = (Parts)property.Item;
                                AddPartsPropertyFontToFontFileDic(refFontFileDic, fontFileList, propertyParts);
                                break;
                        }
                    }
                }
            }
        }

        /// <summary>
        /// 実際にテキストボックスから参照されているフォントファイルのソートされたリストを作る。
        /// </summary>
        /// <param name="rlyt"></param>
        /// <returns></returns>
        static SortedList<string, FontFile> MakeRefFontFileList(IEnumerable<Pane> paneEnum, SortedList<string, FontFile> fontFileList)
        {
            SortedDictionary<string, FontFile> refFontFileDic = new SortedDictionary<string, FontFile>(fontFileList.Comparer);

            foreach (Pane pane in paneEnum)
            {
                if (pane.kind == PaneKind.TextBox) {
                    TextBox textBox = pane.Item as TextBox;
                    AddFontToFontFileDic(refFontFileDic, textBox.font, fontFileList);
                } else if (pane.kind == PaneKind.Parts) {
                    Parts parts = pane.Item as Parts;
                    AddPartsPropertyFontToFontFileDic(refFontFileDic, fontFileList, parts);
                }
            }

            return new SortedList<string, FontFile>(refFontFileDic, fontFileList.Comparer);
        }

        static void RegisterShapeToList(ref List<ShapeInfo> list, Pane pane, PaneKind kind, object item)
        {
            switch (kind)
            {
                case PaneKind.Picture:
                    {
                        Picture picture = (Picture)item;

                        if (picture.shapeType != ShapeType.NormalQuad)
                        {
                            ShapeInfo newShapeInfo = new ShapeInfo();
                            bool newShape = true;

                            newShapeInfo.ShapeType = picture.shapeType;
                            newShapeInfo.GfxShapeParam = picture.shapeParam;

                            foreach (ShapeInfo info in list)
                            {
                                if (info.EqualsContent(newShapeInfo))
                                {
                                    newShapeInfo = info;
                                    newShape = false;
                                    break;
                                }
                            }

                            if (newShape)
                            {
                                newShapeInfo.BinaryIndex = (uint)list.Count;
                                list.Add(newShapeInfo);
                            }

                            picture.ShapeInfo = newShapeInfo;
                        }
                        else
                        {
                            picture.ShapeInfo = null;
                        }
                    }
                    break;
                case PaneKind.Parts:
                    {
                        Parts parts = (Parts)item;
                        if (parts.property != null)
                        {
                            foreach (Property property in parts.property)
                            {
                                // 上書き設定はしてあるが実際に上書きは行われていないときに、Itemがnullになる
                                // ことがある。このような場合は登録しない。
                                if (property.Item != null)
                                {
                                    RegisterShapeToList(ref list, pane, property.kind, property.Item);
                                }
                            }
                        }
                    }
                    break;
            }
         }

        /// <summary>
        /// 渡されたペインのリストから ShapeInfo のリストを作成します。
        /// </summary>
        /// <param name="paneDic"></param>
        /// <returns></returns>
        static List<ShapeInfo> MakeShapeInfoList(Dictionary<string, Pane> paneDic)
        {
            List<ShapeInfo> shapeInfoList = new List<ShapeInfo>();

            foreach (Pane pane in paneDic.Values)
            {
                RegisterShapeToList(ref shapeInfoList, pane, pane.kind, pane.Item);
            }

            return shapeInfoList;
        }

        /// <summary>
        /// キャプチャテクスチャ名をキーとしたリストを作成する。
        /// </summary>
        /// <param name="files"></param>
        /// <returns></returns>
        static SortedList<string, CaptureTexture> MakeCaptureTextureList(Dictionary<string, Pane> paneDic, CaptureTexture[] captureTextures)
        {
            IComparer<string> comparer = StringComparer.InvariantCultureIgnoreCase;

            if (captureTextures == null)
            {
                return new SortedList<string, CaptureTexture>(0, comparer);
            }

            SortedDictionary<string, CaptureTexture> dic = new SortedDictionary<string, CaptureTexture>(comparer);
            foreach (var iter in paneDic)
            {
                foreach (CaptureTexture tex in captureTextures)
                {
                    if (tex.paneName == iter.Value.name)
                    {
                        string texName = iter.Value.name;
                        // LayoutEditor の CaptureTexture.GetNameSuffix と同じ処理です。
                        switch (tex.usage)
                        {
                            case CaptureTextureUsage.Normal:
                                break;
                            case CaptureTextureUsage.Mask:
                                texName += "_M";
                                break;
                            case CaptureTextureUsage.DropShadow:
                                texName += "_D";
                                break;
                            default:
                                Debug.Assert(false);
                                break;
                        }
                        dic.Add(texName, tex);
                    }
                }
            }

            return new SortedList<string, CaptureTexture>(dic, comparer);
        }


        /// <summary>
        /// 実際にマテリアルからから参照されているキャプチャテクスチャのソートされたリストを作る。
        /// </summary>
        /// <param name="rlyt"></param>
        /// <param name="isHiddenNull"></param>
        /// <returns></returns>
        static SortedList<string, CaptureTexture> MakeRefCaptureTextureList(Dictionary<string, Pane> paneDic, IEnumerable<MaterialInfo> matInfoEnum, SortedList<string, CaptureTexture> captureTextureList)
        {
            SortedDictionary<string, CaptureTexture> refCapTexDic = new SortedDictionary<string, CaptureTexture>(StringComparer.InvariantCultureIgnoreCase);

            // ペインの設定から参照されているかどうかチェック
            foreach (var pane in paneDic)
            {
                // マスク機能
                if (pane.Value.IsMaskEnabled())
                {
                    TexMap maskTexMap = pane.Value.mask.maskTexMap;
                    TexMap[] texMapList =
                    {
                        pane.Value.mask.maskTexMap,
                        pane.Value.mask.captureTexMap,
                    };

                    foreach (var texMap in texMapList)
                    {
                        if (texMap.textureResourceType == TextureResourceType.LocalFile)
                        {
                            continue;
                        }

                        if (captureTextureList.ContainsKey(texMap.imageName) &&
                            !refCapTexDic.ContainsKey(texMap.imageName))
                        {
                            refCapTexDic.Add(texMap.imageName, captureTextureList[texMap.imageName]);
                        }
                    }
                }

                // ドロップシャドウ機能
                if (pane.Value.IsDropShadowEnabled())
                {
                    string imageName = pane.Value.dropShadow.captureTexMap.imageName;
                    if (captureTextureList.ContainsKey(imageName))
                    {
                        if (!refCapTexDic.ContainsKey(imageName))
                        {
                            refCapTexDic.Add(imageName, captureTextureList[imageName]);
                        }
                    }
                }
            }

            // マテリアルから参照されているかどうかチェックする
            foreach (MaterialInfo matInfo in matInfoEnum)
            {
                foreach (TexMap texMap in matInfo.texMap)
                {
                    if (texMap.textureResourceType == TextureResourceType.LocalFile)
                    {
                        continue;
                    }

                    // キャプチャペインを参照していたらキャプチャテクスチャを参照しているとみなす。
                    if (captureTextureList.ContainsKey(texMap.imageName))
                    {
                        if (!refCapTexDic.ContainsKey(texMap.imageName))
                        {
                            refCapTexDic.Add(texMap.imageName, captureTextureList[texMap.imageName]);
                        }
                    }
                    else
                    {
                        // 同一レイアウト内のキャプチャテクスチャが見つからないときのみエラーとして警告する。
                        if (texMap.textureResourceType == TextureResourceType.LocalCaptured)
                        {
                            // マテリアルで参照されている名前のキャプチャテクスチャがシーン中に見つからなかった。
                            throw new LayoutDataException(string.Format(Properties.Resources.ErrorCaptureTextureNotFound, matInfo.Pane.name, texMap.imageName));
                        }
                    }
                }
            }

            return new SortedList<string, CaptureTexture>(refCapTexDic, refCapTexDic.Comparer);
        }

        class MaterialRegister
        {
            readonly List<MaterialInfo> _matList;

            public MaterialRegister(List<MaterialInfo> matList)
            {
                _matList = matList;
            }

            public MaterialInfo Add(MaterialInfo matInfo)
            {
                foreach (MaterialInfo mat in _matList)
                {
                    if (matInfo.name == mat.name && matInfo.EqualsContent(mat))
                    {
                        return mat;
                    }
                }
                matInfo.BinaryIndex = _matList.Count;   // MaterialListのインデックス値を覚えておく
                _matList.Add(matInfo);
                return matInfo;
            }
        }

        /// <summary>
        /// ピクチャペインごとのフィルレート最適化結果情報
        /// </summary>
        public class PictureFillrateOptimizeInformation
        {
            /// <summary>
            /// 最適化結果の阻害要因。
            /// </summary>
            public enum UnoptimizedReason
            {
                InvalidShapeType,
                MultiTexture,
                NoTexture,
                NoAlphaChannel,
                TransparentImage,
                OpaquePixelInEdgeForOBB,
                DetailedCombiner,
                VertexColorAnimation,
                VertexColor,
                TextureSRTAnimation,
                IndTextureSRTAnimation,
                TexturePatternAnimation,
                TextureMatrix,
                TextureProjection,
                ComplexTextureCoordinate,
                InternalError,
                BlackColorAlphaIsNotZero,
                FragmentBlendSettingIsNotDefault,
                BlackColorAlphaAnimation,
                Max
            }

            public string Name { get; set; } = "";
            public string ImagePath { get; set; } = "";
            public int TexWidth { get; set; } = 0;
            public int TexHeight { get; set; } = 0;
            public ImageFitMesh.OptimizationType OptimizationType { get; set; } = ImageFitMesh.OptimizationType.None;
            public int OptimizedTexWidth { get; set; } = 0;
            public int OptimizedTexHeight { get; set; } = 0;
            public List<UnoptimizedReason> Reasons { get; set; } = new List<UnoptimizedReason>();
        };

        /// <summary>
        /// フィルレート最適化結果情報
        /// </summary>
        public class FillrateOptimizeInformation
        {
            public enum FailReason{
                FailReason_VertexColorAnimation,
            };

            public string FileName { get; set; } = "";
            public string Date { get; set; } = "";
            public int TotalPaneCount { get; set; } = 0;
            public int PicturePaneCount { get; set; } = 0;
            public int TextBoxPaneCount { get; set; } = 0;
            public int WindowPaneCount { get; set; } = 0;
            public int PartsPaneCount { get; set; } = 0;
            public int NullPaneCount { get; set; } = 0;
            public int BoundingPaneCount { get; set; } = 0;
            public int CapturePaneCount { get; set; } = 0;
            public int AlignmentPaneCount { get; set; } = 0;
            public int ScissorPaneCount { get; set; } = 0;
            public int IncreasedMemorySize { get; set; } = 0;
            public int FillRateOptimizedPicturePaneCount { get; set; } = 0;

            public Dictionary<string, PictureFillrateOptimizeInformation> PictInfo { get; set; } = new Dictionary<string, PictureFillrateOptimizeInformation>();
        }
    }
}
