﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Diagnostics;

namespace NW4F.LayoutBinaryConverter
{
    public class FontFileUtil
    {
        /// <summary>
        /// フォントファイルをコピーする
        /// </summary>
        /// <param name="srcBaseDir"></param>
        /// <param name="fontFiles"></param>
        /// <param name="dstBaseDir"></param>
        /// <param name="resType"></param>
        public static void CopyFontFile(string srcBasePath, ICollection<Schema.Flyt.FontFile> fontFileEnum, string dstBaseDir)
        {
            for (int i = 0; i < 2; ++i)
            {
                string dstDir = null;

                foreach (Schema.Flyt.FontFile fontFile in fontFileEnum)
                {
                    if (string.IsNullOrEmpty(fontFile.path))
                    {
                        // パスが空の場合はコピーしない
                        continue;
                    }

                    string extName = Path.GetExtension(fontFile.path).ToLowerInvariant();
                    bool bArcFont = (".brfna" == extName);

                    if (".ttf" == extName || ".otf" == extName || ".bfttf" == extName || ".bfotf" == extName)
                    {
                        // TTFファイルはコピーしない
                        continue;
                    }

                    if (".fcpx" == extName)
                    {
                        // fcpx ファイルはコピーしない
                        continue;
                    }

                    if (".ffnt" == extName)
                    {
                        // ffnt ファイルはコピーしない
                        continue;
                    }

                    if (i == 0 ? bArcFont: !bArcFont)
                    {
                        continue;
                    }

                    if (dstDir == null)
                    {
                        dstDir = FileUtil.MakeResourceDirectory(dstBaseDir, i == 0 ? BinaryLytWriter.TypeOfFontResource : BinaryLytWriter.TypeOfArchiveFontResource);
                    }

                    string srcPath = FileUtil.GetAbsolutePath(srcBasePath, fontFile.path);
                    string dstFilePath = Path.Combine(dstDir, Path.GetFileName(srcPath));
                    CopyFile(srcPath, dstFilePath);
                }
            }
        }

        static void CopyFile(string srcPath, string dstPath)
        {
            /*
             * File.Copy() は内部で、Win32 の CopyFile() を呼び出すが、
             * このAPIは コピー先ファイルに Hidden, ReadOnly属性が付いていると、ERROR_ACCESS_DENIED となり、
             * 結果として、UnauthorizedAccessException が throw される。
             * そのため、あらかじめ属性をチェックして、これら属性が付いている場合は外しておく。
             */
            FileInfo dstFile = new FileInfo(dstPath);
            if (dstFile.Exists)
            {
                try
                {
                    RemoveReadOnlyAndHiddenAttribute(dstFile);
                }
                catch (IOException)
                {
                    // 属性を落とすだけなので、ここでIOException例外が発生しても
                    // コピー操作を実行させてみる。
                }
            }

            File.Copy(srcPath, dstPath, true);  // 上書きする

            /*
             * srcPathのファイルにHidden, ReadOnly属性が付いていた場合はこれもコピーされる。
             * レイアウトを構成するバイナリファイルのうち、コピーだけですむバイナリファイルのみが
             * 属性を受け継いでしまうのは構成上望ましくないので、これらの属性を外す。
             */
            dstFile = new FileInfo(dstPath);
            try
            {
                RemoveReadOnlyAndHiddenAttribute(dstFile);
            }
            catch (IOException)
            {
                // 属性を落とすだけなので、ここでIOException例外が発生しても
                // コンバート処理は継続させる
            }
        }

        /// <summary>
        /// Hidden, ReadOnly属性を取り除く
        /// </summary>
        /// <param name="dstFile"></param>
        static void RemoveReadOnlyAndHiddenAttribute(FileInfo dstFile)
        {
            FileAttributes attributes = dstFile.Attributes;
            if (0 != (attributes & FileAttributes.Hidden)
              || 0 != (attributes & FileAttributes.ReadOnly))
            {
                attributes &= ~FileAttributes.Hidden;
                attributes &= ~FileAttributes.ReadOnly;
                dstFile.Attributes = attributes;
            }
        }

    }
}
