﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using System.Text.RegularExpressions;
using System.IO;
using System.Diagnostics;
using System.Runtime.InteropServices;

namespace NW4F.LayoutBinaryConverter
{
    public enum FileType
    {
        unknown,
        flyt,
        flan,
        clpa,
        clvi,
        clvc,
        clmc,
        clts,
        cltp,
        fcpx,
    }

    public class FileUtil
    {
        /// <summary>
        /// WiiのDVD関数のファイル名で無効な文字を表現する正規表現。
        /// </summary>
        static readonly Regex sInvalidFileNameChar_ = new Regex(@"[^0-9A-Za-z !#$%&'()+.=\[\]^_@`{}~-]");
        /*
            WiiのDVD関数のファイル名で有効な文字
                数字
                アルファベット
                スペース
                ! # $ % & ' ( ) + - . = [ ] ^ _ @ ` { } ~

            文字の集合を指定する [] も含まれるので、これは \ によりエスケープしている。
            - は範囲指定として扱われないように最後に置いている。
            ^ も含まれるが、先頭以外にあるので問題なし。
         */

        /// <summary>
        /// ファイルのタイプを返します。
        /// </summary>
        /// <param name="fileName"></param>
        /// <returns></returns>
        public static FileType GetFileType(string fileName)
        {
            switch (Path.GetExtension(fileName).ToLowerInvariant())
            {
            case ".flyt": return FileType.flyt;
            case ".flan": return FileType.flan;
            case ".clpa": return FileType.clpa;
            case ".clvi": return FileType.clvi;
            case ".clvc": return FileType.clvc;
            case ".clmc": return FileType.clmc;
            case ".clts": return FileType.clts;
            case ".cltp": return FileType.cltp;
            case ".fcpx": return FileType.fcpx;
            default: return FileType.unknown;
            }
        }

        /// <summary>
        /// NintendoWareツールのパスを取得します。
        /// </summary>
        public static string GetNwToolPath()
        {
            string executingFileName = System.Reflection.Assembly.GetExecutingAssembly().Location;
            string executingFolder = Path.GetDirectoryName(executingFileName);
            string converterFolderParentPath = Path.GetDirectoryName(executingFolder);
            if (converterFolderParentPath.Contains("Tools\\Graphics"))
            {
                if (converterFolderParentPath.EndsWith("Tools\\Graphics"))
                {
                    // パッケージフォルダ構成での動作
                    return converterFolderParentPath;
                }
                else
                {
                    // リポジトリフォルダ構成での動作
                    // 親フォルダをたどって基点を見つけ、Toolフォルダを決定する。
                    string parentDir = converterFolderParentPath;
                    try
                    {
                        while (!parentDir.EndsWith("Programs"))
                        {
                            parentDir = Path.GetDirectoryName(parentDir);
                        }
                        parentDir = Path.GetDirectoryName(parentDir);
                    }
                    catch
                    {
                        // 想定外のフォルダ構成だった。
                        // 適当な値を設定しておく。
                        parentDir = converterFolderParentPath;
                    }

                    return Path.Combine(parentDir, "Tools\\Graphics");
                }
            }
            else
            {
                return Path.Combine(Environment.GetEnvironmentVariable("NW4F_ROOT"), "Tool");
            }
        }

        /// <summary>
        /// フルパスを指定して、拡張子をレイアウトファイルに変更したファイル名を取得します。
        /// </summary>
        /// <param name="fullFileName"></param>
        /// <returns></returns>
        public static string GetLayoutFileName(string fullFileName)
        {
            return Path.ChangeExtension(fullFileName, ".flyt");
        }

        /// <summary>
        /// フルパスを指定して、拡張子をアニメーションファイルに変更したファイル名を取得します。
        /// </summary>
        /// <param name="fullFileName"></param>
        /// <returns></returns>
        public static string GetAnimFileName(string fullFileName)
        {
            return Path.ChangeExtension(fullFileName, ".flan");
        }

        public static bool IsDirectorySeparatorEnd(string pathStr)
        {
            return pathStr != null && pathStr.Length >= 1 && pathStr[pathStr.Length - 1] == Path.DirectorySeparatorChar;
        }

        public static String TrimDirectorySeparatorEnd(String pathStr)
        {
            // 末尾の'\'を取る
            return Path.GetDirectoryName(pathStr + '*');
        }

        public static string[] ExpandInPath(string inPath, bool recursive, out bool bInDirPath)
        {
            if (IsDirectorySeparatorEnd(inPath))
            {
                // 末尾の'\'を取る
                inPath = TrimDirectorySeparatorEnd(inPath);
                if (!Directory.Exists(inPath))
                {
                    throw new LayoutConverterException(string.Format("not found input directory. - {0}", inPath));
                }

                bInDirPath = true;
            }
            else
            {
                if (Directory.Exists(inPath))
                {
                    bInDirPath = true;
                }
                else if (File.Exists(inPath))
                {
                    bInDirPath = false;
                }
                else
                {
                    throw new LayoutConverterException(string.Format("not found input directory/file. - {0}", inPath));
                }
            }

            if (bInDirPath)
            {
                return Directory.GetFiles(inPath, "*", recursive ? System.IO.SearchOption.AllDirectories : System.IO.SearchOption.TopDirectoryOnly);
            }
            else
            {
                return new string[] { inPath };
            }
        }

        public static string MakeOutFileNameStr(string outDirName, string prefixStr, string inFileName, string suffixStr)
        {
            // 入力ファイル名に拡張子をバイナリの"b"を付加した名前にする。
            if (prefixStr != null)
            {
                prefixStr = prefixStr + "_";
            }

            if (suffixStr != null)
            {
                suffixStr = "_" + suffixStr;
            }

            return Path.Combine(
                        outDirName,
                        prefixStr
                        + Path.GetFileNameWithoutExtension(inFileName)
                        + suffixStr
                        + Path.GetExtension(inFileName).Insert(1, "b"));
        }

        /// <summary>
        /// WiiのDVD関数のファイル名として使用可能な文字で構成されているかどうかを判定します。
        /// </summary>
        /// <param name="fileName">ファイル名。拡張子を含まない純粋なファイル名である必要があります。</param>
        /// <returns></returns>
        public static bool IsValidStringForFileName(string fileName)
        {
            return !sInvalidFileNameChar_.IsMatch(fileName);
        }

        /// <summary>
        /// テンポラリディレクトリを作成し、そのパスを返す。
        /// </summary>
        /// <returns></returns>
        public static string CreateTempDirectory()
        {
            string exePath = Environment.GetCommandLineArgs()[0]; // 実行ファイルへのパス
            Guid guidValue = Guid.NewGuid();
            string tempDirName = Path.Combine(Path.GetTempPath(), Path.GetFileNameWithoutExtension(exePath) + "_" + guidValue.ToString());
            DirectoryInfo tempDir = new DirectoryInfo(tempDirName);
            if (tempDir.Exists)
            {
                tempDir.Delete(true);
            }
            tempDir.Create();
            return tempDirName;
        }

        public static string GetAbsolutePath(string basePath, string relPath)
        {
            // uriクラスを使ってローカルパスを合成する。
            //
            //      Path.GetFullPath()では、combPath が 260 を超える場合は、
            //      最終的なパス文字列が260を超えなくても
            //      なぜか PathTooLongException が throw されてしまう。

            // 相対uriとの合成がうまくいくように、baseUri の末尾に'/'が付くようにしておく。
            Uri baseUri = new Uri(basePath);
            Uri absUri = new Uri(baseUri, relPath);
            return absUri.LocalPath;
        }

        /// <summary>
        /// 更新されているのみファイルを移動する
        /// </summary>
        /// <param name="dstDir"></param>
        /// <param name="srcDir"></param>
        public static void MoveUpdateFiles(string dstDir, string srcDir)
        {
            foreach (DirectoryInfo di in new DirectoryInfo(srcDir).GetDirectories())
            {
                foreach (FileInfo fi in di.GetFiles())
                {
                    string dstFileName = Path.Combine(Path.Combine(dstDir, di.Name), fi.Name);

                    if (!File.Exists(dstFileName)) // 出力ディレクトリにファイルが無い場合
                    {
                        DirectoryInfo parentDir = new DirectoryInfo(Path.GetDirectoryName(dstFileName));
                        if (!parentDir.Exists)
                        {
                            parentDir.Create();
                        }
                        File.Move(fi.FullName, dstFileName);
                    }
                    else if (!EqualFile(fi.FullName, dstFileName)) // 出力ディレクトリに同じ名前のファイルがあり、内容が異なる場合
                    {
                        string tempName = dstFileName + "_temp";
                        File.Move(dstFileName, tempName);
                        File.Move(fi.FullName, dstFileName);
                        File.Delete(tempName);
                    }
                }
            }
        }

        // ファイルの内容比較
        static bool EqualFile(string file1, string file2)
        {
            FileInfo fi1 = new FileInfo(file1);
            FileInfo fi2 = new FileInfo(file2);

            if (!fi1.Exists || !fi2.Exists)
            {
                return false;
            }

            if (fi1.Length != fi2.Length)
            {
                return false;
            }

            byte[] buf1 = new byte[8192];
            byte[] buf2 = new byte[8192];

            using (FileStream fs1 = new FileStream(fi1.FullName, FileMode.Open, FileAccess.Read))
            using (FileStream fs2 = new FileStream(fi2.FullName, FileMode.Open, FileAccess.Read))
            {
                while (true)
                {
                    int readByte1 = fs1.Read(buf1, 0, buf1.Length);
                    int readByte2 = fs2.Read(buf2, 0, buf2.Length);
                    if (readByte1 != readByte2)
                    {
                        return false;
                    }

                    if (readByte1 == 0) // ファイルの終端に達した?
                    {
                        break;
                    }

                    for (int i = 0; i < readByte1; ++i)
                    {
                        if (buf1[i] != buf2[i])
                        {
                            return false;
                        }
                    }
                }
            }

            return true;
        }

        /// <summary>
        /// アニメーションにプリフィックスとして付ける名前を生成する。
        /// </summary>
        /// <param name="inFileName"></param>
        /// <returns></returns>
        public static string MakeLayoutPrefixName(string inFileName)
        {
            string[] layoutFileNames = Directory.GetFiles(Path.GetDirectoryName(inFileName), "*.rlyt");
            if (layoutFileNames.Length == 0)
            {
                return null;
            }
            else
            {
                return Path.GetFileNameWithoutExtension(layoutFileNames[0]);
            }
        }

        public static void RoundUpFileSize(Stream fs, int alignment)
        {
            long currentFilePos = fs.Position;
            long alignFilePos = (currentFilePos + (alignment - 1)) & ~(alignment - 1);
            for (long filePos = currentFilePos; filePos < alignFilePos; ++filePos)
            {
                fs.WriteByte(0);
            }
        }

        /// <summary>
        /// 文字列プールの書き込み
        /// </summary>
        /// <param name="fs"></param>
        /// <param name="nameTable"></param>
        public static void WriteStringPool(Stream fs, NameTable nameTable)
        {
            fs.Write(nameTable.namePool, 0, nameTable.namePool.Length);
        }

        /// <summary>
        /// // オフセットテーブルの書き込み
        /// </summary>
        /// <param name="stream"></param>
        /// <param name="offsetTable"></param>
        /// <param name="offsetTableFilePos"></param>
        public static void WriteOffsetTable(Stream stream, uint[] offsetTable)
        {
            BinaryWriter bw = EndianAwareBinaryWriter.CreateResouceEndianBinaryWriter(stream);
            foreach (uint offset in offsetTable)
            {
                bw.Write(offset);
            }
            bw.Flush();
        }

        public static void WriteArray<T>(Action<T> writeAction, Stream stream, ICollection<T> collection, Int64 blockHeadFilePos)
        {
            // オフセットテーブル分移動させる。
            Int64 offsetTableFilePos = stream.Position;
            stream.Seek(Marshal.SizeOf(typeof(uint)) * collection.Count, SeekOrigin.Current);
            uint[] offsetTable = new uint[collection.Count];

            // 各データの書き込み
            int index = 0;
            foreach (T data in collection)
            {
                offsetTable[index] = (uint)(stream.Position - blockHeadFilePos);
                offsetTable[index] = DataUtil.HostToTarget(offsetTable[index]);
                ++index;

                writeAction(data);
            }

            long currentFilePos = stream.Position;
            stream.Seek(offsetTableFilePos, SeekOrigin.Begin);
            FileUtil.WriteOffsetTable(stream, offsetTable);
            stream.Seek(currentFilePos, SeekOrigin.Begin);    // シーク位置を戻す
        }

        public delegate void WriteFunc(uint blockSize);

        public static void WriteDataBlock(Stream stream, WriteFunc writeFunc, long offset)
        {
            long crFilePos = stream.Position;
            stream.Position = offset;
            writeFunc((uint)(crFilePos - offset));
            stream.Position = crFilePos;
        }

        /// <summary>
        /// リソース用ディレクトリを作成し、そのディレクトリパスを返します。
        /// </summary>
        /// <param name="baseDirName"></param>
        /// <param name="resType"></param>
        /// <returns></returns>
        public static string MakeResourceDirectory(string baseDirName, uint resType)
        {
            DirectoryInfo di = new DirectoryInfo(Path.Combine(baseDirName, DataUtil.GetResourceTypeString(resType)));
            if (!di.Exists)
            {
                di.Create();
            }
            return di.FullName;
        }
    }
}
