﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.IO;
using CombinerEditor.Service;

namespace NW4F.LayoutBinaryConverter
{
    using Lyt = Schema.Flyt;

    partial class LytWriter
    {
        /// <summary>
        /// 標準のシェーダパラメータの名前変換テーブルです。
        /// </summary>
        private readonly Dictionary<string, string> baseParameterNameTable = new Dictionary<string, string>()
        {
            // Textures
            { "_a0", "テクスチャ0" },
            { "_a1", "テクスチャ1" },
            { "_a2", "テクスチャ2" },
        };

        private struct ExUserDataInfo
        {
            public string propertyName;
            public string text;
            public Lyt.UserDataElementType kind;
            public string name;
            public int pri;

            public ExUserDataInfo(string _propertyName, string _text, Lyt.UserDataElementType _kind, string _name, int _pri)
            {
                propertyName = _propertyName;
                text = _text;
                kind = _kind;
                name = _name;
                pri = _pri;
            }
        };

        /// <summary>
        /// 拡張ユーザ情報パラメータの名前変換テーブルです。
        /// </summary>
        static private readonly ExUserDataInfo[] extUserDataParameterTable = new []
        {
            // Float
            new ExUserDataInfo("m_VectorParameter8.x", "ExUserDataFloat_0", Lyt.UserDataElementType.AnmFloat, "__CUS_Float_0", 0),
            new ExUserDataInfo("m_VectorParameter8.y", "ExUserDataFloat_1", Lyt.UserDataElementType.AnmFloat, "__CUS_Float_1", 1),
            new ExUserDataInfo("m_VectorParameter8.z", "ExUserDataFloat_2", Lyt.UserDataElementType.AnmFloat, "__CUS_Float_2", 2),
            new ExUserDataInfo("m_VectorParameter8.w", "ExUserDataFloat_3", Lyt.UserDataElementType.AnmFloat, "__CUS_Float_3", 3),

            // Vec2
            new ExUserDataInfo("m_VectorParameter9.xy", "ExUserDataVec2_0", Lyt.UserDataElementType.AnmFloatVec2, "__CUS_Vec2_0", 4),
            new ExUserDataInfo("m_VectorParameter9.zw", "ExUserDataVec2_1", Lyt.UserDataElementType.AnmFloatVec2, "__CUS_Vec2_1", 5),
            new ExUserDataInfo("m_VectorParameter10.xy", "ExUserDataVec2_2", Lyt.UserDataElementType.AnmFloatVec2, "__CUS_Vec2_2", 6),
            new ExUserDataInfo("m_VectorParameter10.zw", "ExUserDataVec2_3", Lyt.UserDataElementType.AnmFloatVec2, "__CUS_Vec2_3", 7),

            // Vec3
            new ExUserDataInfo("m_VectorParameter11.xyz", "ExUserDataVec3_0", Lyt.UserDataElementType.AnmFloatVec3, "__CUS_Vec3_0", 8),
            new ExUserDataInfo("m_VectorParameter12.xyz", "ExUserDataVec3_1", Lyt.UserDataElementType.AnmFloatVec3, "__CUS_Vec3_1", 9),
            new ExUserDataInfo("m_VectorParameter13.xyz", "ExUserDataVec3_2", Lyt.UserDataElementType.AnmFloatVec3, "__CUS_Vec3_2", 10),
            new ExUserDataInfo("m_VectorParameter14.xyz", "ExUserDataVec3_3", Lyt.UserDataElementType.AnmFloatVec3, "__CUS_Vec3_3", 11),

            // Int
            new ExUserDataInfo("m_IVectorParameter0.x", "ExUserDataInt_0", Lyt.UserDataElementType.AnmInt, "__CUS_Int_0", 12),
            new ExUserDataInfo("m_IVectorParameter0.y", "ExUserDataInt_1", Lyt.UserDataElementType.AnmInt, "__CUS_Int_1", 13),
            new ExUserDataInfo("m_IVectorParameter0.z", "ExUserDataInt_2", Lyt.UserDataElementType.AnmInt, "__CUS_Int_2", 14),
            new ExUserDataInfo("m_IVectorParameter0.w", "ExUserDataInt_3", Lyt.UserDataElementType.AnmInt, "__CUS_Int_3", 15),

            // IVec2
            new ExUserDataInfo("m_IVectorParameter1.xy", "ExUserDataIvec2_0", Lyt.UserDataElementType.AnmIntVec2, "__CUS_IVec2_0", 16),
            new ExUserDataInfo("m_IVectorParameter1.zw", "ExUserDataIvec2_1", Lyt.UserDataElementType.AnmIntVec2, "__CUS_IVec2_1", 17),
            new ExUserDataInfo("m_IVectorParameter2.xy", "ExUserDataIvec2_2", Lyt.UserDataElementType.AnmIntVec2, "__CUS_IVec2_2", 18),
            new ExUserDataInfo("m_IVectorParameter2.zw", "ExUserDataIvec2_3", Lyt.UserDataElementType.AnmIntVec2, "__CUS_IVec2_3", 19),

            // Rgba
            new ExUserDataInfo("m_IVectorParameter3", "ExUserDataRgba_0", Lyt.UserDataElementType.AnmByteRGBA4, "__CUS_Rgba_0", 20),
            new ExUserDataInfo("m_IVectorParameter4", "ExUserDataRgba_1", Lyt.UserDataElementType.AnmByteRGBA4, "__CUS_Rgba_1", 21),
            new ExUserDataInfo("m_IVectorParameter5", "ExUserDataRgba_2", Lyt.UserDataElementType.AnmByteRGBA4, "__CUS_Rgba_2", 22),
            new ExUserDataInfo("m_IVectorParameter6", "ExUserDataRgba_3", Lyt.UserDataElementType.AnmByteRGBA4, "__CUS_Rgba_3", 23),
        };

        private int GetTexturePropertyToTexMapNo(string propertyName)
        {
            int _texMapNo = -1;
            if (propertyName.Equals("_a0")) _texMapNo = 0;
            else if (propertyName.Equals("_a1")) _texMapNo = 1;
            else if (propertyName.Equals("_a2")) _texMapNo = 2;
            else throw new InvalidOperationException();
            return _texMapNo;
        }

        /// <summary>
        /// コンバイナエディタサービスの読み込み
        /// </summary>
        /// <param name="combinerEditorPath"></param>
        private bool InitializeCombinerEditorService(string combinerEditorPath)
        {
            Debug.Assert(!string.IsNullOrEmpty(combinerEditorPath));

            if(CombinerEditor.Service.CombinerEditorService.IsInitialized)
            {
                // 既に初期化(LayoutEditor 側も含む)されている場合は再初期化をしない。
                return true;
            }

            string combinerEditorServiceImpl = "CombinerEditor.Service.Impl.dll";
            string combinerEditorDirectry = Path.GetDirectoryName(combinerEditorPath);
            string path = Path.Combine(combinerEditorDirectry,　"CoreModules", combinerEditorServiceImpl);
            bool resInitialize = CombinerEditor.Service.CombinerEditorService.Initialize(path, "CombinerEditor.Service.CombinerEditorServiceImpl");
            if (resInitialize == false)
            {
                Report.Out.WriteLine(string.Format(Properties.Resources.ErrorCombinerEditorService, combinerEditorServiceImpl, path ));
                return false;
            }
            return true;
        }

        public void CheckCombinerUserShader(LytInfo rlyt, string shaderEnvDirectories, string combinerEditorPath)
        {
            if (string.IsNullOrEmpty(shaderEnvDirectories))
            {
                throw new LayoutConverterException(
                    string.Format(
                        Properties.Resources.ErrorShaderEvnPathNotExist,
                        shaderEnvDirectories
                    )
                );
            }
            if (string.IsNullOrEmpty(combinerEditorPath))
            {
                throw new LayoutConverterException(Properties.Resources.ErrorCombinerEditorUnset);
            }
            // 設定されパスに CombinerEditor.exe が存在しない場合。
            if (!File.Exists(combinerEditorPath))
            {
                throw new LayoutConverterException(
                    string.Format(
                        Properties.Resources.ErrorCombinerEditorNotExist,
                        combinerEditorPath
                    )
                );
            }

            // コンバイナエディタサービスの読み込み
            if (!InitializeCombinerEditorService(combinerEditorPath))
            {
                // サービスの読み込みに失敗した場合は、検査処理をスキップする。
                return;
            }

            // ブロック定義データをロード
            var blockDefinitions = CombinerEditorService.LoadBlockDefinitions(new string[]{shaderEnvDirectories});

            if (blockDefinitions == null || blockDefinitions.Count() == 0)
            {
                throw new LayoutConverterException(
                    string.Format(
                        Properties.Resources.ErrorLoadBlockDefinitions,
                        shaderEnvDirectories
                    )
                );
            }

            foreach (Lyt.MaterialInfo materialInfo in rlyt.MaterialList)
            {
                if (materialInfo.UseCombinerUserShaderSettings)
                {
                    // パーツに設定されたコンバイナファイルは検査を行いません。
                    Lyt::Pane pane = materialInfo.Pane;
                    Debug.Assert(pane != null);
                    if (pane.kind == Lyt.PaneKind.Parts)
                    {
                        continue;
                    }

                    string combinerFile = materialInfo.combinerUserShader.fileName;
                    if (string.IsNullOrEmpty(combinerFile))
                    {
                        continue;
                    }

                    // コンバイナファイルをロード
                    IEnumerable<BlockElementInfo> blockElements = null;
                    try
                    {
                        blockElements = CombinerEditorService.LoadCombinerFile(combinerFile, blockDefinitions);
                    }
                    catch
                    {
                        throw new LayoutConverterException(
                            string.Format(Properties.Resources.ErrorConvertCombinerFile, combinerFile)
                        );
                    }

                    if (blockElements == null || blockElements.Count() == 0)
                    {
                        // 判定される uniform 変数が存在しない
                        continue;
                    }

                    // テクスチャ
                    var textureBlockPropertyInfos = blockElements
                                            .SelectMany(s => s.BlockProperties)
                                            .Where(t => t.PropertyType == BlockPropertyType.ShaderParameter
                                                        && baseParameterNameTable.ContainsKey(t.PropertyName))
                                            .GroupBy(i => i.PropertyName).Select(t => t.First())            // 同一のプロパティ名を除外 ※Distinct() 相当。 override して GetHashCode, Equals 用意しない為。
                                            .OrderBy(t => GetTexturePropertyToTexMapNo(t.PropertyName));     // _a0~_a2 に並べ直す

                    foreach(var blockProperty in textureBlockPropertyInfos)
                    {
                        int texMapNo = GetTexturePropertyToTexMapNo(blockProperty.PropertyName);
                        if (texMapNo >= materialInfo.texMap.Count())
                        {
                            // コンバイナファイルで利用されているマップ№にテクスチャが設定されていない
                            throw new LayoutConverterException(
                                string.Format(Properties.Resources.ErrorTexMapNoForCombinerFile, combinerFile, baseParameterNameTable[blockProperty.PropertyName])
                            );
                        }
                    }

                    // 拡張ユーザ情報(priority 収集用に匿名型の利用)
                    var extUserDataBlockPropertyInfos = blockElements
                                            .SelectMany(s => s.BlockProperties)
                                            .Where(t => t.PropertyType == BlockPropertyType.ShaderParameter
                                                && extUserDataParameterTable.Any(p => p.propertyName == t.PropertyName))
                                            .GroupBy(i => i.PropertyName).Select(t => t.First())            // 同一のプロパティ名を除外 ※Distinct() 相当。 override して GetHashCode, Equals 用意しない為。
                                            .Select((info, pri) => new { pri = extUserDataParameterTable.First(k => k.propertyName == info.PropertyName).pri, info }) // ソートを行う為、配列の順を加える
                                            .OrderBy(z => z.pri);

                    // 拡張ユーザ情報に登録するものがあれば、ペイン基本の「拡張ユーザ情報のアニメーションを有効にする」も有効にします。
                    if (extUserDataBlockPropertyInfos.Any() && !pane.extUserDataAnimEnabled)
                    {
                        throw new LayoutConverterException(
                            string.Format(Properties.Resources.ErrorExtUserDataAnimEnabled, combinerFile)
                        );
                    }

                    foreach(var blockProperty in extUserDataBlockPropertyInfos)
                    {
                        ExUserDataInfo parameter = extUserDataParameterTable.First(s => s.propertyName.Equals(blockProperty.info.PropertyName));
                        //object value = GetObjectFromValueText(blockProperty.info.ValueTypeName, blockProperty.info.ValuesText);
                        var exUserDataInfo = extUserDataParameterTable.First(s => s.propertyName.Equals(blockProperty.info.PropertyName));

                        // システム用ユーザーデータと通常ユーザーデータを一つの配列にまとめる。
                        // システム用がある場合は 0 番固定。
                        int combinedCount = pane.userData != null ? pane.userData.Length - 1 : 0;
                        combinedCount += pane.SystemExtData != null ? 1 : 0;

                        if (combinedCount > 0)
                        {
                            object[] combinedUserData = new object[combinedCount];

                            int writeIndex = 0;
                            if (pane.SystemExtData != null)
                            {
                                combinedUserData[writeIndex++] = pane.SystemExtData;
                            }

                            bool basicUserData = true;
                            foreach (var userData in pane.userData)
                            {
                                // "__BasicUserDataString" を書き出さないようにスキップする。
                                // ユーザーデータの 0 番に必ず入っている。
                                if (basicUserData)
                                {
                                    basicUserData = false;
                                }
                                else
                                {
                                    combinedUserData[writeIndex++] = userData;
                                }
                            }

                            // 拡張ユーザー情報の中から、目的のものを得る
                            var param = combinedUserData
                            .Select(obj => new{obj, ExtUserData=extUserDataParameterTable.Where(t => {
                                string nameStr = null;
                                string valStr = null;
                                UserDataSubWriter.ExUserDataType type;
                                RlytSubWriter.ReadUserDataElement_(obj, out nameStr, out valStr, out type);
                                return t.name == nameStr;
                            }) })
                            .Where(s => s.ExtUserData.Count() != 0 && s.ExtUserData.Any(t => t.Equals(exUserDataInfo))); // フィルタ

                            if (param.Count() == 0)
                            {
                                throw new LayoutConverterException(
                                    string.Format(Properties.Resources.ErrorExtUserData_Unset, combinerFile, exUserDataInfo.text, exUserDataInfo.name, exUserDataInfo.kind.ToString())
                                );
                            }

                            // 誤った型で利用されていなかの検査
                            Lyt.UserDataElementType kind;
                            RlytSubWriter.ReadUserDataKind(param.First().obj, out kind);
                            if (kind != param.First().ExtUserData.First().kind)
                            {
                                throw new LayoutConverterException(
                                    string.Format(Properties.Resources.ErrorExtUserDataKind_IncorrectKind,
                                        combinerFile,
                                        exUserDataInfo.name,
                                        exUserDataInfo.kind.ToString(),
                                        kind.ToString())
                                );
                            }
                        }
                        else
                        {
                            // 目的の型が拡張ユーザー情報に見つからない
                            throw new LayoutConverterException(
                                string.Format(Properties.Resources.ErrorExtUserData_Unset, combinerFile, exUserDataInfo.text, exUserDataInfo.name, exUserDataInfo.kind.ToString())
                            );
                        }
                    }
                }
            }
        }
    }
}
