﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "stdafx.h"

#include <vector>
#include <cstring>  // for memset()
#include <revolution/types.h>
#include <S3_intrf.h>

#include "S3TCConverter.h"

using namespace System;
using namespace System::Diagnostics;

namespace NW4R
{
namespace LayoutBinaryConverter
{
namespace TexConv
{

/*>*******************************(*)*******************************<*/
// convert an rgb and alpha layer to a cmp layer using s3.lib.

// note: the only cmp alpha we support is 1 bit color key- alphas
//       should be either 0 or 255 in the alpha layer
/*>*******************************(*)*******************************<*/
Layer^
S3TCConverter::ConvertToS3(Layer^ lyColor, Layer^ lyAlpha)
{
    // threshold alpha
    const s32 alphaRef   = 1;

    // quantization weights for compression.
    // note: these are the default weight from S3TC.doc
    const float redWgt   = 0.3086f;
    const float greenWgt = 0.6094f;
    const float blueWgt  = 0.0820f;

    // encode with 1-bit alpha
    const u32 encodeFlag = S3TC_ENCODE_RGB_ALPHA_COMPARE;



    //-------------------------------------------------------

    // combine the color and alpha layers into an rgba s3 texture

    std::vector<S3_COLOR> surfaceBuf(lyColor->Width * lyColor->Height);

    // combine both layers into the pSurface buffer
    {
        std::vector<S3_COLOR>::iterator pClr = surfaceBuf.begin();
        for (s32 row = 0; row < lyColor->Height; ++row)
        {
            for (s32 col = 0; col < lyColor->Width; ++col, ++pClr)
            {
                u16 r, a;
                u8  g, b;
                lyColor->GetValue( col, row, r, g, b );

                a = 255;  // if no alpha layer, set a to max.
                if (lyAlpha)
                {
                    u8 dmyG, dmyB;
                    lyAlpha->GetValue( col, row, a, dmyG, dmyB);
                }

                // pack as S3_COLORs
                pClr->cRed   = char(r);
                pClr->cGreen = char(g);
                pClr->cBlue  = char(b);
                pClr->cAlpha = char(a);
            }
        }
    }

    //-------------------------------------------------------

    S3_TEXTURE s3RGB, s3CMP;  // input, output s3 textures

    // zero out structures, then set only the fields we need
    std::memset(&s3RGB, 0, sizeof(s3RGB));
    std::memset(&s3CMP, 0, sizeof(s3CMP));

    s3RGB.lWidth   = lyColor->Width;
    s3RGB.lHeight  = lyColor->Height;
    s3RGB.lPitch   = lyColor->Width * sizeof(S3_COLOR);

    s3RGB.pSurface = &surfaceBuf[0];

    // s3RGB.ColorKey;      // color key is unused

    s3RGB.PixelFormat.nFlags        =  S3_TF_HASALPHA;
    s3RGB.PixelFormat.nARGBBitCount =  32;

    // pixel masks match rgba packing order of pSurface buffer
    // pSurface color is array of u8[4] = {r,g,b,a}. Masks are little-endian.
    s3RGB.PixelFormat.nRedMask      =  0x000000FF;
    s3RGB.PixelFormat.nGreenMask    =  0x0000FF00;
    s3RGB.PixelFormat.nBlueMask     =  0x00FF0000;
    s3RGB.PixelFormat.nAlphaMask    =  0xFF000000;

    // s3RGB.pPalette;      //palette is unused

    // create a cmp texture from the rgb texture
    // and a cmp color layer from the cmp texture.
    // S3TCencode() can compress directly to the layer buffer.

    // set color weights
    S3TC_SetColorWeighting( redWgt, greenWgt, blueWgt );

    // set threshold for alpha compare
    S3TC_SetAlphaReference( alphaRef );

    // get the required buffer size.
    // encodeFlag signals 1-bit alpha encoding

    // set cmp layer attributes
    Layer^ lyCmp = gcnew Layer(
        LayerType::COLOR_CMP,
        lyColor->Width,
        lyColor->Height,

        // note: set 'data' explicitly ( not with lySetBuffer )
        //       to use the size returned by S3TCgetEncodeSize
        S3TC_GetEncodeSize(s3RGB.lWidth, s3RGB.lHeight, encodeFlag)
    );

    // compress the texture
    {
        pin_ptr<u8> pDestBuf = &lyCmp->Data[0];
        s32 ok = S3TC_Encode( &s3RGB, &s3CMP, pDestBuf, encodeFlag, NULL, NULL, NULL );
        Debug::Assert( ok == 0, "error: TCConvertToDDS: s3 encoding failed" );
    }

    return lyCmp;

    //---------------------------------------------------------------------------
}

}   // namespace TexConv
}   // namespace LayoutBinaryConverter
}   // namespace NW4R
