﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "sharcUtil.h"
#include "sharcDebug.h"
#include <algorithm>


namespace sharc {
//------------------------------------------------------------------------------

uint32_t calcHash32( const std::string& str, uint32_t key )
{
    const char* buffer = str.c_str();
    uint32_t ret = 0;
    for( int32_t i = 0; buffer[ i ] != '\0'; ++i )
    {
        ret = ret * key + buffer[ i ];
    }
    return ret;
}

// multNumber の倍数に切り上げ
uint32_t roundUp(uint32_t val, uint32_t multNumber)
{
    uint32_t remainder = val % multNumber;
    if (remainder == 0)
    {
        return val;
    }
    else
    {
        return val - remainder + multNumber;
    }
}

// エンディアンを変換する関数
uint16_t toHostU16(EndianTypes from, uint16_t x)
{
    if (from == EndianTypes::cBig)
    {
        return ( x << 8 ) | ( x >> 8 );
    }
    return x;
}

uint32_t toHostU32(EndianTypes from, uint32_t x)
{
    if (from == EndianTypes::cBig)
    {
        return ( x << 24 ) | ( ( x & 0x0000ff00 ) << 8 ) | ( ( x & 0x00ff0000 ) >> 8 ) | ( x >> 24 );
    }
    return x;
}

uint16_t fromHostU16(EndianTypes to, uint16_t x)
{
    if (to == EndianTypes::cBig)
    {
        return ( x << 8 ) | ( x >> 8 );
    }
    return x;
}

uint32_t fromHostU32(EndianTypes to, uint32_t x)
{
    if (to == EndianTypes::cBig)
    {
        return ( x << 24 ) | ( ( x & 0x0000ff00 ) << 8 ) | ( ( x & 0x00ff0000 ) >> 8 ) | ( x >> 24 );
    }
    return x;
}

// エンディアンマーク(Byte Order Mark)からエンディアンを判定する関数
EndianTypes markToEndian(uint16_t mark)
{
    if (mark == 0xfeff)
    {
        return EndianTypes::cLittle;
    }
    else if (mark == 0xfffe)
    {
        return EndianTypes::cBig;
    }
    NW_ERR("Undefined endian mark(0x%02x 0x%02x).", mark & 0xff, (mark >> 8) & 0xff);
    return EndianTypes::cLittle;
}

// 指定したエンディアンを表わすエンディアンマーク(Byte Order Mark)を返す関数
uint16_t endianToMark(EndianTypes endian)
{
    if (endian == EndianTypes::cBig)
    {
        return 0xfffe;
    }
    return 0xfeff;
}

// 拡張子を含んでいるかを判定
bool checkExistenceExt(const std::string& path)
{
    size_t idx = path.rfind('.');
    return (idx != std::string::npos) && !(path.find('/', idx) != std::string::npos || path.find('\\', idx) != std::string::npos);
}

// ディレクトリ名を取得
bool getDirectoryName(PathString& dst, const std::string& src)
{
    dst.resize(0);
    size_t idx_s = src.rfind( "/" );
    size_t idx_y = src.rfind( "\\" );
    size_t idx = std::max(idx_s + 1, idx_y + 1) - 1;
    if( idx > 0 )
    {
        dst = src.substr(0, idx);
        return true;
    }
    return false;
}

// ファイル名を取得
bool getFileName(PathString& dst, const std::string& src)
{
    dst.resize(0);
    size_t idx_s = src.rfind( "/" );
    size_t idx_y = src.rfind( "\\" );
    size_t idx = std::max(idx_s+1, idx_y+1) - 1;
    dst = src.substr( idx + 1 );
    return true;
}

// ディレクトリを作成
bool makeDirectory(const std::string& path)
{
    // 環境変数を展開する
    static const int cFileNameLen = 512;
    char ex_path[cFileNameLen];
    ExpandEnvironmentStringsA(path.c_str(), ex_path, cFileNameLen);

    return (CreateDirectoryA(ex_path, nullptr) == TRUE);
}

// ファイルが存在するか調べる
bool tryIsExistFile(bool* is_exist, const std::string& path)
{
    // 環境変数を展開する
    static const int cFileNameLen = 512;
    char ex_path[cFileNameLen];
    ExpandEnvironmentStringsA(path.c_str(), ex_path, cFileNameLen);

    WIN32_FIND_DATAA find_data;
    HANDLE h = FindFirstFileA(ex_path, &find_data);
    if(h == INVALID_HANDLE_VALUE)
    {
        *is_exist = false;
        return true; // ここでは見つからなかったらそれでOKなので成功
    }
    *is_exist = ((find_data.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) == 0);
    BOOL success = FindClose(h);
    return success != FALSE;
}

// ディレクトリが存在するか調べる
bool tryIsExistDirectory(bool* is_exist, const std::string& path)
{
    // 環境変数を展開する
    static const int cFileNameLen = 512;
    char ex_path[cFileNameLen];
    ExpandEnvironmentStringsA(path.c_str(), ex_path, cFileNameLen);
    // 末尾が \ か / で終わってたら、FindFirstFileがディレクトリ内を
    // 探しに行ってしまうため、ディレクトリ自身があるかないかのチェックにならない。
    // 消してからチェックする。
    PathString tmp(ex_path);
    if (!tmp.empty() && (tmp.back() == '\\' || tmp.back() == '/'))
    {
        tmp.pop_back();
    }

    WIN32_FIND_DATAA find_data;
    HANDLE h = FindFirstFileA(ex_path, &find_data);
    if(h == INVALID_HANDLE_VALUE)
    {
        *is_exist = false;
        return true; // ここでは見つからなかったらそれでOKなので成功
    }
    *is_exist = ((find_data.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) != 0);
    BOOL success = FindClose(h);
    return success != FALSE;
}

// 親ディレクトリを含めてディレクトリを作成
bool makeDirectoryWithParent(const std::string& path)
{
    // パスそのものがあればもう何もしない
    bool is_exist = false;
    bool success = tryIsExistDirectory(&is_exist, path);
    if( !success )
    {
        return false;
    }
    if( is_exist )
    {
        return true;
    }
    // 親ディレクトリがあるかを調べる
    PathString tmp;
    success = getDirectoryName(tmp, path);
    int del_num = 1;
    while( success )
    {
        is_exist = false;
        success = tryIsExistDirectory(&is_exist, tmp);
        if( !success )
        {
            return false;
        }
        if( is_exist )
        {
            break;
        }
        success = getDirectoryName(tmp, tmp);
        del_num++;
    }
    if( !success )
    {
        tmp.resize(0);
    }
    const char delimiter = '/';
    size_t it1 = 0;
    size_t end = std::string::npos;
    // 数を数えて、削った部分をつなぐための数を求める
    int token_num = 0;
    while( it1 != end )
    {
        token_num++;
        it1 = path.find(delimiter, it1);
        if (it1 != end)
        {
            it1++;
        }
    }

    int skip_num = token_num - del_num;
    // 数えた数まではスキップし、そのあとのものをmakeDirectoryしながらつないでいく
    size_t it2 = 0;
    while( it2 != end )
    {
        size_t next = path.find(delimiter, it2);
        if( skip_num > 0 )
        {
            skip_num--;
        }
        else
        {
            PathString token;
            token = path.substr(it2, next - it2);
            if( tmp != "" )
            {
                tmp.append("/");
            }
            tmp.append(token);
            success = makeDirectory(tmp);
            if( !success )
            {
                return false;
            }
        }
        it2 = next;
        if (it2 != end)
        {
            it2++;
        }
    }
    return true;
}

//------------------------------------------------------------------------------

char GetOpt::next()
{
    // 位置引数を読み込み
    while (mOptind < mArgc && mArgv[mOptind][0] != '-')
    {
        mNonOptionArgs.push_back(mArgv[mOptind]);
        mOptind++;
    }

    // -- 以降の位置引数を読み込み
    if (mOptind < mArgc && std::string(mArgv[mOptind]) == "--")
    {
        mOptind++;
        while (mOptind < mArgc)
        {
            mNonOptionArgs.push_back(mArgv[mOptind]);
            mOptind++;
        }
    }

    if (mOptind >= mArgc)
    {
        mOptopt = END;
        return END;
    }

    char c = mArgv[mOptind][1];

    int endOfOption = 0;

    // ロングネーム
    if (c=='-')
    {
        for (int i = 0; i < mLongOptc; i++)
        {
            auto opt = std::string(mLongOpts[i].pLongName);
            auto len = opt.size();
            if (std::string(mArgv[mOptind]+2).find(opt) == 0 && (mArgv[mOptind][2+len] == '\0' || mArgv[mOptind][2+len] == '='))
            {
                // 先頭の文字のオプション
                c = mLongOpts[i].shortName;
                endOfOption = 2 + len;
                break;
            }
        }
    }
    else
    {
        endOfOption = 2;
    }

    mOptopt = c;

    const char find_str[2] = {c, '\0'};
    size_t find_result = std::string::npos;
    if(c == ':' || (find_result = mOpts.find(find_str)) == std::string::npos) {
        fprintf(stderr, "%s%s%s\n", mArgv[0], ": illegal option : ", mArgv[mOptind]);
        mOptind++;
        return '?';
    }

    if(mOpts.at(find_result + 1) == ':') {
        if(mArgv[mOptind][endOfOption] != '\0') {
            if (mArgv[mOptind][endOfOption] == '=')
            {
                if (mArgv[mOptind][endOfOption+1] != '\0')
                {
                    mOptarg = &mArgv[mOptind][endOfOption+1];
                    mOptind++;
                }
                else
                {
                    fprintf(stderr, "%s%s%s\n", mArgv[0], ": option requires an argument : ", mArgv[mOptind]);
                    mOptind++;
                    return '?';
                }
            }
            else
            {
                mOptarg = &mArgv[mOptind][endOfOption];
                mOptind++;
            }
        } else if(mOptind+1 >= mArgc) {
            fprintf(stderr, "%s%s%s\n", mArgv[0], ": option requires an argument : ", mArgv[mOptind]);
            mOptind++;
            return '?';
        } else {
            mOptarg = mArgv[mOptind+1];
            mOptind +=2;
        }
    } else {

        if(mArgv[mOptind][endOfOption] == '\0') {
            mOptind++;
        }
        else
        {
            fprintf(stderr, "%s%s%s endOfOption=%d code=%d\n", mArgv[0], ": illegal option (invalid termination) : ", mArgv[mOptind], endOfOption, mArgv[mOptind][endOfOption]);
            mOptind++;

            return '?';
        }
        mOptarg = nullptr;
    }
    return c;
}

//------------------------------------------------------------------------------
} // namespace sharc
