﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace FcpxMaker.DataModel
{
    using static FcpxMaker.Type;

    [Serializable]
    public class FontDataModel : DataModelBase
    {
        /// <summary>
        /// フォントファイルの相対パスです。
        /// </summary>
        public string RelativeFontFilePath
        {
            get
            {
                return StringUtility.GetRelativePath(this.AbsoluteFontFilePath, FileIOManager.FcpxOpenFilePath);
            }
        }

        /// <summary>
        /// フォントファイルの絶対パスです。
        /// </summary>
        public string AbsoluteFontFilePath { get; set; } = string.Empty;

        /// <summary>
        /// フォントの種類です。
        /// </summary>
        public FontFileType FontType { get; set; }

        /// <summary>
        /// データの種類です。
        /// </summary>
        override public DataType DataType
        {
            get { return DataType.Font; }
        }

        /// <summary>
        /// データの検証結果です。
        /// </summary>
        override public bool Validate
        {
            get
            {
                // フォントが存在するか
                return !String.IsNullOrEmpty(AbsoluteFontFilePath) && File.Exists(AbsoluteFontFilePath);
            }
        }

        /// <summary>
        /// データモデルを比較します。
        /// </summary>
        override public bool Equals(DataModelBase dataModel)
        {
            FontDataModel dm = dataModel as FontDataModel;
            if (dm == null) return false;

            if ((this.AbsoluteFontFilePath != dm.AbsoluteFontFilePath) ||
                (this.FontType != dm.FontType))
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// データモデルを複製します。
        /// </summary>
        override public DataModelBase Clone()
        {
            FontDataModel newDataModel = new FontDataModel();
            newDataModel.AbsoluteFontFilePath = this.AbsoluteFontFilePath;
            newDataModel.FontType = this.FontType;

            return newDataModel;
        }

        /// <summary>
        /// データモデルをコピーします。
        /// </summary>
        override public bool CopyParam(DataModelBase dataModel, bool copyChild)
        {
            FontDataModel dm = dataModel as FontDataModel;
            if (dm == null) return false;

            // FontTypeが同じ場合のみ貼り付け可能
            if (this.FontType != dm.FontType) return false;

            this.AbsoluteFontFilePath = dm.AbsoluteFontFilePath;

            return true;
        }

        /// <summary>
        /// 表示名です。
        /// </summary>
        override public string ToString()
        {
            return GetDispName();
        }

        private string GetDispName()
        {
            if (String.IsNullOrEmpty(RelativeFontFilePath))
            {
                return DataType.ToString();
            }

            return StringUtility.GetFileName(RelativeFontFilePath);
        }
    }
}
