﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//------------------------------------------------------------------------
// .NET モジュールからデータを扱いやすいように、RGBAデータを新規生成します。


#include "stdafx.h"

#include "NWFontGeneric.h"

#if NW_TOOL_BUILD_CTR

#include <windows.h>

namespace NW4R{
    namespace Font{
        namespace UnManaged{

            bool TextureConvertUtility::InitializeOpenGL(void* deviceContext){ return true; }
            bool TextureConvertUtility::FinalizeOpenGL(){ return true; }
            void* TextureConvertUtility::BuildNewTextureRGBAImage(void* pFontInfo, void* pBaseBlock){ return NULL; }
            bool TextureConvertUtility::IsNeededToSrgbFetchFormat(void* pFontTextureGlyph){ return true; }
        }
    }
}

#else

#include <windows.h>
#include <GL/glew.h>

#include "fontResourceFormat.h"
#include <dolphin/gx/GXEnum.h>
#include <nw4r/misc.h>

#include <cafe/gx2/gx2Enum.h>


#include "ByteOrderUtil.h"

namespace NW4R
{
    namespace Font
    {
        namespace UnManaged
        {
            //------------------------------------------------------------------------
            u32 GetU32FromARBG_(u8 A, u8 R, u8 G, u8 B)
            {
                u32 vA = (u32)A << 24;
                u32 vR = (u32)R << 16;
                u32 vG = (u32)G << 8;
                u32 vB = (u32)B << 0;
                // vA = 0x000000FF;// TEST
                return (u32)(vA | vR | vG | vB);
            }

            static inline void SwapU32_(u32* pLhs, u32* pRhs)
            {
                u32 temp = *pLhs;
                *pLhs = *pRhs;
                *pRhs = temp;
            }

            static inline void AdjustU32PixcelOneChannel_(u8* pDst)
            {
                u8 i = pDst[0];

                pDst[0] = i;
                pDst[1] = i;
                pDst[2] = i;
                pDst[3] = i;
            }

            static inline void AdjustU32PixcelTwoChannel_(u8* pDst)
            {
                u8 i = pDst[0];
                u8 a = pDst[3];

                pDst[0] = i;
                pDst[1] = i;
                pDst[2] = i;
                pDst[3] = a;
            }

            static inline void AdjustU32PixcelBC1_(u8* pDst)
            {
                pDst[0] = pDst[0];
                pDst[1] = pDst[1];
                pDst[2] = pDst[2];
                pDst[3] = pDst[3];
            }

            static inline void AdjustU32PixcelBC3_(u8* pDst)
            {
                u8 r = pDst[0];
                u8 b = pDst[2];

                pDst[0] = b;
                pDst[1] = pDst[1];
                pDst[2] = r;
                pDst[3] = pDst[3];
            }

            static inline void AdjustU32PixcelBGRAToRGBA_(u8* pDst)
            {
                u8 r = pDst[0];
                u8 g = pDst[1];
                u8 b = pDst[2];
                u8 a = pDst[3];

                pDst[0] = b;
                pDst[1] = g;
                pDst[2] = r;
                pDst[3] = a;
            }

            // シェーダで描画対応するようにしたので利用されなくなりました。
            static inline void AdjustPackedA8_(u8* pDst)
            {
                u8 r = pDst[3];
                if (r <= 127)
                {
                    pDst[0] = 0;
                    pDst[1] = 0;
                    pDst[2] = 0;
                    pDst[3] = r << 1;
                }
                else{
                    r = (r - 128) << 1;
                    pDst[0] = r;
                    pDst[1] = r;
                    pDst[2] = r;
                    pDst[3] = 255;
                }
            }


            //------------------------------------------------------------------------
            // sheetFormat をコピーしています。
            // ランタイムライブラリ の font_ResourceFormat.h 内 FontSheetFormat 列挙子と対応しています。
            enum ImageFormat
            {
                BC3_SRGB = 17,
                BC2_SRGB = 16,
                BC1_SRGB = 15,
                RGBA8_SRGB = 14,
                BC7 = 18,
                BC7_SRGB = 19,
                BC5 = 13,
                BC4 = 12,
                BC3 = 11,
                BC2 = 10,
                BC1 = 9,
                A8 = 8, // 順番に注意
                A4 = 7, // 順番に注意
                LA4 = 6,
                LA8 = 5,
                RGB565 = 4,
                RGB5A1 = 3,
                RGBA4 = 2,
                RGB8 = 1,
                RGBA8 = 0,
            };

            //------------------------------------------------------------------------

            static inline GX2SurfaceFormat GetGX2SurfaceFormat_(u16 sheetFormat)
            {
                sheetFormat &= FONT_SHEETFORMAT_TEXFMT_MASK;
                switch (sheetFormat)
                {
                case ImageFormat::RGBA8:    return GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_UNORM;
                case ImageFormat::A4:       return GX2_SURFACE_FORMAT_TC_R8_UNORM;
                case ImageFormat::A8:       return GX2_SURFACE_FORMAT_TC_R8_UNORM;
                case ImageFormat::BC1:      return GX2_SURFACE_FORMAT_T_BC1_UNORM;
                case ImageFormat::BC2:      return GX2_SURFACE_FORMAT_T_BC2_UNORM;
                case ImageFormat::BC3:      return GX2_SURFACE_FORMAT_T_BC3_UNORM;
                case ImageFormat::BC4:      return GX2_SURFACE_FORMAT_T_BC4_UNORM;
                case ImageFormat::BC5:      return GX2_SURFACE_FORMAT_T_BC5_UNORM;
                case ImageFormat::BC1_SRGB:      return GX2_SURFACE_FORMAT_T_BC1_SRGB;
                case ImageFormat::BC2_SRGB:      return GX2_SURFACE_FORMAT_T_BC2_SRGB;
                case ImageFormat::BC3_SRGB:      return GX2_SURFACE_FORMAT_T_BC3_SRGB;
                case ImageFormat::RGBA8_SRGB:      return GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_SRGB;
                default: NW4R_ASSERTMSG(false, "The unknown pixel format is found."); return GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_UNORM;
                }
            }

            static inline const wchar_t* GetNXSurfaceFormatString(u16 sheetFormat)
            {
                sheetFormat &= FONT_SHEETFORMAT_TEXFMT_MASK;
                switch (sheetFormat)
                {
                case ImageFormat::RGBA8:    return L"unorm_8_8_8_8";
                case ImageFormat::A4:       return L"unorm_8";
                case ImageFormat::A8:       return L"unorm_8";
                case ImageFormat::BC1:      return L"unorm_bc1";
                case ImageFormat::BC2:      return L"unorm_bc2";
                case ImageFormat::BC3:      return L"unorm_bc3";
                case ImageFormat::BC4:      return L"unorm_bc4";
                case ImageFormat::BC5:      return L"unorm_bc5";
                case ImageFormat::BC1_SRGB:      return L"srgb_bc1";
                case ImageFormat::BC2_SRGB:      return L"srgb_bc2";
                case ImageFormat::BC3_SRGB:      return L"srgb_bc3";
                case ImageFormat::RGBA8_SRGB:      return L"srgb_8_8_8_8";
                case ImageFormat::BC7:           return L"unorm_bc7";
                case ImageFormat::BC7_SRGB:      return L"srgb_bc7";
                default: NW4R_ASSERTMSG(false, "The unknown pixel format is found."); return L"unorm_8_8_8_8";
                }
            }

            static inline bool IsBNTXData(u8* pSrc)
            {
                return (pSrc[0] == 'B' && pSrc[1] == 'N' && pSrc[2] == 'T' && pSrc[3] == 'X');
            }

            //------------------------------------------------------------------------

            static void GetImagePlatform( u8* pSrc, char* pPlatform )
            {
                if(IsBNTXData(pSrc))
                {
                    // C# ツールから、nn ライブラリを利用できない制限があるため自前でフォーマットを解釈しています。
                    u8* pBase = pSrc;
                    // バイナリフォーマットに従って生画像を得る
                    // http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=92312258
                    // 共通バイナリヘッダ
                    const uintptr_t offsetToPlatformDataPtr = 32;
                    pSrc = pSrc + offsetToPlatformDataPtr;

                    for (int i = 0; i < 4; i++)
                    {
                        pPlatform[i] = pSrc[i];
                    }
                    pPlatform[4] = 0;
                }
                else
                {
                    pPlatform[0] = 'G';
                    pPlatform[1] = 'e';
                    pPlatform[2] = 'n';
                    pPlatform[3] = ' ';
                    pPlatform[4] = 0;
                }
            }

            //------------------------------------------------------------------------

            static u8 GetTextureFlags(u8* pSrc)
            {
                if(!IsBNTXData(pSrc))
                {
                    return 0;
                }

                // C# ツールから、nn ライブラリを利用できない制限があるため自前でフォーマットを解釈しています。
                u8* pBase = pSrc;
                // バイナリフォーマットに従って生画像を得る
                // http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=92312258
                // 共通バイナリヘッダ(テクスチャ情報配列へのポインタを得る)
                pSrc = pSrc + 32 + 4 + 4;
                unsigned __int64 offsetToPtrArray = *reinterpret_cast<unsigned __int64*>(pSrc);

                // テクスチャ情報の先頭を得る
                u8* pTextureInfoHeadPtr = pBase + offsetToPtrArray;
                unsigned __int64 offsetToInfo = *reinterpret_cast<unsigned __int64*>(pTextureInfoHeadPtr);

                // nn::gfx::TextureInfoData を読み飛ばす。
                u8* pTextureInfo = pBase + offsetToInfo;
                const uintptr_t offsetToTextureLayout = 16;
                u8* pTexFlags = pTextureInfo + offsetToTextureLayout;
                return *pTexFlags;
            }

            //------------------------------------------------------------------------

            static u8* GetTextureLayout(u8* pSrc)
            {
                u8* pTexLayout = nullptr;
                if(IsBNTXData(pSrc))
                {
                    // C# ツールから、nn ライブラリを利用できない制限があるため自前でフォーマットを解釈しています。
                    u8* pBase = pSrc;
                    // バイナリフォーマットに従って生画像を得る
                    // http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=92312258
                    // 共通バイナリヘッダ(テクスチャ情報配列へのポインタを得る)
                    pSrc = pSrc + 32 + 4 + 4;
                    unsigned __int64 offsetToPtrArray = *reinterpret_cast<unsigned __int64*>(pSrc);

                    // テクスチャ情報の先頭を得る
                    u8* pTextureInfoHeadPtr = pBase + offsetToPtrArray;
                    unsigned __int64 offsetToInfo = *reinterpret_cast<unsigned __int64*>(pTextureInfoHeadPtr);

                    // nn::gfx::TextureInfoData を読み飛ばす。
                    u8* pTextureInfo = pBase + offsetToInfo;
                    const uintptr_t offsetToTextureLayout = 16 + 1 + 1 + 2 + 2 + 2 + 2 + 2 + 4 + 4 + 4 + 4 + 4 + 4;
                    pTexLayout = pTextureInfo + offsetToTextureLayout;
                }
                return pTexLayout;
            }

            //------------------------------------------------------------------------

            static u8* GetRawImagePtr(u8* pSrc)
            {
                if(IsBNTXData(pSrc))
                {
                    // C# ツールから、nn ライブラリを利用できない制限があるため自前でフォーマットを解釈しています。
                    u8* pBase = pSrc;
                    // バイナリフォーマットに従って生画像を得る
                    // http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=92312258
                    // 共通バイナリヘッダ
                    const uintptr_t offsetToImageDataPtr = 32 + 4 + 4 + 8;
                    pSrc = pSrc + offsetToImageDataPtr;

                    unsigned __int64 offset = *reinterpret_cast<unsigned __int64*>(pSrc);

                    // テクスチャ画像の先頭を得る
                    u8* pTextureDateHead = pBase + offset;

                    // 現在、1画像が 1bntx に格納されているので先頭をそのまま返せばよい。
                    // nn::util::BinaryBlockHeader (16) を読み飛ばす。
                    const uintptr_t sizeOfBinaryBlockHeader = 16;
                    return pTextureDateHead + sizeOfBinaryBlockHeader;

                }else{
                    return pSrc;
                }
            }

            //------------------------------------------------------------------------

            static size_t GetRawImageSize(u8* pSrc, size_t sheetSize)
            {
                if(IsBNTXData(pSrc))
                {
                    // C# ツールから、nn ライブラリを利用できない制限があるため自前でフォーマットを解釈しています。
                    u8* pBase = pSrc;
                    // バイナリフォーマットに従って生画像を得る
                    // http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=92312258
                    // 共通バイナリヘッダ(テクスチャ情報配列へのポインタを得る)
                    pSrc = pSrc + 32 + 4 + 4;
                    unsigned __int64 offsetToPtrArray = *reinterpret_cast<unsigned __int64*>(pSrc);

                    // テクスチャ情報の先頭を得る
                    u8* pTextureInfoHeadPtr = pBase + offsetToPtrArray;
                    unsigned __int64 offsetToInfo = *reinterpret_cast<unsigned __int64*>(pTextureInfoHeadPtr);

                    // nn::gfx::TextureInfoData を読み飛ばす。
                    u8* pTextureInfo = pBase + offsetToInfo;
                    const uintptr_t offsetToImageSize = 16 + 1 + 1 + 2 + 2 + 2 + 2 + 2 + 4 + 4 + 4 + 4 + 4 + 4 + 4 + 24;
                    pTextureInfo = pTextureInfo + offsetToImageSize;

                    // 画像サイズを取得します。
                    size_t rawImageSize = *reinterpret_cast<u32*>(pTextureInfo);
                    return rawImageSize;

                }else{
                    return sheetSize;
                }
            }

            std::wstring TextureConvertUtility::PlatformName = std::wstring(L"");
            std::wstring TextureConvertUtility::ToolsPath = std::wstring(L"");

            void TextureConvertUtility::SetPlatform(void* platformName, void* toolsPath)
            {
                PlatformName =	std::wstring(reinterpret_cast<wchar_t*>(platformName));
                ToolsPath =		std::wstring(reinterpret_cast<wchar_t*>(toolsPath));
           }

            //------------------------------------------------------------------------

            bool TextureConvertUtility::InitializeOpenGL(void* deviceContext)
            {
                PIXELFORMATDESCRIPTOR pfd = {
                    sizeof(PIXELFORMATDESCRIPTOR),   // size of this pfd
                    1,                     // version number
                    PFD_DRAW_TO_WINDOW |   // support window
                    PFD_SUPPORT_OPENGL |   // support OpenGL ,
                    PFD_TYPE_RGBA |         // RGBA type
                    PFD_SUPPORT_COMPOSITION,
                    24,                    // 24-bit color depth
                    0, 0, 0, 0, 0, 0,      // color bits ignored
                    8,                     // no alpha buffer
                    0,                     // shift bit ignored
                    0,                     // no accumulation buffer
                    0, 0, 0, 0,            // accum bits ignored
                    24,                    // 32-bit z-buffer
                    8,                     // no stencil buffer
                    0,                     // no auxiliary buffer
                    PFD_MAIN_PLANE,        // main layer
                    0,                     // reserved
                    0, 0, 0                // layer masks ignored
                };

                GLenum error;
                BOOL result;

                HDC hdc = (HDC)deviceContext;

                int pixelFormat = ChoosePixelFormat(hdc, &pfd);
                result = SetPixelFormat(hdc, pixelFormat, &pfd);
                result = wglMakeCurrent(0, 0);

                HGLRC hcnt = wglCreateContext(hdc);
                result = wglMakeCurrent(hdc, hcnt);
                error = glewInit();

                if (error != GLEW_OK)
                {
                    return false;
                }

                // OpenGL 3.3 以上をサポートしているかを判定します。
                return (GLEW_VERSION_3_3) == GL_TRUE;
            }

            //------------------------------------------------------------------------

            bool TextureConvertUtility::FinalizeOpenGL()
            {
                HGLRC mainContext = wglGetCurrentContext();
                wglMakeCurrent(0, 0);
                wglDeleteContext(mainContext);

                return true;
            }


            //------------------------------------------------------------------------
            // SRGB フェッチが必要なフォーマットか
            bool  TextureConvertUtility::IsNeededToSrgbFetchFormat(void* pData)
            {
                FontTextureGlyph* pGlyph = reinterpret_cast<FontTextureGlyph*>(pData);

                return IsNeededToSrgbFetchFormat(pGlyph->sheetFormat);
            }

            bool TextureConvertUtility::IsNeededToSrgbFetchFormat(u16 sheetFormat)
            {
                switch (sheetFormat & FONT_SHEETFORMAT_TEXFMT_MASK)
                {
                case ImageFormat::BC1_SRGB:
                case ImageFormat::BC2_SRGB:
                case ImageFormat::BC3_SRGB:
                case ImageFormat::RGBA8_SRGB:
                case ImageFormat::BC7_SRGB:
                    return true;
                }
                return false;
            }

            //------------------------------------------------------------------------
            // .NET モジュールからデータを扱いやすいように、RGBAデータを新規生成します。
            void* TextureConvertUtility::BuildNewTextureRGBAImage(void* pData, void* pBaseBlock)
            {
                FontInformation* pFontInfo = (FontInformation*)pData;
                char* pOffsetBase = (char*)pBaseBlock;
                FontTextureGlyph* pGlyph = reinterpret_cast<FontTextureGlyph*>(pOffsetBase + pFontInfo->pGlyph);
                // sheet １つ分のサイズ
                int szOneARBGSheet = pGlyph->sheetWidth * pGlyph->sheetHeight * sizeof(u32);
                int szARBGSheet = szOneARBGSheet * pGlyph->sheetNum;
                u8* pRGBASheetImage = new u8[szARBGSheet];

                GLint fmt = GL_RGBA; // TODO : 変換テーブルを作る。
                GLint type = GL_ALPHA; // TODO : 変換テーブルを作る。
                std::wstring fmtStr;
                bool isCompress = false;
                const u16 sheetFormat = pGlyph->sheetFormat & FONT_SHEETFORMAT_TEXFMT_MASK;
                switch (sheetFormat)
                {
                case ImageFormat::A4:       fmt = GL_ALPHA;             type = GL_UNSIGNED_BYTE; break;
                case ImageFormat::A8:       fmt = GL_ALPHA;             type = GL_UNSIGNED_BYTE; break;
                case ImageFormat::LA4:      fmt = GL_LUMINANCE_ALPHA;   type = GL_UNSIGNED_BYTE; break;
                case ImageFormat::LA8:      fmt = GL_LUMINANCE_ALPHA;   type = GL_UNSIGNED_BYTE; break;
                case ImageFormat::RGBA4:    fmt = GL_RGBA;              type = GL_UNSIGNED_SHORT_4_4_4_4; break;
                case ImageFormat::RGB5A1:   fmt = GL_RGBA;              type = GL_UNSIGNED_SHORT_5_5_5_1; break;
                case ImageFormat::RGBA8:    fmt = GL_RGBA;              type = GL_UNSIGNED_BYTE; break;
                case ImageFormat::RGB8:     fmt = GL_RGB;               type = GL_UNSIGNED_BYTE; break;
                case ImageFormat::BC1:      fmt = GL_COMPRESSED_RGB_S3TC_DXT1_EXT;          type = GL_UNSIGNED_BYTE; isCompress = true; break;
                case ImageFormat::BC2:      fmt = GL_COMPRESSED_RGBA_S3TC_DXT3_EXT;         type = GL_UNSIGNED_BYTE; isCompress = true; break;
                case ImageFormat::BC3:      fmt = GL_COMPRESSED_RGBA_S3TC_DXT5_EXT;         type = GL_UNSIGNED_BYTE; isCompress = true; break;
                case ImageFormat::BC4:      fmt = GL_COMPRESSED_LUMINANCE_LATC1_EXT;        type = GL_UNSIGNED_BYTE; isCompress = true; break;
                case ImageFormat::BC5:      fmt = GL_COMPRESSED_LUMINANCE_ALPHA_LATC2_EXT;  type = GL_UNSIGNED_BYTE; isCompress = true; break;
                case ImageFormat::BC1_SRGB: fmt = GL_COMPRESSED_RGB_S3TC_DXT1_EXT;  type = GL_UNSIGNED_BYTE; isCompress = true; break;
                case ImageFormat::BC2_SRGB: fmt = GL_COMPRESSED_RGBA_S3TC_DXT3_EXT;  type = GL_UNSIGNED_BYTE; isCompress = true; break;
                case ImageFormat::BC3_SRGB: fmt = GL_COMPRESSED_RGBA_S3TC_DXT5_EXT;  type = GL_UNSIGNED_BYTE; isCompress = true; break;
                case ImageFormat::RGBA8_SRGB: fmt = GL_RGBA;  type = GL_UNSIGNED_BYTE; break;
                case ImageFormat::BC7: fmt = GL_COMPRESSED_RGBA_BPTC_UNORM_ARB;  type = GL_UNSIGNED_BYTE; isCompress = true; break;
                case ImageFormat::BC7_SRGB: fmt = GL_COMPRESSED_SRGB_ALPHA_BPTC_UNORM_ARB;  type = GL_UNSIGNED_BYTE; isCompress = true; break;
                default: NW4R_ASSERTMSG(false, "The unknown pixel format is found."); break;
                }

                u8* pSrc = reinterpret_cast<u8*>(pOffsetBase + pGlyph->sheetImage);
                u8* pDetiled = nullptr;

                int imageSize = (int)GetRawImageSize(pSrc, pGlyph->sheetSize);

                // プラットフォーム名を取得
                char platform[8];
                GetImagePlatform(pSrc, platform);

                // タイリング処理が行われていれば、タイリングを解除します。
                bool isTiled = ((pGlyph->sheetFormat & FONT_SHEETFORMAT_LINEAR_FLAG) == 0);
                if(isTiled)
                {
#ifdef GENERIC
                    // プラットフォームに応じたデタイル処理
                    if(_stricmp(platform, "Gen ") == 0)
                    {

                        ; // Genericの場合は常にLinearのはず
                    }
                    else if(_stricmp(platform, "NX  ") == 0)
                    {
                        // TODO: 毎回生成しないで使いまわす用に
                        auto textureWriter = gcnew NW4F::LayoutBinaryConverter::GenericTextureWriter();
                        textureWriter->InitializeEncoder(gcnew System::String(ToolsPath.c_str()));

                        System::String^ dimStr = L"2d_array";
                        System::String^ fmtStr = gcnew System::String(GetNXSurfaceFormatString(sheetFormat));
                        System::String^ platformStr = L"NX";
                        System::UInt64 dstDataSize;
                        auto textureFlags = GetTextureFlags(pSrc);
                        auto isSparseTiled = ((textureFlags & 0x04) != 0);
                        auto pTextureLayout = GetTextureLayout(pSrc);

                        System::IntPtr tex = textureWriter->ConvertTiling(
                            false,
                            (System::IntPtr)GetRawImagePtr(pSrc),
                            (long long)GetRawImageSize(pSrc, pGlyph->sheetSize),
                            dimStr,
                            fmtStr,
                            (int)pGlyph->sheetWidth,
                            (int)pGlyph->sheetHeight,
                            1,
                            (int)pGlyph->sheetNum,
                            1,
                            platformStr,
                            dstDataSize,
                            (System::IntPtr)pTextureLayout,
                            isSparseTiled);

                        pDetiled = reinterpret_cast<u8*>((void*)tex);

                        // タイリング解除に失敗した場合、そのままソース画像を使います。
                        if(pDetiled != NULL)
                        {
                            pSrc = pDetiled;
                        }else{
                            throw std::string("Can't convert tiling.");
                        }
                    }
                    else if(_stricmp(platform, "Cafe") == 0)
                    {
                        // 何もしません。
                    }
#else
                    NW4F::LayoutBinaryConverter::GX2TextureWriter::Detiling(
                        reinterpret_cast<u8*>(pOffsetBase + pGlyph->sheetImage),
                        imageSize,
                        pGlyph->sheetWidth,
                        pGlyph->sheetHeight,
                        pGlyph->sheetNum,
                        GX2_SURFACE_DIM_2D_ARRAY,
                        GetGX2SurfaceFormat_(sheetFormat),
                        GX2_TILE_MODE_2D_TILED_THIN1);
#endif
                }

                // タイリング解除後の sheet １つ分のサイズ
                int szOneSrcSheet = (int)(imageSize / pGlyph->sheetNum);

                u8* pDst = pRGBASheetImage;

                // sheetNum の分だけすべての sheet について
                int sheetNum = 0;
                while (sheetNum < pGlyph->sheetNum)
                {
                    // 生画像データを取得します。
                    pSrc = GetRawImagePtr(pSrc);

                    if (sheetFormat == ImageFormat::RGBA8)
                    {
                        // szOneSrcSheet == szOneARBGSheet なはず
                        memcpy(pDst, pSrc, szOneARBGSheet);
                    }
                    else{
                        GLuint tempHandle = 0;
                        glGenTextures(1, &tempHandle);
                        glBindTexture(GL_TEXTURE_2D, tempHandle);

                        if (isCompress)
                        {
                            glCompressedTexImage2D(GL_TEXTURE_2D, 0, fmt, pGlyph->sheetWidth, pGlyph->sheetHeight, 0, szOneSrcSheet, pSrc);
                            glGetTexImage(GL_TEXTURE_2D, 0, GL_RGBA, GL_UNSIGNED_BYTE, pDst);
                        }
                        else{
                            glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
                            glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, pGlyph->sheetWidth, pGlyph->sheetHeight, 0, fmt, type, pSrc);
                            glGetTexImage(GL_TEXTURE_2D, 0, GL_RGBA, GL_UNSIGNED_BYTE, pDst);
                        }
                        glDeleteTextures(1, &tempHandle);
                    }

                    // 縦のループ
                    for (int y = 0; y < pGlyph->sheetHeight / 2; y++)
                    {
                        int offY = y * pGlyph->sheetWidth;
                        int invOffY = (pGlyph->sheetHeight - y - 1) * pGlyph->sheetWidth;

                        // 横のループ
                        for (int x = 0; x < pGlyph->sheetWidth; x++)
                        {
                            int off = (offY + x) * 4;
                            int invOff = (invOffY + x) * 4;

                            SwapU32_((u32*)&pDst[off + 0], (u32*)&pDst[invOff + 0]);
                            if (sheetFormat == ImageFormat::BC5)
                            {
                                AdjustU32PixcelTwoChannel_(&pDst[off + 0]);
                                AdjustU32PixcelTwoChannel_(&pDst[invOff + 0]);
                            }
                            else if (sheetFormat == ImageFormat::BC4){
                                AdjustU32PixcelOneChannel_(&pDst[off + 0]);
                                AdjustU32PixcelOneChannel_(&pDst[invOff + 0]);
                            }
                            else if (sheetFormat == ImageFormat::BC1 || sheetFormat == ImageFormat::BC1_SRGB){
                                AdjustU32PixcelBC1_(&pDst[off + 0]);
                                AdjustU32PixcelBC1_(&pDst[invOff + 0]);
                            }
                            else if (sheetFormat == ImageFormat::BC3 || sheetFormat == ImageFormat::BC3_SRGB){
                                AdjustU32PixcelBC3_(&pDst[off + 0]);
                                AdjustU32PixcelBC3_(&pDst[invOff + 0]);
                            }
                            else if (sheetFormat == ImageFormat::BC7 || sheetFormat == ImageFormat::BC7_SRGB) {
                                AdjustU32PixcelBGRAToRGBA_(&pDst[off + 0]);
                                AdjustU32PixcelBGRAToRGBA_(&pDst[invOff + 0]);
                            }
                            else if (sheetFormat == ImageFormat::RGBA8 || sheetFormat == ImageFormat::RGBA8_SRGB){
                                AdjustU32PixcelBGRAToRGBA_(&pDst[off + 0]);
                                AdjustU32PixcelBGRAToRGBA_(&pDst[invOff + 0]);
                            }
                        }
                    }
                    // 次の画像(sheet)へ...
                    pSrc += szOneSrcSheet;
                    pDst += szOneARBGSheet;

                    sheetNum++;
                }

                // 新しいデータに置き換えます。
                pGlyph->sheetSize = szOneARBGSheet;
                // 注意：64bit 動作では、値があふれてしまうことが起こりうるので、LayoutEditor は pGlyph->sheetImage を描画時に利用せず、RgbaSheetImageBase を利用する。
                pGlyph->sheetImage = 0;
                // タイリング解除で画像が生成された場合は開放
                if (pDetiled != nullptr)
                {
                    free(pDetiled);
                    pDetiled = nullptr;
                }

                return pRGBASheetImage;
            }

            // プラットフォーム名を pPlatform に設定します。
            bool TextureConvertUtility::GetPlatform(void* pData, void* pBaseBlock, char* pPlatform, size_t platformBufferSize)
            {
                // 実装は BuildNewTextureRGBAImage にあわせています。
                NW4R_ASSERTMSG(false, platformBufferSize < 5);
                if (platformBufferSize >= 5)
                {
                    FontInformation* pFontInfo = (FontInformation*)pData;
                    char* pOffsetBase = (char*)pBaseBlock;
                    FontTextureGlyph* pGlyph = reinterpret_cast<FontTextureGlyph*>(pOffsetBase + pFontInfo->pGlyph);

                    u8* pSrc = reinterpret_cast<u8*>(pOffsetBase + pGlyph->sheetImage);

                    // プラットフォーム名を取得
                    GetImagePlatform(pSrc, pPlatform);
                    return true;
                }
                else
                {
                    return false;
                }
            }

            // end of namespace
        }
    }
}

#endif // NW_TOOL_BUILD_CTR
