﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Drawing;
using System.Drawing.Drawing2D;

namespace LayoutEditor.Utility
{
    /// <summary>
    /// サムネイル生成
    /// </summary>
    public class TextureThumbnailHelper
    {
        static private readonly HatchBrush _bgOpeqBrush;
        static private readonly SolidBrush _bgBrush;

        /// <summary>
        /// 不正なサムネイル画像が検出された旨を通知します。
        /// </summary>
        public static Action<string> OnInvalidThumbnailAction;

        /// <summary>
        ///
        /// </summary>
        static TextureThumbnailHelper()
        {
            // 背景色は決めうちにしている。
            _bgOpeqBrush = new HatchBrush(HatchStyle.LargeCheckerBoard, Color.White, Color.LightGray);
            _bgBrush = new SolidBrush(SystemColors.ControlLight);
        }

        /// <summary>
        /// 例外を通知する。
        /// </summary>
        static private void NotifyException_(Exception ex)
        {
            if (OnInvalidThumbnailAction != null)
            {
                string msg =
                    "Message : " + Environment.NewLine +
                    ex.Message + Environment.NewLine +
                    "StackTrace : " + Environment.NewLine +
                    ex.StackTrace;

                OnInvalidThumbnailAction(msg);
            }
        }

        /// <summary>
        /// Image が有効か判定する。
        /// </summary>
        static public bool CheckImageValid(Image img)
        {
            if (img == null) { return false; }

            try
            {
                int w = img.Width; // 幅が正しく取得できるかどうか？
                return true;
            }
            catch (System.Exception ex)
            {
                NotifyException_(ex);

                // 登録ハンドラで外部に通知する。
                return false;
            }
        }

        /// <summary>
        /// 矩形サイズを得る
        /// </summary>
        static public Rectangle GetRect(int srcBitMapW, int srcBitMapH, int outBoundW, int outBoundH)
        {
            float scaleW = 1.0F;
            float scaleH = 1.0F;

            if (srcBitMapW < srcBitMapH)
            {
                scaleW = (float)srcBitMapW / (float)srcBitMapH;
                scaleH = 1.0F;
            }
            else
            {
                if (srcBitMapW > srcBitMapH)
                {
                    scaleW = 1.0F;
                    scaleH = (float)srcBitMapH / (float)srcBitMapW;
                }
            }
            int dstW = (int)((float)outBoundW * scaleW);
            int dstH = (int)((float)outBoundH * scaleH);
            int dstX = (outBoundW - dstW) / 2;
            int dstY = (outBoundH - dstH) / 2;

            return new Rectangle(dstX, dstY, dstW, dstH);
        }

        /// <summary>
        /// 矩形の調整
        /// </summary>
        static public Rectangle AdjustRect(Rectangle rect, bool hasBorder)
        {
            // ちょっと小さめに書かないと全体が表示されない
            if (rect.Width > 1)
            {
                rect.Width = rect.Width - 1;
            }

            if (rect.Height > 1)
            {
                rect.Height = rect.Height - 1;
            }

            // ボーダー付きの場合は2ピクセル小さくすると丁度よい
            if (hasBorder)
            {
                if (rect.Width > 2)
                {
                    rect.Width = rect.Width - 2;
                }

                if (rect.Height > 2)
                {
                    rect.Height = rect.Height - 2;
                }
            }

            return rect;
        }

        /// <summary>
        /// サムネイルビットマップの作成
        /// </summary>
        static public Bitmap CreateThumbnail(int srcBitMapW, int srcBitMapH, Bitmap srcBitMap, int W, int H)
        {
            return TextureThumbnailHelper.CreateThumbnail(srcBitMapW, srcBitMapH, srcBitMap, W, H, false);
        }

        /// <summary>
        /// サムネイルビットマップの作成
        /// </summary>
        static public Bitmap CreateThumbnail(int srcBitMapW, int srcBitMapH, Bitmap srcBitMap, int W, int H, bool hasBorder)
        {
            try
            {
                // サムネイル画像を作成
                Bitmap thumbnailImage = new Bitmap(W, H);
                using (Graphics gc = Graphics.FromImage(thumbnailImage))
                {
                    DrawThumbnail(gc, srcBitMapW, srcBitMapH, srcBitMap, W, H, 0, 0, hasBorder);
                }

                return thumbnailImage;
            }
            catch(Exception ex)
            {
                // 深刻なエラーが起こったことを通知する。
                NotifyException_(ex);

                // 何かしらの理由で失敗したのでとりあえず仮の画像を返す。
                Bitmap thumbnailImage = new Bitmap(1, 1);
                thumbnailImage.SetPixel(0, 0, Color.Red);

                return thumbnailImage;
            }
        }

        static public void DrawThumbnail(Graphics gc, int srcBitMapW, int srcBitMapH, Bitmap srcBitMap, int W, int H, int offsetX, int offsetY, bool hasBorder)
        {
            Rectangle imgRect = GetRect(srcBitMapW, srcBitMapH, W, H);
            imgRect = AdjustRect(imgRect, hasBorder);
            imgRect.X += offsetX;
            imgRect.Y += offsetY;

            gc.InterpolationMode = InterpolationMode.NearestNeighbor;
            gc.PixelOffsetMode = PixelOffsetMode.Half;

            gc.FillRectangle(_bgBrush, offsetX, offsetY, W, H);

            gc.PixelOffsetMode = PixelOffsetMode.Half;
            gc.FillRectangle(_bgOpeqBrush, imgRect);
            gc.DrawImage(srcBitMap,
                         imgRect,
                         new Rectangle(0, 0, srcBitMapW, srcBitMapH),
                         GraphicsUnit.Pixel);

            gc.PixelOffsetMode = PixelOffsetMode.None;
            gc.DrawRectangle(Pens.DarkGray, imgRect);
        }
    }
}
