﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.ComponentModel;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Imaging;
using System.Drawing.Text;
using System.Windows.Forms;

namespace LayoutEditor.Utility
{
    using LECore.Util;
    /// <summary>
    /// カスタムメニュークラス。
    /// </summary>
    public class CustomMenu
    {
        // 画像リスト
        private ImageList _imageList;
        private Hashtable _imageIdxTable = new Hashtable();

        // 描画ハンドラ
        private DrawItemEventHandler    _handlerDrawItem;
        private MeasureItemEventHandler _handlerMeasureItem;
        private DrawItemEventHandler    _handlerDrawTopItem;
        private MeasureItemEventHandler _handlerMeasureTopItem;

        // 描画に使用する定数
        private const int cItemLeftMargin     = 22;	// 項目左側余白
        private const int cItemTextMargin     = 8;	// 項目テキスト左余白
        private const int cItemShortcutMargin = 10;	// 項目ショートカット左余白
        private const int cItemRightMargin    = 20;	// 項目右側余白
        private const int cItemBasicHeight    = 22;	// 項目既定高さ
        private const int cItemTextBlank      = 7;	// 項目テキスト上下余白
        private const int cSeparatorHeight    = 5;	// セパレータ高さ

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public CustomMenu()
        {
            // ハンドラ作成
            _handlerDrawItem       = new DrawItemEventHandler(Event_MenuItem_DrawItem);
            _handlerMeasureItem    = new MeasureItemEventHandler(Event_MenuItem_MeasureItem);
            _handlerDrawTopItem    = new DrawItemEventHandler(Event_TopItem_DrawItem);
            _handlerMeasureTopItem = new MeasureItemEventHandler(Event_TopItem_MeasureItem);
        }

        /// <summary>
        /// 項目画像リスト。
        /// </summary>
        public ImageList ItemImageList
        {
            get { return _imageList;  }
            set { _imageList = value; }
        }

        /// <summary>
        /// メインメニューをセットアップ。
        /// </summary>
        public void SetupMenu(MainMenu menu)
        {
            foreach (MenuItem item in menu.MenuItems)
            {
                // トップレベル項目の設定
                SetupItem(item, _handlerMeasureTopItem, _handlerDrawTopItem);

                // 全サブ項目を設定
                SetupItemCollection(item.MenuItems);
            }
        }

        /// <summary>
        /// コンテキストメニューをセットアップ。
        /// </summary>
        public void SetupMenu(ContextMenu menu)
        {
            // 全サブ項目を設定
            SetupItemCollection(menu.MenuItems);
        }

        /// <summary>
        /// 項目画像番号設定。
        /// </summary>
        public bool SetItemImageIdx(MenuItem item, int imageIdx)
        {
            string key = MakeItemKey(item);

            if (!_imageIdxTable.Contains(key))
            {
                _imageIdxTable.Add(key, imageIdx);
                return true;
            }
            return false;
        }

        /// <summary>
        /// 項目コレクションをセットアップ。
        /// </summary>
        private void SetupItemCollection(Menu.MenuItemCollection menuItems)
        {
            foreach (MenuItem item in menuItems)
            {
                // 自身の設定
                SetupItem(item, _handlerMeasureItem, _handlerDrawItem);

                // サブ項目の設定
                SetupItemCollection(item.MenuItems);
            }
        }

        /// <summary>
        /// 項目をセットアップ。
        /// </summary>
        private void SetupItem(MenuItem item, MeasureItemEventHandler measureItemHandler, DrawItemEventHandler drawItemHandler)
        {
            if (!item.OwnerDraw)
            {
                // UNDONE: 順序を変えるとトップレベル項目のMeasureItemイベントが発生しなくなる...
                item.MeasureItem += measureItemHandler;
                item.DrawItem    += drawItemHandler;
                item.OwnerDraw = true;
            }
        }

        /// <summary>
        /// 画像番号を取得。
        /// </summary>
        private int GetItemImageIdx(MenuItem item)
        {
            string key = MakeItemKey(item);

            if (_imageIdxTable.Contains(key))
            {
                return (int)_imageIdxTable[key];
            }
            return -1;
        }

        /// <summary>
        /// 項目キーを作成。
        /// </summary>
        private string MakeItemKey(MenuItem item)
        {
            // ”テキスト”＋”ニーモニック”＋”ショートカット”をキーにする
            return item.Text + item.Mnemonic.ToString() + item.Shortcut.ToString();
        }

        /// <summary>
        /// 項目テキスト描画。
        /// </summary>
        private void DrawItemText(MenuItem item, DrawItemEventArgs e)
        {
            // セパレータ
            if (item.Text == "-")
            {
                e.Graphics.DrawLine(
                    SystemPens.GrayText,
                    e.Bounds.X + cItemLeftMargin + cItemTextMargin,
                    e.Bounds.Y + cSeparatorHeight / 2,
                    e.Bounds.Width,
                    e.Bounds.Y + cSeparatorHeight / 2
                );
            }
            // メニュー項目
            else
            {
                // ブラシ
                Brush brush;
                if (item.Enabled) { brush = new SolidBrush(SystemColors.MenuText); }
                else              { brush = new SolidBrush(SystemColors.GrayText); }

                // 領域
                Rectangle rect = new Rectangle(
                    cItemLeftMargin + cItemTextMargin,
                    e.Bounds.Y,
                    e.Bounds.Width,
                    e.Bounds.Height
                );

                using (brush)
                using (StringFormat sf = new StringFormat())
                {
                    // フォーマット
                    sf.LineAlignment = StringAlignment.Center;
                    sf.HotkeyPrefix  = HotkeyPrefix.Show;

                    // テキスト描画
                    e.Graphics.DrawString(item.Text, SystemInformation.MenuFont, brush, rect, sf);

                    // ショートカット描画
                    if (item.Shortcut != Shortcut.None && item.ShowShortcut)
                    {
                        DrawShortcutText(item, e, brush, sf);
                    }
                }
            }
        }

        /// <summary>
        /// ショートカットテキスト描画。
        /// </summary>
        private void DrawShortcutText(MenuItem item, DrawItemEventArgs e, Brush brush, StringFormat sf)
        {
            // テキスト
            // 数字キーは"D0"～"D9"に変換される為、直接指定する（...何かいい方法ないの？）
            string text;
            switch (item.Shortcut)
            {
            case Shortcut.Ctrl0:      text = "Ctrl+0";       break;
            case Shortcut.Ctrl1:      text = "Ctrl+1";       break;
            case Shortcut.Ctrl2:      text = "Ctrl+2";       break;
            case Shortcut.Ctrl3:      text = "Ctrl+3";       break;
            case Shortcut.Ctrl4:      text = "Ctrl+4";       break;
            case Shortcut.Ctrl5:      text = "Ctrl+5";       break;
            case Shortcut.Ctrl6:      text = "Ctrl+6";       break;
            case Shortcut.Ctrl7:      text = "Ctrl+7";       break;
            case Shortcut.Ctrl8:      text = "Ctrl+8";       break;
            case Shortcut.Ctrl9:      text = "Ctrl+9";       break;
            case Shortcut.CtrlShift0: text = "Ctrl+Shift+0"; break;
            case Shortcut.CtrlShift1: text = "Ctrl+Shift+1"; break;
            case Shortcut.CtrlShift2: text = "Ctrl+Shift+2"; break;
            case Shortcut.CtrlShift3: text = "Ctrl+Shift+3"; break;
            case Shortcut.CtrlShift4: text = "Ctrl+Shift+4"; break;
            case Shortcut.CtrlShift5: text = "Ctrl+Shift+5"; break;
            case Shortcut.CtrlShift6: text = "Ctrl+Shift+6"; break;
            case Shortcut.CtrlShift7: text = "Ctrl+Shift+7"; break;
            case Shortcut.CtrlShift8: text = "Ctrl+Shift+8"; break;
            case Shortcut.CtrlShift9: text = "Ctrl+Shift+9"; break;
            case Shortcut.Alt0:       text = "Alt+0";        break;
            case Shortcut.Alt1:       text = "Alt+1";        break;
            case Shortcut.Alt2:       text = "Alt+2";        break;
            case Shortcut.Alt3:       text = "Alt+3";        break;
            case Shortcut.Alt4:       text = "Alt+4";        break;
            case Shortcut.Alt5:       text = "Alt+5";        break;
            case Shortcut.Alt6:       text = "Alt+6";        break;
            case Shortcut.Alt7:       text = "Alt+7";        break;
            case Shortcut.Alt8:       text = "Alt+8";        break;
            case Shortcut.Alt9:       text = "Alt+9";        break;
            // スケマティックビューのコンテキストメニュー用
            case Shortcut.AltF1:      text = "+";            break;
            case Shortcut.AltF2:      text = "-";            break;
            case Shortcut.AltF3:      text = "A";            break;
            case Shortcut.AltF4:      text = "F";            break;
            default:
                text = TypeDescriptor.GetConverter(typeof(Keys)).ConvertToString((Keys)item.Shortcut);
                break;
            }

            // サイズ
            SizeF size = e.Graphics.MeasureString(text, SystemInformation.MenuFont);

            // 描画領域
            Rectangle rect = new Rectangle(
                e.Bounds.Right - ((int)size.Width + cItemRightMargin),
                e.Bounds.Y,
                (int)size.Width + cItemRightMargin,
                e.Bounds.Height
            );

            // 描画
            e.Graphics.DrawString(text, SystemInformation.MenuFont, brush, rect, sf);
        }

        /// <summary>
        /// 項目画像描画。
        /// </summary>
        private void DrawItemImage(MenuItem item, DrawItemEventArgs e)
        {
            // 画像番号
            int imageIdx = GetItemImageIdx(item);
            if (imageIdx != -1)
            {
                if (_imageList == null)
                {
                    imageIdx = -1;
                }
                else if (imageIdx >= _imageList.Images.Count)
                {
                    imageIdx = -1;
                }
            }

            // 領域（サイズ固定）
            int       size = 20;
            Rectangle rect = new Rectangle(
                e.Bounds.X + (cItemLeftMargin - size) / 2,
                e.Bounds.Y + (e.Bounds.Height - size) / 2,
                size, size
            );

            // チェック有り
            if (item.Checked)
            {
                // 枠
                DrawImageFrame(item, e, rect);

                // 画像あり
                if (imageIdx != -1)
                {
                    DrawImageData(item, e, rect, imageIdx);
                }
                // チェックマーク
                else
                {
                    DrawCheckMark(item, e, rect);
                }
            }
            // チェック無し
            else
            {
                // 画像あり
                if (imageIdx != -1)
                {
                    DrawImageData(item, e, rect, imageIdx);
                }
            }
        }

        /// <summary>
        /// 画像データ描画。
        /// </summary>
        private void DrawImageData(MenuItem item, DrawItemEventArgs e, Rectangle rect, int imageIdx)
        {
            Image     image     = _imageList.Images[imageIdx];
            Rectangle imageRect = Rectangle.Inflate(rect, -2, -2);

            // 有効状態
            if (item.Enabled)
            {
                // 選択状態
                if ((e.State & DrawItemState.Selected) != 0)
                {
                    // 影用グレースケール化
                    ColorMatrix cm = new ColorMatrix();

                    cm.Matrix00 = 0.0f;
                    cm.Matrix10 = 0.0f;
                    cm.Matrix20 = 0.0f;
                    cm.Matrix30 = 0.0f;
                    cm.Matrix40 = 0.6f;

                    cm.Matrix01 = 0.0f;
                    cm.Matrix11 = 0.0f;
                    cm.Matrix21 = 0.0f;
                    cm.Matrix31 = 0.0f;
                    cm.Matrix41 = 0.6f;

                    cm.Matrix02 = 0.0f;
                    cm.Matrix12 = 0.0f;
                    cm.Matrix22 = 0.0f;
                    cm.Matrix32 = 0.0f;
                    cm.Matrix42 = 0.6f;

                    using (ImageAttributes ia = new ImageAttributes())
                    {
                        // 右下にずらして描画
                        imageRect.Offset(1, 1);
                        ia.SetColorMatrix(cm);
                        e.Graphics.DrawImage(image, imageRect, 0, 0, 16, 16, GraphicsUnit.Pixel, ia);
                    }

                    // 画像を浮かして描画
                    imageRect.Offset(-2, -2);
                    e.Graphics.DrawImage(image, imageRect);
                }
                // 非選択状態
                else
                {
                    e.Graphics.DrawImage(image, imageRect);
                }
            }
            // 無効状態
            else
            {
                // グレースケール化して描画
                ColorMatrix cm = new ColorMatrix();

                cm.Matrix00 = 0.3f;
                cm.Matrix10 = 0.3f;
                cm.Matrix20 = 0.3f;
                cm.Matrix30 = 0.0f;
                cm.Matrix40 = 0.3f * (float)SystemColors.GrayText.R / 255.0f;

                cm.Matrix01 = 0.3f;
                cm.Matrix11 = 0.3f;
                cm.Matrix21 = 0.3f;
                cm.Matrix31 = 0.0f;
                cm.Matrix41 = 0.3f * (float)SystemColors.GrayText.G / 255.0f;

                cm.Matrix02 = 0.3f;
                cm.Matrix12 = 0.3f;
                cm.Matrix22 = 0.3f;
                cm.Matrix32 = 0.0f;
                cm.Matrix42 = 0.3f * (float)SystemColors.GrayText.B / 255.0f;

                using (ImageAttributes ia = new ImageAttributes())
                {
                    ia.SetColorMatrix(cm);
                    e.Graphics.DrawImage(image, imageRect, 0, 0, 16, 16, GraphicsUnit.Pixel, ia);
                }
            }
        }

        /// <summary>
        /// チェックマーク描画。
        /// </summary>
        private void DrawCheckMark(MenuItem item, DrawItemEventArgs e, Rectangle rect)
        {
            // ラジオマーク
            if (item.RadioCheck)
            {
                Brush brush;
                if (item.Enabled) { brush = new SolidBrush(SystemColors.MenuText); }
                else              { brush = new SolidBrush(SystemColors.GrayText); }

                Rectangle markRect = Rectangle.Inflate(rect, -7, -7);
                markRect.Offset(-1, -1);
                markRect.Width  += 1;
                markRect.Height += 1;

                using (brush)
                {
                    e.Graphics.FillEllipse(brush, markRect);
                }
            }
            // チェックマーク
            else
            {
                Pen pen;
                if (item.Enabled) { pen = SystemPens.MenuText; }
                else              { pen = SystemPens.GrayText; }

                e.Graphics.DrawLine(pen, rect.X +  7, rect.Y +  9, rect.X +  7, rect.Y + 10);
                e.Graphics.DrawLine(pen, rect.X +  8, rect.Y + 10, rect.X +  8, rect.Y + 11);
                e.Graphics.DrawLine(pen, rect.X +  9, rect.Y + 11, rect.X +  9, rect.Y + 12);
                e.Graphics.DrawLine(pen, rect.X + 10, rect.Y + 10, rect.X + 10, rect.Y + 11);
                e.Graphics.DrawLine(pen, rect.X + 11, rect.Y +  9, rect.X + 11, rect.Y + 10);
                e.Graphics.DrawLine(pen, rect.X + 12, rect.Y +  8, rect.X + 12, rect.Y +  9);
                e.Graphics.DrawLine(pen, rect.X + 13, rect.Y +  7, rect.X + 13, rect.Y +  8);
            }
        }

        /// <summary>
        /// トップレベル項目テキスト描画。
        /// </summary>
        private void DrawTopItemText(MenuItem item, DrawItemEventArgs e)
        {
            // ブラシ
            Brush brush;
            if (item.Enabled) { brush = new SolidBrush(SystemColors.MenuText); }
            else              { brush = new SolidBrush(SystemColors.GrayText); }

            using (brush)
            using (StringFormat sf = new StringFormat())
            {
                // フォーマット
                sf.Alignment     = StringAlignment.Center;
                sf.LineAlignment = StringAlignment.Center;
                sf.HotkeyPrefix  = HotkeyPrefix.Show;

#if true
                // 描画
                Rectangle rect = e.Bounds;
                if ((e.State & DrawItemState.Selected) != 0)
                {
                    rect.X += 1;
                    rect.Y += 1;
                }
                e.Graphics.DrawString(item.Text, SystemInformation.MenuFont, brush, rect, sf);
#else
                // 描画
                e.Graphics.DrawString(item.Text, SystemInformation.MenuFont, brush, e.Bounds, sf);
#endif
            }
        }

        /// <summary>
        /// 選択枠描画。
        /// </summary>
        private void DrawSelectedFrame(Graphics g, Rectangle rect)
        {
            using (Brush brush = new SolidBrush(ColorBlender.Blend(SystemColors.Window, SystemColors.Highlight, 0.25f)))
            {
                // 背景
                g.FillRectangle(brush, rect);

                // 枠
                g.DrawRectangle(
                    SystemPens.Highlight,
                    rect.X,
                    rect.Y,
                    rect.Width  - 1,
                    rect.Height - 1
                );
            }
        }

        /// <summary>
        /// 画像枠描画。
        /// </summary>
        private void DrawImageFrame(MenuItem item, DrawItemEventArgs e, Rectangle rect)
        {
            Color color;
            Pen   pen;
            if (item.Enabled)
            {
                color = SystemColors.Highlight;
                pen   = SystemPens.Highlight;
            }
            else
            {
                color = SystemColors.GrayText;
                pen   = SystemPens.GrayText;
            }

            Brush brush;
            if (item.Enabled && ((e.State & DrawItemState.Selected) != 0))
            {
                brush = new SolidBrush(ColorBlender.Blend(SystemColors.Window, color, 0.5f));
            }
            else
            {
                brush = new SolidBrush(ColorBlender.Blend(SystemColors.Window, color, 0.25f));
            }

            using (brush)
            {
                // 背景
                e.Graphics.FillRectangle(brush, rect);

                // 枠
                e.Graphics.DrawRectangle(
                    pen,
                    rect.X,
                    rect.Y,
                    rect.Width  - 1,
                    rect.Height - 1
                );
            }
        }

        #region イベントハンドラ
        #region MenuItem_DrawItem
        /// <summary>
        /// イベントハンドラ。
        /// </summary>
        private void Event_MenuItem_DrawItem(object sender, DrawItemEventArgs e)
        {
            MenuItem item = (MenuItem)sender;

            // 有効状態で選択状態
            if (item.Enabled && (e.State & DrawItemState.Selected) != 0)
            {
                DrawSelectedFrame(e.Graphics, e.Bounds);
            }
            // それ以外
            else
            {
                // 背景
                e.Graphics.FillRectangle(SystemBrushes.Window, e.Bounds);

                // 画像領域
                Rectangle gradRect = new Rectangle(
                    e.Bounds.X      + cItemLeftMargin / 3,
                    e.Bounds.Y,
                    cItemLeftMargin - cItemLeftMargin / 3,
                    e.Bounds.Height
                );
                using (LinearGradientBrush gradBrush = new LinearGradientBrush(gradRect, SystemColors.Window, SystemColors.Control, LinearGradientMode.Horizontal))
                {
                    gradBrush.WrapMode = WrapMode.TileFlipX;
                    e.Graphics.FillRectangle(gradBrush, gradRect);
                }

                // キーボード処理時
                if ((e.State & DrawItemState.NoAccelerator) == 0)
                {
                    // ・通常項目
                    // ・無効状態
                    // ・選択状態
                    if (item.Text != "-" && !item.Enabled && (e.State & DrawItemState.Selected) != 0)
                    {
                        // 選択枠
                        e.Graphics.DrawRectangle(
                            SystemPens.Highlight,
                            e.Bounds.X,
                            e.Bounds.Y,
                            e.Bounds.Width  - 1,
                            e.Bounds.Height - 1
                        );
                    }
                }
            }

            // テキスト＆画像
            DrawItemText(item, e);
            DrawItemImage(item, e);
        }
        #endregion

        #region MenuItem_MeasureItem
        /// <summary>
        /// イベントハンドラ。
        /// </summary>
        private void Event_MenuItem_MeasureItem(object sender, MeasureItemEventArgs e)
        {
            MenuItem item = (MenuItem)sender;

            // セパレータ
            if (item.Text == "-")
            {
                e.ItemWidth  = cItemLeftMargin + cItemRightMargin;
                e.ItemHeight = cSeparatorHeight;
            }
            // メニュー項目
            else
            {
                using (StringFormat sf = new StringFormat())
                {
                    // フォーマット
                    sf.Alignment     = StringAlignment.Near;
                    sf.LineAlignment = StringAlignment.Center;
                    sf.HotkeyPrefix  = HotkeyPrefix.Show;

                    // テキストサイズ
                    SizeF textSize = e.Graphics.MeasureString(
                        item.Text,
                        SystemInformation.MenuFont,
                        int.MaxValue,
                        sf
                    );

                    // ショートカット幅
                    int shortcutW = 0;
                    if (item.Shortcut != Shortcut.None && item.ShowShortcut)
                    {
                        SizeF size = e.Graphics.MeasureString(
                            TypeDescriptor.GetConverter(typeof(Keys)).ConvertToString((Keys)item.Shortcut),
                            SystemInformation.MenuFont,
                            int.MaxValue,
                            sf
                        );
                        shortcutW = (int)size.Width;
                    }

                    // 高さ
                    int itemH = (int)textSize.Height + cItemTextBlank;
                    if (itemH < cItemBasicHeight)
                    {
                        itemH = cItemBasicHeight;
                    }

                    // サイズ設定
                    e.ItemWidth  = cItemLeftMargin + (int)textSize.Width + cItemShortcutMargin + shortcutW + cItemRightMargin;
                    e.ItemHeight = itemH;
                }
            }
        }
        #endregion

        #region TopItem_DrawItem
        /// <summary>
        /// イベントハンドラ。
        /// </summary>
        private void Event_TopItem_DrawItem(object sender, DrawItemEventArgs e)
        {
            MenuItem item = (MenuItem)sender;

#if true
            // 背景
            e.Graphics.FillRectangle(SystemBrushes.Control, e.Bounds);

            // 有効状態
            if (item.Enabled)
            {
                // トラッキング状態
                if ((e.State & DrawItemState.HotLight) != 0)
                {
                    GraphicsUtil.Draw3DRectangle(e.Graphics, SystemPens.ControlLightLight, SystemPens.ControlDark, e.Bounds);
                }
                // 選択状態
                else if ((e.State & DrawItemState.Selected) != 0)
                {
                    GraphicsUtil.Draw3DRectangle(e.Graphics, SystemPens.ControlDark, SystemPens.ControlLightLight, e.Bounds);
                }
            }
#else
            // トラッキング状態
            if (item.Enabled && (e.State & DrawItemState.HotLight) != 0)
            {
                DrawSelectedFrame(e.Graphics, e.Bounds);
            }
            // 選択状態
            else if (item.Enabled && (e.State & DrawItemState.Selected) != 0)
            {
                e.Graphics.FillRectangle(SystemBrushes.ControlLight, e.Bounds);
                e.Graphics.DrawRectangle(SystemPens.ControlDark, e.Bounds.X, e.Bounds.Y, e.Bounds.Width - 1, e.Bounds.Height - 1);
            }
            // それ以外
            else
            {
                e.Graphics.FillRectangle(SystemBrushes.Control, e.Bounds);
            }
#endif
            // テキスト
            DrawTopItemText(item, e);
        }
        #endregion

        #region TopItem_MeasureItem
        /// <summary>
        /// イベントハンドラ。
        /// </summary>
        private void Event_TopItem_MeasureItem(object sender, MeasureItemEventArgs e)
        {
            MenuItem item = (MenuItem)sender;

            using (StringFormat sf = new StringFormat())
            {
                // フォーマット
                sf.Alignment     = StringAlignment.Center;
                sf.LineAlignment = StringAlignment.Center;
                sf.HotkeyPrefix  = HotkeyPrefix.Show;

                // テキストサイズ
                SizeF textSize = e.Graphics.MeasureString(
                    item.Text,
                    SystemInformation.MenuFont,
                    int.MaxValue,
                    sf
                );

                // サイズ設定
                e.ItemWidth  = (int)textSize.Width;
                e.ItemHeight = SystemInformation.MenuHeight;
            }
        }
        #endregion
        #endregion
    }
}
