﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System.Diagnostics;
using System.IO;

namespace LayoutEditor.Utility
{
    using LECore;
    using LECore.Structures;

    public static class CombinerEditorHelper
    {
        /// <summary>
        /// コンバイナエディタのプロセスを起動。
        /// </summary>
        /// <param name="filepath">コンバイナエディタのファイルパス</param>
        /// <param name="launchOption">オプション</param>
        private static void StartProcessCombinerEditor(string filePath, string launchOption)
        {
            Process proc = new Process();
            proc.StartInfo.FileName = filePath;
            proc.StartInfo.Arguments = launchOption;
            proc.StartInfo.CreateNoWindow = false;
            proc.StartInfo.UseShellExecute = true;
            proc.Start();
        }

        /// <summary>
        /// コンバイナエディタを起動します。
        /// </summary>
        /// <param name="combinerEditorPath"></param>
        /// <param name="multipleStartup"></param>
        /// <param name="launchOption"></param>
        ///
        /// <remarks>
        /// 多重起動を行わず、プロセスの切替を行う場合はオプションは利用出来ません。
        /// </remarks>
        public static void LaunchCombinerEditor(string combinerEditorPath, bool multipleStartup, string launchOption)
        {
            Debug.Assert(!(multipleStartup == false && !string.IsNullOrEmpty(launchOption)));

            string filePath = combinerEditorPath;
            if (!string.IsNullOrWhiteSpace(filePath))
            {
                // 絶対パスか相対パスか調べます。
                // 先頭の文字が '\'(または '/')か、二文字目が ':' の時に True になります。
                if (!Path.IsPathRooted(combinerEditorPath))
                {
                    filePath = SceneHelper.GetAbsolutePathFromPartsRootBasedPath(LayoutEditorCore.Scene, filePath);
                }
            }

            if (!File.Exists(filePath))
            {
                string title = StringResMgr.Get("LAUNCHCOMBINEREDITOR_ERROR_DLG_TITLE");
                LayoutEditorCore.MsgReporter.OnShowMessageDialogAsync(title, StringResMgr.Get("LAUNCHCOMBINEREDITOR_ERROR_CANTFIND_EXE", filePath));
                return;
            }

            if (!multipleStartup)
            {
                // 既にコンバイナエディタが起動している場合は多重起動は行わなず、プロセスの変更を行う
                Process[] processses = Process.GetProcesses();
                foreach (Process p in processses)
                {
                    const string combinerEditorName = "NintendoWare CombinerEditor";

                    // コンバイナエディタの名称はバージョンや読み込んでいるファイルで変わる為 StartsWith で判定します。
                    if (p.MainWindowTitle.StartsWith(combinerEditorName))
                    {
                        // 外部アプリケーションのウインドウをアクティブにする
                        LECore.Win32.User32.SetForegroundWindow(p.MainWindowHandle);
                        return;
                    }
                }
            }

            StartProcessCombinerEditor(filePath, launchOption);
        }
    }
}
