﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Drawing;
using System.Xml.Serialization;
using System.Linq;
using System.Text;

namespace LayoutEditor.Structures.SerializableObject
{
    using LayoutEditor.Forms.ToolWindows;
    #region ViewManagerSetting
    /// <summary>
    /// ビューマネージャの設定を表すクラス。
    /// XMLシリアライズされて、ビューの状態永続化に利用されます。
    /// </summary>
    [XmlInclude( typeof( LEToolFormSetting ) )]
    public class ViewManagerSetting
    {
        List<LEToolFormSetting> _toolFormSettings = new List<LEToolFormSetting>();// LEToolFormSetting

        Size	_appFormSize;
        Point	_appFormLocation;
        bool	_snapToolWindows;
        List<string> _windowZOrder = new List<string>();

        #region プロパティ
        public List<LEToolFormSetting> ToolFormSettings
        {
            get { return _toolFormSettings; }
            set { _toolFormSettings = value; }
        }

        public Size AppFormSize
        {
            get { return _appFormSize; }
            set { _appFormSize = value; }
        }

        public Point AppFormLocation
        {
            get { return _appFormLocation; }
            set { _appFormLocation = value; }
        }

        public bool SnapToolWindows
        {
            get { return _snapToolWindows; }
            set { _snapToolWindows = value; }
        }

        public bool HideToolbar
        {
            get; set;
        }

        public bool AllowEndUserDocking { get; set; } = true;

        public bool Mdi { get; set; }

        /// <summary>
        /// 描画順
        /// </summary>
        public List<string> WindowZOrder
        {
            get { return _windowZOrder; }
            set { _windowZOrder = value; }
        }

        [XmlIgnore]
        public const string DockPanelModeLayout = "Layout";

        [XmlIgnore]
        public const string DockPanelModeAnimation = "Animation";

        [XmlIgnore]
        public const string DockPanelModelParts = "Parts";

        public string WorkspaceName { get; set; } = DockPanelModeLayout;

        public List<DockPanelSetting> DockPanelSettings { get; set; } = new List<DockPanelSetting>();

        public void SetDockPanelSetting(string mode, string xml)
        {
            var setting = DockPanelSettings.FirstOrDefault(x => x.id == mode);
            if (setting == null)
            {
                setting = new DockPanelSetting() { id = mode };
                DockPanelSettings.Add(setting);
            }

            setting.DockPanel = xml;
        }

        public class DockPanelSetting
        {
            [XmlAttribute]
            public string id { get; set; }

            [XmlAttribute]
            public string label { get; set; }

            [XmlIgnore]
            public string DockPanel
            {
                get
                {
                    return XmlElementToString(DockPanelXmlElement);
                }
                set
                {
                    DockPanelXmlElement = StringToXmlElement(value);
                }
            }

            [XmlAnyElement]
            public System.Xml.XmlElement DockPanelXmlElement;

            private System.Xml.XmlElement StringToXmlElement(string content)
            {
                var document = new System.Xml.XmlDocument();
                document.LoadXml(content);
                return document.DocumentElement;
            }

            private string XmlElementToString(System.Xml.XmlElement element)
            {
                if (element == null)
                {
                    return string.Empty;
                }

                return element.OuterXml;
            }
        }

        #endregion プロパティ

        #region コンストラクタ

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public ViewManagerSetting()
        {

        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public ViewManagerSetting( ViewManagerSetting src )
        {
            _appFormSize = src.AppFormSize;
            _appFormLocation = src.AppFormLocation;
            _snapToolWindows = src.SnapToolWindows;
            HideToolbar = src.HideToolbar;
            _windowZOrder = src.WindowZOrder;
            WorkspaceName = src.WorkspaceName;
            DockPanelSettings = src.DockPanelSettings.Select(x => new DockPanelSetting() { id = x.id, label = x.label, DockPanel = x.DockPanel }).ToList();
        }

        #endregion コンストラクタ
    }


    #endregion ViewManagerSetting

}
