﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.Windows;


namespace LayoutEditor.Forms.ToolWindows.common
{
    using LECore.Structures;
    using LECore.Manipulator;
    using LECore.Util;

    /// <summary>
    ///
    /// </summary>
    public class PaneTreeStatePanel : Panel
    {
        /// <summary>
        /// 対象となるTreeViewに使用されるクラス
        /// </summary>
        public class PaneHierarchyTreeView : TreeView
        {
            public event EventHandler VScroll;

            protected override void WndProc( ref Message m )
            {
                if( m.Msg == LECore.Win32.WM.WM_VSCROLL )
                {
                    if( VScroll != null )
                    {
                        VScroll( this, null );
                    }
                }
                base.WndProc( ref m );
            }

            /// <summary>
            /// コンストラクタです。
            /// </summary>
            public PaneHierarchyTreeView()
            {
                this.ShowRootLines = false;
            }
        }

        //----------------------------------------------------------

        PaneHierarchyTreeView _tvwTargetHierarchy;
        bool _isDragging = false;
        bool _updateHidden = false;
        bool _bNewHidden = false;
        bool _updateLocked = false;
        bool _bNewbLocked = false;
        List<KeyValuePair<string, bool>> _beforeState = new List<KeyValuePair<string, bool>>();
        List<KeyValuePair<string, bool>> _afterState = new List<KeyValuePair<string, bool>>();
        string _lastAltClickNode = null;

        //----------------------------------------------------------

        /// <summary>
        ///
        /// </summary>
        public ISubScene _CurrentSubScene
        {
            get { return LECore.LayoutEditorCore.Scene.CurrentISubScene; }
        }

        /// <summary>
        ///
        /// </summary>
        public void Initialize( PaneHierarchyTreeView tvwTargetHierarchy )
        {
            Debug.Assert( _tvwTargetHierarchy == null );
            Debug.Assert( tvwTargetHierarchy != null );

            this.Paint += Event_PnlState_Paint;

            this.MouseDown += Event_PnlState_MouseDown;
            this.MouseUp += Event_PnlState_MouseUp;
            this.MouseLeave += Event_PnlState_MouseLeave;
            this.MouseMove += Event_PnlState_MouseMove;

            _tvwTargetHierarchy = tvwTargetHierarchy;

            _tvwTargetHierarchy.VScroll += Event_TvwHierarchy_VScroll;
            _tvwTargetHierarchy.MouseWheel += Event_TvwTargetHierarchy_MouseWheel;

            _tvwTargetHierarchy.AfterCollapse += Event_TvwHierarchy_AfterCollapse;
            _tvwTargetHierarchy.AfterExpand += Event_TvwHierarchy_AfterExpand;
            _tvwTargetHierarchy.AfterSelect += Event_TvwHierarchy_AfterSelect;
        }

        /// <summary>
        ///
        /// </summary>
        public PaneTreeStatePanel()
        {
            this.DoubleBuffered = true;
        }

        /// <summary>
        ///
        /// </summary>
        void CheckNodeFlag_( TreeNode node, out bool bHidden, out bool bLocked, out bool bReadOnlyLocked )
        {
            bHidden = true;
            bLocked = false;
            bReadOnlyLocked = false;

            if( node.Tag is IPane )
            {
                IPane pane = node.Tag as IPane;

                bHidden = pane.Hidden;
                bLocked = pane.Locked;
                bReadOnlyLocked = pane.IsReadOnlyLocked;
            }
            else
            {
                foreach( TreeNode childNode in node.Nodes )
                {
                    if( childNode.Tag is IPane )
                    {
                        IPane childPane = childNode.Tag as IPane;

                        bHidden &= childPane.Hidden;
                        bLocked |= childPane.Locked;
                        bReadOnlyLocked |= childPane.IsReadOnlyLocked;
                    }
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void SetNodePaneState_( TreeNode node, PaneManipulator paneMnp,
            bool updateHidden, bool hiddenFlag,
            bool updateLocked , bool lockedFlag)
        {
            if( node.Tag is IPane )
            {
                paneMnp.BindTarget( node.Tag as IPane );

                // Hidden編集
                if( updateHidden && paneMnp.IPane.Hidden != hiddenFlag )
                {
                    paneMnp.Hidden = hiddenFlag;
                }

                // Lock編集
                if( updateLocked && paneMnp.IPane.Locked != lockedFlag )
                {
                    paneMnp.Locked = lockedFlag;
                }
            }

            foreach( TreeNode childNode in node.Nodes )
            {
                SetNodePaneState_( childNode, paneMnp, updateHidden, hiddenFlag, updateLocked, lockedFlag );
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void Event_PnlState_Paint( object sender, PaintEventArgs e )
        {
            if( _tvwTargetHierarchy == null )
            {
                return;
            }

            Icon eyeIcon = ImageResMgr.GetManifestResourceIcon( "eye_16.ico" );
            Icon lockIcon = ImageResMgr.GetManifestResourceIcon( "lock_16.ico" );
            Bitmap readonlyIcon = ImageResMgr.GetManifestResourcePng("readonly_lock.png");

            TreeNode visibleNode = _tvwTargetHierarchy.TopNode;
            while( visibleNode != null )
            {
                Point pos = visibleNode.TreeView.PointToScreen( visibleNode.Bounds.Location );
                pos = this.PointToClient( pos );

                if( visibleNode.Parent == null )
                {
                    e.Graphics.FillRectangle( SystemBrushes.Control, 0, pos.Y, 34, 16 );
                }
                else
                {
                    bool bHidden;
                    bool bLocked;
                    bool bReadOnlyLocked;

                    CheckNodeFlag_( visibleNode, out bHidden, out bLocked, out bReadOnlyLocked );

                    if( !bHidden )
                    {
                        e.Graphics.DrawIcon( eyeIcon, 0, pos.Y );
                    }

                    if (bReadOnlyLocked)
                    {
                        e.Graphics.DrawImage(readonlyIcon, 0 + eyeIcon.Width, pos.Y, lockIcon.Width, lockIcon.Height);
                    }
                    else if (bLocked)
                    {
                        e.Graphics.DrawIcon(lockIcon, 0 + eyeIcon.Width, pos.Y);
                    }
                }
                visibleNode = visibleNode.NextVisibleNode;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void Event_PnlState_MouseDown(object sender, MouseEventArgs e)
        {
            if (_tvwTargetHierarchy == null)
            {
                return;
            }

            Point pos = _tvwTargetHierarchy.PointToClient(this.PointToScreen(e.Location));
            pos.X = 0;

            TreeNode node = _tvwTargetHierarchy.GetNodeAt(pos);
            if (node != null && node.Parent != null)
            {
                bool bReadOnlyLocked;
                CheckNodeFlag_(node, out _bNewHidden, out _bNewbLocked, out bReadOnlyLocked);

                bool isAltPressed = Control.ModifierKeys == Keys.Alt;
                _isDragging = !isAltPressed; // Alt押下中はドラッグ判定無効

                var hierarchyNodes = GenericUtil.GetNodeRecursively(_tvwTargetHierarchy.Nodes[0], (x) => x.Nodes.OfType<TreeNode>()).Skip(1).ToArray();

                // クリック位置によって、変更するフラグを決定します。
                if (e.Location.X < this.Size.Width / 2)
                {
                    // 表示・非表示の設定

                    // 更新オプション
                    _bNewHidden = !(_bNewHidden | isAltPressed);
                    _updateHidden = true;
                    _updateLocked = false;

                    // Altキーが押されている場合
                    if (isAltPressed)
                    {
                        List<KeyValuePair<string, bool>> currentState = new List<KeyValuePair<string, bool>>();
                        // 現在の状態をDictionary形式で保持します
                        currentState = StoreNodesState_(hierarchyNodes);

                        // 前回の変更後の状態と現在の状態が等しいか？
                        if (_lastAltClickNode == node.FullPath &&
                            _beforeState.Count != 0 &&
                            currentState.SequenceEqual(_afterState))
                        {
                            // 直前の状態を復元します
                            RestoreNodesState_(hierarchyNodes.Zip(_beforeState, (x, y) => new KeyValuePair<TreeNode, bool>(x, y.Value)));

                            // 復元情報をクリア
                            _beforeState.Clear();
                            _afterState.Clear();
                            _lastAltClickNode = null;

                            return;
                        }
                        else
                        {
                            // 変更前の状態を保持します
                            _beforeState = StoreNodesState_(hierarchyNodes);

                            // 一旦すべてのノードを非表示にします
                            ChangeNodesState_(hierarchyNodes, _updateHidden, true, _updateLocked, false);

                            _lastAltClickNode = node.FullPath;
                        }
                    }
                }
                else
                {
                    // ロックの設定
                    _bNewbLocked = !_bNewbLocked;
                    _updateHidden = false;
                    _updateLocked = true;
                }

                // 選択ノード以下のステート更新
                ChangeNodeState_(node);

                // 変更後の状態を保持します
                if (_updateHidden)
                {
                    _afterState = StoreNodesState_(hierarchyNodes);
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void Event_PnlState_MouseUp(object sender, MouseEventArgs e)
        {
            _isDragging = false;
        }

        /// <summary>
        ///
        /// </summary>
        private void Event_PnlState_MouseLeave(object sender, EventArgs e)
        {
            _isDragging = false;
        }

        /// <summary>
        ///
        /// </summary>
        private void Event_PnlState_MouseMove(object sender, MouseEventArgs e)
        {
            if (_tvwTargetHierarchy == null || !_isDragging)
            {
                return;
            }

            Point pos = _tvwTargetHierarchy.PointToClient(this.PointToScreen(e.Location));
            pos.X = 0;

            TreeNode node = _tvwTargetHierarchy.GetNodeAt(pos);
            if (node != null && node.Parent != null)
            {
                ChangeNodeState_(node);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void ChangeNodeState_(TreeNode node)
        {
            _CurrentSubScene.BeginMassiveModify();

            SetNodePaneState_(node, new PaneManipulator(), _updateHidden, _bNewHidden, _updateLocked, _bNewbLocked);

            _CurrentSubScene.EndMassiveModify();
        }

        /// <summary>
        ///
        /// </summary>
        private void ChangeNodesState_(IEnumerable<TreeNode> nodes, bool updateHidden, bool newHidden, bool updateLocked, bool newLocked)
        {
            _CurrentSubScene.BeginMassiveModify();

            foreach (TreeNode node in nodes)
            {
                SetNodePaneState_(node, new PaneManipulator(), updateHidden, newHidden, updateLocked, newLocked);
            }

            _CurrentSubScene.EndMassiveModify();
        }

        /// <summary>
        ///
        /// </summary>
        private List<KeyValuePair<string, bool>> StoreNodesState_(IEnumerable<TreeNode> nodes)
        {
            List<KeyValuePair<string, bool>> storeData = new List<KeyValuePair<string, bool>>();

            foreach (TreeNode node in nodes)
            {
                bool bHidden;
                bool bLocked;
                bool bReadOnlyLocked;
                CheckNodeFlag_(node, out bHidden, out bLocked, out bReadOnlyLocked);

                // 目玉アイコンの表示・非表示だけ保持する
                storeData.Add(new KeyValuePair<string, bool>(node.FullPath, bHidden));
            }

            return storeData;
        }

        /// <summary>
        ///
        /// </summary>
        private void RestoreNodesState_(IEnumerable<KeyValuePair<TreeNode, bool>> restoreData)
        {
            foreach (var item in restoreData)
            {
                // 目玉アイコンの表示・非表示だけ更新する
                SetNodePaneState_(item.Key, new PaneManipulator(), true, item.Value, false, false);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void Event_TvwHierarchy_AfterCollapse( object sender, TreeViewEventArgs e )
        {
            this.Invalidate();
        }

        /// <summary>
        ///
        /// </summary>
        private void Event_TvwHierarchy_AfterExpand( object sender, TreeViewEventArgs e )
        {
            this.Invalidate();
        }

        /// <summary>
        ///
        /// </summary>
        private void Event_TvwHierarchy_AfterSelect( object sender, TreeViewEventArgs e )
        {
            this.Invalidate();
        }

        /// <summary>
        ///
        /// </summary>
        void Event_TvwHierarchy_VScroll( object sender, EventArgs e )
        {
            this.Invalidate();
        }

        /// <summary>
        ///
        /// </summary>
        void Event_TvwTargetHierarchy_MouseWheel( object sender, MouseEventArgs e )
        {
            this.Invalidate();
        }
    }
}
