﻿using LayoutEditor.src.Forms.ToolWindows.StateSettingWindow;
using LECore;
using LECore.Manipulator;
using LECore.Structures;
using LECore.Structures.Core;
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;

namespace LayoutEditor.Forms.ToolWindows.StateWindow
{
    /// <summary>
    ///
    /// </summary>
    public partial class StateSettingWindow : LayoutEditor.Forms.ToolWindows.LEToolWindow
    {
        //----------------------------------------------------------

        AddPropertyDlg _addPropertyDlg = new AddPropertyDlg();
        AddStateDlg _addStateDlg = new AddStateDlg();
        EditTransitionDlg _editTransitionDlg = new EditTransitionDlg();
        EditFeatureParamaterDlg _editFeatureParamaterDlg = new EditFeatureParamaterDlg();

        IState _currentEditState;
        IState _currentEditNextState;

        System.Timers.Timer _timer = new System.Timers.Timer();
        DateTime _lastUpdateTime = DateTime.Now;

        static readonly TimeSpan _FrameSpanFps30X2 = new TimeSpan(10000000 / 15);
        static readonly TimeSpan _FrameSpanFps30 = new TimeSpan(10000000 / 30);
        static readonly TimeSpan _FrameSpanFps60 = new TimeSpan(10000000 / 60);

        //----------------------------------------------------------

        /// <summary>
        ///
        /// </summary>
        void DrawIcon_(Graphics g, string text)
        {
            var font = new Font(this.Font, FontStyle.Bold);
            var size = g.MeasureString(text, font);

            g.FillRectangle(SystemBrushes.ControlLight, 0, 0, 16, 16);
            g.DrawRectangle(SystemPens.InactiveBorder, 0, 0, 15, 15);
            g.DrawString(text, font, SystemBrushes.GrayText, (16 - size.Width) / 2.0f, (16 - size.Height) / 2.0f);
        }

        //----------------------------------------------------------

        void RegisterIcon_(FeatureParamaterKind kind, Icon icon)
        {
            _lvwStateParamaters.SmallImageList.Images.Add(kind.ToString(), icon);
            _lvwStateParamatersStart.SmallImageList.Images.Add(kind.ToString(), icon);
            _lvwStateParamatersEnd.SmallImageList.Images.Add(kind.ToString(), icon);
        }

        //----------------------------------------------------------

        public StateSettingWindow()
        {
            InitializeComponent();

            // 暫定アイコンの作成
            {
                _lvwStateParamaters.SmallImageList = new ImageList();
                _lvwStateParamaters.SmallImageList.ColorDepth = ColorDepth.Depth32Bit;
                _lvwStateParamatersStart.SmallImageList = new ImageList();
                _lvwStateParamatersEnd.SmallImageList = new ImageList();

                {
                    Bitmap bmp = new Bitmap(16, 16);
                    using (Graphics g = Graphics.FromImage(bmp))
                    {
                        DrawIcon_(g, "T");
                        RegisterIcon_(FeatureParamaterKind.Position, Icon.FromHandle(bmp.GetHicon()));
                        DrawIcon_(g, "S");
                        RegisterIcon_(FeatureParamaterKind.Scale, Icon.FromHandle(bmp.GetHicon()));
                        DrawIcon_(g, "R");
                        RegisterIcon_(FeatureParamaterKind.RotateZ, Icon.FromHandle(bmp.GetHicon()));
                        DrawIcon_(g, "W");
                        RegisterIcon_(FeatureParamaterKind.WhiteColor, Icon.FromHandle(bmp.GetHicon()));
                        DrawIcon_(g, "B");
                        RegisterIcon_(FeatureParamaterKind.BlackColor, Icon.FromHandle(bmp.GetHicon()));
                        DrawIcon_(g, "E");
                        RegisterIcon_(FeatureParamaterKind.StateMachineEvent, Icon.FromHandle(bmp.GetHicon()));
                    }
                }
            }

            // 更新タイマー
            _timer.Interval = 5;
            _timer.Elapsed += (s, e) => this.Invoke(new Action(() => { Event_Timer_Tick(s, e); }));

            UpdatePropertyForSelectedLayerEditItem_(null, null);
        }

        //----------------------------------------------------------

        private void StateSettingWindow_Load(object sender, EventArgs e)
        {
            UpdateProperty_();
        }

        //----------------------------------------------------------

        IScene _Scene
        {
            get { return LayoutEditorCore.Scene; }
        }

        //----------------------------------------------------------
        /// <summary>
        /// 対象サブシーンを取得します。
        /// </summary>
        ISubScene _CurrentSubScene
        {
            get { return _Scene?.CurrentISubScene; }
        }

        //----------------------------------------------------------
        public override bool IsNeededToHide
        {
            get
            {
                var setting = ViewManager?.AppSetting?.ProjectSettings;
                return setting != null ? !setting.IsStateMachineEnalbed : true;
            }
        }
        //----------------------------------------------------------

        void UpdatePropertyEnableState_()
        {
            if (_CurrentSubScene == null)
            {
                this.Enabled = false;
                return;
            }

            this.Enabled = true;

            IStateLayer selectedLayer = _CurrentSubScene.IStateMachine.GetSelectedIStateLayer();
            if(selectedLayer != null && _CurrentSubScene.IStateMachine.IsEnabled)
            {
                _pnlStateLayerEditItem.Enabled = true;
                var editItem = selectedLayer.GetSelectedEditItem();
                _pnlPropertyEdit.Enabled = editItem != null;
                _btnStoreCurrentToProperty.Enabled = editItem != null;
            }
            else
            {
                _pnlStateLayerEditItem.Enabled = false;
                _pnlPropertyEdit.Enabled = false;
                _btnStoreCurrentToProperty.Enabled = false;
            }

            _btnEditProperty.Enabled = !_CurrentSubScene.ISelectedSet.Empty;
        }

        //----------------------------------------------------------

        private void SetTransitionUIVisible_(bool visible)
        {
            _pnlTransitionEdit.Visible = visible;
            _lvwStateParamatersEnd.Visible = visible;

            _picStartState.Visible = visible;
            _picEndState.Visible = visible;
        }

        //----------------------------------------------------------

        private void UpdatePropertyForSelectedLayerEditItem_(IStateLayer selectedLayer, IStateLayerEditItem selectedEditItem)
        {
            {
                _btnCreateStateFromSelected.Enabled = selectedEditItem is IState;
                _btnEditTransition.Enabled = selectedEditItem != null;
                _btnDeleteSelected.Enabled = selectedEditItem != null && selectedEditItem.IsDeletable();
            }

            {
                _lvwStateParamaters.Items.Clear();
                _lvwStateParamatersStart.Items.Clear();
                _lvwStateParamatersEnd.Items.Clear();

                _currentEditState = null;
                _currentEditNextState = null;

                if (selectedLayer != null && selectedEditItem != null)
                {
                    if (selectedEditItem is IState)
                    {
                        var state = selectedEditItem as IState;
                        SetParamatersToListView_(_lvwStateParamaters, state);
                        SetParamatersToListView_(_lvwStateParamatersStart, state);

                        _currentEditState = state;

                        SetTransitionUIVisible_(false);
                    }
                    else if (selectedEditItem is IStateTransition)
                    {
                        var transition = selectedEditItem as IStateTransition;

                        var startState = selectedLayer.FindStateByName(transition.StartStateName);
                        _currentEditState = startState;
                        SetParamatersToListView_(_lvwStateParamaters, startState);
                        SetParamatersToListView_(_lvwStateParamatersStart, startState);

                        var endState = selectedLayer.FindStateByName(transition.EndStateName);
                        _currentEditNextState = endState;
                        SetParamatersToListView_(_lvwStateParamatersEnd, endState);

                        SetTransitionUIVisible_(true);

                        _pnlTransitionEdit.Setup(_lvwStateParamaters, _CurrentSubScene.IStateMachine, transition);

                        _pnlTransitionEdit.Invalidate();

                        _ttpMain.SetToolTip(_picStartState, transition.StartStateName);
                        _ttpMain.SetToolTip(_picEndState, transition.EndStateName);
                    }
                    else
                    {
                        Debug.Assert(false);
                    }
                }
                else
                {
                    SetTransitionUIVisible_(false);
                }
            }
        }

        //----------------------------------------------------------
        /// <summary>
        /// データモデルを UI に設定します。
        /// </summary>
        void UpdateProperty_()
        {
            UpdatePropertyEnableState_();

            if (_CurrentSubScene == null)
            {
                _btnSelectTargetLayer.DropDownItems.RemoveByKey("dynamically");
                _lvwStateLayerEditItems.Items.Clear();
                UpdatePropertyForSelectedLayerEditItem_(null, null);

                return;
            }

            IStateLayer selectedLayer = _CurrentSubScene.IStateMachine.GetSelectedIStateLayer();

            _btnEnableState.Checked = _CurrentSubScene.IStateMachine.IsEnabled;

            // レイヤーリストを更新、選択復元
            {
                _btnSelectTargetLayer.DropDownItems.RemoveByKey("dynamically");

                _btnSelectTargetLayer.Text = selectedLayer != null ? selectedLayer.Name : string.Empty;

                foreach (var layer in _CurrentSubScene.IStateMachine.Layers)
                {
                    var item = new ToolStripMenuItem(layer.Name);
                    item.Name = "dynamically";
                    item.Click += _selectLayerItem_Click;

                    _btnSelectTargetLayer.DropDownItems.Add(item);
                }
            }

            // ステートリスト を更新、選択復元
            {
                _lvwStateLayerEditItems.SuspendLayout();
                _lvwStateLayerEditItems.Items.Clear();

                foreach (var item in selectedLayer.EnumrateEditItems())
                {
                    ListViewItem lvi = new ListViewItem(item.EditName);
                    lvi.Tag = item;
                    lvi.Selected = item.IsSelected;
                    lvi.ImageIndex = item is IStateTransition ? 1 : 0;

                    _lvwStateLayerEditItems.Items.Add(lvi);
                }

                _lvwStateLayerEditItems.ResumeLayout();
            }

            // 特徴パラメーター類
            {
                var selectedItem = selectedLayer.GetSelectedEditItem();
                UpdatePropertyForSelectedLayerEditItem_(selectedLayer, selectedItem);
            }
        }

        //----------------------------------------------------------
        void SetParamatersToListView_(ListView lv, IState state)
        {
            if(state == null)
            {
                return;
            }

            foreach (var param in state.Paramaters)
            {
                var item = new ListViewItem(param.Name);
                item.Tag = param;
                if(lv.SmallImageList != null)
                {
                    item.ImageIndex = lv.SmallImageList.Images.IndexOfKey(param.Kind.ToString());
                }

                lv.Items.Add(item);
            }

            lv.Tag = state;
        }

        //----------------------------------------------------------
        private void _selectLayerItem_Click(object sender, EventArgs e)
        {
            // 現在の選択レイヤーを変更する。
            StateMachineManipulator mnp = new StateMachineManipulator();
            mnp.BindTarget(_CurrentSubScene.IStateMachine);

            string stateLayerName = (sender as ToolStripMenuItem).Text;

            mnp.SetLayerSelected(stateLayerName);
        }

        public override void OnTimeChangedHandler(int time, TimeChageEventType type)
        {
            switch (type)
            {
                case TimeChageEventType.Tick:
                    _pnlTransitionEdit.Invalidate();
                    break;

            }
            base.OnTimeChangedHandler(time, type);
        }

        //----------------------------------------------------------
        public override void OnSceneModifyHandler(object sender, LECore.Structures.SceneModifyEventArgs e)
        {
            //DbgConsole.WriteLine(string.Format("================== StateWindow {0}", e.Target));
            switch (e.Target)
            {
                case SceneModifyEventArgs.Kind.CurrentSubSceneChanged:
                    UpdateProperty_();
                    break;
                case SceneModifyEventArgs.Kind.StateMachineModify:
                    UpdateProperty_();
                    break;
                case SceneModifyEventArgs.Kind.SelectedSetModify:
                    UpdatePropertyEnableState_();
                    break;
                case SceneModifyEventArgs.Kind.StateMachineSelectionChanged:
                    UpdatePropertyEnableState_();

                    IStateLayer selectedLayer = _CurrentSubScene.IStateMachine.GetSelectedIStateLayer();
                    var selectedItem = selectedLayer.GetSelectedEditItem();

                    UpdatePropertyForSelectedLayerEditItem_(selectedLayer, selectedItem);
                    break;
            }
            base.OnSceneModifyHandler(sender, e);
        }

        //----------------------------------------------------------

        private void _btnEnableState_CheckedChanged(object sender, EventArgs e)
        {
            if(_CurrentSubScene == null)
            {
                return;
            }

            if(_CurrentSubScene.IStateMachine.IsEnabled == _btnEnableState.Checked)
            {
                return;
            }

            StateMachineManipulator mnp = new StateMachineManipulator();
            mnp.BindTarget(_CurrentSubScene.IStateMachine);

            if (_btnEnableState.Checked)
            {
                // 有効にする
                mnp.IsEnabled = true;
                // 初期設定が必要なら整える
                if(_CurrentSubScene.IStateMachine.NeedToInitialSetup())
                {
                    mnp.InitialSetup();
                }
            }
            else
            {
                // ダイアログを出して、消してもOKか確認して消す。
                if(MessageBox.Show("ステートをOFFにします。よいですか？","ステート編集", MessageBoxButtons.OKCancel) == DialogResult.OK)
                {
                    mnp.IsEnabled = false;
                }
            }
        }

        //----------------------------------------------------------

        private void _lvwStateProperties_ItemSelectionChanged(object sender, ListViewItemSelectionChangedEventArgs e)
        {
            // 最後にクリックされた場所にノードある場合、そのノードからの通知以外は無視する。
            // ノードが無い場合は選択解除の可能性があるので処理する。
            // 選択解除と、選択の2度メッセージが来てちらつくのを回避するため。
            if (_lvwStateLayerEditItems.Tag != null)
            {
                Point lastMouseDownPoint = (Point)_lvwStateLayerEditItems.Tag;
                var item = _lvwStateLayerEditItems.GetItemAt(lastMouseDownPoint.X, lastMouseDownPoint.Y);
                if (item != null)
                {
                    if (item != e.Item)
                    {
                        return;
                    }
                }
            }

            var steteLayer = _CurrentSubScene.IStateMachine.GetSelectedIStateLayer();
            if (steteLayer == null)
            {
                return;
            }

            StateMachineManipulator mnp = new StateMachineManipulator();
            mnp.BindTarget(_CurrentSubScene.IStateMachine);

            IStateLayerEditItem editItem = _lvwStateLayerEditItems.SelectedIndices.Count > 0 ?
                _lvwStateLayerEditItems.Items[_lvwStateLayerEditItems.SelectedIndices[0]].Tag as IStateLayerEditItem : null;

            string selectedName = editItem != null ? editItem.EditName : string.Empty;
            mnp.SetLayerEditItemSelected(steteLayer, selectedName);
        }

        //----------------------------------------------------------

        private void _btnEditProperty_Click(object sender, EventArgs e)
        {
            if (_addPropertyDlg.Visible)
            {
                return;
            }

            var onPaneAdded = new Action<IEnumerable<IPane>, FeatureParamaterKind, bool>((panes, flag, isEdit) =>
            {
                if (_CurrentSubScene == null)
                {
                    return;
                }

                IStateLayer selectedLayer = _CurrentSubScene.IStateMachine.GetSelectedIStateLayer();
                if (selectedLayer == null)
                {
                    return;
                }

                StateMachineManipulator mnp = new StateMachineManipulator();
                mnp.BindTarget(_CurrentSubScene.IStateMachine);

                if (isEdit)
                {
                    mnp.EditParamatersToLayer(selectedLayer, panes, flag);
                }
                else
                {
                    mnp.AddParamatersToLayer(selectedLayer, panes, flag);
                }
            });

            _addPropertyDlg.Setup(_CurrentSubScene.IStateMachine, _CurrentSubScene.ISelectedSet.IPaneArray, onPaneAdded);
            _addPropertyDlg.ShowInEditMode();

            _addPropertyDlg.SetLocationToToolStripButtonPos(sender);
        }

        //----------------------------------------------------------

        private void _lvwStateParamaters_ItemSelectionChanged(object sender, ListViewItemSelectionChangedEventArgs e)
        {
            // 最後にクリックされた場所にノードある場合、そのノードからの通知以外は無視する。
            // ノードが無い場合は選択解除の可能性があるので処理する。
            // 選択解除と、選択の2度メッセージが来てちらつくのを回避するため。
            if (_lvwStateParamaters.Tag != null)
            {
                Point lastMouseDownPoint = (Point)_lvwStateParamaters.Tag;
                var item = _lvwStateParamaters.GetItemAt(lastMouseDownPoint.X, lastMouseDownPoint.Y);
                if (item != null)
                {
                    if (item != e.Item)
                    {
                        return;
                    }
                }
            }

            List<IPane> paneSet = new List<IPane>();
            if (_lvwStateParamaters.SelectedIndices.Count > 0)
            {
                foreach (int index in _lvwStateParamaters.SelectedIndices)
                {
                    var prop = _lvwStateParamaters.Items[index].Tag as IFeatureParamater;

                    var pane = _CurrentSubScene.FindPaneByName(prop.Name);
                    if (pane != null && !paneSet.Contains(pane))
                    {
                        paneSet.Add(pane);
                    }
                }
            }

            {
                var selectedPanes = _CurrentSubScene.ISelectedSet.IPaneArray;
                if (selectedPanes.Length != paneSet.Count || selectedPanes.Any(p => !paneSet.Contains(p)))
                {
                    SubSceneManipulator mnp = new SubSceneManipulator();
                    mnp.BindTarget(_CurrentSubScene);
                    mnp.BeginSelectSetChange();
                    mnp.ResetSelectedSet();
                    foreach (var pane in paneSet)
                    {
                        mnp.SelectPanesByPaneRef(pane);
                    }

                    mnp.EndSelectSetChange();
                }
            }
        }

        //----------------------------------------------------------

        private void _btnCreateStateFromSelected_Click(object sender, EventArgs e)
        {
            if(_CurrentSubScene == null)
            {
                return;
            }

            var steteLayer = _CurrentSubScene.IStateMachine.GetSelectedIStateLayer();
            if(steteLayer == null)
            {
                return;
            }

            var selectedProp = _CurrentSubScene.IStateMachine.GetSelectedEditItem();
            if(!(selectedProp is IState))
            {
                return;
            }

            var onOk = new Action(() =>
            {
                if (_CurrentSubScene == null)
                {
                    return;
                }

                IStateLayer selectedLayer = _CurrentSubScene.IStateMachine.GetSelectedIStateLayer();
                if (selectedLayer == null)
                {
                    return;
                }

                StateMachineManipulator mnp = new StateMachineManipulator();
                mnp.BindTarget(_CurrentSubScene.IStateMachine);

                mnp.AddStateToLayerByCloneSource(selectedLayer, _addStateDlg.NewStateName, _addStateDlg.SourceStateName);

                if (_addStateDlg.IsCreateTransition)
                {
                    mnp.AddTranstionToLayer(selectedLayer, _addStateDlg.OriginStateName, _addStateDlg.NewStateName);
                }
            });

            _addStateDlg.Setup(_CurrentSubScene.IStateMachine, (selectedProp as IState).Name, onOk);
            _addStateDlg.Show();

            _addStateDlg.SetLocationToToolStripButtonPos(sender);
        }

        //----------------------------------------------------------

        private void _btnEditTransition_Click(object sender, EventArgs e)
        {
            Debug.Assert(_CurrentSubScene != null);
            Debug.Assert(_CurrentSubScene.IStateMachine.GetSelectedIStateLayer() != null);

            var onOk = new Action(() =>
            {
                StateMachineManipulator mnp = new StateMachineManipulator();
                mnp.BindTarget(_CurrentSubScene.IStateMachine);

                IStateLayer selectedLayer = _CurrentSubScene.IStateMachine.GetSelectedIStateLayer();
                mnp.AddTranstionToLayer(selectedLayer,
                    _editTransitionDlg.StartStateName,
                    _editTransitionDlg.EndStateName);

                mnp.EditTranstion(selectedLayer,
                    _editTransitionDlg.StartStateName,
                    _editTransitionDlg.EndStateName,
                    _editTransitionDlg.TotalDuration,
                    IStateEasingHelper.Create(_editTransitionDlg.EasingType, _editTransitionDlg.EasingExtraParamater),
                    _editTransitionDlg.IsCancelable,
                    _editTransitionDlg.IsLoop);

                mnp.EditTranstionTrigger(selectedLayer,
                    _editTransitionDlg.StartStateName,
                    _editTransitionDlg.EndStateName,
                    _editTransitionDlg.TriggerKind,
                    _editTransitionDlg.StateMachineOperator,
                    _editTransitionDlg.VariableName,
                    _editTransitionDlg.VariableValue);
            });

            var selectedItem = _CurrentSubScene.IStateMachine.GetSelectedEditItem();
            if (selectedItem is IState)
            {
                _editTransitionDlg.Setup(_CurrentSubScene.IStateMachine, (selectedItem as IState).Name, string.Empty, IStateEasingHelper.Empty, TriggerKind.IsAuto, onOk);
            }
            else if(selectedItem is IStateTransition)
            {
                // トリガー関連の設定を足す！
                var transition = selectedItem as IStateTransition;
                _editTransitionDlg.Setup(_CurrentSubScene.IStateMachine, transition.StartStateName, transition.EndStateName, transition.StateEasing, transition.Trigger.TriggerKind, onOk);
                _editTransitionDlg.SetupVariableChangedTrigger(_CurrentSubScene.IStateMachine, transition.Trigger.StateMachineOperator, string.Empty, 0.0f);

                _editTransitionDlg.IsCancelable = transition.IsCancelable;
                _editTransitionDlg.IsLoop = transition.IsLoop;
                _editTransitionDlg.TotalDuration = transition.TotalDuration;
            }
            else
            {
                Debug.Assert(false);
            }

            _editTransitionDlg.Show();
            _editTransitionDlg.SetLocationToToolStripButtonPos(sender);
        }

        //----------------------------------------------------------

        private void _lvwStateParamaters_MouseDoubleClick(object sender, MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Left)
            {
                // ダブルクリック
                if (e.Clicks == 2)
                {
                    var lv = sender as ListView;
                    // 行全体で判定するため、x に適当な値を設定
                    var item = lv.GetItemAt(1, e.Y);
                    if (item == null)
                    {
                        return;
                    }

                    var onOk = new Action(() =>
                    {
                        if (_CurrentSubScene == null)
                        {
                            return;
                        }

                        IStateLayer selectedLayer = _CurrentSubScene.IStateMachine.GetSelectedIStateLayer();
                        if (selectedLayer == null)
                        {
                            return;
                        }

                        StateMachineManipulator mnp = new StateMachineManipulator();
                        mnp.BindTarget(_CurrentSubScene.IStateMachine);

                        mnp.EditStateFeatureParamater(
                            selectedLayer,
                            _editFeatureParamaterDlg.State,
                            _editFeatureParamaterDlg.FeatureParamater.Name,
                            _editFeatureParamaterDlg.FeatureParamater.Kind,
                            _editFeatureParamaterDlg.X,
                            _editFeatureParamaterDlg.Y,
                            _editFeatureParamaterDlg.Z,
                            _editFeatureParamaterDlg.W);
                    });

                    var state = sender == _lvwStateParamatersStart ? _currentEditState : _currentEditNextState;
                    var paramater = item.Tag as IFeatureParamater;
                    _editFeatureParamaterDlg.Setup(_CurrentSubScene.IStateMachine, state, paramater, onOk);

                    _editFeatureParamaterDlg.Show();
                    _editFeatureParamaterDlg.SetLocationToMousePos(lv, e.X, e.Y);
                }

            }
        }

        //----------------------------------------------------------

        private void _lvwStateParamaters_MouseDown(object sender, MouseEventArgs e)
        {
            _lvwStateParamaters.Tag = new Point(e.X, e.Y);
        }

        //----------------------------------------------------------

        private void _lvwStateProperties_MouseDown(object sender, MouseEventArgs e)
        {
            _lvwStateLayerEditItems.Tag = new Point(e.X, e.Y);
        }

        /// <summary>
        /// 現在の値を FeatureParamaters に取り込む。
        /// </summary>
        private void _btnStoreCurrentToProperty_Click(object sender, EventArgs e)
        {
            if (_CurrentSubScene == null)
            {
                return;
            }

            var editItem = _CurrentSubScene.IStateMachine.GetSelectedEditItem();
            Debug.Assert(editItem != null);

            StateMachineManipulator mnp = new StateMachineManipulator();
            mnp.BindTarget(_CurrentSubScene.IStateMachine);
            mnp.StoreParamatersFromCurrent(_CurrentSubScene.IStateMachine.GetSelectedIStateLayer(), editItem);
        }

        private void _btnDeleteSelected_Click(object sender, EventArgs e)
        {
            if (_CurrentSubScene == null)
            {
                return;
            }

            var steteLayer = _CurrentSubScene.IStateMachine.GetSelectedIStateLayer();
            if (steteLayer == null)
            {
                return;
            }

            var selectedEditItem = _CurrentSubScene.IStateMachine.GetSelectedEditItem();
            if(selectedEditItem == null)
            {
                return;
            }

            StateMachineManipulator mnp = new StateMachineManipulator();
            mnp.BindTarget(_CurrentSubScene.IStateMachine);

            if(selectedEditItem.IsDeletable())
            {
                mnp.RemoveEditItemFromLayer(steteLayer, selectedEditItem);
            }
        }

        /// <summary>
        /// タイマイベントハンドラ
        /// </summary>
        private void Event_Timer_Tick(object sender, System.EventArgs e)
        {
            if (!_timer.Enabled)
            {
                // すでに Timer が止まっている場合は処理しない
                return;
            }

            if(_CurrentSubScene == null)
            {
                return;
            }

            var now = DateTime.Now;
            var elapsed = now - _lastUpdateTime;

            {
                if (elapsed < _FrameSpanFps60)
                {
                    return;
                }
                // 1フレームのレンダリングに2フレーム分以上時間がかかっている場合は、次のイベントでもレンダリングされるように適当に補正
                if (elapsed > _FrameSpanFps30)
                {
                    _lastUpdateTime = now - _FrameSpanFps60;
                }
                else
                {
                    _lastUpdateTime = _lastUpdateTime + _FrameSpanFps60;
                }
            }

            StateMachineManipulator mnp = new StateMachineManipulator();
            mnp.BindTarget(_CurrentSubScene.IStateMachine);

            mnp.RunUpdate(1.0f, FVec2.Empty, (Control.MouseButtons & MouseButtons.Left) != 0);
        }

        private void _btnRunStop_Click(object sender, EventArgs e)
        {
            StateMachineManipulator mnp = new StateMachineManipulator();
            mnp.BindTarget(_CurrentSubScene.IStateMachine);

            if(_btnRunStop.Checked)
            {
                mnp.SetStateMachineRunMode(StateMachineRunMode.Run);
                // 更新タイマーを走らす
                _timer.Start();
            }
            else
            {
                mnp.SetStateMachineRunMode(StateMachineRunMode.Edit);
                // 更新タイマーを止める
                _timer.Stop();
            }
        }
    }
}
