﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Drawing;
using System.Diagnostics;
using System.Linq;
using System.Windows.Forms;

namespace LayoutEditor.Forms.ToolWindows
{
    using LECore.Structures;
    using LECore.Structures.Core;

    using Structures.SerializableObject;
    using PropertyEditWindow;
    using PropertyEditWindow.Adapters;

    using GUISrcElem = LECore.Structures.Nsrif.Elements;
    using GUISrcAttr = LECore.Structures.Nsrif.Attributes;

    using ReservedWndMap = Dictionary<string, PropertyEditWindow.PropertyWindow>;

    /// <summary>
    /// PropertyWindowMgr
    ///
    /// 芸算表示されている、 複数個の PropertyWindow を管理するクラスです。
    /// ”選択アイテムが変更された”イベント等は一旦、本クラスに通知されます。
    /// 本クラスは、メッセージハンドラの中で、PropertyWindowインタンスを走査し、
    /// 更新の必要がある、インスタンスに対して、ターゲットアイテムの変更を行います。
    ///
    /// 本クラスの、原型である 3D Editorの実装のように、
    /// PropertyWindowインスタンスは、自ら、選択アイテムを参照することはしません。
    ///
    /// </summary>
    public class PropertyWindowMgr :
        ILEToolWindow
    {
        //-------------------------------------------------------
        #region 型定義

        /// <summary>
        /// プロパティウインドウの状態クラス
        ///
        /// ウインドウが閉じられた時の、位置、サイズ保存に利用されます。
        /// </summary>
        struct PropertyWindowSetting
        {
            Point _windowPosition;
            Size _windowSize;

            public Point WindowPosition { get { return _windowPosition; } }
            public Size WindowSize { get { return _windowSize; } }


            public PropertyWindowSetting( PropertyWindow wnd )
            {
                _windowPosition = wnd.FloatPane?.FloatWindow?.Location ?? wnd.Location;
                _windowSize = wnd.FloatPane?.FloatWindow?.Size ?? wnd.Size;
            }

            public PropertyWindowSetting( Point pos, Size size )
            {
                _windowPosition = pos;
                _windowSize = size;
            }

            public void SetToForm( Form form )
            {
                form.Location = _windowPosition;
                form.Size = _windowSize;
            }

            public bool Equals( PropertyWindowSetting another )
            {
                return this._windowPosition.Equals( another._windowPosition ) &&
                    this._windowSize.Equals( another._windowSize );
            }

            public static PropertyWindowSetting Empty =
                new PropertyWindowSetting( Point.Empty, Size.Empty );
        }
        #endregion 型定義

        //-------------------------------------------------------
        #region フィールド
        /// <summary>
        /// 芸算表示されている、 複数個の PropertyWindow
        /// </summary>
        List<PropertyWindow> _propertyDlgSet = new List<PropertyWindow>();

        /// <summary>
        /// 最後に閉じられた、プロパティウインドウの大きさ
        /// </summary>
        PropertyWindowSetting _windowSetting = PropertyWindowSetting.Empty;

        /// <summary>
        /// アプリケーション設定
        /// </summary>
        private AppSetting _appSetting
        {
            get { return this.ViewManager != null ? this.ViewManager.AppSetting : null; }
        }

        /// <summary>
        /// ビューマネージャです。
        /// </summary>
        public IViewManager ViewManager { get; set; }

        /// <summary>
        /// アプリケーションフォーム
        /// </summary>
        private AppForm _theAppForm = null;

        /// <summary>
        /// LEWindowName
        /// </summary>
        readonly string _LEWindowName;

        readonly Action<ViewManagerMessage> _messageSendHandler;
        #endregion フィールド

        //-------------------------------------------------------
        #region --------------- 外部非公開メソッド ---------------

        /// <summary>
        /// アダプタクラスを生成します。
        /// </summary>
        /// <param name="paneAdapter"></param>
        /// <returns></returns>
        PaneGUIAdapter MakeGUIAdapter_( IPane targetPane )
        {
            return new PaneGUIAdapter(targetPane);
        }

        /// <summary>
        ///
        /// </summary>
        private PaneGUIAdapter[] MakeGUIAdapters( IPane[] panes)
        {
            List<PaneGUIAdapter> list = new List<PaneGUIAdapter>();
            foreach( IPane pane in panes )
            {
                list.Add( MakeGUIAdapter_( pane));
            }
            return list.ToArray();
        }

        /// <summary>
        /// SelectedSetModify から GuiObjectGroup を作成します。
        /// </summary>
        PaneGuiAdapterGroup MakeGuiObjectGroupFromSelectedSet_( IPane[] selSet )
        {
            if( selSet.Length != 0 )
            {
                PaneGuiAdapterGroup tempGroup = new PaneGuiAdapterGroup();

                foreach( IPane targetPane in selSet )
                {
                    // アダプタオブジェクトを生成してグループに登録します。
                    PaneGUIAdapter paneAdapter = MakeGUIAdapter_( targetPane );
                    tempGroup.Add( paneAdapter );
                }
                return tempGroup;
            }
            else
            {
                // 空のグループを返す。
                return new PaneGuiAdapterGroup();
            }
        }

        /// <summary>
        ///
        /// </summary>
        static bool IsWindowTargetCurrent_( PropertyWindow propDlg, ISubScene subScene )
        {
            foreach( PaneGUIAdapter adapter in propDlg.Target.Objects )
            {
                if( adapter.PaneManipulator.IPane.OwnerSubScene != subScene )
                {
                    return false;
                }
            }
            return true;
        }

        #endregion // --------------- 外部非公開メソッド ---------------


        /// <summary>
        /// コンストラクタ
        /// </summary>
        public PropertyWindowMgr(AppForm theAppForm, Action<ViewManagerMessage> messageSendHandler)
        {
            _theAppForm = theAppForm;

            TextureManager.Initialize();

            _LEWindowName = StringResMgr.Get( "PROP_WNDMGR_NAME" );

            // シーン変更イベントハンドラを実行し、内部テクスチャマネージャ等を初期化します。
            SceneModifyEventArgs args =
                new SceneModifyEventArgs( SceneModifyEventArgs.Kind.CurrentSubSceneChanged, null );

            _messageSendHandler = messageSendHandler;

            OnSceneModifyHandler( this, args );
        }

        //-------------------------------------------------------
        #region プロパティウインドウの新規登録
        /// <summary>
        /// 指定ペインを表示対象にしている、プロパティダイアログを検索します。
        /// </summary>
        /// <param name="paneAdapter"></param>
        /// <returns></returns>
        PropertyWindow FindPropertyDlgShowingThePane_(IBaseGuiAdapter adapter, out bool bShouldFreeze )
        {
            Debug.Assert( adapter != null );
            bShouldFreeze = false;

            // 無ければ、全体からさがす。
            foreach( PropertyWindow propDlg in _propertyDlgSet )
            {
                foreach(IBaseGuiAdapter taget in propDlg.Target.Objects)
                {
                    if( taget.HasSameTargets( adapter ) )
                    {
                        bShouldFreeze = propDlg.Freeze;
                        return propDlg;
                    }
                }
            }
            return null;
        }


        /// <summary>
        /// マネージャが管理している、ダイアログが閉じられたときに呼び出されるハンドラ。
        /// </summary>
        /// <param name="closingDlg">閉じられたダイアログ</param>
        void OnDlgClose_( PropertyWindow closingDlg )
        {
            if (_propertyDlgSet.Remove(closingDlg))
            {
                closingDlg.SendMessageToViewManager -= _messageSendHandler;

                // 直前のフローティング状態がフローティングウインドウ全体を占める場合のみ保存
                if (closingDlg?.FloatPane?.Contents?.Count == 1 &&
                    closingDlg?.FloatPane?.FloatWindow?.NestedPanes?.Count == 1)
                {
                    _windowSetting = new PropertyWindowSetting(closingDlg);
                }

                if (!_theAppForm.IsDisposed)
                {
                    _theAppForm.Activate();
                }
            }
        }

        /// <summary>
        /// 新規にダイアログを生成します。
        /// </summary>
        /// <param name="paneAdapter"></param>
        public PropertyWindow RegisterNewDialog( IPane[] targetPanes, bool show)
        {
            // ラッパーを生成する
            PaneGUIAdapter[] paneAdapters = MakeGUIAdapters( targetPanes);

            // すでに paneAdapter を表示しているダイアログが存在しないか検索します。
            bool bShouldFreeze = false;
            PropertyWindow propDlg = null;

            // 表示まで要求されているときは既存のウインドウがあるならそちらを使う
            if (paneAdapters.Any() && show)
            {
                propDlg = FindPropertyDlgShowingThePane_
                 (paneAdapters[0], out bShouldFreeze);
            }

            bool created = false;
            if( propDlg == null )
            {
                // シーンに表示対象を失ったダイアログが存在しないか検索します。
                // 新規にダイアログを生成し、対象として指定ペインを設定します。
                propDlg = new PropertyWindow( this, null, _appSetting );
                propDlg.OwnerCmdKeyAcceptor = _theAppForm;
                propDlg.SendMessageToViewManager += _messageSendHandler;

                _propertyDlgSet.Add( propDlg );

                propDlg.OnClose += new PropertyWindow.OnCloseHandler(OnDlgClose_);
                created = true;
            }
            Debug.Assert( propDlg != null );

            // 一度もウインドウ情報が保存されていない場合には作成したウインドウの情報を
            // 最新の情報として保存する
            if( _windowSetting.Equals( PropertyWindowSetting.Empty) != false )
            {
                _windowSetting = new PropertyWindowSetting( propDlg);
            }

            //--------------------------
            // 対象が設定されていなければ設定します。
            PaneGuiAdapterGroup tempGroup = new PaneGuiAdapterGroup();
            foreach( PaneGUIAdapter paneAdapter in paneAdapters )
            {
                tempGroup.Add( paneAdapter);
            }

            SetDlgTarget_( propDlg, tempGroup );

            //--------------------------
            // 固定設定をします。
            propDlg.Freeze = bShouldFreeze;

            //--------------------------
            // 表示します。
            if(show)
            {
                if (created)
                {
                    propDlg.Show(_theAppForm.DockPanel, new Rectangle(_windowSetting.WindowPosition, _windowSetting.WindowSize));
                }
                else
                {
                    propDlg.Show(_theAppForm.DockPanel);
                }
            }

            propDlg.UpdateCurrentPane();

            return propDlg;
        }

        /// <summary>
        /// 新規にダイアログを生成します。
        /// 指定ダイアログと同じTargetを指します
        /// </summary>
        public void DuplicatePropertyDialog(ILEToolWindow dlg)
        {
            PropertyWindow srcDlg = dlg as PropertyWindow;
            PropertyWindow newDlg = null;

            newDlg = new PropertyWindow(this, null, _appSetting);
            newDlg.OwnerCmdKeyAcceptor = _theAppForm;
            newDlg.SendMessageToViewManager += _messageSendHandler;

            // 設定が保存されていれば、適用します。
            if (!_windowSetting.Equals(PropertyWindowSetting.Empty))
            {
                newDlg.StartPosition = FormStartPosition.Manual;
                newDlg.Location = _windowSetting.WindowPosition;
                newDlg.Size = _windowSetting.WindowSize;
            }
            _propertyDlgSet.Add(newDlg);
            Debug.Assert(newDlg != null);

            //
            int add = SystemInformation.CaptionHeight * 2;
            newDlg.Location = new Point(srcDlg.Left + add, srcDlg.Top + add);
            if (newDlg.Top + newDlg.Height > Screen.PrimaryScreen.Bounds.Height ||
                newDlg.Left + newDlg.Width > Screen.PrimaryScreen.Bounds.Width)
            {
                newDlg.Location = new Point(0, 0);
            }

            newDlg.OnClose += OnDlgClose_;

            newDlg.Target = srcDlg.Target;
            newDlg.Freeze = true;

            var size = newDlg.Size;
            newDlg.Show(_theAppForm.DockPanel, WeifenLuo.WinFormsUI.Docking.DockState.Float);
            if (newDlg.FloatPane?.FloatWindow != null)
            {
                newDlg.FloatPane.FloatWindow.ClientSize = size;
            }
        }
        #endregion プロパティウインドウの新規登録

        //-------------------------------------------------------
        #region 表す・隠す
        /// <summary>
        /// すべてのダイアログを表示します。
        /// </summary>
        public void ShowAll()
        {
            if (!_propertyDlgSet.Any())
            {
                ViewManager.MakeNewPropertyDlg(true, true);
                return;
            }

            ISubScene subScene = LECore.LayoutEditorCore.Scene.CurrentISubScene;
            foreach( PropertyWindow propDlg in _propertyDlgSet )
            {
                if( IsWindowTargetCurrent_( propDlg, subScene ) )
                {
                    if (propDlg.DockPanel == null)
                    {
                        Size size = propDlg.Size;
                        propDlg.Show(_theAppForm.DockPanel);
                        if (propDlg.FloatPane?.FloatWindow != null)
                        {
                            propDlg.FloatPane.FloatWindow.ClientSize = size;
                        }
                    }
                    else
                    {
                        propDlg.Show();
                    }
                }
            }
        }

        /// <summary>
        /// すべてのダイアログを隠します。
        /// </summary>
        public void HideAll()
        {
            foreach( PropertyWindow propDlg in _propertyDlgSet )
            {
                propDlg.Hide();
            }
        }

        public void ClearAll()
        {
            foreach (PropertyWindow propDlg in _propertyDlgSet.ToArray() )
            {
                propDlg.Close();
            }
        }
        #endregion 表す・隠す

        //-------------------------------------------------------
        #region テクスチャ、パレット、フォントなど、リソースマネージャの状態更新。
        /// <summary>
        /// プロパティダイアログが参照するテクスチャマネージャの状態を
        /// 最新の状態に更新します。
        /// ViewManagerの テクスチャマネージャ更新ハンドラから呼び出されます。
        ///
        /// ランタイムライブラリのリソース参照関係と同様に
        /// すべての SubScene を取りまとめてグローバルなテクスチャリストを作成します。
        /// つまり、配下の部品レイアウト（孫部品など再帰的にすべて）のテクスチャもすべて
        /// UI レンダリングは参照できるようにします。
        ///
        /// ランタイムライブラリでは、テクスチャはすべての部品間で統合され、一つのリソースアクセサから検索、参照されています。
        /// </summary>
        void UpdateTextureMgrState_(ISubScene subScene)
        {
            LECore.DbgConsole.WriteLine("UI-TextureManager is rebuilt.");

            // すべての内容を更新します。(MEMO：処理を効率化したい。)
            TextureManager.Reset();
            var textureMangers = new List<ITextureMgr>();

            textureMangers.Add(subScene.ITextureMgr);

            textureMangers.AddRange(subScene.FindPanesByKindRecursively(PaneKind.Parts).
                Select((partsPane) => partsPane.IPartsLayout?.PartsSubScene?.ITextureMgr).
                Where((mgr) => mgr != null));

            foreach (var textureManger in textureMangers)
            {
                bool isOwnedByCurrentSubScene = subScene.ITextureMgr == textureManger;
                foreach (ITextureImage texImg in textureManger.ITextureImageSet)
                {
                    // ファイルサイズを知りたい。
                    // サムネイルが必要。
                    // ファイルフォーマットが必要。
                    Texture newTexture = new Texture(texImg, isOwnedByCurrentSubScene);
                    TextureManager.Add(newTexture);
                }
            }
        }

        /// <summary>
        /// プロパティダイアログが参照するテクスチャマネージャの状態を
        /// 最新の状態に更新します。
        /// ViewManagerの テクスチャマネージャ更新ハンドラから呼び出されます。
        /// </summary>
        void UpdateFontMgrState_(ISubScene subScene)
        {
            LECore.DbgConsole.WriteLine("UI-FontManager is rebuilt.");

            // すべての未登録フォントを登録します。
            FontManager.Reset();
            var fontMangers = new List<ILEFontManager>();

            fontMangers.Add(subScene.ILEFontManager);

            fontMangers.AddRange(subScene.FindPanesByKindRecursively(PaneKind.Parts).
                Select((partsPane) => partsPane.IPartsLayout?.PartsSubScene?.ILEFontManager).
                Where((mgr) => mgr != null));

            foreach (var fontManger in fontMangers)
            {
                foreach (ILEFont font in fontManger.ILEFontSet)
                {
                    if (FontManager.FindFontByFontName(font.FontName) == null)
                    {
                        FontManager.Add(new GuiFont(font.FontPath, font.FontName));
                    }
                }
            }
        }
        #endregion テクスチャ、パレット、フォントなど、リソースマネージャの状態更新。

        //-------------------------------------------------------
        #region ILEToolWindow メンバ

        public string LEWindowName { get { return _LEWindowName; } }
        public string Description { get { return string.Empty; } }
        public Keys CustomShortcut { get { return Keys.Control | Keys.Shift | Keys.P; } }
        public bool AllowToChangeVisible { get { return true; } }
        public bool IsNeededToHide { get { return false; } }

        /// <summary>
        /// コマンドキー処理の委譲先です。
        /// </summary>
        public ILECmdKeyAcceptor OwnerCmdKeyAcceptor { get; set; }

        #region ITimeChageEventListener メンバ
        public void OnTimeChangedHandler( int time, TimeChageEventType type )
        {
            if( type == TimeChageEventType.Stop )
            {
                // プロパティダイアログの変更
                RedrawAll();
            }
        }
        #endregion ITimeChageEventListener メンバ

        #region ISceneModifyListener メンバ

        /// <summary>
        /// シーン更新イベントハンドラ：カレントサブシーン変更
        /// </summary>
        void OnSceneModifyHandler_CurrentSubSceneChanged_(
            object sender, SceneModifyEventArgs e )
        {
            ISubScene subScene = LECore.LayoutEditorCore.Scene.CurrentISubScene;
            if( subScene != null )
            {
                // 内部テクスチャ情報の更新
                UpdateTextureMgrState_(subScene);
                // 内部テクスチャ情報の更新
                UpdateFontMgrState_(subScene);

                // 新しいサブシーンに含まれないペインを
                // 固定してターゲットしているウインドウを隠します。
                foreach( PropertyWindow wnd in _propertyDlgSet )
                {
                    if( wnd.Freeze )
                    {
                        wnd.Freeze = false;
                    }
                }
            }
        }
        /// <summary>
        /// シーン更新イベントハンドラ
        /// </summary>
        public void OnSceneModifyHandler( object sender, SceneModifyEventArgs e )
        {
            var subScene = LECore.LayoutEditorCore.Scene.CurrentISubScene;
            switch ( e.Target )
            {
                //------------------------------------------------------
                case SceneModifyEventArgs.Kind.SelectedSetModify:
                {
                    IPane[] selSet = e.Paramaters[0] as IPane[];
                    Debug.Assert( selSet != null );

                    OnSelectedSetChangedEventHandler( selSet );
                    break;
                }
                //------------------------------------------------------
                // カレントサブシーンの更新
                case SceneModifyEventArgs.Kind.CurrentSubSceneChanged:
                {
                    OnSceneModifyHandler_CurrentSubSceneChanged_( sender, e );
                    break;
                }
                //------------------------------------------------------
                case SceneModifyEventArgs.Kind.PartsLayout:
                {
                    if (subScene != null)
                    {
                        UpdateTextureMgrState_(subScene);
                        UpdateFontMgrState_(subScene);
                    }
                    break;
                }
                case SceneModifyEventArgs.Kind.TextureManager:
                {
                    if (subScene != null)
                    {
                        UpdateTextureMgrState_(subScene);
                    }
                    break;
                }
                //------------------------------------------------------
                case SceneModifyEventArgs.Kind.FontManager:
                {
                    if (subScene != null)
                    {
                        UpdateFontMgrState_(subScene);
                    }
                    break;
                }
            }

            //----------------------------------------------------------
            // すべてのプロパティウインドウにメッセージを伝播します。
            foreach( PropertyWindow propDlg in _propertyDlgSet )
            {
                propDlg.OnSceneModifyHandler( sender, e );
            }
        }
        #endregion ISceneModifyListener メンバ

        #region IAppEventListener メンバ

        public void OnAppEvent( object sender, AppEventArgs args )
        {
            switch (args.Kind)
            {
                case AppEventKind.AppSettingChanged: RedrawAll(); break;
            }
        }

        #endregion AppEventListener

        #region 設定保存関連
        /// <summary>
        /// 状態を保存します。
        /// 最小化状態のウインドウの状態は保存しません。
        /// </summary>
        public void SaveSetting(LEToolFormSetting setting, SaveSettingOption option)
        {
            // 直前のフローティング状態がフローティングウインドウ全体を占める場合のみ保存
            var dlg = _propertyDlgSet.FirstOrDefault(x =>
                x?.FloatPane?.Contents?.Count == 1 &&
                x?.FloatPane?.FloatWindow?.NestedPanes?.Count == 1);
            if (dlg != null)
            {
                _windowSetting = new PropertyWindowSetting(dlg);
            }

            // 最後に保存されたウインドウサイズを永続化します。
            setting.Name = this.LEWindowName;
            setting.Position = _windowSetting.WindowPosition;
            setting.Size = _windowSetting.WindowSize;
        }

        /// <summary>
        /// 状態を読み込みます。
        /// </summary>
        public void LoadSetting( LEToolFormSetting setting, LoadSettingOption option)
        {
            // 最後に保存されたウインドウサイズとして入力します。
            _windowSetting = new PropertyWindowSetting( setting.Position, setting.Size );

            // LEToolWIndow.LoadFormBasicData は実行しない
        }
        #endregion 設定保存関連

        #endregion ILEToolWindow メンバ

        //-------------------------------------------------------
        #region ISelectedSetChangedEventListener メンバ

        void SetDlgTarget_
            (
            PropertyWindow listenerDlg,
            PaneGuiAdapterGroup objGroup
            )
        {
            // すでに設定されている場合は、設定を解除します。
            if( listenerDlg.Target != null )
            {
                PaneGuiAdapterGroup oldGroup = listenerDlg.Target;
            }

            // 新規に登録します。
            listenerDlg.Target = objGroup;
        }

        /// <summary>
        /// 選択セット変更ハンドラ
        /// </summary>
        public void OnSelectedSetChangedEventHandler(IPane[] selSet)
        {
            if (_propertyDlgSet.Count == 0)
            {
                return;
            }

            PaneGuiAdapterGroup guiObjectGroup = MakeGuiObjectGroupFromSelectedSet_(selSet);

            ISubScene subScene = LECore.LayoutEditorCore.Scene.CurrentISubScene;
            bool deleted = false;

            foreach (PropertyWindow propDlg in _propertyDlgSet)
            {
                deleted = false;
                if (propDlg.Target.IsEmpty == false)
                {
                    PaneGUIAdapter paneAdapter = propDlg.Target.Active as PaneGUIAdapter;
                    string name = paneAdapter.Name;
                    if (subScene == null ||
                        SubSceneHelper.FindPaneByName(subScene, name) == null)
                    {
                        deleted = true;
                    }
                }

                // 対象が固定されていなければ...
                // ただし、guiObjectGroupが無指定なった場合には対象を設定し直すようにする
                if (propDlg.Freeze == false || deleted)
                {
                    if (deleted)
                    {
                        propDlg.Freeze = false;
                    }

                    // 必要があれば、対象を設定しなおす。
                    if (!propDlg.Target.Equals(guiObjectGroup))
                    {
                        SetDlgTarget_(propDlg, guiObjectGroup);
                    }
                }
            }
        }

        /// <summary>
        /// 指定したペインのプロパティダイアログをすべて再描画します。
        /// </summary>>
        public void RedrawAll()
        {
            foreach( PropertyWindow propDlg in _propertyDlgSet )
            {
                propDlg.Redraw();
            }
        }

        #endregion ISelectedSetChangedEventListener メンバ
    }
}
