﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.Drawing;
using System.Windows.Forms;


namespace LayoutEditor.Forms.ToolWindows.PropertyEditWindow
{
    using System.Collections.Generic;
    using Adapters;
    using LayoutEditor.Controls;
    using LayoutEditor.Utility;
    using LECore.Structures;
    using LECore;
    using System.Threading;
    using LECore.Win32;
    using Structures.SerializableObject;


    /// <summary>
    /// オブジェクトプロパティパネルクラス。
    /// </summary>
    public class PropertyPanel : UserControl, ISceneModifyListener
    {
        #region デザイナ変数

        protected System.Windows.Forms.Panel pnlRight;
        protected System.Windows.Forms.Panel pnlLeft;
        protected LayoutEditor.Controls.UITreeView tvwCategory;

        private System.Windows.Forms.Splitter splBase;
        private LayoutEditor.Controls.BorderPanel pnlPage;
        private System.Windows.Forms.ImageList imgCategory;
        private System.Windows.Forms.ImageList imgButtons;
        private CollapsiblePanel pnlPages;
        private System.ComponentModel.IContainer components;
        #endregion

        #region デザイナコード
        /// <summary>
        /// デザイナ サポートに必要なメソッドです。このメソッドの内容を
        /// コード エディタで変更しないでください。
        /// </summary>
        private void InitializeComponent()
        {
            this.components = new System.ComponentModel.Container();
            System.ComponentModel.ComponentResourceManager resources = new System.ComponentModel.ComponentResourceManager(typeof(PropertyPanel));
            this.pnlRight = new System.Windows.Forms.Panel();
            this.pnlLeft = new System.Windows.Forms.Panel();
            this.imgCategory = new System.Windows.Forms.ImageList(this.components);
            this.splBase = new System.Windows.Forms.Splitter();
            this.imgButtons = new System.Windows.Forms.ImageList(this.components);
            this.pnlPages = new LayoutEditor.Controls.CollapsiblePanel();
            this.pnlPage = new LayoutEditor.Controls.BorderPanel();
            this.tvwCategory = new LayoutEditor.Controls.UITreeView();
            this.pnlRight.SuspendLayout();
            this.pnlLeft.SuspendLayout();
            this.pnlPages.Panel2.SuspendLayout();
            this.pnlPages.SuspendLayout();
            this.SuspendLayout();
            //
            // pnlRight
            //
            this.pnlRight.AutoScroll = true;
            this.pnlRight.Controls.Add(this.pnlPages);
            this.pnlRight.Dock = System.Windows.Forms.DockStyle.Fill;
            this.pnlRight.Location = new System.Drawing.Point(176, 0);
            this.pnlRight.Name = "pnlRight";
            this.pnlRight.Size = new System.Drawing.Size(200, 368);
            this.pnlRight.TabIndex = 2;
            //
            // pnlLeft
            //
            this.pnlLeft.Controls.Add(this.tvwCategory);
            this.pnlLeft.Dock = System.Windows.Forms.DockStyle.Left;
            this.pnlLeft.Location = new System.Drawing.Point(0, 0);
            this.pnlLeft.Name = "pnlLeft";
            this.pnlLeft.Size = new System.Drawing.Size(168, 368);
            this.pnlLeft.TabIndex = 0;
            //
            // imgCategory
            //
            this.imgCategory.ColorDepth = System.Windows.Forms.ColorDepth.Depth8Bit;
            this.imgCategory.ImageSize = new System.Drawing.Size(16, 16);
            this.imgCategory.TransparentColor = System.Drawing.Color.Transparent;
            //
            // splBase
            //
            this.splBase.Location = new System.Drawing.Point(168, 0);
            this.splBase.MinExtra = 200;
            this.splBase.MinSize = 100;
            this.splBase.Name = "splBase";
            this.splBase.Size = new System.Drawing.Size(8, 368);
            this.splBase.TabIndex = 1;
            this.splBase.TabStop = false;
            //
            // imgButtons
            //
            this.imgButtons.ImageStream = ((System.Windows.Forms.ImageListStreamer)(resources.GetObject("imgButtons.ImageStream")));
            this.imgButtons.TransparentColor = System.Drawing.Color.Magenta;
            this.imgButtons.Images.SetKeyName(0, "");
            this.imgButtons.Images.SetKeyName(1, "");
            //
            // pnlPages
            //
            this.pnlPages.BorderStyle = System.Windows.Forms.BorderStyle.FixedSingle;
            this.pnlPages.Dock = System.Windows.Forms.DockStyle.Fill;
            this.pnlPages.FixedPanel = System.Windows.Forms.FixedPanel.Panel1;
            this.pnlPages.IsSplitterFixed = true;
            this.pnlPages.Location = new System.Drawing.Point(0, 0);
            this.pnlPages.Name = "pnlPages";
            this.pnlPages.Orientation = System.Windows.Forms.Orientation.Horizontal;
            //
            // pnlPages.Panel1
            //
            this.pnlPages.Panel1.BackColor = System.Drawing.SystemColors.ButtonFace;
            this.pnlPages.Panel1MinSize = 0;
            this.pnlPages.Panel1Size = new System.Drawing.Size(200, 100);
            //
            // pnlPages.Panel2
            //
            this.pnlPages.Panel2.Controls.Add(this.pnlPage);
            this.pnlPages.Panel2MinSize = 0;
            this.pnlPages.Size = new System.Drawing.Size(200, 368);
            this.pnlPages.SplitterDistance = 100;
            this.pnlPages.TabIndex = 1;
            this.pnlPages.TitleString = "";
            //
            // pnlPage
            //
            this.pnlPage.AutoScroll = true;
            this.pnlPage.Dock = System.Windows.Forms.DockStyle.Fill;
            this.pnlPage.Location = new System.Drawing.Point(0, 0);
            this.pnlPage.Name = "pnlPage";
            this.pnlPage.Size = new System.Drawing.Size(198, 262);
            this.pnlPage.TabIndex = 0;
            //
            // tvwCategory
            //
            this.tvwCategory.Dock = System.Windows.Forms.DockStyle.Fill;
            this.tvwCategory.HideSelection = false;
            this.tvwCategory.ImageIndex = 0;
            this.tvwCategory.ImageList = this.imgCategory;
            this.tvwCategory.Location = new System.Drawing.Point(0, 0);
            this.tvwCategory.Name = "tvwCategory";
            this.tvwCategory.SelectedImageIndex = 0;
            this.tvwCategory.ShowNodeToolTips = true;
            this.tvwCategory.ShowRootLines = false;
            this.tvwCategory.Size = new System.Drawing.Size(168, 368);
            this.tvwCategory.TabIndex = 0;
            //
            // PropertyPanel
            //
            this.Controls.Add(this.pnlRight);
            this.Controls.Add(this.splBase);
            this.Controls.Add(this.pnlLeft);
            this.Name = "PropertyPanel";
            this.Size = new System.Drawing.Size(376, 368);
            this.Load += new System.EventHandler(this.Event_PropWndPanel_Load);
            this.pnlRight.ResumeLayout(false);
            this.pnlLeft.ResumeLayout(false);
            this.pnlPages.Panel2.ResumeLayout(false);
            this.pnlPages.ResumeLayout(false);
            this.ResumeLayout(false);

        }
        #endregion

        // オーナー
        private PropertyWindow _owner = null;

        // プロパティページ
        private PropertyPage _activePage    = null;

        // カスタムメニュー
        private CustomMenu _customMenu = new CustomMenu();

        // オーバレイ番号
        protected const int _CommentOverrayIconIdx = 1;

        // ViewManagerMessage
        protected Action<ViewManagerMessage> _sendMessageToViewManagerAction = null;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public PropertyPanel()
        {
            InitializeComponent();

            // カテゴリイメージ登録
            tvwCategory.ImageList = this.imgCategory;
            this.imgCategory.ColorDepth = ColorDepth.Depth32Bit;
            this.imgCategory.Images.Add("PropertyCategory1", LayoutEditor.ImageResMgr.GetManifestResourceIcon("PropertyCategory1.ico"));
            this.imgCategory.Images.Add("PropertyCategory2", LayoutEditor.ImageResMgr.GetManifestResourceIcon("PropertyCategory2.ico"));
            this.imgCategory.Images.Add("PropertyCategory3", LayoutEditor.ImageResMgr.GetManifestResourceIcon("PropertyCategory3.ico"));

            // ペイン種類毎のアイコン（有効状態・無効状態）
            foreach (PaneKind paneKind in Enum.GetValues(typeof(PaneKind)))
            {
                //if(paneKind == PaneKind.None)
                //{
                //    continue;
                //}

                this.imgCategory.Images.Add(PaneItemImageHelper.GetPaneImageKindAsString(paneKind, true), PaneItemImageHelper.GetPaneImage(paneKind, true));
                this.imgCategory.Images.Add(PaneItemImageHelper.GetPaneImageKindAsString(paneKind, false), PaneItemImageHelper.GetPaneImage(paneKind, false));
            }

            // オーバーレイアイコン
            this.imgCategory.Images.Add(PaneItemImageHelper.GetCommentOverrayIcon());
            int commentOverrayIconIndex = this.imgCategory.Images.Count - 1;
            OverlayIconHelper.ImageList_SetOverlayImage(this.imgCategory.Handle, commentOverrayIconIndex, _CommentOverrayIconIdx);

            // イベント登録
            tvwCategory.AfterSelect += new TreeViewEventHandler(Event_TvwCategory_AfterSelect);
            tvwCategory.ContextMenuPopup += new ContextMenuPopupEventHandler(Event_TvwCategory_ContextMenuPopup);

            // 拡張領域は無効化しておく
            pnlPages.Panel1Collapsed = true;
            this.RightPageExtraPanel = null;

            new MessageForwarder(pnlPage, WM.WM_MOUSEWHEEL);
        }

        /// <summary>
        /// オーナー。
        /// </summary>
        public PropertyWindow Owner
        {
            get { return _owner;  }
            set { _owner = value; }
        }

        /// <summary>
        /// ターゲット。
        /// </summary>
        public PaneGuiAdapterGroup Target
        {
            get { return _owner != null ? _owner.Target : null; }
        }

        public AppSetting AppSetting
        {
            get{ return Owner?.AppSetting; }
        }

        /// <summary>
        /// ViewManagerMessageを取得します。
        /// </summary>
        public Action<ViewManagerMessage> SendMessageToViewManagerAction
        {
            get { return _sendMessageToViewManagerAction; }
            set { _sendMessageToViewManagerAction = value; }
        }

        /// <summary>
        /// 左側領域幅。
        /// </summary>
        public int LeftWidth
        {
            get { return pnlLeft.Width;  }
            set { pnlLeft.Width = value; }
        }

        /// <summary>
        /// 拡張パネル領域
        /// </summary>
        protected Control RightPageExtraPanel
        {
            set
            {
                var oldPanel = pnlPages.Panel1.Controls.Count > 0 ? pnlPages.Panel1.Controls[0] : null;
                if (oldPanel != value)
                {
                    pnlPages.Panel1.Controls.Clear();

                    if (value == null)
                    {
                        pnlPages.Panel1Collapsed = true;
                    }
                    else
                    {
                        // 収まるように高さを調整。
                        const int SplitterMergine = 8;
                        pnlPages.SplitterDistance = value.Height + SplitterMergine;

                        value.Dock = DockStyle.Fill;
                        pnlPages.Panel1.Controls.Add(value);
                        pnlPages.Panel1Collapsed = false;
                    }
                }
            }
        }

        /// <summary>
        /// オーナー。
        /// </summary>
        protected PropertyPage _ActivePage
        {
            get { return _activePage; }
        }

        /// <summary>
        /// フォーム更新。
        /// </summary>
        public virtual void UpdateForm()
        {
            // 自身の更新
            UpdateProperty();

            // アクティブページ更新
            if (_activePage != null)
            {
                _activePage.UpdateForm();
            }
        }

        /// <summary>
        /// ページIDの取得。
        /// </summary>
        public PropertyPageID GetPageID()
        {
            if(_activePage != null)
            {
                return _activePage.PageID;
            }
            return PropertyPageID.Null;
        }

        public PropertyPageID ActivePageID
        {
            get{ return GetPageID();}
        }

        /// <summary>
        /// コピーが可能か。
        /// </summary>
        public bool CanCopy()
        {
            if(_activePage != null)
            {
                return _activePage.CanCopy();
            }
            return false;
        }


        /// <summary>
        /// コピー。
        /// </summary>
        public Object Copy()
        {
            Debug.Assert(CanCopy());
            if(_activePage != null)
            {
                return _activePage.Copy();
            }
            return null;
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public void Paste(Object pasteObject)
        {
            Debug.Assert((_activePage != null) && (pasteObject != null));
            _activePage.Paste(pasteObject);
        }

        /// <summary>
        /// カテゴリビュー。
        /// </summary>
        protected TreeView CategoryView
        {
            get { return this.tvwCategory; }
        }

        /// <summary>
        /// カテゴリイメージリスト。
        /// </summary>
        protected ImageList CategoryImageList
        {
            get { return this.imgCategory; }
        }

        /// <summary>
        /// カテゴリ登録。
        /// </summary>
        protected static PropWndTreeNode RegisterCategory(string textStrID, int imageIndex,CreatePropertyPageFunction func, object obj, bool isOverwrite = false)
        {
            Debug.Assert(func != null);

            // ノード作成
            PropWndTreeNode node = new PropWndTreeNode(StringResMgr.Get(textStrID), imageIndex, imageIndex, func, obj, isOverwrite);
            return node;
        }

        /// <summary>
        /// ページプラグインカテゴリ登録。
        /// </summary>
        protected static PropWndTreeNode RegisterPluginCategory(string nodeName, int imageIndex,CreatePropertyPageFunction func, object obj)
        {
            Debug.Assert(func != null);

            // ノード作成
            PropWndTreeNode node = new PropWndTreeNode(nodeName, imageIndex, imageIndex, func, obj);
            return node;
        }

        /// <summary>
        /// コンテキストメニューを表示。
        /// </summary>
        public virtual void ShowContextMenu(Point point)
        {
            ContextMenu menu = new ContextMenu();
            _customMenu.SetupMenu(menu);
            _customMenu.ItemImageList = imgButtons;
            menu.Show(this, point);
        }

        /// <summary>
        /// ターゲット変更時の更新処理
        /// </summary>
        public virtual void OnTargetChangeCategoryUpdate()
        {
            // デフォルトでは何もしません。
        }

        /// <summary>
        /// ターゲット変更時の更新処理
        /// </summary>
        public void OnTargetChangePageUpdate()
        {
            // 状態修復処理
            foreach( TreeNode newSelNode in tvwCategory.Nodes )
            {
                PropertyPage page = newSelNode.Tag as PropertyPage;
                if( page != null )
                {
                    page.OnTargetChangeHandler();
                }
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                if (components != null)
                {
                    components.Dispose();
                }
            }
            base.Dispose(disposing);
        }

        /// <summary>
        /// 明示的にすべてのページを破棄します。
        /// </summary>
        public List<PropertyPage> ResetAllNodesAndGatherOldPages()
        {
            List<PropertyPage> pages = new List<PropertyPage>();
            foreach (TreeNode node in tvwCategory.Nodes)
            {
                ResetNodesAndGatherOldNodePages_(node, pages);
            }

            tvwCategory.Nodes.Clear();

            return pages;
        }

        /// <summary>
        /// ノードのページを破棄します。
        /// </summary>
        private void ResetNodesAndGatherOldNodePages_(TreeNode node, List<PropertyPage> pages)
        {
            PropertyPage page = node.Tag as PropertyPage;
            if (page != null)
            {
                node.Tag = null;
                pages.Add(page);
            }

            foreach (TreeNode childNode in node.Nodes)
            {
                ResetNodesAndGatherOldNodePages_(childNode, pages);
            }

            node.Nodes.Clear();
        }

        protected virtual void UpdateProperty()
        {
            // 必要なら派生クラスで実装する
            // 抽象にしないのはフォームデザイナでインスタンス化させる為
        }

        /// <summary>
        /// ダイアログのサイズ計算
        /// </summary>
        /// <param name="in_page">計算元のページ</param>
        public void SetCalculationDialogSize(PropertyPage in_page)
        {
            //
            // フォームサイズの自動計算機能は現在無効化されています。
            //
#if false
           PropertyPage	page	= ( in_page == null ) ? _activePage : in_page;
            //	表示サイズ変更
            if( page != null )
            {
                //	サイズ変更ページの場合かつデフォルトが設定されているときはデフォルトを使用
                Size	size	= (!page.Sizable || page.DefaultPageSize == Size.Empty) ? page.Size : page.DefaultPageSize;
                // TODO: X=12,Y=16の決めうちをどうするか
                int		x		= size.Width + 12, y	= size.Height + 16;
                for( Control parent = page; parent != null && parent != this.Owner; parent = parent.Parent )
                {
                    x	+= parent.Location.X;
                    y	+= parent.Location.Y;
                }
                //	ダイアログ枠部分の考慮
                Size	result	= new Size( this.Owner.Size.Width - this.Owner.ClientSize.Width + x, this.Owner.Size.Height - this.Owner.ClientSize.Height + y );
                if( !this.Owner.Size.Equals( result ) )
                    this.Owner.Size	= result;
            }
            else
            {
                //	ページのデフォルトサイズが360なので、変更するときはここも変更する必要あり
                int		x		= 360 + 12;
                for( Control parent = pnlRight; parent != null && parent != this.Owner; parent = parent.Parent )
                {
                    x	+= parent.Location.X;
                }
                Size	result	= new Size( this.Owner.Size.Width - this.Owner.ClientSize.Width + x, this.Owner.MinimumSize.Height );
                if( !this.Owner.Size.Equals( result ) )
                    this.Owner.Size	= result;
            }
#endif
        }

        /// <summary>
        /// ダイアログのサイズからサイズ変更ページを設定
        /// </summary>
        public void SetCalculationPageSize()
        {
            if( _activePage == null || !_activePage.Sizable )
                return;

            Size	size	= this.Owner.ClientSize;
            int		x		= size.Width - 12, y	= size.Height - 16;
            for( Control parent = _activePage; parent != null && parent != this.Owner; parent = parent.Parent )
            {
                x	-= parent.Location.X;
                y	-= parent.Location.Y;
            }
            DockStyle	style	= DockStyle.Fill;
            if( _activePage.DefaultPageSize.Width > x )
            {
                y	-= SystemInformation.HorizontalScrollBarHeight;
            }
            if( _activePage.DefaultPageSize.Height > y )
            {
                x	-= SystemInformation.VerticalScrollBarWidth;
            }
            if( _activePage.DefaultPageSize.Width > x )
            {
                style	= DockStyle.Left;
            }
            if( _activePage.DefaultPageSize.Height > y )
            {
                style	= (style == DockStyle.Fill ) ? DockStyle.Top : DockStyle.None;
            }
            this._activePage.Dock	= style;
        }

        /// <summary>
        /// 初期値を同じページにする
        /// </summary>
        /// <param name="panel"></param>
        public void InitializeSamePage( PropertyPanel panel )
        {
            Debug.Assert( this.GetType().Equals( panel.GetType() ) );
            if( this.CategoryView.SelectedNode != null || panel.CategoryView.SelectedNode == null )
                return;
            foreach( PropWndTreeNode node in this.CategoryView.Nodes )
            {
                TreeNode	result	= node.SearchSameNode( (PropWndTreeNode)panel.CategoryView.SelectedNode );
                if( result == null )
                    continue;

                this.CategoryView.SelectedNode	= result;
                return;
            }
            //	見つからなかった
            Debug.Assert( false );
        }

        //-----------------------------------------------------
        #region イベントハンドラ
        /// <summary>
        /// イベントハンドラ。
        /// </summary>
        private void Event_TvwCategory_ContextMenuPopup(
            object sender, ContextMenuPopupEventArgs e )
        {
            ShowContextMenu( PointToClient( Cursor.Position ) );
        }

        /// <summary>
        /// イベントハンドラ。
        /// </summary>
        private void Event_TvwCategory_AfterSelect( object sender, TreeViewEventArgs e )
        {
            SetActiveCategory(e.Node as PropWndTreeNode);
        }

        /// <summary>
        /// アクティブカテゴリが変更された時の処理です。
        /// </summary>
        protected virtual void OnChangedActiveCategory(PropWndTreeNode categoryNode, PropertyPage newActivePage)
        {
        }

        /// <summary>
        /// アクティブカテゴリを設定します。
        /// </summary>
        protected void SetActiveCategory(PropWndTreeNode categoryNode)
        {
            PropertyPage page = categoryNode.Tag as PropertyPage;
            if( page == null )
            {
                // 作成が重いのでウェイト表示...
                using( WaitCursor wait = new WaitCursor() )
                {
                    page = categoryNode.GetPage();
                }

                page.Visible = false;
                page.Owner = this;
                page.Parent = pnlPage;

                page.InitializeForm();
                categoryNode.Tag = page;
            }

            //	表示サイズ変更
            if( _activePage != page )
            {
                SetCalculationDialogSize( page );
            }

            // アクティブページ切り替え
            if( _activePage == null )
            {
                _activePage = page;
                _activePage.Visible = true;
            }
            else if( _activePage != page )
            {
                page.Visible = true;
                _activePage.OnHide();
                _activePage.Visible = false;
                _activePage = page;
            }

            new MessageForwarder(_activePage, WM.WM_MOUSEWHEEL);

            // サイズ再計算。
            SetCalculationPageSize();

            // 位置が狂うときがあるので明示的に位置ゼロを設定する。
            page.Location = Point.Empty;

            // アクティブページ更新処理
            OnChangedActiveCategory(categoryNode, _activePage);

            // 更新
            _activePage.UpdateForm();

            // 自分自身の更新
            UpdateProperty();
        }

        /// <summary>
        /// ロードハンドラ
        /// </summary>
        private void Event_PropWndPanel_Load( object sender, EventArgs e )
        {
            // カテゴリーをnullに設定します。
            // 初期化処理を行うために、初期化処理時に、null以外に設定している場合が
            // あるためです。
            //
            // CategoryView.SelectedNodeはこの後の、
            // VisibleChangeのタイミングで適切に設定されます。
            CategoryView.SelectedNode = null;
        }

        /// <summary>
        /// 非アクティブになった際に呼び出されるメソッド。
        /// </summary>
        public void OnHide()
        {
            if (_activePage != null)
            {
                _activePage.OnHide();
            }
        }

        #endregion

        //-----------------------------------------------------
        #region ISceneModifyListener メンバ

        public virtual void OnSceneModifyHandler( object sender, SceneModifyEventArgs e )
        {
            if (_activePage != null)
            {
                _activePage.OnSceneModifyHandler(sender, e);
            }
        }

        #endregion
    }

    /// <summary>
    /// オブジェプロパティパネル用のツリークラス
    /// </summary>
    public class PropWndTreeNode : System.Windows.Forms.TreeNode
    {
        private readonly CreatePropertyPageFunction _initializeFunc	= null;
        private readonly object	_arg	= null;
        private PropertyPage	_page	= null;
        private readonly bool _isOverwrite = false;

        /// <summary>
        /// ページを取得
        /// </summary>
        public PropertyPage Page
        {
            get { return _page; }
        }

        /// <summary>
        /// ページが強制的に無効か？
        /// </summary>
        public bool ForceDisabled
        {
            get;
            set;
        }

        /// <summary>
        /// テクスチャだけ上書きをしているか
        /// </summary>
        public bool IsTextureOverwritingEnabled
        {
            get;
            set;
        }

        /// <summary>
        /// 上書き設定か
        /// </summary>
        public bool IsOverwrite
        {
            get { return _isOverwrite; }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public PropWndTreeNode( System.String text , System.Int32 imageIndex , System.Int32 selectedImageIndex,
            CreatePropertyPageFunction func, object arg , bool isOverwrite = false)
            : base( text, imageIndex, selectedImageIndex )
        {
            _initializeFunc	= func;
            _arg			= arg;
            this.Tag = null;
            this.ForceDisabled = false;
            this.IsTextureOverwritingEnabled = false;
            this._isOverwrite = isOverwrite;
        }

        /// <summary>
        /// 設定関数で初期化処理
        /// <remarks>
        /// 初回アクセス時に生成して、内部でキャッシュするようになりました。
        /// </remarks>
        /// </summary>
        public PropertyPage GetPage()
        {
            Debug.Assert( _initializeFunc != null );
            if(_page == null)
            {
                _page = _initializeFunc(_arg);
                _page.ForceDisabled = ForceDisabled;
                _page.IsTextureOverwritingEnabled = IsTextureOverwritingEnabled;
                _page.IsOverwrite = IsOverwrite;
            }

            return _page;
        }

        /// <summary>
        /// ノードの対象を更新します。
        /// </summary>
        public void UpdateTarget(IPane pane)
        {
            this.GetPage().UpdateTarget(pane);
            foreach (PropWndTreeNode chidlNode in this.Nodes)
            {
                chidlNode.UpdateTarget(pane);
            }
        }

        /// <summary>
        /// 与えたれたノードが持つページと同じものを自分のツリー上から探す
        /// </summary>
        /// <param name="in_node"></param>
        /// <returns></returns>
        public TreeNode SearchSameNode( PropWndTreeNode in_node )
        {
            if( in_node._initializeFunc.Equals( this._initializeFunc ) )
                return this;
            foreach( PropWndTreeNode node in this.Nodes )
            {
                TreeNode	result	= node.SearchSameNode( in_node );
                if( result != null )
                    return result;
            }
            return null;
        }

    }

    /// <summary>
    /// ページ初期化用
    /// </summary>
    public delegate PropertyPage CreatePropertyPageFunction(object arg);
}
