﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Data;
using System.Text;
using System.Windows.Forms;

namespace LayoutEditor.Forms.ToolWindows.PropertyEditWindow
{
    using LayoutEditor.Controls;
    using LayoutEditor.Utility;
    using LayoutEditor.Forms.ToolWindows.PropertyEditWindow.Adapters;
    using LECore.Structures.Nsrif.Attributes;
    using LECore.Structures;
    using App = LayoutEditor;
    using LECore;
    using System.Linq;
    using LECore.Structures.Core;
    using src.Controls;

    public partial class TexCombStagePanel : UserControl
    {
        // 編集用データ
        private readonly TevDirectStage _data = new TevDirectStage(null, 0, false);
        private IRevHWMaterial _ownerMaterial = null;

        /// <summary>データ変更イベント。</summary>
        public event EventHandler DataChanged = null;

        ///
        protected const int MaxSourceCount = 3;
        private int _stageNo = -1;

        IAnmAttribute IndirectScaleAttr
        {
            get
            {
                return _ownerMaterial?.GetAnimationTargetAttributeSet(ParamaterKind.Animation_IndirectTextureSRT)?.ElementAt(1);
            }
        }

        IAnmAttribute IndirectRotateAttr
        {
            get
            {
                return _ownerMaterial?.GetAnimationTargetAttributeSet(ParamaterKind.Animation_IndirectTextureSRT)?.ElementAt(2);
            }
        }

        /// <summary>
        /// アニメーションマークを利用するか
        /// </summary>
        public bool UseAnimationMark
        {
            get;
            set;
        }

        /// <summary>
        /// スタティックコンストラクタ
        /// </summary>
        static TexCombStagePanel()
        {
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public TexCombStagePanel()
        {
            InitializeComponent();

            if (DesignMode)
            {
                _grpIndirect.Visible = LECore.LayoutEditorCore.PlatformDetail.IndirectTextureEnabled;
            }

            // コンテキストメニューを追加する
            AnimationMarkContextMenuBuilder.BuildMultiControlMenu(_lblIndirectScale, _nudIndirectScale.AnimationMarkArray);

            // アニメーションマークを関連付ける
            _lblIndirectRotate.BindAnimationMark(_nudIndirectRotate.AnimationMarkInst);
        }

        /// <summary>
        /// ViewManagerへのメッセージハンドラを設定します。
        /// </summary>
        public void SetViewManagerMessageHandler(Action<ViewManagerMessage> handler)
        {
            _nudIndirectScale.SetViewManagerMessageHandler(handler);
            _nudIndirectRotate.SetViewManagerMessageHandler(handler);
        }

        /// <summary>
        /// データ。
        /// </summary>
        public TevDirectStage Data
        {
            get { return _data; }
        }

        protected virtual void SetCombineModeCombo_(UIComboBox combo, TevDirectStage stage, int stageNo)
        {
        }

        public void SetData(int stageNo, TevDirectStage stage, IRevHWMaterial ownerMaterial)
        {
            _stageNo = stageNo;
            _ownerMaterial = ownerMaterial;

            // 設定をコピー
            _data.Set(stage);

            // コンボ項目を更新します。
            SetCombineModeCombo_(cmbCombineMode, stage, _stageNo);

            // プロパティ更新
            UpdateProperty();
        }

        /// <summary>
        /// プロパティ初期化。
        /// </summary>
        public virtual void InitializeProperty()
        {
            cmbCombineMode.SelectedIndexChanged += Event_CmbCombineMode_SelectedIndexChanged;
            cmbCombineMode.ExpandDropDownWidth();
        }

        /// <summary>
        /// プロパティ更新。
        /// </summary>
        public virtual void UpdateProperty()
        {
            cmbCombineMode.SetSelectedItemData(_data.CombineMode);

            // アニメーションマークとコンテキストメニューの有効無効
            List<IAnimationMarkUsable> animCtrlList = GetAnimationMarkUsableControl();
            foreach (IAnimationMarkUsable ctrl in animCtrlList)
            {
                ctrl.UseAnimationMark = this.UseAnimationMark;
            }

            _grpIndirect.Enabled = _data.IsIndirectStage;
            if (_data.IsIndirectStage)
            {
                _nudIndirectScale.FVec2Value = _data.IndirectScale;
                _nudIndirectRotate.Value = (Decimal)_data.IndirectRotate;
            }

            // ヘルプ用ツールチップ
            switch (_data.CombineMode)
            {
                case AttrCombineMode.Indirect: this._ttiMain.SetToolTip(this.cmbCombineMode, LECoreStringResMgr.Get("PROP_MAT_TEXCOMB_STAGE_INDIRECT_DESC")); break;
                case AttrCombineMode.BlendIndirect: this._ttiMain.SetToolTip(this.cmbCombineMode, LECoreStringResMgr.Get("PROP_MAT_TEXCOMB_STAGE_BLENDINDIRECT_DESC")); break;
                case AttrCombineMode.EachIndirect: this._ttiMain.SetToolTip(this.cmbCombineMode, LECoreStringResMgr.Get("PROP_MAT_TEXCOMB_STAGE_EACHINDIRECT_DESC")); break;
                default:
                    if (cmbCombineMode.SelectedItem != null)
                    {
                        _ttiMain.SetToolTip(cmbCombineMode, cmbCombineMode.SelectedItem.ToString());
                    }
                    break;
            }

            // アニメーションマークで利用するアトリビュートを関連付け
            _nudIndirectScale.SetTargetAttribute(IndirectScaleAttr);
            _nudIndirectRotate.SetTargetAttribute(IndirectRotateAttr);

            // アニメーションマークの更新
            foreach (IAnimationMarkUsable ctrl in animCtrlList)
            {
                ctrl.UpdateAnimationMark();
            }
        }

        /// <summary>
        /// アニメーションマークが利用可能なコントロールの一覧を取得する。
        /// </summary>
        protected List<IAnimationMarkUsable> GetAnimationMarkUsableControl()
        {
            List<IAnimationMarkUsable> list = new List<IAnimationMarkUsable>();

            list.Add(_lblIndirectScale);
            list.Add(_lblIndirectRotate);
            list.Add(_nudIndirectScale);
            list.Add(_nudIndirectRotate);

            return list;
        }

        /// <summary>
        /// 連動ボタンを更新します
        /// </summary>
        public void UpdateLinkButton()
        {
            if (_data == null) return;

            _nudIndirectScale.UpdateLinkButton(_data.IndirectScale.X == _data.IndirectScale.Y);
        }

        /// <summary>
        /// データ変更ハンドラ。
        /// </summary>
        protected virtual void OnDataChanged(EventArgs e)
        {
            if (this.DataChanged != null)
            {
                this.DataChanged(this, e);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void ShowInvalidSourcesErrorDialog()
        {
            MessageBox.Show(
                            StringResMgr.Get( "PROP_MAT_TEV_MSG_INVALID_SOURCES" ),
                            StringResMgr.Get( "PROP_ERROR_DLG_TITLE" ),
                            MessageBoxButtons.OK,
                            MessageBoxIcon.Error );
        }

        #region イベントハンドラ

        /// <summary>
        /// コンバインモード変更
        /// </summary>
        private void Event_CmbCombineMode_SelectedIndexChanged(object sender, EventArgs e)
        {
            AttrCombineMode prevCombineMode = _data.CombineMode;
            _data.CombineMode = (AttrCombineMode)cmbCombineMode.SelectedItemData;

            OnDataChanged(EventArgs.Empty);
        }

        /// <summary>
        /// インダイレクトスケール
        /// </summary>
        private void Event_NudIndirectScale_OnFVec2Edit(object sender, FVec2EditEventArgs args)
        {
            if (_data.IndirectScale != args.Value)
            {
                _data.IndirectScale = args.Value;
                OnDataChanged(EventArgs.Empty);
            }
        }

        /// <summary>
        /// インダイレクト回転
        /// </summary>
        private void Event_NudIndirectRotate_ValueChanged(object sender, EventArgs e)
        {
            if (_data.IndirectRotate != (float)_nudIndirectRotate.Value)
            {
                _data.IndirectRotate = (float)_nudIndirectRotate.Value;
                OnDataChanged(EventArgs.Empty);
            }
        }

        #endregion
    }

    #region TevColorStagePanel
    /// <summary>
    /// カラーステージパネルクラス。
    /// </summary>
    public sealed class TexCombColorStagePanel : TexCombStagePanel
    {
        /// <summary>
        /// プロパティ初期化。
        /// </summary>
        public override void InitializeProperty()
        {
            base.InitializeProperty();
        }

        protected override void SetCombineModeCombo_(UIComboBox combo, TevDirectStage stage, int stageNo)
        {
            // コンボ項目を更新します。
            combo.Items.Clear();
            foreach (var combDesc in LECore.LayoutEditorCore.PlatformDetail.GetTextreuCombinerInfoColor(stageNo))
            {
                combo.AddItem(combDesc.Descriptions, combDesc.CombineMode);
            }
        }
    }
    #endregion

    #region TevColorStagePanel
    /// <summary>
    /// アルファステージパネルクラス。
    /// </summary>
    public sealed class TexCombAlphaStagePanel : TexCombStagePanel
    {
        /// <summary>
        /// プロパティ初期化。
        /// </summary>
        public override void InitializeProperty()
        {
            cmbCombineMode.Items.Clear();
            foreach (var combDesc in LECore.LayoutEditorCore.PlatformDetail.GetTextreuCombinerInfoAlpha())
            {
                cmbCombineMode.AddItem(combDesc.Descriptions, combDesc.CombineMode);
            }

            // 非表示
            _grpIndirect.Visible = false;

            base.InitializeProperty();
        }
    }
    #endregion
}
