﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Windows.Forms;


namespace LayoutEditor.Forms.ToolWindows.LayoutWindow
{
    using LECore.Util;

    /// <summary>
    /// MainViewBar の概要の説明です。
    /// </summary>
    public class MainViewBar : System.Windows.Forms.UserControl
    {
        #region フィールド
        #region デザイナ変数

        private System.ComponentModel.Container   components = null;
        private System.Windows.Forms.Button       btnZoomOut;
        private LayoutEditor.Controls.UITrackBar  trackBar;
        private System.Windows.Forms.Button       btnZoomIn;

        #endregion
        private System.Windows.Forms.ComboBox _cmbMagnify;



        MainView             _mainView = null;
        #endregion

        #region プロパティ
        public MainView             View
        {
            set
            {
                _mainView = value;
                // MainView 倍率更新ハンドラに登録
                _mainView.MagnifyChanged += new EventHandler(Event_View_MagnifyChanged);
                UpdateComponent_();
            }
        }

        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public MainViewBar()
        {
            InitializeComponent();
        }

        /// <summary>
        /// 使用されているリソースに後処理を実行します。
        /// </summary>
        protected override void Dispose( bool disposing )
        {
            if( disposing )
            {
                if(components != null)
                {
                    components.Dispose();
                }
            }
            base.Dispose( disposing );
        }

        #region プライベート・メソッド

        /// <summary>
        /// UIコンポーネントの値を更新します。
        /// </summary>
        void UpdateComponent_()
        {
            if (this.DesignMode)
            {
                return;
            }
            // 倍率ラベルの更新
            _cmbMagnify.Text = string.Format("{0:f1}%", _mainView.Magnify * 100.0f);
            // トラックバーの値を更新
            trackBar.Value   = MagnifyTrackBarValue_( _mainView.Magnify );
        }

        /// <summary>
        /// 倍率をトラックバー値に変換。
        /// </summary>
        int MagnifyTrackBarValue_(float magnify)
        {
            Range  valRange = new Range(trackBar.Minimum, trackBar.Maximum);
            RangeF magRange = _mainView.MagnifyRange;

            int val = (int)MathUtil.Interpolate(
                magRange.Min,
                magRange.Max,
                magnify,
                valRange.Min,
                valRange.Max);

            return valRange.Truncate(val);
        }

        /// <summary>
        /// トラックバー値を倍率に変換。
        /// </summary>
        float TrackBarValueToMagnify_(int val)
        {
            Range  valRange = new Range(trackBar.Minimum, trackBar.Maximum);
            RangeF magRange = _mainView.MagnifyRange;

            float magnify = MathUtil.Interpolate(
                valRange.Min,
                valRange.Max,
                val,
                magRange.Min,
                magRange.Max
                );

            return magRange.Truncate(magnify);
        }

        #endregion

        #region コンポーネント デザイナで生成されたコード
        /// <summary>
        /// デザイナ サポートに必要なメソッドです。このメソッドの内容を
        /// コード エディタで変更しないでください。
        /// </summary>
        private void InitializeComponent()
        {
            System.ComponentModel.ComponentResourceManager resources = new System.ComponentModel.ComponentResourceManager(typeof(MainViewBar));
            this.btnZoomOut = new System.Windows.Forms.Button();
            this.btnZoomIn = new System.Windows.Forms.Button();
            this._cmbMagnify = new System.Windows.Forms.ComboBox();
            this.trackBar = new LayoutEditor.Controls.UITrackBar();
            ((System.ComponentModel.ISupportInitialize)(this.trackBar)).BeginInit();
            this.SuspendLayout();
            //
            // btnZoomOut
            //
            this.btnZoomOut.BackColor = System.Drawing.SystemColors.GradientInactiveCaption;
            this.btnZoomOut.FlatStyle = System.Windows.Forms.FlatStyle.Flat;
            this.btnZoomOut.ForeColor = System.Drawing.SystemColors.ActiveCaption;
            this.btnZoomOut.Image = ((System.Drawing.Image)(resources.GetObject("btnZoomOut.Image")));
            this.btnZoomOut.Location = new System.Drawing.Point(191, 3);
            this.btnZoomOut.Name = "btnZoomOut";
            this.btnZoomOut.Size = new System.Drawing.Size(20, 20);
            this.btnZoomOut.TabIndex = 2;
            this.btnZoomOut.UseVisualStyleBackColor = false;
            this.btnZoomOut.Click += new System.EventHandler(this.Event_BtnZoomOut_Click);
            //
            // btnZoomIn
            //
            this.btnZoomIn.BackColor = System.Drawing.SystemColors.GradientInactiveCaption;
            this.btnZoomIn.FlatStyle = System.Windows.Forms.FlatStyle.Flat;
            this.btnZoomIn.ForeColor = System.Drawing.SystemColors.ActiveCaption;
            this.btnZoomIn.Image = ((System.Drawing.Image)(resources.GetObject("btnZoomIn.Image")));
            this.btnZoomIn.Location = new System.Drawing.Point(215, 3);
            this.btnZoomIn.Name = "btnZoomIn";
            this.btnZoomIn.Size = new System.Drawing.Size(20, 20);
            this.btnZoomIn.TabIndex = 3;
            this.btnZoomIn.UseVisualStyleBackColor = false;
            this.btnZoomIn.Click += new System.EventHandler(this.Event_BtnZoomIn_Click);
            //
            // _cmbMagnify
            //
            this._cmbMagnify.DropDownWidth = 72;
            this._cmbMagnify.Items.AddRange(new object[] {
            "800.0%",
            "400.0%",
            "200.0%",
            "150.0%",
            "100.0%",
            "50.0%",
            "25.0%",
            "12.5%"});
            this._cmbMagnify.Location = new System.Drawing.Point(3, 3);
            this._cmbMagnify.Margin = new System.Windows.Forms.Padding(0);
            this._cmbMagnify.MaxDropDownItems = 10;
            this._cmbMagnify.Name = "_cmbMagnify";
            this._cmbMagnify.Size = new System.Drawing.Size(64, 20);
            this._cmbMagnify.TabIndex = 0;
            this._cmbMagnify.SelectedIndexChanged += new System.EventHandler(this.Event_CmbMagnify_SelectedIndexChanged);
            this._cmbMagnify.KeyDown += new System.Windows.Forms.KeyEventHandler(this.Event_CmbMagnify_KeyDown);
            //
            // trackBar
            //
            this.trackBar.Location = new System.Drawing.Point(72, 3);
            this.trackBar.Maximum = 100;
            this.trackBar.Name = "trackBar";
            this.trackBar.Size = new System.Drawing.Size(113, 20);
            this.trackBar.TabIndex = 1;
            this.trackBar.Scroll += new System.EventHandler(this.Event_TrackBar_Scroll);
            this.trackBar.MouseDown += new System.Windows.Forms.MouseEventHandler(this.Event_TrackBar_MouseDown);
            //
            // MainViewBar
            //
            this.Controls.Add(this._cmbMagnify);
            this.Controls.Add(this.btnZoomOut);
            this.Controls.Add(this.trackBar);
            this.Controls.Add(this.btnZoomIn);
            this.Name = "MainViewBar";
            this.Size = new System.Drawing.Size(239, 26);
            this.VisibleChanged += new System.EventHandler(this.MainViewBar_VisibleChanged);
            ((System.ComponentModel.ISupportInitialize)(this.trackBar)).EndInit();
            this.ResumeLayout(false);

        }
        #endregion

        #region UIイベントハンドラ

        /// <summary>
        /// - ボタン
        /// </summary>
        private void Event_BtnZoomOut_Click(object sender, System.EventArgs e)
        {
            // 倍率を減らす
            _mainView.ZoomOut();
        }

        /// <summary>
        /// + ボタン
        /// </summary>
        private void Event_BtnZoomIn_Click(object sender, System.EventArgs e)
        {
            // 倍率を増やす
            _mainView.ZoomIn();
        }

        /// <summary>
        /// バー操作
        /// </summary>
        private void Event_TrackBar_Scroll(object sender, System.EventArgs e)
        {
            _mainView.Magnify = TrackBarValueToMagnify_(trackBar.Value);
        }

        /// <summary>
        /// MainView 倍率変更 イベントハンドラ。
        /// </summary>
        private void Event_View_MagnifyChanged(object sender, EventArgs e)
        {
            UpdateComponent_();
        }

        /// <summary>
        /// トラックバークリック
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void Event_TrackBar_MouseDown(object sender, System.Windows.Forms.MouseEventArgs e)
        {
            // 右クリック：倍率リセット
            if( e.Button == MouseButtons.Right )
            {
                _mainView.Magnify = 1.0f;
            }
        }

        /// <summary>
        /// パーセント表記倍率文字列から倍率を取得します。
        /// </summary>
        /// <param name="str">パーセント表記倍率文字列</param>
        /// <param name="magValue">倍率(0.0 - 1.0)失敗した場合は、1.0が返ります。</param>
        /// <returns>成否</returns>
        bool GetMagnityFromPercentString_( string percentStr, out float magValue )
        {
            try
            {
                magValue = Convert.ToSingle( percentStr.Replace( "%", "" ) ) / 100.0f;
                return true;
            }
            catch
            {
                magValue = 1.0f;
                return false;
            }
        }


        /// <summary>
        /// 倍率コンボボック変更ハンドラ
        /// </summary>
        private void Event_CmbMagnify_SelectedIndexChanged(object sender, System.EventArgs e)
        {
            ComboBox cmb = sender as ComboBox;

            float newMagnity;
            if( GetMagnityFromPercentString_( cmb.Text, out newMagnity ) )
            {
                _mainView.Magnify = newMagnity;
            }
            else
            {
                // 不正な文字列が入力された場合は、文字列を現在の状態に戻します。
                UpdateComponent_();
            }
        }

        /// <summary>
        /// コンボボックキーダウンハンドラ
        /// Enterキーが押されていたら、倍率の評価を行います。
        /// </summary>
        private void Event_CmbMagnify_KeyDown(object sender, System.Windows.Forms.KeyEventArgs e)
        {
            if( e.KeyCode == Keys.Enter )
            {
               ComboBox cmb = sender as ComboBox;
                float newMagnity;
                if( GetMagnityFromPercentString_( cmb.Text, out newMagnity ) )
                {
                    _mainView.Magnify = newMagnity;
                }
                else
                {
                    // 不正な文字列が入力された場合は、文字列を現在の状態に戻します。
                    UpdateComponent_();
                }
            }
        }

        /// <summary>
        /// 表示・非表示変更ハンドラ
        /// </summary>
        private void MainViewBar_VisibleChanged(object sender, EventArgs e)
        {
            // レイアウトの初回読み込み時、MainViewBarが非表示状態だと_cmbMagnify.Textが更新されないため、
            // VisibleChangedのタイミングでUIを更新します。
            if (this.Visible)
            {
                UpdateComponent_();
            }
        }

        #endregion
    }
}
