﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.ComponentModel;
using System.Drawing;
using System.Diagnostics;
using System.Windows.Forms;
using System.Collections.Generic;

namespace LayoutEditor.Forms.ToolWindows
{
    using Structures.SerializableObject;
    using Controls;

    using LECore.Structures;
    using LECore.Structures.Core;


    /// <summary>
    /// LEToolWindow の概要の説明です。
    /// </summary>
    public class LEToolWindow :
        WeifenLuo.WinFormsUI.Docking.DockContent,
        ILEToolWindow,
        IViewManagerMessageSender
    {
        #region デザイナ関連コード
        #region Windows フォーム デザイナで生成されたコード

        /// <summary>
        /// デザイナ サポートに必要なメソッドです。このメソッドの内容を
        /// コード エディタで変更しないでください。
        /// </summary>
        private void InitializeComponent()
        {
            this.SuspendLayout();
            //
            // LEToolWindow
            //
            this.AutoScaleBaseSize = new System.Drawing.Size(5, 12);
            this.ClientSize = new System.Drawing.Size(483, 395);
            this.DockAreas = ((WeifenLuo.WinFormsUI.Docking.DockAreas)(((((WeifenLuo.WinFormsUI.Docking.DockAreas.Float | WeifenLuo.WinFormsUI.Docking.DockAreas.DockLeft)
            | WeifenLuo.WinFormsUI.Docking.DockAreas.DockRight)
            | WeifenLuo.WinFormsUI.Docking.DockAreas.DockTop)
            | WeifenLuo.WinFormsUI.Docking.DockAreas.DockBottom)));
            this.Font = new System.Drawing.Font("MS UI Gothic", 9F, System.Drawing.FontStyle.Regular, System.Drawing.GraphicsUnit.Point, ((byte)(128)));
            this.FormBorderStyle = System.Windows.Forms.FormBorderStyle.FixedToolWindow;
            this.HideOnClose = true;
            this.Name = "LEToolWindow";
            this.ShowHint = WeifenLuo.WinFormsUI.Docking.DockState.Float;
            this.ShowInTaskbar = false;
            this.Text = "LEToolWindow";
            this.DockStateChanged += new System.EventHandler(this.LEToolWindow_DockStateChanged);
            this.ResumeLayout(false);

        }
        #endregion

        /// <summary>
        /// 必要なデザイナ変数です。
        /// </summary>
        private System.ComponentModel.Container components = null;
        #endregion デザイナ関連コード

        public LEToolWindow()
        {
            // これを書き換えるときは DefaultValue も書き換えること
            HideOnClose = true;
            ShowHint = WeifenLuo.WinFormsUI.Docking.DockState.Float;

            //
            // Windows フォーム デザイナ サポートに必要です。
            //
            InitializeComponent();
            //_stickyWindow = new common.StickyWindow( this );
        }

        // DockPanel Suite と組み合わせたときに例外が起きることがある。
        // 直接触らせない
        [Obsolete]
        public new Form Owner { get { return base.Owner; } set { base.Owner = value; } }


        private Size _minimumSize = new Size(0, 0);
        // DockPanel Suite と組み合わせると表示が崩れることがあるので使用しないこと
        // TODO: デザイナからは見えるので何とかしたい。継承したクラスでは、設定しても取り消される模様
        [Obsolete()]
        public new Size MinimumSize
        {
            get
            {
                return _minimumSize;
            }
            set
            {
                Debug.Assert(value == new Size(0, 0));
                _minimumSize = value;
            }
        }

        #region デザイナのデフォルトの書き換え
        [DefaultValue(true)]
        public new bool HideOnClose
        {
            get { return base.HideOnClose; }
            set { base.HideOnClose = value; }
        }

        [DefaultValue(WeifenLuo.WinFormsUI.Docking.DockState.Float)]
        public new WeifenLuo.WinFormsUI.Docking.DockState ShowHint
        {
            get { return base.ShowHint; }
            set { base.ShowHint = value; }
        }
        #endregion

        /// <summary>
        /// 使用されているリソースに後処理を実行します。
        /// </summary>
        protected override void Dispose( bool disposing )
        {
            if( disposing )
            {
                if( components != null )
                {
                    components.Dispose();
                }
            }
            base.Dispose( disposing );
        }

        /// <summary>
        /// オーナーが ILECmdKeyAccepter ならメッセージを伝播します。
        /// </summary>
        protected bool NotifyCmdKeyMessageToOwner( ref Message msg, Keys keyData )
        {
            ILECmdKeyAcceptor accepter = this.OwnerCmdKeyAcceptor;
            if( accepter != null )
            {
                // 伝播する相手のメッセージとして、
                // ウインドウハンドルを書き換えます。
                var origHandle = msg.HWnd;
                msg.HWnd = (accepter as Form).Handle;
                var ret = accepter.DoProcessCmdKey( ref msg, keyData );
                msg.HWnd = origHandle;
                return ret;
            }

            return false;
        }

        /// <summary>
        /// キーメッセージを処理します。
        /// 登録されたイベントハンドラを実行します。
        ///
        /// キーメッセージの詳細については、以下のページが詳しい。
        /// http://www.atmarkit.co.jp/fdotnet/dotnettips/243winkeyproc/winkeyproc.html
        /// </summary>
        protected override bool ProcessCmdKey(ref Message msg, Keys keyData)
        {
            // 保存ショートカットを処理する。

            if ((keyData == (Keys.S | Keys.Control)) ||
                (keyData == (Keys.S | Keys.Control | Keys.Shift)))
            {
                FixFocusedControlValue();

                if (NotifyCmdKeyMessageToOwner(ref msg, keyData))
                {
                    return true;
                }
            }

            try
            {
                // 伝播は NotifyCmdKeyMessageToOwner で処理する
                OwnerCmdKeyAcceptor.SuspendProcessCmdKey++;

                // 既定の処理
                if (base.ProcessCmdKey(ref msg, keyData))
                {
                    return true;
                }
            }
            finally
            {
                OwnerCmdKeyAcceptor.SuspendProcessCmdKey--;
            }

            // TODO: メインウインドウ側で制御したい
            switch (keyData)
            {
                case Keys.Control | Keys.C:
                case Keys.Control | Keys.X:
                case Keys.Control | Keys.V:
                case Keys.Control | Keys.A:
                case Keys.Delete: // Delete はコマンドキー
                    return false;
                case Keys.Control | Keys.Z:
                case Keys.Control | Keys.Shift | Keys.Z:
                    {
                        FixFocusedControlValue();
                    }
                    break;
            }

            return NotifyCmdKeyMessageToOwner(ref msg, keyData);
        }

        /// <summary>
        // Undo実行前に編集中の値を確定します
        // 対象となるコントロールはValueTextBoxとUINumericUpDownです
        /// </summary>
        private void FixFocusedControlValue()
        {
            FixEditingState focusedControl = LECore.Manipulator.SubSceneUndoRedoHelper.GetFocusedControl(this) as FixEditingState;
            if (focusedControl != null)
            {
                focusedControl.FixValue();
            }
        }

        /// <summary>
        /// 更新メッセージを通知します。
        /// </summary>
        static public void DbgReportUpdate( ILEToolWindow wnd )
        {
            LECore.DbgConsole.WriteLine( "upadated -- [{0}]", wnd.LEWindowName );
        }




        #region 設定保存、読み込み関連

        /// <summary>
        /// フォーム基本データを保存します。
        /// </summary>
        static public void SaveToBasicData( Form form, LEToolFormSetting setting )
        {
            setting.FormWindowState = form.WindowState;

            Point formPosition;
            Size fromSize;

            // NativeAPIコールを利用して、
            // this.WindowState の状態にかかわらない
            // 位置、サイズを取得します。
            common.WindowPlacementHelper.GetWindowNormalPosition(
                    form,
                    out formPosition,
                    out fromSize );

            setting.Position = formPosition;
            setting.Size = fromSize;
        }

        /// <summary>
        /// 状態を保存します。
        /// 最小化状態のウインドウの状態は保存しません。
        /// </summary>
        public virtual void SaveSetting(LEToolFormSetting setting, SaveSettingOption option)
        {
            SaveToBasicData( this, setting );

            if( this.WindowState == FormWindowState.Normal )
            {
                setting.Name = this.LEWindowName;
            }

            // フォームの状態を保存します。
            LEControlUserDataChunk[]		userDataSet;
            ILESerializableControlHelper.SaveFormState( this, out userDataSet );
            if( userDataSet.Length > 0 )
            {
                setting.UserDataSet.AddRange( userDataSet );
            }
        }

        /// <summary>
        /// フォーム基本データを読み込む
        /// </summary>
        static public void LoadFormBasicData( Form form, LEToolFormSetting setting )
        {
            // 表示状態の設定は、ここでは行いません。
            // 読み込み終了時に、おこないます。

            // 設定は Load イベントで行いますが、
            // Loadイベントが呼ばれない場合もあるので一度明示的に呼び出しておきます。
            LoadLEToolFormSettingFunctor loadSettingFunctor = new LoadLEToolFormSettingFunctor( setting );
            form.Load += loadSettingFunctor.Form_Load;
            loadSettingFunctor.Form_Load(form, null);
        }

        #region struct LoadSettingFuctor

        /// <summary>
        /// 設定データをFormに設定するファンクタ
        /// </summary>
        class LoadLEToolFormSettingFunctor
        {
            readonly LEToolFormSetting _setting;
            /// <summary>
            /// コンストラクタ
            /// </summary>
            public LoadLEToolFormSettingFunctor( LEToolFormSetting setting )
            {
                _setting = setting;
            }

            /// <summary>
            ///
            /// </summary>
            public void Form_Load( object sender, EventArgs e )
            {
                Form form = sender as Form;

                //-----------------------------------------
                // 表示状態の設定は、ここでは行いません。
                // 読み込み終了時に、おこないます。
                // サイズや位置の更新前の、タイミングで表示してしまうと、
                // 初期化時にウインドウがちらついて見えてしまうので注意が必要です。
                // また、WindowState の設定との順番が結果に影響するようです。
                SetFormPosAndSize_(form, _setting.Position, _setting.Size);

                if (form.WindowState != _setting.FormWindowState)
                {
                    form.WindowState = _setting.FormWindowState;
                }
            }

            /// <summary>
            ///
            /// </summary>
            static private void SetFormPosAndSize_(Form form, Point position, Size size)
            {
                //-----------------------------------------
                // 位置と、サイズを設定します。
                //
                // form.Visible より後に実行しないと、Loadイベントで位置、
                // サイズがリセットされます。
                //
                // また、form.WindowStateの設定より前に実行しないと、
                // 正しく再描画が行われないようです。
                //
                // 注意: DockPanel Suite の MDI 表示のときは、DesktopLocation はドキュメント領域内での座標を示している
                form.DesktopLocation = LEToolFormSetting.ClampWindowPosition(position, size);

                // サイズ変更可能なものに限ってサイズ設定を読み込みます。
                // サイズゼロは普通に考えて何かの間違えなので無視。
                if (size.Width == 0 || size.Height == 0)
                {
                    return;
                }

                // サイズ制約にマッチする値だけ設定
                if (form.FormBorderStyle == FormBorderStyle.Sizable ||
                    form.FormBorderStyle == FormBorderStyle.SizableToolWindow)
                {
                    if ((form.MinimumSize.Width == 0 || (form.MinimumSize.Width < size.Width)) &&
                        (form.MaximumSize.Width == 0 || (form.MaximumSize.Width > size.Width)) &&
                        (form.MinimumSize.Height == 0 || (form.MinimumSize.Height < size.Height)) &&
                        (form.MaximumSize.Height == 0 || (form.MaximumSize.Height > size.Height)))
                    {
                        form.Size = size;
                    }
                }
            }
        }

        #endregion struct LoadSettingFuctor



        /// <summary>
        /// 状態を読み込みます。
        /// </summary>
        public virtual void LoadSetting( LEToolFormSetting setting , LoadSettingOption option)
        {
            // ウインドウの位置やサイズの情報はここでは復元しない

            // フォームの状態を読み込みます。
            LEControlUserDataChunk[] userDataSet = setting.UserDataSet.ToArray();
            ILESerializableControlHelper.LoadFormState( this, userDataSet );
        }

        #endregion 設定保存、読み込み関連

        /// <summary>
        /// WndProc オーバーライド
        /// </summary>
        protected override void WndProc( ref Message m )
        {
            switch( m.Msg )
            {
                case LECore.Win32.WM.WM_NCLBUTTONDBLCLK:
                {
                    // メニューバーのダブルクリックによってウインドウをたたむ処理は廃止されました。
                    // ただし、規定の処理(ウインドウの最大化)は行わないようにします。
                    return;
                }
            }
            base.WndProc( ref m );
        }

#region ILEToolWindow メンバ

        /// <summary>
        ///
        /// </summary>
        public virtual string LEWindowName
        {
            get { return this.Text; }
        }

        /// <summary>
        ///
        /// </summary>
        public string Description
        {
            get { return "ToolWindow"; }
        }

        /// <summary>
        ///
        /// </summary>
        public virtual bool AllowToChangeVisible { get { return true; } }

        public virtual bool IsNeededToHide { get { return false; } }

        /// <summary>
        ///
        /// </summary>
        public virtual Keys CustomShortcut { get { return Keys.None; } }


        /// <summary>
        /// ビューマネージャです。
        /// </summary>
        public IViewManager ViewManager { get; set; }

        /// <summary>
        /// コマンドキー処理の委譲先です。
        /// </summary>
        public ILECmdKeyAcceptor OwnerCmdKeyAcceptor { get; set; }

#region 規定の処理：派生クラスで具体的な処理をオーバーライドしてください
        public virtual void OnTimeChangedHandler( int time, TimeChageEventType type )
        {
            // DoNothig
        }

        public virtual void OnSceneModifyHandler( object sender, SceneModifyEventArgs e )
        {
            // DoNothig
        }
#endregion 規定の処理：派生クラスで具体的な処理をオーバーライドしてください

#region IAppEventListener メンバ
        public virtual void OnAppEvent( object sender, AppEventArgs args )
        {
            // 規定の処理。何も処理を行いません。
        }
#endregion IAppEventListener

#endregion ILEToolWindow メンバ

#region ILECmdKeyAccepter メンバ

        /// <summary>
        /// 子ウインドウからのコマンドキーを受け取って、
        /// オーナー AppFormにメッセージを伝播します。
        /// </summary>
        public bool DoProcessCmdKey( ref Message msg, System.Windows.Forms.Keys keyData )
        {
            // 自らの、メッセージとして解釈するように、
            // ウインドウハンドルを書き換えます。
            var origHandle = msg.HWnd;
            msg.HWnd = this.Handle;

            var ret = base.ProcessCmdKey( ref msg, keyData );
            msg.HWnd = origHandle;
            return ret;
        }

#endregion

#region  IViewManagerMessageSender メンバ

        public event Action<ViewManagerMessage> SendMessageToViewManager;

        /// <summary>
        /// マネージャにメッセージを送る。
        /// 派生クラスが、メッセージを送信する場合に使用されます。
        /// </summary>
        /// <param name="args"></param>
        public void DoSendMessageToViewManager( ViewManagerMessage args )
        {
            if( SendMessageToViewManager != null )
            {
                SendMessageToViewManager(args);
            }
        }

        #endregion  IViewManagerMessageSender

        private void LEToolWindow_DockStateChanged(object sender, EventArgs e)
        {
            // フローティングにした直後にキーボードショートカットが効かない問題への対応
            if (IsHandleCreated)
            {
                BeginInvoke((Action)(() => Focus()));
            }
            else
            {
                Focus();
            }
        }
    }

    #region IViewManagerMessageSender 関連

    /// <summary>
    /// ビューマネージャメッセージ種類
    /// </summary>
    public enum ViewManagerMessageKind
    {
        Invalid,
        ShowProperty,
        ShowCurveEditor,
        FindLayoutPathBySubScene,
        OpenLayout,
        ForceCloseLayout,
        ForceCloseLayoutAll,
        CreateDerivativeLayout,
        ReloadPartsAll,
        WriteMessage,
        WriteWarning,
        SaveStateChanged,
        SendViewer,
        CreateKeyFrameOnTargetCurve,
        KeyCmd,
        UpdateWindowTitle,
        PreviewCurveInTargetTag,
        UpdatePanePasteControlWindow,
        DuplicatePanes,
        ProjectSettingChanged,
    }

    /// <summary>
    /// ビューマネージャメッセージ
    /// </summary>
    public class ViewManagerMessage
    {
        //---------- フィールド
        readonly ViewManagerMessageKind _kind = ViewManagerMessageKind.Invalid;
        readonly object _paramater = null;
        object _result = null;

#region プロパティ
        public ViewManagerMessageKind MessageKind { get { return _kind; } }
        public object MessageParam { get { return _paramater; } }
        public object Result { get { return _result; } set { _result = value; } }
        public bool ResultAsBool
        {
            get
            {
                if( _result == null )
                {
                    return false;
                }
                else
                {
                    return (bool)_result;
                }
            }
        }
#endregion プロパティ

        //---------- コンストラクタ
        public ViewManagerMessage( ViewManagerMessageKind kind )
            : this( kind, null )
        {
        }

        public ViewManagerMessage( ViewManagerMessageKind kind, object param )
        {
            _kind = kind;
            _paramater = param;
        }

#region 慣用メッセージ
        public static readonly ViewManagerMessage ShowPropertyWndMsg = new ViewManagerMessage( ViewManagerMessageKind.ShowProperty, null );
#endregion 慣用メッセージ
    }

    /// <summary>
    /// ビューマネージャメッセージ
    /// KeyCmdに対応するパラメータを設定します
    /// </summary>
    public class KeyCmdViewManagerMessage : ViewManagerMessage
    {
        readonly Keys _keyData = Keys.None;
        readonly IEnumerable<IPane> _paneSet = null;

        public Keys KeyData { get { return _keyData; }  }
        public IEnumerable<IPane> PaneSet { get { return _paneSet; } }

        //---------- コンストラクタ
        public KeyCmdViewManagerMessage(ViewManagerMessageKind kind, Keys keyData, IEnumerable<IPane> paneSet)
            : base( kind, null )
        {
            _keyData = keyData;
            _paneSet = paneSet;
        }
    }

    /// <summary>
    /// ViewManagerに対してリクエストを要求するクラスが実装するインタフェース
    /// </summary>
    public interface IViewManagerMessageSender
    {
        event Action<ViewManagerMessage> SendMessageToViewManager;
    }
#endregion IViewManagerMessageSender 関連

    /// <summary>
    /// ユーザインタフェースの状態をシリアライズして、
    /// 永続化するクラスが実装すべきインタフェースです。
    /// </summary>
    public interface ILEDataSerializebleWindow
    {
        /// <summary>
        /// ツールウインドウの名前を取得します。
        /// </summary>
        string LEWindowName { get; }
        /// <summary>
        /// ツールウインドウの説明文字列を取得します。
        /// </summary>
        string Description { get; }

        /// <summary>
        /// 状態を保存します。
        /// </summary>
        void SaveSetting( LEToolFormSetting setting, SaveSettingOption option);

        /// <summary>
        /// 状態を読み込みます。
        /// </summary>
        void LoadSetting( LEToolFormSetting setting, LoadSettingOption option);
    }

    /// <summary>
    /// 保存オプション
    /// </summary>
    public class SaveSettingOption
    {
        // ワークスペース以外の情報も保存する
        public bool AlsoSaveOtherThanWorkspace { get; set; } = false;
    }

    /// <summary>
    /// 読み込みオプション
    /// </summary>
    public class LoadSettingOption
    {
        // ワークスペース以外の情報も読み込む
        public bool AlsoLoadOtherThanWorkspace { get; set; } = false;
    }


    /// <summary>
    /// ユーティリティ
    /// </summary>
    public static class LEToolWindowHelper
    {
        /// <summary>
        /// ツールウインドウが全面か判定します。
        /// </summary>
        public static bool CheckToolWindowIsFront(IntPtr handle)
        {
            HashSet<IntPtr> handleSet = new HashSet<IntPtr>();
            handleSet.Add(handle);
            IntPtr myOwnerHandle = LECore.Win32.User32.GetWindow(handle, LECore.Win32.GW.GW_OWNER);

            IntPtr prevHandle = LECore.Win32.User32.GetWindow(handle, LECore.Win32.GW.GW_HWNDPREV);
            while (!handleSet.Contains(prevHandle))
            {
                handleSet.Add(prevHandle);

                // 自分より前面にあり、親が一致する
                if (myOwnerHandle == LECore.Win32.User32.GetWindow(prevHandle, LECore.Win32.GW.GW_OWNER))
                {
                    if (LECore.Win32.User32.IsWindowVisible(prevHandle))
                    {
                        // 自分が最前面ではなかった
                        return false;
                    }
                }

                prevHandle = LECore.Win32.User32.GetWindow(prevHandle, LECore.Win32.GW.GW_HWNDPREV);
            }

            // 自分が最前面
            return true;
        }
    }

    /// <summary>
    /// ツールウインドウ、インターフェース
    /// AppFormに公開する機能
    /// </summary>
    public interface ILEToolWindow :
        ITimeChageEventListener,              // グローバル時間変更イベントの通知を受け取ります。
        ISceneModifyListener,                 // シーンの変更イベントを受け取ります
        IAppEventListener,                    // アプリケーション更新イベントを受け取ります
        ILEDataSerializebleWindow            // データをシリアライズ可能である。
    {
        /// <summary>
        /// 可視状態の変更機能を提供します
        /// </summary>
        bool AllowToChangeVisible { get;}

        /// <summary>
        /// 可視状態の変更メニューを隠す必要があるかどうか？
        /// 隠し機能として提供する場合は、この部分をオーバーライドして対応します。
        /// </summary>
        bool IsNeededToHide { get; }

        /// <summary>
        /// メニュー項目でのショートカットを設定します。
        /// </summary>
        Keys CustomShortcut { get;}

        /// <summary>
        /// ビューマネージャです。
        /// </summary>
        IViewManager ViewManager { get; set; }

        /// <summary>
        /// コマンドキー処理を委譲します。
        /// </summary>
        ILECmdKeyAcceptor OwnerCmdKeyAcceptor { get; set; }
    }

    /// <summary>
    /// LoadSetting 後の処理
    /// </summary>
    public interface IInitializeAfterLoadSetting
    {
        void InitializeAfterLoadSetting();
    }
}
