﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using System.Text;
using System.Text.RegularExpressions;
using System.Windows.Forms;

namespace LayoutEditor.Forms.ToolWindows.CurveEditWindow
{
    using LECore.Structures;
    using LECore.Structures.Core;

    using common;
    using System.Drawing;

    /// <summary>
    /// UI アニメーションアトリビュートヘルパ
    /// </summary>
    public class AttributerUIHelper
    {
        const bool IsFontColor = true;

        /// <summary>
        /// 色を求める
        /// </summary>
        public static Color GetAttrColor(IAnmAttribute attr, Color defaultColor)
        {
            return GetAttrColor_(attr, defaultColor, IsFontColor);
        }

        /// <summary>
        /// 色を求める
        /// </summary>
        public static Color GetCurveColor(IAnmAttribute attr, Color defaultColor)
        {
            return GetAttrColor_(attr, defaultColor, !IsFontColor);
        }

        /// <summary>
        /// 色を求める
        /// </summary>
        static Color GetAttrColor_(IAnmAttribute attr, Color defaultColor, bool isFontColor)
        {
            if (attr.HasSubAttribute)
            {
                return defaultColor;
            }

            // 拡張ユーザーデータのスカラー値はdefaultColorにする
            if (attr.EnableLocalize == false)
            {
                return defaultColor;
            }

            int index = attr.GetSelfSubAttrIndex();
            switch (index)
            {
                case 0: return (attr.OwnerNode.NumSubAttribute > 1) ? Color.FromArgb(230, 0, 0) : defaultColor;
                case 1: return isFontColor ? Color.FromArgb(0, 230, 0) : Color.FromArgb(190, 255, 0);
                case 2: return Color.FromArgb(0, 0, 230);
                default:
                    return defaultColor;
            }
        }
    }

    /// <summary>
    /// カーブエディットウインドウのアトリビュートツリー部分の
    /// 再構築処理を行うクラスです。
    ///
    /// 必要のない、ツリーの更新をできるだけ回避するようにします。
    ///
    /// TreeNodeFactoryクラスを包含しています。
    /// </summary>
    class AttributeTreeFactory
    {
        #region Viewの再構築処理

        #region CurveEditWindowTargetState
        /// <summary>
        /// ターゲットに関する状態
        /// </summary>
        class CurveEditWindowTargetState
        {
            /// <summary>
            /// キー文字列取得時に、行うペイン名置換処理の正規表現
            /// </summary>
            static Regex _FullPathForKeyStringReplaceRegex = new Regex(string.Format("^ \\[{0}*\\]", LECore.AppConstants.RegexStrValidCharForObjectName), RegexOptions.Compiled);

            IAnmCurve[] _targetCurveSet = null;
            // ノードのフルパスと、ノードのテーブル
            Hashtable _treeNodeExpandStateTable = new Hashtable();

            /// <summary>
            /// 復元可能か取得します。
            /// </summary>
            public bool ReadyToResotore
            {
                get { return _targetCurveSet != null; }
            }

            /// <summary>
            /// 状態をリセットします。
            /// </summary>
            void Reset_()
            {
                _targetCurveSet = null;
                _treeNodeExpandStateTable.Clear();
            }

            #region 状態の保存

            /// <summary>
            /// テーブルのキーに使用する、FullPath文字列を取得します。
            /// </summary>
            string GetFullPathForKeyString_( TreeNode node )
            {
                string fullPath = node.FullPath;

                fullPath = _FullPathForKeyStringReplaceRegex.Replace(fullPath, @" [---]");

                return fullPath;
            }


            /// <summary>
            /// windowの状態を保存します。
            /// </summary>
            void StoreExpandState_( TreeNode node )
            {
                string path = GetFullPathForKeyString_( node );
                if( !_treeNodeExpandStateTable.Contains( path ) )
                {
                    _treeNodeExpandStateTable.Add( path, node );
                }

                foreach( TreeNode child in node.Nodes )
                {
                    StoreExpandState_( child );
                }
            }

            /// <summary>
            /// windowの状態を保存します。
            /// </summary>
            public void Store( IAnmCurve[] targetCurve, TreeNode[] attributeTreeNode )
            {
                if( !ReadyToResotore )
                {
                    _targetCurveSet = targetCurve;

                    _treeNodeExpandStateTable.Clear();
                    foreach( TreeNode node in attributeTreeNode )
                    {
                        StoreExpandState_( node );
                    }
                }
            }
            #endregion 状態の保存

            #region 状態の復元

            /// <summary>
            /// 保存状態をwindowに復元します。
            /// </summary>
            void RestoreExpandState_( List<IAnmAttribute> newTargetSet, TreeNode node )
            {
                // 自らの完全パスでテーブルを引き、以前の状態を調べます。
                string keyString = GetFullPathForKeyString_( node );
                object stateData = _treeNodeExpandStateTable[keyString];
                if( stateData != null )
                {
                    TreeNode srcNode = stateData as TreeNode;
                    if( srcNode.IsExpanded )
                    {
                        node.Expand();
                    }
                }

                // 以前にターゲット設定されていたのならば...
                IAnmAttribute attr = node.Tag as IAnmAttribute;
                if( Array.IndexOf( _targetCurveSet, attr ) != -1 )
                {
                    newTargetSet.Add( attr );
                }

                // 子供ノードに対しても、再帰的に処理を実行します。
                foreach( TreeNode child in node.Nodes )
                {
                    RestoreExpandState_( newTargetSet, child );
                }
            }

            /// <summary>
            /// 保存状態をwindowに復元します。
            /// </summary>
            public void Restore( TreeNode[] attributeTreeNode )
            {
                if (ReadyToResotore && attributeTreeNode.Length > 0)
                {
                    List<IAnmAttribute> newTargetSet = new List<IAnmAttribute>();
                    foreach( TreeNode node in attributeTreeNode )
                    {
                        RestoreExpandState_( newTargetSet, node );
                    }

                    Reset_();
                }
            }
            #endregion 状態の復元
        }

        #endregion CurveEditWindowTargetState

        #region ITreeNodeStateCheckPolicy 実装
        /// <summary>
        /// ノードステート確認ポリシークラス
        /// </summary>
        class AttributeTreeNodeStateCheckPolicy : ITreeNodeStateCheckPolicy
        {

            #region ITreeNodeStateCheckPolicy メンバ

            public bool CheckChanged( object current, object old )
            {
                return true;
            }

            public void MakeClone( object current, out object clone )
            {
                clone = current;
            }

            public void UpdateClone( object current, object clone )
            {
                // 何もしません。
            }
            #endregion
        }
        #endregion ITreeNodeStateCheckPolicy 実装

        // ツリーノードの開閉状態を記憶しているクラス。
        // ツリー再構築の際、前回の内容の復元を試みます。
        CurveEditWindowTargetState _targetState = new CurveEditWindowTargetState();
        // Treeの状態が古く、アップデートが必要である。
        bool _isViewObsolete = false;
        // Active状態であるアトリビュートのみ扱います。
        bool _showActiveOnly = true;
        // 基本的なTreeNodeFactory
        readonly TreeNodeFactory _treeNodeFactory;
        // アトリビュートフィルタ
        readonly AttributeFilterManager _attributeFilterManager;

        // パスから、ノード文字列を取得するマップ。
        // ユーザにとってより理解しやすい文字列に置き換えるために利用する。
        readonly List<KeyValuePair<string, string>> _nodeTextConversionMap = new List<KeyValuePair<string, string>>();

        /// <summary>
        /// アクティブなノードのみ表示するか？
        /// </summary>
        public bool ShowActiveOnly
        {
            get { return _showActiveOnly; }
            set { _showActiveOnly = value; }
        }

        /// <summary>
        /// アトリビュートフィルタ
        /// </summary>
        public Color DefaultBackColor
        {
            get;
            set;
        }

        /// <summary>
        /// アトリビュートフィルタ
        /// </summary>
        public AttributeFilterManager AttributeFilters
        {
            get { return _attributeFilterManager; }
        }

        /// <summary>
        /// ビューを再構築する必要があるか判定します。
        /// </summary>
        /// <returns></returns>
        bool _IsViewObsolete
        {
            get { return _isViewObsolete; }
            set { _isViewObsolete = value; }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public AttributeTreeFactory()
        {
            _treeNodeFactory = new TreeNodeFactory(new AttributeTreeNodeStateCheckPolicy());
            _attributeFilterManager = new AttributeFilterManager();
        }

        /// <summary>
        /// アトリビュートがフィルタ対象か判定します。
        /// </summary>
        bool CheckAttributeFiltered_( IAnmAttribute attribute )
        {
            return
                _attributeFilterManager.CurrentFilter != null &&
                _attributeFilterManager.CurrentFilter.Contains(attribute);
        }

        /// <summary>
        /// ノードテキストを取得します。
        /// <remarks>
        /// 変換マップを操作して置換処理を施して最終的なノード文字列を取得します。
        /// そこそこのオーバーヘッドがありそうで置換後文字列のキャッシュも検討しましたが、
        /// 実用上はほとんど問題にならないのでシンプルにいまの形にしています。
        /// </remarks>
        /// </summary>
        private string GetNodeText_(string originalNodeText, bool enableLocalize)
        {
            // ローカライズ無効なら置換はしない。
            if (enableLocalize == false)
            {
                return originalNodeText;
            }

            foreach (var conversionPair in _nodeTextConversionMap)
            {
                originalNodeText = originalNodeText.Replace(conversionPair.Key, conversionPair.Value);
            }

            return originalNodeText;
        }

        /// <summary>
        /// アトリビュートの情報をツリーノードに設定します。
        /// サブアトリビュートは、子ノードとして、再帰的に登録処理されます。
        /// </summary>
        TreeNode BuildTreeNode_( TreeNode parentNode, string attrNamePreFixStr, IAnmAttribute attr )
        {
            TreeNode node = null;
            if( attr.IsActiveAttribute || !_showActiveOnly )
            {
                // フィルタに一致しなかったらノード追加
                if( attr.OwnerNode == null || !CheckAttributeFiltered_( attr ) )
                {
                    node = _treeNodeFactory.MakeNewNode(attr, attr);

                    // ノードテキストの設定。トップノード以外では
                    node.Text = GetNodeText_(attrNamePreFixStr + attr.DescriptionName, parentNode != null && attr.EnableLocalize);

                    node.ForeColor = AttributerUIHelper.GetAttrColor(attr, Color.DimGray);
                    node.BackColor = this.DefaultBackColor;
                    AttributeTreeAppearanceUpdater.ResetNodeAppearenceSelected(node);
                    parentNode = node;
                }
                else
                {
                    attrNamePreFixStr += attr.DescriptionName + "/";
                    parentNode.ExpandAll();
                }

                if( parentNode != null )
                {
                    // サブアトリビュートについて再帰的の処理をおこない、子ノードとして登録します。
                    for( int i = 0 ; i < attr.NumSubAttribute ; i++ )
                    {
                        IAnmAttribute childAttr = attr.FindSubAttributeByIdx( i );
                        TreeNode childNode = BuildTreeNode_( parentNode, attrNamePreFixStr, childAttr );

                        if( childNode != null)
                        {
                            // 子ノード子供要素を持つか、
                            // 自身がアニメーションカーブを持つ場合には
                            if( childNode.Nodes.Count != 0 || childAttr.ICurrentAnimationCurve != null )
                            {
                                // 子供ノードとして登録します。
                                parentNode.Nodes.Add( childNode );
                            }
                        }
                    }
                }
            }
            return node;
        }

        /// <summary>
        /// TreeViewの再構築を行います。
        /// </summary>
        public List<TreeNode> RebuildTreeView( IAnmAttribute[] targetAttrSet )
        {
            LECore.DbgConsole.WriteLine( "GraphEdit-View Controls are rebuilt." );

            List<TreeNode> nodeSet = new List<TreeNode>();
            TreeView tv = new TreeView();

            _treeNodeFactory.Clear();

            foreach( IAnmAttribute attr in targetAttrSet )
            {
                TreeNode newNode = BuildTreeNode_( null, string.Empty, attr );
                if( newNode != null )
                {
                    nodeSet.Add( newNode );
                    tv.Nodes.Add( newNode );
                }
            }


            // 状態の復元を試みます。
            if( _targetState.ReadyToResotore )
            {
                _targetState.Restore( nodeSet.ToArray() );
            }

            // 削除します。
            foreach( TreeNode treeNode in nodeSet )
            {
                tv.Nodes.Remove( treeNode );
            }

            return nodeSet;
        }

        /// <summary>
        /// 状態を保存する
        /// </summary>
        public void StoreState(
            IAnmCurve[] targetAttrSet,
            TreeNodeCollection attributeTreeNode )
        {
            List<TreeNode> nodeSet = new List<TreeNode>();
            foreach( TreeNode tn in attributeTreeNode )
            {
                nodeSet.Add( tn );
            }

            _targetState.Store( targetAttrSet, nodeSet.ToArray() );
        }

        /// <summary>
        /// 末端のアトリビュートから、ノードセットを検索します。
        /// </summary>
        public List<TreeNode> GetUpdatedNodeByContents( IAnmAttribute curve )
        {
            return _treeNodeFactory.FindUpdatedNodeByContents( curve );
        }

        /// <summary>
        /// ノード文字列変換テーブルを設定します。
        /// </summary>
        public void InitNodeTextConversionMap(KeyValuePair<string,string>[] nodePathSet)
        {
            _nodeTextConversionMap.Clear();
            _nodeTextConversionMap.AddRange(nodePathSet);
        }

        #endregion Viewの再構築処理
    }
}
