﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Diagnostics;
using System.Text;
using System.Windows.Forms;
using System.Xml;
using LayoutEditor.Properties;

namespace LayoutEditor.Forms.ToolWindows.ColorSetWindow
{
    using Structures.SerializableObject;
    using LayoutEditor.Controls;
    using ColorTranslator = System.Drawing.ColorTranslator;
    using common;
    using LECore.Structures;

    /// <summary>
    /// 色見本ウインドウ
    /// </summary>
    public partial class ColorSetWindow : LEToolWindow
    {
        #region フィールド
        const View	_ImageViewMode = View.Tile;
        const string _DummyComment = "";
        const string _UserData_ColorNamePreFix = "ColorData_Color_{0}";
        const string _UserData_CommentPreFix = "ColorData_Comment_{0}";
        const int _ColorCommentSubItemIndex = 1;
        SolidBrush		_ownerDrawBrush = new SolidBrush( Color.Empty );
        Pen _ownerDrawPen = new Pen( Color.Empty );
        TextBox _labelEditTextBox = new TextBox();
        ColorEditPanel.DragDropAction _dragDropAction;

        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public ColorSetWindow()
        {
            InitializeComponent();

            _tcbViewMode.Items.Add( new UIListControlItem(
                    LayoutEditor.StringResMgr.Get( "COLORSET_COMBOBOX_VIEW_SICON" ),
                    _ImageViewMode ) );

            _tcbViewMode.Items.Add( new UIListControlItem(
                    LayoutEditor.StringResMgr.Get( "COLORSET_COMBOBOX_VIEW_DETAILS" ),
                    System.Windows.Forms.View.Details ) );

            // Details を指定
            _tcbViewMode.SelectedIndex = 1;

            _clhColorIcon.Text = LayoutEditor.StringResMgr.Get( "COLORSET_LISTVIEW_HEADER_COLORICON" );
            _clhComment.Text = LayoutEditor.StringResMgr.Get( "TAG_COMMENT" );

            _lvwColors.TileSize = new Size( 16, 16 );

            // ラベル編集処理の設定
            _labelEditTextBox.Parent = _lvwColors.Parent;
            _lvwColors.SubItemClicked += (obj, e) =>
            {
                if (e.SubItem == _ColorCommentSubItemIndex)
                {
                    _lvwColors.StartEditing(_labelEditTextBox, e.Item, e.SubItem);
                }
            };

            // クリックによる編集時は、SubItemClicked内でStartEditingしているため、
            // SubItemBeginEditingを利用することができない。
            // そのため、Event_LvwColors_BeforeSubItemLabelEditもSubItemClickedで呼び出すようにする。
            _lvwColors.SubItemClicked += Event_LvwColors_BeforeSubItemLabelEdit;

            // F2キーによる編集時はクリックイベントが発生しないためBeginEditingで
            // Event_LvwColors_BeforeSubItemLabelEditを呼び出す。
            _lvwColors.SubItemBeginEditing += Event_LvwColors_BeforeSubItemLabelEdit;

            _lvwColors.SubItemEndEditing += Event_LvwColors_AfterSubItemLabelEdit;

            _lvwColors.InsertionMark.Index = -1;


            // メッセージフィルタの設定
            ToolStripMenuItemHelper.ToolStripMessageFilter.BindMessageFilter( _tspMain );

            UpdateProperty_();
        }

      /// <summary>
      ///
      /// </summary>
      protected override void OnLoad( EventArgs e)
      {
          // 一度、Tileにしてから Detailsを設定する必要があります。
          // Tile時に描画が乱れるのを防ぐ為です。
          // また、これらの処理は OnLoadで行う必要があります。
          _tcbViewMode.SelectedIndex = 0;
          _tcbViewMode.SelectedIndex = 1;
      }

        //------------------------------------------------------
        #region GUI状態更新

        /// <summary>
        /// カラーの新規登録
        /// </summary>
        ListViewItem GetNewItem_( FloatColor color, string commnet )
        {
            ListViewItem lvi = new ListViewItem();

            ColorData cd = new ColorData( color, commnet );

            lvi.Text = commnet;
            lvi.ToolTipText = commnet;

            lvi.Tag = cd;
            lvi.SubItems.Add( cd.Comment );

            return lvi;
        }

        /// <summary>
        ///
        /// </summary>
        void UpdateProperty_ListViewItem_( ListViewItem lvi )
        {
            ColorData cd = lvi.Tag as ColorData;
            Debug.Assert( cd != null );

            string commentStr = cd.Comment;

            if( lvi.SubItems[_ColorCommentSubItemIndex].Text != commentStr )
            {
                lvi.Text = commentStr;
                lvi.ToolTipText = commentStr;
                lvi.SubItems[_ColorCommentSubItemIndex].Text = commentStr;
            }
        }

        /// <summary>
        ///
        /// </summary>
        void UpdateProperty_ListView_()
        {
            _lvwColors.BeginUpdate();

            foreach( ListViewItem lvi in _lvwColors.Items )
            {
                UpdateProperty_ListViewItem_( lvi );
            }

            _lvwColors.EndUpdate();
        }

        /// <summary>
        /// GUI状態更新
        /// </summary>
        void UpdateProperty_()
        {
            //----------------------------------
            // ボタンの状態更新
            bool bSelected = ( _lvwColors.SelectedIndices.Count > 0 ) ;

            _tsbDelete.Enabled = bSelected;
            _tsbEdit.Enabled = bSelected;
            _ttbColorComment.Enabled = bSelected;

            if( bSelected )
            {
                ColorData cd = _lvwColors.SelectedItems[0].Tag as ColorData;
                _ttbColorComment.Text = cd.Comment;
            }

            _tsbClear.Enabled = _lvwColors.Items.Count > 0;
        }

        #endregion GUI状態更新

        //------------------------------------------------------
        #region オーバーライド

        //------------------------------------------------------
        #region 設定保存、読み込み関連
        /// <summary>
        /// 状態を保存します。
        /// 最小化状態のウインドウの状態は保存しません。
        /// </summary>
        public override void SaveSetting(LEToolFormSetting setting, SaveSettingOption option)
        {
            if (option.AlsoSaveOtherThanWorkspace)
            {
                foreach (ListViewItem lvi in _lvwColors.Items)
                {
                    ColorData cd = lvi.Tag as ColorData;
                    Debug.Assert(cd != null);

                    // カラー
                    setting.AddUserData(String.Format(_UserData_ColorNamePreFix, lvi.Index), ConverColorToHtmlString(cd.Color.ToSystemColor()));
                    // コメント
                    setting.AddUserData(String.Format(_UserData_CommentPreFix, lvi.Index), cd.Comment);
                }
            }

            base.SaveSetting(setting, option);
        }

        /// <summary>
        /// 状態を読み込みます。
        /// </summary>
        public override void LoadSetting(LEToolFormSetting setting, LoadSettingOption option)
        {
            if (option.AlsoLoadOtherThanWorkspace)
            {
                int numColors = 0;
                LEControlUserDataChunk dataChunk = null;
                _lvwColors.Items.Clear();
                do
                {
                    dataChunk = setting.FindUserDataByName(String.Format(_UserData_ColorNamePreFix, numColors));

                    if (dataChunk != null)
                    {
                        FloatColor color;
                        try
                        {
                            color = new FloatColor(ColorTranslator.FromHtml(dataChunk.Value));
                        }
                        catch
                        {
                            color = FloatColor.Empty;
                        }

                        string commnetStr;
                        setting.TryToFindUserDataStrByName(
                            String.Format(_UserData_CommentPreFix, numColors), out commnetStr);
                        ListViewItem lvi = GetNewItem_(color, commnetStr);

                        _lvwColors.Items.Add(lvi);
                    }

                    numColors++;
                }
                while (dataChunk != null);
            }

            base.LoadSetting(setting, option);
        }

        /// <summary>
        /// ColorをHtmlStringに変換します。
        /// </summary>
        private string ConverColorToHtmlString(Color color)
        {
            return String.Format("#{0:X2}{1:X2}{2:X2}{3:X2}", color.A, color.R, color.G, color.B);
        }
        #endregion 設定保存、読み込み関連

        /// <summary>
        /// ショートカットキー
        /// </summary>
        public override Keys CustomShortcut{ get{return Keys.Alt | Keys.F7;}}


        #endregion  オーバーライド

        //------------------------------------------------------
        #region イベントハンドラ
        /// <summary>
        /// 新規作成ボタンクリック
        /// </summary>
        private void Event_TsbNew_Click( object sender, EventArgs e )
        {
            ColorPickerOKCancelDialog colorPickerOKCancelDialog = new ColorPickerOKCancelDialog();
            ColorData srcColorData = null;
            if( _lvwColors.SelectedItems.Count > 0 )
            {
                srcColorData =  _lvwColors.Items[_lvwColors.SelectedIndices[0]].Tag as ColorData;
            }
            else if( _lvwColors.Items.Count > 0 )
            {
                srcColorData = _lvwColors.Items[_lvwColors.Items.Count - 1].Tag as ColorData;
            }

            if( srcColorData != null )
            {
                colorPickerOKCancelDialog.Color = srcColorData.Color;
            }

            if( colorPickerOKCancelDialog.ShowDialog( this ) == DialogResult.OK )
            {
                _lvwColors.Items.Add( GetNewItem_( colorPickerOKCancelDialog.Color, colorPickerOKCancelDialog.Comment ) );
            }

            UpdateProperty_();
        }

        /// <summary>
        /// 表示モードコンボボックス選択変更
        /// </summary>
        private void Event_TcbViewMode_SelectedIndexChanged( object sender, EventArgs e )
        {
            UIListControlItem item = _tcbViewMode.SelectedItem as UIListControlItem;
            if( item != null )
            {
                View view = (View)item.Data;

#if false
                //--------------------------------
                // HACK : 初回時のみ、Tileに切り替えた場合に、表示が崩れることがある。
                // 対策として、2度適用する。
                _lvwColors.BeginUpdate();
                _lvwColors.View = view;
                _lvwColors.View = view;
                _lvwColors.EndUpdate();
#else
        // OnLoadで初期化を行うことにより、ここで小細工する必要が無くなりました。
        // BeginUpdate()、EndUpdate()を行っていたことにより Windows7で不具合が発生していました。
        _lvwColors.View = view;
#endif

                UpdateProperty_();
            }
        }

        /// <summary>
        /// リストビュー選択アイテム変更
        /// </summary>
        private void Event_LvwColors_SelectedIndexChanged( object sender, EventArgs e )
        {
            UpdateProperty_();
        }

        /// <summary>
        /// 削除
        /// </summary>
        private void Evnet_TsbDelete_Click( object sender, EventArgs e )
        {
            // 選択アイテムを削除
            foreach( ListViewItem lvi in _lvwColors.SelectedItems )
            {
                _lvwColors.Items.Remove( lvi );
            }

            UpdateProperty_();
        }

        /// <summary>
        /// クリア
        /// </summary>
        private void Evnet_TsbClear_Click(object sender, EventArgs e)
        {
            if (MessageBox.Show(
                StringResMgr.Get("COLORSET_ALL_CLEAR_MESSAGE"),
                StringResMgr.Get("COLORSET_ALL_CLEAR_CAPTION"),
                MessageBoxButtons.OKCancel) == DialogResult.OK)
            {
                // 全アイテムを削除
                foreach (ListViewItem lvi in _lvwColors.Items)
                {
                    _lvwColors.Items.Remove(lvi);
                }

                UpdateProperty_();
            }
        }

        /// <summary>
        /// 編集
        /// </summary>
        private void Event_TsbEdit_Click( object sender, EventArgs e )
        {
            if( _lvwColors.SelectedItems.Count > 0 )
            {
                ColorPickerOKCancelDialog colorPickerOKCancelDialog =
                    new ColorPickerOKCancelDialog();

                ColorData srcColorData = ( _lvwColors.Items[_lvwColors.SelectedIndices[0]].Tag as ColorData );
                colorPickerOKCancelDialog.Color = srcColorData.Color;
                colorPickerOKCancelDialog.Comment = srcColorData.Comment;

                if( colorPickerOKCancelDialog.ShowDialog( this ) == DialogResult.OK )
                {
                    foreach( ListViewItem lvi in _lvwColors.SelectedItems )
                    {
                        ColorData cd = lvi.Tag as ColorData;
                        Debug.Assert( cd != null );

                        cd.Color = colorPickerOKCancelDialog.Color;
                        cd.Comment = colorPickerOKCancelDialog.Comment;
                    }
                    _ttbColorComment.Text = colorPickerOKCancelDialog.Comment;
                    UpdateProperty_ListView_();
                }
            }
        }

        #region リストビューアイテムのオーナー描画
        /// <summary>
        /// アイテムオーナードロー：
        /// 本Formでは、SmallIcon時のみ、実行されています。
        /// </summary>
        private void Event_LvwColors_DrawItem( object sender, DrawListViewItemEventArgs e )
        {
            if( _lvwColors.View != _ImageViewMode )
            {
                return;
            }

            _ownerDrawBrush.Color = ( e.Item.Tag as ColorData ).Color.ToSystemColor();
            e.Graphics.FillRectangle( _ownerDrawBrush, e.Bounds );

            _ownerDrawPen.Color = Color.Black;
            e.Graphics.DrawRectangle( _ownerDrawPen, e.Bounds );

            e.DrawFocusRectangle();
        }

        private void Event_LvwColors_DrawSubItem( object sender, DrawListViewSubItemEventArgs e )
        {
            if( e.ColumnIndex != 0 )
            {
                e.DrawDefault = true;
                return;
            }

            // Color
            int boundsWidth = e.Bounds.Width / 2;
            Rectangle colorBounds = new Rectangle(e.Bounds.X, e.Bounds.Y, boundsWidth, e.Bounds.Height);
            Rectangle bounds = colorBounds;
            bounds.Inflate( -1, -2 );

            Color origColor = (e.Item.Tag as ColorData).Color.ToSystemColor();
            _ownerDrawBrush.Color = Color.FromArgb(255, origColor.R, origColor.G, origColor.B);
            e.Graphics.FillRectangle( _ownerDrawBrush, bounds );

            _ownerDrawPen.Color = Color.Black;
            e.Graphics.DrawRectangle( _ownerDrawPen, bounds );

            // Alpha
            Rectangle alphaBounds = new Rectangle(colorBounds.X + colorBounds.Width, colorBounds.Y, boundsWidth, e.Bounds.Height);
            bounds = alphaBounds;
            bounds.Inflate(-1, -2);
            bounds.Offset(-2, 0);

            _ownerDrawBrush.Color = Color.FromArgb(255 - origColor.A, 0, 0, 0);
            e.Graphics.FillRectangle(_ownerDrawBrush, bounds);

            _ownerDrawPen.Color = Color.Black;
            e.Graphics.DrawRectangle(_ownerDrawPen, bounds);

            e.DrawFocusRectangle( e.Bounds );

            // 挿入マークの描画テスト
            _ownerDrawBrush.Color = Color.Black;
            if( _lvwColors.InsertionMark.Index != -1 )
            {
                int heightItem = 3;
                if( _lvwColors.InsertionMark.Index == e.Item.Index )
                {
                    int y = ( _lvwColors.InsertionMark.AppearsAfterItem ) ?
                        bounds.Bottom - heightItem : bounds.Top;

                    e.Graphics.FillRectangle( _ownerDrawBrush, bounds.X, y, bounds.Width, heightItem );
                }
            }
        }

        private void Event_LvwColors_DrawColumnHeader( object sender, DrawListViewColumnHeaderEventArgs e )
        {
            e.DrawDefault = true;
        }

        #endregion オーナー描画

        #region テキストボックス
        /// <summary>
        /// コメントテキストボックス変更
        /// </summary>
        private void Event_TtbColorComment_Validated( object sender, EventArgs e )
        {
            foreach( ListViewItem lvi in _lvwColors.SelectedItems )
            {
                ColorData cd = lvi.Tag as ColorData;
                Debug.Assert( cd != null );
                cd.Comment = _ttbColorComment.Text;
            }
            UpdateProperty_ListView_();
        }

        private void Event_TtbColorComment_Leave( object sender, EventArgs e )
        {
            Event_TtbColorComment_Validated( sender, e );
        }

        private void Event_TtbColorComment_KeyDown( object sender, KeyEventArgs e )
        {
            if( e.KeyData == Keys.Enter )
            {
                Event_TtbColorComment_Validated( sender, e );
            }
        }
        #endregion テキストボックス

        #region ドラッグ・ドロップ処理関連
        /// <summary>
        /// 開始
        /// </summary>
        private void Event_LvwColors_ItemDrag( object sender, ItemDragEventArgs e )
        {
            ListViewItem lvi = e.Item as ListViewItem;
            Debug.Assert( lvi != null );

            _lvwColors.DoDragDrop( new Tuple<ColorData, ColorEditPanel.DragDropAction>(lvi.Tag as ColorData, _dragDropAction), DragDropEffects.Move );
        }

        /// <summary>
        /// ドラッグ開始
        /// </summary>
        private void Event_LvwColors_DragEnter( object sender, DragEventArgs e )
        {
            if( ColorData.CheckDropItemHasColorDataWithAction( e ) )
            {
                e.Effect = e.AllowedEffect;
            }
            else
            {
                e.Effect = DragDropEffects.None;
            }
        }

        /// <summary>
        /// ドラッグ中
        /// </summary>
        private void Event_LvwColors_DragOver( object sender, DragEventArgs e )
        {
            if( !ColorData.CheckDropItemHasColorDataWithAction( e ) )
            {
                return;
            }

            Point targetPoint = _lvwColors.PointToClient( new Point( e.X, e.Y ) );

            // マウス位置に近い位置のアイテム番号を取得する
            int oldIndex = _lvwColors.InsertionMark.Index;
            int targetIndex = _lvwColors.InsertionMark.NearestIndex( targetPoint );
            if( targetIndex > -1 )
            {
                // InsertionMark.AppearsAfterItem を正しく設定します。
                Rectangle itemBounds = _lvwColors.GetItemRect( targetIndex );
                if( targetPoint.X > itemBounds.Left + ( itemBounds.Width / 2 ) )
                {
                    _lvwColors.InsertionMark.AppearsAfterItem = true;
                }
                else
                {
                    _lvwColors.InsertionMark.AppearsAfterItem = false;
                }
            }
            else
            {
                ColorData colorData;
                ColorData.TryToGetColorDataFromDropItem( e, out colorData );

                ListViewItem lvi = _lvwColors.GetItemAt( targetPoint.X, targetPoint.Y );
                if( lvi != null && colorData != lvi.Tag )
                {
                    Rectangle rect = _lvwColors.GetItemRect( lvi.Index );

                    // 挿入マークを設定します。
                    targetIndex = lvi.Index;
                    _lvwColors.InsertionMark.AppearsAfterItem = ( targetPoint.Y > rect.Top + rect.Height / 2 );
                }
            }

            // 挿入マークを設定します。
            _lvwColors.InsertionMark.Index = targetIndex;

            _lvwColors.RedrawItems( oldIndex, oldIndex, false );
            _lvwColors.RedrawItems( targetIndex, targetIndex, false );
        }



        /// <summary>
        /// ドラッグ終了
        /// </summary>
        private void Event_LvwColors_DragLeave( object sender, EventArgs e )
        {
            int		index = _lvwColors.InsertionMark.Index;

            if (index >= 0)
            {
                // 挿入マークを消します
                _lvwColors.InsertionMark.Index = -1;
                _lvwColors.InsertionMark.AppearsAfterItem = false;

                _lvwColors.RedrawItems(index, index, false);
            }
        }

        /// <summary>
        ///
        /// </summary>
        ListViewItem FindTreeViewItemByColorData_( ColorData colorData  )
        {
            foreach( ListViewItem lvi in _lvwColors.Items )
            {
                if( lvi.Tag == colorData )
                {
                    return lvi;
                }
            }
            return null;
        }

        /// <summary>
        /// ドロップの実行
        /// </summary>
        private void Event_LvwColors_DragDrop( object sender, DragEventArgs e )
        {
            Tuple<ColorData, ColorEditPanel.DragDropAction> data;
            if ( ColorData.TryToGetColorDataFromDropItem( e, out data ) )
            {
                ColorData colorData = data.Item1;

                int targetIndex = _lvwColors.InsertionMark.Index;
                    if( targetIndex != -1 )
                    {
                        if( _lvwColors.InsertionMark.AppearsAfterItem )
                        {
                            targetIndex++;
                        }


                        ListViewItem newItem = GetNewItem_(
                            colorData.Color,
                            colorData.Comment );

                        _lvwColors.Items.Insert( targetIndex, newItem );

                        // ドラッグアイテムを検索します。
                        // 他コントロールからのドロップの場合は、発見されません。
                        ListViewItem lviDragged = FindTreeViewItemByColorData_( colorData );
                        if( lviDragged != null )
                        {
                            _lvwColors.Items.Remove( lviDragged );
                        }

                        //--------------------------------------------------
                        // HACK : 原因については、はっきりしないが、
                        // View プロパティの変更を行わないと、並び替えの結果が反映されない
                        View viewCurrent = _lvwColors.View;
                        _lvwColors.View = View.List;
                        _lvwColors.View = viewCurrent;
                    }
                    Event_LvwColors_DragLeave( null, null );
                //}
            }
        }

        /// <summary>
        /// ドラッグドロップ時のアクションを決める。
        /// </summary>
        ColorEditPanel.DragDropAction selectDragDropAction(int xPos)
        {
            if (xPos <= this._clhColorIcon.Width / 2)
            {
                return ColorEditPanel.DragDropAction.Copy_RGB;
            }
            else
            {
                return ColorEditPanel.DragDropAction.Copy_Alpha;
            }
        }

        #endregion ドラッグ・ドロップ処理関連

        //------------------------------------------------------
        #region インポート・エクスポート

        /// <summary>
        /// lcc インポート
        /// </summary>
        private void _tsbImport_Click( object sender, EventArgs e )
        {
            OpenFileDialog   ofd = new OpenFileDialog();
            ofd.InitialDirectory = Environment.GetFolderPath(Environment.SpecialFolder.Desktop);
            ofd.Filter = "layout color chart files (*.lcc)|*.lcc";
            ofd.RestoreDirectory = true;
            if( ofd.ShowDialog( this ) == DialogResult.OK )
            {
                XmlDocument		document = new XmlDocument();
                try
                {
                    document.Load( ofd.FileName );
                    XmlNodeList xmlNodeList = document.SelectNodes( "ctr_layout_color_chart/ColorData" );
                    foreach( XmlNode node in xmlNodeList )
                    {
                        FloatColor color = new FloatColor(ColorTranslator.FromHtml( node.Attributes["color"].Value ));
                        string commnet = node.Attributes["comment"].Value;

                        _lvwColors.Items.Add( GetNewItem_( color, commnet ) );
                    }
                }
                catch
                {
                    LECore.DbgConsole.WriteLine( "Error in *.lcc import." );
                }
            }

            UpdateProperty_();
        }

        /// <summary>
        /// lcc エクスポート
        /// </summary>
        private void _tsbExport_Click( object sender, EventArgs e )
        {
            SaveFileDialog sfd = new SaveFileDialog();

            sfd.Filter = "layout color chart files (*.lcc)|*.lcc";
            sfd.RestoreDirectory = true;
            sfd.AddExtension = true;
            sfd.InitialDirectory = Environment.GetFolderPath(Environment.SpecialFolder.Desktop);

            if( sfd.ShowDialog( this ) == DialogResult.OK )
            {
                try
                {
                    XmlDocument document = new XmlDocument();
                    XmlElement rootElement = document.CreateElement( "ctr_layout_color_chart" );

                    foreach( ListViewItem lvi in _lvwColors.Items )
                    {
                        ColorData cd = lvi.Tag as ColorData;
                        Debug.Assert( cd != null );

                        XmlElement element = document.CreateElement( "ColorData" );
                        element.SetAttribute( "color", ConverColorToHtmlString( cd.Color.ToSystemColor() ) );
                        element.SetAttribute( "comment", cd.Comment );

                        rootElement.AppendChild( element );
                    }
                    XmlDeclaration decl = document.CreateXmlDeclaration( "1.0", "UTF-8", "yes" );
                    document.AppendChild( decl );
                    document.AppendChild( rootElement );
                    document.Save( sfd.FileName );
                }
                catch
                {
                    LECore.DbgConsole.WriteLine( "Error in *.lcc export." );
                }
            }
        }

        #endregion インポート・エクスポート

        /// <summary>
        /// ラベル編集前ハンドラ
        /// </summary>
        private void Event_LvwColors_BeforeSubItemLabelEdit(object sender, SubItemEventArgs e)
        {
            _tsbDelete.Enabled = false;
            _tsbClear.Enabled = false;
        }

        /// <summary>
        /// ラベル編集後ハンドラ
        /// </summary>
        private void Event_LvwColors_AfterSubItemLabelEdit(object sender, SubItemEndEditingEventArgs e)
        {
            ColorData cd = e.Item.Tag as ColorData;
            cd.Comment = e.DisplayText;
            _ttbColorComment.Text = e.DisplayText;
            _tsbDelete.Enabled = true;
            _tsbClear.Enabled = true;
        }

        /// <summary>
        ///
        /// </summary>
        private void ColorSetWindow_KeyDown( object sender, KeyEventArgs e )
        {
            if( e.KeyData == Keys.F2 )
            {
                if( _lvwColors.SelectedItems.Count == 1 )
                {
                    // デフォルトのラベル編集機能との併用は考慮されていないため
                    // 直接StartEditing()を呼び出す
                    _lvwColors.StartEditing(_labelEditTextBox, _lvwColors.SelectedItems[0], 1);
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void Event_LvwColors_MouseDoubleClick( object sender, MouseEventArgs e )
        {
            if( _lvwColors.SelectedItems.Count > 0 )
            {
                Point posClicked = new Point( e.X, e.Y );
                if( _lvwColors.SelectedItems[0].Bounds.Contains( posClicked ) )
                {
                    Event_TsbEdit_Click( null, null );
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void Event_LvwColors_MouseDown( object sender, MouseEventArgs e )
        {
            // アイテム外をマウスダウンで、新規作成。
            if ( _lvwColors.GetItemAt( e.X, e.Y ) == null )
            {
                Event_TsbNew_Click( null, null );
            }
            else
            {
                _dragDropAction = selectDragDropAction(e.X);
            }
        }
        #endregion イベントハンドラ
    }
}
