﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using System.Diagnostics;
using System.Drawing;
using System.Windows.Forms;
using LECore;
using LECore.Win32;
using System.Runtime.InteropServices;

namespace LayoutEditor.Controls
{

    public class MessageForwarder : NativeWindow, IMessageFilter
    {
        readonly private Control _Control;
        readonly private Form _Form;
        readonly private HashSet<int> _Messages;

        private Control _PreviousParent;
        private bool _IsMouseOverControl;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public MessageForwarder(Control control, int message)
            : this(control, new int[] { message })
        {
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public MessageForwarder(Control control, IEnumerable<int> messages)
        {
            _Control = control;
            _Form = _Control.FindForm();
            AssignHandle(control.Handle);

            _Messages = new HashSet<int>(messages);
            _PreviousParent = control.Parent;
            _IsMouseOverControl = false;

            if (control.Parent != null)
            {
                Application.AddMessageFilter(this);
            }
        }

        public bool PreFilterMessage(ref Message m)
        {
            if (_Control.Parent == null)
            {
                Application.RemoveMessageFilter(this);
                return false;
            }

            // マウスの内外判定を行う
            if (WM.WM_MOUSEMOVE == m.Msg)
            {
                POINT pos = new POINT();
                pos.x = Control.MousePosition.X;
                pos.y = Control.MousePosition.Y;

                // アクティブでなければマウスOFF。
                _IsMouseOverControl = _Control.Bounds.Contains(_Control.Parent.PointToClient(Control.MousePosition));
            }

            if (_IsMouseOverControl && _Control.CanFocus && !_Control.Focused && _Messages.Contains(m.Msg))
            {
                var origHandle = m.HWnd;
                m.HWnd = _Control.Handle;
                WndProc(ref m);
                m.HWnd = origHandle;
                return true;
            }

            return false;
        }
    }

    /// <summary>
    /// ヘルパ関数
    /// </summary>
    static class UIHelper
    {
        #region HeaderUtility
        /// <summary>
        /// ヘッダユーティリティクラス。
        /// </summary>
        public static class HeaderUtility
        {
            /// <summary>
            /// HDM_SETITEM。
            /// </summary>
            public static void SetItem(IntPtr handle, int index, ref LECore.Win32.HDITEM hdItem)
            {
                LECore.Win32.User32.SendMessage(handle, LECore.Win32.HDM.HDM_SETITEMW, (IntPtr)index, ref hdItem);
            }

            /// <summary>
            /// HDM_GETITEM。
            /// </summary>
            public static void GetItem(IntPtr handle, int index, ref LECore.Win32.HDITEM hdItem)
            {
                LECore.Win32.User32.SendMessage(handle, LECore.Win32.HDM.HDM_GETITEMW, (IntPtr)index, ref hdItem);
            }

            /// <summary>
            /// HDM_GETITEMRECT。
            /// </summary>
            public static Rectangle GetItemRect(IntPtr handle, int index)
            {
                LECore.Win32.RECT rect = new LECore.Win32.RECT();
                LECore.Win32.User32.SendMessage(handle, LECore.Win32.HDM.HDM_GETITEMRECT, (IntPtr)index, ref rect);
                return rect.ToRectangle();
            }
        }
        #endregion

        public static byte ClampDecimalToByte( Decimal val )
        {
            if( val > byte.MaxValue )
            {
                return byte.MaxValue;
            }
            else if( val < byte.MinValue )
            {
                return byte.MinValue;
            }
            else
            {
                return Convert.ToByte( val );
            }
        }

        /// <summary>
        /// 少々強引な、デザインモードの判定。
        /// @ITのスレッド書き込みより、
        /// http://72.14.235.104/search?q=cache:xmRfM1Lpt7IJ:www.atmarkit.co.jp/bbs/phpBB/viewtopic.php%3Ftopic%3D3314%26forum%3D7+Component.DesignMode&hl=ja&ct=clnk&cd=4&gl=jp
        /// </summary>
        static bool _IsDesignMode
        {
            get { return
                AppDomain.CurrentDomain != null &&
                AppDomain.CurrentDomain.FriendlyName == "DefaultDomain"; }
        }

        /// <summary>
        /// 非デザインモードの場合にのみ機能するアサート
        /// </summary>
        public static void UIAssert( bool bCondition )
        {
            if( !_IsDesignMode )
            {
                Debug.Assert( bCondition );
            }
        }

        /// <summary>
        /// 複数行の文字列をスペース区切りの一行にします。
        /// </summary>
        public static string ChangeLinesMultiToSingle(string multiLine)
        {
            return multiLine != null ? multiLine.Replace("\n", " ") : null;
        }
    }
}
