﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.ComponentModel;
using System.Collections;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Windows.Forms;
using System.Windows.Forms.Design;

namespace LayoutEditor.Controls
{
    using Win32 = LECore.Win32;

    /// <summary>
    /// ＵＩツリービュークラス。
    /// </summary>
    [ToolboxBitmap(typeof(TreeView))]
    public class UITreeView : TreeView
    {
        // 描画バッファ
        private Bitmap   _internalBmp = null;
        private Graphics _internalGfx = null;

        /// <summary>コンテキストメニューポップアップイベント。</summary>
        public event ContextMenuPopupEventHandler ContextMenuPopup = null;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UITreeView()
        {
        }

        /// <summary>
        /// コンテキストメニューポップアップイベント処理。
        /// </summary>
        protected virtual void OnContextMenuPopup(ContextMenuPopupEventArgs e)
        {
            if (this.ContextMenuPopup != null)
            {
                this.ContextMenuPopup(this, e);
            }
        }

        /// <summary>
        /// 内部コンポーネントを破棄。
        /// </summary>
        private void DisposeComponent()
        {
            if (_internalGfx != null)
            {
                _internalGfx.Dispose();
                _internalGfx = null;
            }
            if (_internalBmp != null)
            {
                _internalBmp.Dispose();
                _internalBmp = null;
            }
        }

        #region オーバーライド
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                DisposeComponent();
            }
            base.Dispose(disposing);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void WndProc(ref Message m)
        {
            switch (m.Msg)
            {
            case Win32.WM.WM_ERASEBKGND:
                return;
            case Win32.WM.WM_PAINT:
            {
                // 描画バッファ未作成なら作成
                if (_internalGfx == null)
                {
                    OnResize(EventArgs.Empty);
                }

                // 更新領域
                Win32.RECT rcUpdate = new Win32.RECT();
                if (!Win32.User32.GetUpdateRect(m.HWnd, ref rcUpdate, false))
                {
                    break;
                }

                // 描画処理
                Win32.PAINTSTRUCT ps = new Win32.PAINTSTRUCT();
                IntPtr pdc = Win32.User32.BeginPaint(m.HWnd, ref ps);
                {
                    using (Graphics g = Graphics.FromHdc(pdc))
                    {
                        // 描画バッファに既定描画をさせる
                        IntPtr hdc = _internalGfx.GetHdc();
                        {
                            Message msg = Message.Create(this.Handle, Win32.WM.WM_PRINTCLIENT, hdc, IntPtr.Zero);
                            base.DefWndProc(ref msg);
                        }
                        _internalGfx.ReleaseHdc(hdc);

                        // 派生クラス用にイベント発行
                        OnPaint(new PaintEventArgs(
                            _internalGfx,
                            Rectangle.FromLTRB(rcUpdate.left, rcUpdate.top, rcUpdate.right, rcUpdate.bottom)
                        ));

                        // ビットマップ転送
                        g.DrawImage(_internalBmp, 0, 0);
                    }
                }
                Win32.User32.EndPaint(m.HWnd, ref ps);
                return;
            }
            default:
                break;
            }
            base.WndProc(ref m);
        }

        #region Resize
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnResize(EventArgs e)
        {
            // 描画バッファを再作成
            if (_internalBmp        == null       ||
                _internalBmp.Width  != this.Width ||
                _internalBmp.Height != this.Height)
            {
                if (this.Width > 0 && this.Height > 0)
                {
                    DisposeComponent();
                    _internalBmp = new Bitmap(this.Width, this.Height);
                    _internalGfx = Graphics.FromImage(_internalBmp);
                }
            }
            base.OnResize(e);
        }
        #endregion

        #region KeyUp
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnKeyUp(KeyEventArgs e)
        {
            base.OnKeyUp(e);

            // コンテキストメニュー
            if (e.KeyCode == Keys.Apps)
            {
                // コンテキストメニュー
                // ノード位置が取れないのでカーソル位置に表示
                OnContextMenuPopup(new ContextMenuPopupEventArgs(Cursor.Position));
            }
        }
        #endregion

        #region MouseDown
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnMouseDown(MouseEventArgs e)
        {
            base.OnMouseDown(e);

            // 左右ボタン
            if (e.Button == MouseButtons.Left || e.Button == MouseButtons.Right)
            {
                // 選択済みじゃなければ選択
                TreeNode node = GetNodeAt(e.X, e.Y);
                if (node != null && node != this.SelectedNode)
                {
                    this.SelectedNode = node;
                }
            }
        }
        #endregion

        #region MouseUp
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnMouseUp(MouseEventArgs e)
        {
            base.OnMouseUp(e);

            // 右ボタン
            if (e.Button == MouseButtons.Right)
            {
                // コンテキストメニュー
                OnContextMenuPopup(new ContextMenuPopupEventArgs(new Point(e.X, e.Y)));
            }
        }
        #endregion

        private void InitializeComponent()
        {
            this.SuspendLayout();
            this.ResumeLayout( false );

        }
        #endregion
    }

    #region ContextMenuPopupEvent
    /// <summary>
    /// コンテキストメニューポップアップイベントハンドラデリゲート。
    /// </summary>
    public delegate void ContextMenuPopupEventHandler(object sender, ContextMenuPopupEventArgs e);

    /// <summary>
    /// コンテキストメニューポップアップイベントデータクラス。
    /// </summary>
    public sealed class ContextMenuPopupEventArgs : EventArgs
    {
        private readonly Point _point;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ContextMenuPopupEventArgs(Point point)
        {
            _point = point;
        }

        /// <summary>
        /// ポイント。
        /// </summary>
        public Point Point
        {
            get { return _point; }
        }
    }
    #endregion
}
