﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Data;
using System.Text;
using System.Windows.Forms;
using System.Drawing.Drawing2D;
using LayoutEditor.Utility;

namespace LayoutEditor.Controls
{
    static class PictureBoxHelper
    {
        /// <summary>
        /// 今設定されている画像を破棄しつつ新しく画像を設定する。
        /// </summary>
        public static void SetImageWithDispose(PictureBox pb, Image newImage)
        {
            Image currentImage = pb.Image;

            pb.Image = newImage;

            if (currentImage != null)
            {
                currentImage.Dispose();
            }
        }
    }

    /// <summary>
    /// 画像の表示制御を一括して行う機能を追加したPictureBox
    ///
    /// </summary>
    public partial class UIPictureBox : PictureBox
    {
        public delegate bool GetVisiblityFlagHandler();
        public static HatchBrush _bgBrush = new HatchBrush(HatchStyle.LargeCheckerBoard, Color.White, Color.LightGray);

        GetVisiblityFlagHandler _getVisiblityFlagHandler = null;

        /// <summary>
        /// カスタム描画イベント
        /// </summary>
        public event Action<UIPictureBox, PaintEventArgs> OnCustumPaint;

        /// <summary>
        /// 引き伸ばして描画するか
        /// </summary>
        public bool StretchDraw
        {
            get;
            set;
        }

        /// <summary>
        /// チェックバッグを描画するか？
        /// </summary>
        public bool DrawCheckBG
        {
            get;
            set;
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public UIPictureBox()
        {
            InitializeComponent();
        }

        /// <summary>
        /// 可視状態を取得するハンドラを設定します。
        /// 初期化は一度だけ行われることを想定しています。
        /// </summary>
        public void InitializeGetVisiblityHandler( GetVisiblityFlagHandler handler )
        {
            UIHelper.UIAssert( handler != null );
            UIHelper.UIAssert( _getVisiblityFlagHandler == null );

            _getVisiblityFlagHandler = handler;
        }

        /// <summary>
        /// OnPaint
        /// </summary>
        protected override void OnPaint(PaintEventArgs pe)
        {
            // 可視状態以外は、背景色でクリアする。
            bool bVisible = (_getVisiblityFlagHandler != null) ? _getVisiblityFlagHandler() : true;
            if (!bVisible)
            {
                pe.Graphics.Clear(this.BackColor);
                return;
            }

            if (!TextureThumbnailHelper.CheckImageValid(this.Image))
            {
                pe.Graphics.Clear(this.BackColor);
                return;
            }

            if (OnCustumPaint != null)
            {
                OnCustumPaint(this, pe);
                return;
            }
            else
            {
                if (StretchDraw)
                {
                    Debug.Assert(this.Image != null);

                    Rectangle imgRect = TextureThumbnailHelper.GetRect(this.Image.Width, this.Image.Height, this.Width, this.Height);
                    imgRect = TextureThumbnailHelper.AdjustRect(imgRect, this.BorderStyle != BorderStyle.None);

                    pe.Graphics.InterpolationMode = InterpolationMode.NearestNeighbor;

                    if (DrawCheckBG)
                    {
                        pe.Graphics.PixelOffsetMode = PixelOffsetMode.None;
                        pe.Graphics.FillRectangle(_bgBrush, imgRect);
                    }
                    else
                    {
                        pe.Graphics.Clear(this.BackColor);
                    }

                    pe.Graphics.PixelOffsetMode = PixelOffsetMode.Half;
                    pe.Graphics.DrawImage(this.Image, imgRect);

                    pe.Graphics.PixelOffsetMode = PixelOffsetMode.None;
                    pe.Graphics.DrawRectangle(Pens.DarkGray, imgRect);

                }
                else
                {
                    base.OnPaint(pe);
                }
            }
        }
    }
}
