﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Windows.Forms;
using LayoutEditor.Controls;
using LECore.Util;
using LayoutEditor.Utility;

namespace LayoutEditor.Controls.UI
{
    /// <summary>
    /// ＵＩコントロールクラス。
    /// </summary>
    public abstract class UIControl : Control
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UIControl()
        {
        }
    }

    //-------------------------------------------------------------------------
    // UIControl 派生クラス
    //-------------------------------------------------------------------------
    #region UnselectableControl
    /// <summary>
    /// 選択不可コントロールクラス。
    /// </summary>
    public abstract class UnselectableControl : UIControl
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UnselectableControl()
        {
            base.SetStyle(ControlStyles.Opaque, true);
            base.SetStyle(ControlStyles.ResizeRedraw, true);
            base.SetStyle(ControlStyles.Selectable, false);
            base.DoubleBuffered = true;
            base.TabStop = false;
        }

        #region デザイナ制御
        /// <summary>
        /// 再定義。
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public new int TabIndex
        {
            get { return base.TabIndex; }
            set { base.TabIndex = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public new bool TabStop
        {
            get { return base.TabStop; }
            set { base.TabStop = value; }
        }
        #endregion
    }
    #endregion

    //-------------------------------------------------------------------------
    // UIControl ユーティリティクラス
    //-------------------------------------------------------------------------
    #region UIControlEventSuppressBlock
    /// <summary>
    /// ＵＩコントロールイベント抑制ブロッククラス。
    /// </summary>
    public sealed class UIControlEventSuppressBlock : IDisposable
    {
        // インスタンスリスト
        private static readonly List<UIControlEventSuppressBlock> _instances = new List<UIControlEventSuppressBlock>();

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UIControlEventSuppressBlock()
        {
            // リストに追加
            _instances.Add(this);
        }

        /// <summary>
        /// 開放処理。
        /// </summary>
        public void Dispose()
        {
            // リストから削除
            _instances.Remove(this);
        }

        /// <summary>
        /// 抑制中かどうか。
        /// </summary>
        public static bool Enabled
        {
            get
            {
                // インスタンスがあれば抑制中
                return _instances.Count > 0;
            }
        }
    }
    #endregion

    #region UIControlHelper
    /// <summary>
    /// ＵＩコントロールヘルパクラス。
    /// </summary>
    public static class UIControlHelper
    {
        //---------------------------------------------------------------------------
        // 汎用カテゴリ名
        /// <summary>オリジナルプロパティカテゴリ名。</summary>
        public const string OriginalPropertyCategoryName = "[独自プロパティ]";
        /// <summary>オリジナルイベントカテゴリ名。</summary>
        public const string OriginalEventCategoryName = "[独自イベント]";

        //---------------------------------------------------------------------------
        // 汎用プロパティ説明（DescriptionAttribute 用）
        /// <summary>拡張ボーダースタイルプロパティ説明。</summary>
        public const string PropertyDesc_ExtendedBorderStyle = "拡張ボーダースタイル（WS_EX_STATICEDGE、WS_EX_DLGMODALFRAME）を示します。";
        /// <summary>ダブルバッファリングプロパティ説明。</summary>
        public const string PropertyDesc_DoubleBufferring = "ダブルバッファリングするかどうかを示します。";

        //---------------------------------------------------------------------------
        // ユーティリティメソッド
        /// <summary>
        /// 拡張ボーダースタイル用に CreateParams を設定。
        /// </summary>
        public static void SetExtendedBorderStyleCreateParams(CreateParams cp, ExtendedBorderStyle style)
        {
            switch (style)
            {
                case ExtendedBorderStyle.StaticEdge:
                    cp.ExStyle |= LECore.Win32.WS_EX.WS_EX_STATICEDGE;
                    break;
                case ExtendedBorderStyle.DlgModalFrame:
                    cp.ExStyle |= LECore.Win32.WS_EX.WS_EX_DLGMODALFRAME;
                    break;
                default:
                    break;
            }
        }

        /// <summary>
        /// 浮動小数値の表示用文字列を取得。
        /// </summary>
        public static string GetFloatValueDisplayString(float value, FloatValueDisplayStyle style)
        {
            switch (style)
            {
                case FloatValueDisplayStyle.AutoDecimal:
                    return GetFloatValueAutoDecimalString(value);
                case FloatValueDisplayStyle.Standard:
                    return value.ToString();
                case FloatValueDisplayStyle.Fixed1:
                    return value.ToString("f1");
                case FloatValueDisplayStyle.Fixed2:
                    return value.ToString("f2");
                case FloatValueDisplayStyle.Fixed3:
                    return value.ToString("f3");
                case FloatValueDisplayStyle.Fixed4:
                    return value.ToString("f4");
                case FloatValueDisplayStyle.Fixed5:
                    return value.ToString("f5");
                case FloatValueDisplayStyle.Fixed6:
                    return value.ToString("f6");
                default:
                    Debug.Assert(false);
                    break;
            }
            return value.ToString();
        }

        /// <summary>
        /// 浮動小数値の自動小数形式文字列を取得。
        /// </summary>
        private static string GetFloatValueAutoDecimalString(float value)
        {
            string valueString = value.ToString("f6");

            // 小数点位置
            int periodIndex = valueString.LastIndexOf('.');
            if (periodIndex == -1)
            {
                return valueString + ".0";
            }

            // 小数部文字列
            string decimalString = valueString.Substring(periodIndex + 1);
            if (decimalString.Length <= 1)
            {
                return valueString;
            }

            // 右側から「０」の文字数を計測
            int rightZeroCount = 0;
            for (int i = decimalString.Length - 1; i >= 0; i--)
            {
                if (decimalString[i] == '0')
                {
                    rightZeroCount++;
                }
                else
                {
                    break;
                }
            }
            // 最低１桁は表示する
            if (rightZeroCount == decimalString.Length)
            {
                rightZeroCount--;
            }

            return valueString.Substring(0, valueString.Length - rightZeroCount);
        }
    }
    #endregion

    //-------------------------------------------------------------------------
    // イベント定義
    //-------------------------------------------------------------------------
    #region MouseBeginDragEvent
    /// <summary>
    /// マウスドラッグ開始イベントハンドラデリゲート。
    /// </summary>
    public delegate void MouseBeginDragEventHandler(object sender, MouseBeginDragEventArgs e);

    /// <summary>
    /// マウスドラッグ開始イベントデータクラス。
    /// </summary>
    public sealed class MouseBeginDragEventArgs : MouseEventArgs
    {
        // キャンセルフラグ
        private bool _cancel = false;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public MouseBeginDragEventArgs(MouseEventArgs e)
            : base(e.Button, e.Clicks, e.X, e.Y, e.Delta)
        {
        }

        /// <summary>
        /// ドラッグ開始をキャンセルするかどうか。
        /// </summary>
        public bool Cancel
        {
            get { return _cancel; }
            set { _cancel = value; }
        }
    }
    #endregion

    #region ContextMenuPopupEvent
    /// <summary>
    /// コンテキストメニューポップアップイベントハンドラデリゲート。
    /// </summary>
    public delegate void ContextMenuPopupEventHandler(object sender, ContextMenuPopupEventArgs e);

    /// <summary>
    /// コンテキストメニューポップアップイベントデータクラス。
    /// </summary>
    public class ContextMenuPopupEventArgs : EventArgs
    {
        // 位置
        private readonly Point _location;
        // キー入力からかどうか
        private readonly bool _fromAppKey;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ContextMenuPopupEventArgs(Point location, bool fromAppKey)
        {
            _location = location;
            _fromAppKey = fromAppKey;
        }

        /// <summary>
        /// 位置。
        /// </summary>
        public Point Location
        {
            get { return _location; }
        }

        /// <summary>
        /// 位置Ｘ。
        /// </summary>
        public int X
        {
            get { return _location.X; }
        }

        /// <summary>
        /// 位置Ｙ。
        /// </summary>
        public int Y
        {
            get { return _location.Y; }
        }

        /// <summary>
        /// キー入力からかどうか。
        /// </summary>
        public bool FromAppKey
        {
            get { return _fromAppKey; }
        }
    }
    #endregion

    #region ColumnContextMenuPopupEvent
    /// <summary>
    /// 列項目コンテキストメニューポップアップイベントハンドラデリゲート。
    /// </summary>
    public delegate void ColumnContextMenuPopupEventHandler(object sender, ColumnContextMenuPopupEventArgs e);

    /// <summary>
    /// 列項目コンテキストメニューポップアップイベントデータクラス。
    /// </summary>
    public sealed class ColumnContextMenuPopupEventArgs : ContextMenuPopupEventArgs
    {
        // 列項目
        private readonly ColumnHeader _header;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ColumnContextMenuPopupEventArgs(Point location, ColumnHeader header)
            : base(location, false)
        {
            _header = header;
        }

        /// <summary>
        /// 列項目。
        /// </summary>
        public ColumnHeader Header
        {
            get { return _header; }
        }
    }
    #endregion

    #region ColumnDividerDoubleClickEvent
    /// <summary>
    /// 列分割線ダブルクリックイベントハンドラデリゲート。
    /// </summary>
    public delegate void ColumnDividerDoubleClickEventHandler(object sender, ColumnDividerDoubleClickEventArgs e);

    /// <summary>
    /// 列分割線ダブルクリックイベントデータクラス。
    /// </summary>
    public sealed class ColumnDividerDoubleClickEventArgs : HandledEventArgs
    {
        // 列項目
        private readonly ColumnHeader _header;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ColumnDividerDoubleClickEventArgs(ColumnHeader header)
        {
            _header = header;
        }

        /// <summary>
        /// 列項目。
        /// </summary>
        public ColumnHeader Header
        {
            get { return _header; }
        }
    }
    #endregion

    #region CustomDrawItemEventArgs
    /// <summary>
    /// 項目カスタム描画イベントデータクラス。
    /// </summary>
    public class CustomDrawItemEventArgs : HandledEventArgs
    {
        //---------------------------------------------------------------------
        // ユーザー指定項目（既定描画用）
        //---------------------------------------------------------------------
        // 背景色
        private Color _backColor = Color.Empty;
        // 前景色
        private Color _foreColor = Color.Empty;
        // テキスト
        private string _text = null;
        // フォント
        private Font _font = null;
        // イメージ番号
        private int _imageIndex = -1;
        // イメージキー
        private string _imageKey = null;
        // イメージ
        private Image _image;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public CustomDrawItemEventArgs()
        {
        }

        /// <summary>
        /// 背景色の指定。
        /// </summary>
        public Color SpecificBackColor
        {
            get { return _backColor; }
            set { _backColor = value; }
        }

        /// <summary>
        /// 前景色の指定。
        /// </summary>
        public Color SpecificForeColor
        {
            get { return _foreColor; }
            set { _foreColor = value; }
        }

        /// <summary>
        /// テキストの指定。
        /// </summary>
        public string SpecificText
        {
            get { return _text; }
            set { _text = value; }
        }

        /// <summary>
        /// フォントの指定。
        /// </summary>
        public Font SpecificFont
        {
            get { return _font; }
            set { _font = value; }
        }

        /// <summary>
        /// イメージ番号の指定。
        /// </summary>
        public int SpecificImageIndex
        {
            get { return _imageIndex; }
            set { _imageIndex = value; }
        }

        /// <summary>
        /// イメージキーの指定。
        /// </summary>
        public string SpecificImageKey
        {
            get { return _imageKey; }
            set { _imageKey = value; }
        }

        /// <summary>
        /// イメージの指定。
        /// </summary>
        public Image SpecificImage
        {
            get { return _image; }
            set { _image = value; }
        }

        /// <summary>
        /// イメージを取得。
        /// </summary>
        protected Image GetImage(ImageList imageList, int index, string key)
        {
            if (imageList != null)
            {
                if (index >= 0 && index < imageList.Images.Count)
                {
                    return imageList.Images[index];
                }
                else if (key != null && imageList.Images.ContainsKey(key))
                {
                    return imageList.Images[key];
                }
            }
            return null;
        }
    }
    #endregion

    #region CustomDrawListViewItemEvent
    /// <summary>
    /// リストビュー項目カスタム描画イベントハンドラデリゲート。
    /// </summary>
    public delegate void CustomDrawListViewItemEventHandler(object sender, CustomDrawListViewItemEventArgs e);

    /// <summary>
    /// リストビュー項目カスタム描画イベントデータクラス。
    /// </summary>
    public sealed class CustomDrawListViewItemEventArgs : CustomDrawItemEventArgs
    {
        // リストビュー
        private readonly UIListView _listView;
        // 元イベントデータ
        private readonly DrawListViewSubItemEventArgs _eventArgs;
        // 領域
        private readonly Rectangle _bounds;
        // 選択状態
        private readonly bool _selected;
        // 既定のイメージ描画を無視
        private bool _ignoreDefaultImage = false;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public CustomDrawListViewItemEventArgs(UIListView listView, DrawListViewSubItemEventArgs eventArgs, Rectangle bounds, bool selected)
        {
            // eventArgs.Bounds は列順序入れ替え時に
            // 正しい値にならないので参照しない
            _listView  = listView;
            _eventArgs = eventArgs;
            _bounds    = bounds;
            _selected  = selected;
        }

        /// <summary>
        /// グラフィクス。
        /// </summary>
        public Graphics Graphics
        {
            get { return _eventArgs.Graphics; }
        }

        /// <summary>
        /// 領域。
        /// </summary>
        public Rectangle Bounds
        {
            get { return _bounds; }
        }

        /// <summary>
        /// 項目背景色。
        /// </summary>
        public Color ItemBackColor
        {
            get { return _eventArgs.SubItem.BackColor; }
        }

        /// <summary>
        /// 項目前景色。
        /// </summary>
        public Color ItemForeColor
        {
            get { return _eventArgs.SubItem.ForeColor; }
        }

        /// <summary>
        /// 項目テキスト。
        /// </summary>
        public string ItemText
        {
            get { return _eventArgs.SubItem.Text; }
        }

        /// <summary>
        /// 項目フォント。
        /// </summary>
        public Font ItemFont
        {
            get { return _eventArgs.SubItem.Font; }
        }

        /// <summary>
        /// 既定描画時の背景カラー。
        /// </summary>
        public Color DefaultDrawBackColor
        {
            get
            {
                if (this.IsSelected)
                {
                    if (_listView.Focused)
                    {
                        return SystemColors.Highlight;
                    }
                    else
                    {
                        return SystemColors.InactiveBorder;
                    }
                }
                else
                {
                    return SystemColors.Window;
                }
            }
        }

        /// <summary>
        /// 既定描画時の前景カラー。
        /// </summary>
        public Color DefaultDrawForeColor
        {
            get
            {
                if (this.IsSelected && (_listView.Focused))
                {
                    return SystemColors.HighlightText;
                }
                else
                {
                    return SystemColors.WindowText;
                }
            }
        }

        /// <summary>
        /// リストビュー。
        /// </summary>
        public UIListView ListView
        {
            get { return _listView; }
        }

        /// <summary>
        /// 項目。
        /// </summary>
        public ListViewItem Item
        {
            get { return _eventArgs.Item; }
        }

        /// <summary>
        /// 項目番号。
        /// </summary>
        public int ItemIndex
        {
            get { return _eventArgs.ItemIndex; }
        }

        /// <summary>
        /// サブ項目。
        /// </summary>
        public ListViewItem.ListViewSubItem SubItem
        {
            get { return _eventArgs.SubItem; }
        }

        /// <summary>
        /// 列番号。
        /// </summary>
        public int ColumnIndex
        {
            get { return _eventArgs.ColumnIndex; }
        }

        /// <summary>
        /// 列ヘッダ。
        /// </summary>
        public ColumnHeader ColumnHeader
        {
            get { return _eventArgs.Header; }
        }

        /// <summary>
        /// 既定のイメージ描画を無視。
        /// </summary>
        public bool IgnoreDefaultImage
        {
            get { return _ignoreDefaultImage; }
            set { _ignoreDefaultImage = value; }
        }

        /// <summary>
        /// 選択状態か。
        /// </summary>
        public bool IsSelected
        {
            get { return _selected; }
        }

        /// <summary>
        /// フォーカス状態か。
        /// </summary>
        public bool IsFocused
        {
            get { return (_eventArgs.ItemState & ListViewItemStates.Focused) != 0; }
        }

        /// <summary>
        /// チェック状態か。
        /// </summary>
        public bool IsChecked
        {
            get { return (_eventArgs.ItemState & ListViewItemStates.Checked) != 0; }
        }

        /// <summary>
        /// 中間状態か（３状態チェック時）。
        /// </summary>
        public bool IsIndeterminate
        {
            get { return (_eventArgs.ItemState & ListViewItemStates.Indeterminate) != 0; }
        }

        /// <summary>
        /// 既定の描画処理。
        /// </summary>
        public void DrawDefault()
        {
            //-----------------------------------------------------------------
            // 背景
            if (this.SpecificBackColor != Color.Empty)
            {
                Rectangle rect = this.Bounds;

                // 選択時は選択色とブレンド
                Color color = this.SpecificBackColor;
                if (this.IsSelected)
                {
                    if (_listView.Focused)
                    {
                        color = ColorBlender.Blend(this.SpecificBackColor, SystemColors.Highlight);
                    }
                    else
                    {
                        color = ColorBlender.Blend(this.SpecificBackColor, SystemColors.InactiveBorder);
                    }
                }
                using (Brush brush = new SolidBrush(color))
                {
                    this.Graphics.FillRectangle(brush, rect);
                }
            }

            //-----------------------------------------------------------------
            // 前景色
            Color foreColor = this.ItemForeColor;
            if (this.SpecificForeColor != Color.Empty)
            {
                foreColor = this.SpecificForeColor;
            }
            else if (this.IsSelected)
            {
                if (_listView.Focused)
                {
                    foreColor = SystemColors.HighlightText;
                }
            }

            //-----------------------------------------------------------------
            // テキスト
            string text = this.ItemText;
            if (this.SpecificText != null)
            {
                text = this.SpecificText;
            }

            //-----------------------------------------------------------------
            // フォント
            Font font = this.ItemFont;
            if (this.SpecificFont != null)
            {
                font = this.SpecificFont;
            }

            //-----------------------------------------------------------------
            // イメージ
            Image image = null;
            if (this.SpecificImage != null)
            {
                image = this.SpecificImage;
            }
            else
            {
                // 指定イメージ番号＆イメージキーで検索
                image = GetImage(_listView.SmallImageList, this.SpecificImageIndex, this.SpecificImageKey);
                if (image == null)
                {
                    // 項目０の既定イメージ
                    if (_eventArgs.ColumnIndex == 0 && !_ignoreDefaultImage)
                    {
                        ListViewItem item = _eventArgs.Item;
                        image = GetImage(_listView.SmallImageList, item.ImageIndex, item.ImageKey);
                    }
                }
            }

            //-----------------------------------------------------------------
            // ラベル描画
            Rectangle rcLabel = this.Bounds;
            rcLabel.Inflate(-1, 0);
            GraphicsUtil.DrawText(
                this.Graphics,
                text,
                font,
                foreColor,
                rcLabel,
                _eventArgs.Header.TextAlign,
                image
            );
        }
    }
    #endregion

    #region CompareListViewItemEvent
    /// <summary>
    /// リストビュー項目比較イベントハンドラデリゲート。
    /// </summary>
    public delegate int CompareListViewItemEventHandler(object sender, CompareListViewItemEventArgs e);

    /// <summary>
    /// リストビュー項目比較イベントデータクラス。
    /// </summary>
    public sealed class CompareListViewItemEventArgs : EventArgs
    {
        // 列番号
        private readonly int _columnIndex;
        // 列ヘッダ
        private readonly ColumnHeader _columnHeader;
        // 比較項目１
        private readonly ListViewItem _item1;
        // 比較項目２
        private readonly ListViewItem _item2;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public CompareListViewItemEventArgs(int columnIndex, ColumnHeader columnHeader, ListViewItem item1, ListViewItem item2)
        {
            _columnIndex  = columnIndex;
            _columnHeader = columnHeader;
            _item1        = item1;
            _item2        = item2;
        }

        /// <summary>
        /// 列番号。
        /// </summary>
        public int ColumnIndex
        {
            get { return _columnIndex; }
        }

        /// <summary>
        /// 列ヘッダ。
        /// </summary>
        public ColumnHeader ColumnHeader
        {
            get { return _columnHeader; }
        }

        /// <summary>
        /// 比較項目１。
        /// </summary>
        public ListViewItem Item1
        {
            get { return _item1; }
        }

        /// <summary>
        /// 比較項目２。
        /// </summary>
        public ListViewItem Item2
        {
            get { return _item2; }
        }

        #region 比較
        /// <summary>
        /// 整数値比較。
        /// </summary>
        public int CompareInt(int value1, int value2)
        {
            if      (value1 < value2) { return -1; }
            else if (value1 > value2) { return  1; }
            return 0;
        }

        /// <summary>
        /// 実数値比較。
        /// </summary>
        public int CompareFloat(float value1, float value2)
        {
            if      (value1 < value2) { return -1; }
            else if (value1 > value2) { return  1; }
            return 0;
        }

        /// <summary>
        /// 真偽値比較。
        /// </summary>
        public int CompareBool(bool value1, bool value2)
        {
            if      (!value1 &&  value2) { return -1; }
            else if ( value1 && !value2) { return  1; }
            return 0;
        }

        /// <summary>
        /// 文字列比較。
        /// </summary>
        public int CompareString(string value1, string value2)
        {
            {
                int intVal1;
                if (int.TryParse(value1, out intVal1))
                {
                    int intVal2;
                    if (int.TryParse(value2, out intVal2))
                    {
                        return CompareInt(intVal1, intVal2);
                    }
                }
            }

            {
                float floatVal1;
                if (float.TryParse(value1, out floatVal1))
                {
                    float floatVal2;
                    if (float.TryParse(value2, out floatVal2))
                    {
                        return CompareFloat(floatVal1, floatVal2);
                    }
                }
            }

            return string.Compare(value1, value2);
        }

        /// <summary>
        /// カラー比較（ＲＧＢ成分）。
        /// </summary>
        public int CompareColorRGB(Color value1, Color value2)
        {
            // 明度で比較
            int intensity1 = value1.R + value1.G + value1.B;
            int intensity2 = value2.R + value2.G + value2.B;

            if      (intensity1 < intensity2) { return -1; }
            else if (intensity1 > intensity2) { return  1; }

            // 成分で比較
            int component1 = value1.B << 16 | value1.G << 8 | value1.R;
            int component2 = value2.B << 16 | value2.G << 8 | value2.R;

            if      (component1 < component2) { return -1; }
            else if (component1 > component2) { return  1; }
            return 0;
        }

        /// <summary>
        /// カラー比較（ＲＧＢＡ成分）。
        /// </summary>
        public int CompareColorRGBA(Color value1, Color value2)
        {
            // 明度で比較
            int intensity1 = value1.R + value1.G + value1.B + value1.A;
            int intensity2 = value2.R + value2.G + value2.B + value2.A;

            if      (intensity1 < intensity2) { return -1; }
            else if (intensity1 > intensity2) { return  1; }

            // 成分で比較
            int component1 = value1.A << 24 | value1.B << 16 | value1.G << 8 | value1.R;
            int component2 = value2.A << 24 | value2.B << 16 | value2.G << 8 | value2.R;

            if      (component1 < component2) { return -1; }
            else if (component1 > component2) { return  1; }
            return 0;
        }

        /// <summary>
        /// デフォルト比較。
        /// </summary>
        public int CompareDefault()
        {
            // メイン項目
            if (_columnIndex == 0)
            {
                return CompareString(_item1.Text, _item2.Text);
            }
            // サブ項目
            else
            {
                int subItemIndex = _columnIndex;

                // 項目１
                ListViewItem.ListViewSubItem subItem1 = null;
                if (subItemIndex < _item1.SubItems.Count)
                {
                    subItem1 = _item1.SubItems[subItemIndex];
                }
                // 項目２
                ListViewItem.ListViewSubItem subItem2 = null;
                if (subItemIndex < _item2.SubItems.Count)
                {
                    subItem2 = _item2.SubItems[subItemIndex];
                }

                // 条件を分けて比較する
                if      (subItem1 == null && subItem2 == null) { return  0; }
                else if (subItem1 != null && subItem2 == null) { return -1; }
                else if (subItem1 == null && subItem2 != null) { return  1; }
                else
                {
                    return CompareString(subItem1.Text, subItem2.Text);
                }
            }
        }
        #endregion
    }
    #endregion

    //-------------------------------------------------------------------------
    // 列挙型定義
    //-------------------------------------------------------------------------
    #region ExtendedBorderStyle
    /// <summary>
    /// 拡張ボーダースタイル。
    /// </summary>
    public enum ExtendedBorderStyle
    {
        /// <summary>なし。</summary>
        None = 0,
        /// <summary>スタティックエッジ（WS_EX_STATICEDGE）。</summary>
        StaticEdge,
        /// <summary>ダイアログモーダルフレーム（WS_EX_DLGMODALFRAME）。</summary>
        DlgModalFrame,
    }
    #endregion

    #region FloatValueDisplayStyle
    /// <summary>
    /// 実数値表示スタイル。
    /// </summary>
    public enum FloatValueDisplayStyle
    {
        /// <summary>自動小数部桁数。</summary>
        AutoDecimal,
        /// <summary>標準（書式指定無し）。</summary>
        Standard,
        /// <summary>小数部固定１桁（{f1}）。</summary>
        Fixed1,
        /// <summary>小数部固定２桁（{f2}）。</summary>
        Fixed2,
        /// <summary>小数部固定３桁（{f3}）。</summary>
        Fixed3,
        /// <summary>小数部固定４桁（{f4}）。</summary>
        Fixed4,
        /// <summary>小数部固定５桁（{f5}）。</summary>
        Fixed5,
        /// <summary>小数部固定６桁（{f6}）。</summary>
        Fixed6,
    }
    #endregion
}
