﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;

namespace LayoutEditor.Controls
{
    /// <summary>
    /// 保存用クラスに保持されるデータの塊
    /// </summary>
    [Serializable]
    public class LEControlUserDataChunk
    {
        public string Name;
        public string Value;

        public LEControlUserDataChunk()
        {
        }

        public LEControlUserDataChunk( string name, string val )
        {
            Name = name;
            Value = val;
        }
    }

    /// <summary>
    /// 状態をシリアライズするコントロールが実装するインタフェース。
    /// </summary>
    public interface ILESerializableControl
    {
        void SaveState( out LEControlUserDataChunk[] dataSet );
        void LoadState( LEControlUserDataChunk[] dataSet );
    }

    /// <summary>
    /// ILESerializableControl関連ヘルパ
    /// </summary>
    public static class ILESerializableControlHelper
    {
        //------------------------------------------------------------
        #region private
        /// <summary>
        /// シリアライズする必要があるコントロールは、シリアライズを行います。
        /// 子供コントロールについて、再帰的に処理を行います。
        /// </summary>
        static void SaveControlState_( Control ctrl, List<LEControlUserDataChunk> dstDataSet )
        {
            ILESerializableControl serializableControl = ctrl as ILESerializableControl;

            if( serializableControl != null )
            {
                LEControlUserDataChunk[] saveData;
                serializableControl.SaveState( out saveData );
                if( saveData != null )
                {
                    dstDataSet.AddRange( saveData );
                }
            }

            foreach( Control childCtrl in ctrl.Controls )
            {
                SaveControlState_( childCtrl, dstDataSet );
            }
        }

        /// <summary>
        /// シリアライズする必要があるコントロールは、シリアライズを行います。
        /// 子供コントロールについて、再帰的に処理を行います。
        /// </summary>
        static void LoadControlState_( Control ctrl, LEControlUserDataChunk[] dataSet )
        {
            ILESerializableControl serializableControl = ctrl as ILESerializableControl;

            if( serializableControl != null )
            {
                serializableControl.LoadState( dataSet );
            }

            foreach( Control childCtrl in ctrl.Controls )
            {
                LoadControlState_( childCtrl, dataSet );
            }
        }
        #endregion

        /// <summary>
        /// フォームの状態を保存します。
        /// </summary>
        static public void SaveFormState( Form form, out LEControlUserDataChunk[] dataSet )
        {
            List<LEControlUserDataChunk> dstDataSet = new List<LEControlUserDataChunk>();
            foreach( Control c in form.Controls )
            {
                SaveControlState_( c, dstDataSet );
            }

            dataSet = dstDataSet.ToArray();
        }

        /// <summary>
        /// フォームの状態を読み込みます。
        /// </summary>
        static public void LoadFormState( Form form, LEControlUserDataChunk[] dataSet )
        {
            foreach( Control c in form.Controls )
            {
                LoadControlState_( c, dataSet );
            }
        }
    }
}
