﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.ComponentModel;
using System.Collections;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Windows.Forms;
using System.Windows.Forms.Design;

namespace LayoutEditor.Controls
{
    using LECore.Structures;

    /// <summary>
    /// カラーボタンクラス。
    /// </summary>
    [ToolboxBitmap(typeof(Button))]
    [Designer(typeof(ColorButton.CustomDesigner))]
    public sealed class ColorButton : Button
    {
        private FloatColor _color = FloatColor.White;
        private bool  _enableAlpha = false;
        private	bool  _buttonDown  = false;
        private bool  _mousePress  = false;
        private bool  _buttonKey   = false;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ColorButton()
        {
        }

        /// <summary>
        /// カラー。
        /// </summary>
        [Description("カラー。")]
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public FloatColor Color
        {
            get { return _color;  }
            set
            {
                _color = value;
                Invalidate();
            }
        }

        /// <summary>
        /// アルファ値有効フラグ。
        /// </summary>
        [DefaultValue(false)]
        [Description("アルファ値有効フラグ。")]
        public bool EnableAlpha
        {
            get { return _enableAlpha; }
            set
            {
                _enableAlpha = value;
                Invalidate();
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void WndProc(ref Message m)
        {
            switch (m.Msg)
            {
            case LECore.Win32.WM.WM_KEYDOWN:
            case LECore.Win32.WM.WM_KEYUP:
                {
                    Keys keyCode = (Keys)(int)m.WParam & Keys.KeyCode;
                    if (_mousePress && keyCode == Keys.Space)
                    {
                        return;
                    }
                }
                break;
            }
            base.WndProc(ref m);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnPaint(PaintEventArgs e)
        {
            bool	offset	= false;
            e.Graphics.Clear(this.BackColor);
            ButtonState	state	= ButtonState.Normal;
            if( !this.Enabled )
                state	|= ButtonState.Inactive;
            if( _buttonDown )
            {
                state	|= ButtonState.Pushed;
                offset	= true;
            }
            ControlPaint.DrawButton(e.Graphics, this.DisplayRectangle, state);
            {
                Rectangle	rect	= this.DisplayRectangle;
                rect.Inflate(-4, -4);
                if( offset )
                    rect.Offset(1, 1);
                if( this.Enabled )
                {
                    Color color = this.Color.ToSystemColor();
                    using( SolidBrush brush = new SolidBrush(System.Drawing.Color.FromArgb(255,color)) )
                    {
                        e.Graphics.FillRectangle(brush, rect);
                    }
                    //	アルファ部分描画
                    if( _enableAlpha )
                    {
                        Rectangle	half	= rect;
                        half.Offset(rect.Width/2, 0);
                        half.Size	= new Size(rect.Width/2, rect.Height);
                        using( SolidBrush brush = new SolidBrush(System.Drawing.Color.FromArgb(color.A, color.A, color.A)) )
                        {
                            e.Graphics.FillRectangle(brush, half);
                        }
                        e.Graphics.DrawLine( Pens.Black, half.X, half.Y, half.X, half.Y+rect.Height-1 );
                    }
                    LECore.Util.GraphicsUtil.DrawRectangle(e.Graphics, Pens.Black, rect);
                }
                else
                    LECore.Util.GraphicsUtil.DrawRectangle(e.Graphics, SystemPens.ControlDark, rect);
            }
            if( this.Focused )
            {
                Rectangle	rect	= this.DisplayRectangle;
                rect.Inflate(-5, -5);
                if( offset )
                    rect.Offset(1, 1);
                LECore.Util.GraphicsUtil.DrawFocusRectangle(e.Graphics, rect );
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnEnabledChanged(EventArgs e)
        {
            if (!this.Enabled)
            {
                _buttonDown = false;
            }
            base.OnEnabledChanged(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnMouseDown(MouseEventArgs e)
        {
            if ((e.Button & MouseButtons.Left) != 0)
            {
                _buttonDown = true;
                _mousePress = true;
                Invalidate();
            }
            base.OnMouseDown(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnMouseUp(MouseEventArgs mevent)
        {
            _buttonDown = false;
            _buttonKey  = false;
            _mousePress = false;
            Invalidate();

            base.OnMouseUp(mevent);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnMouseMove(MouseEventArgs e)
        {
            if ((e.Button != MouseButtons.None) && _mousePress)
            {
                if (!base.ClientRectangle.Contains(e.X, e.Y))
                {
                    if (_buttonDown)
                    {
                        _buttonDown = false;
                        Invalidate();
                    }
                }
                else if (!_buttonDown)
                {
                    _buttonDown = true;
                    Invalidate();
                }
            }
            base.OnMouseMove(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnKeyDown(KeyEventArgs e)
        {
            if (e.KeyCode == Keys.Space)
            {
                if (!_buttonDown)
                {
                    _buttonDown = true;
                    _buttonKey  = true;
                    Invalidate();
                    e.Handled = true;
                }
                else
                {
                    e.Handled = false;
                }
            }
            base.OnKeyDown(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnKeyUp(KeyEventArgs e)
        {
            if (_buttonDown && _buttonKey)
            {
                _buttonDown	= false;
                _buttonKey	= false;
                _mousePress	= false;
                Invalidate();
                this.OnClick(EventArgs.Empty);
                e.Handled = true;
            }
            base.OnKeyUp(e);
        }

        #region CustomDesigner
        /// <summary>
        /// カスタムデザイナクラス。
        /// </summary>
        internal class CustomDesigner : ControlDesigner
        {
            /// <summary>
            /// オーバーライド。
            /// </summary>
            protected override void PostFilterProperties(IDictionary properties)
            {
//				properties.Remove("AllowDrop");
//				properties.Remove("BackColor");
//				properties.Remove("BackgroundImage");
//				properties.Remove("CausesValidation");
//				properties.Remove("Font");
//				properties.Remove("ForeColor");
//				properties.Remove("ImeMode");
//				properties.Remove("RightToLeft");
//				properties.Remove("TabIndex");
//				properties.Remove("TabStop");
//				properties.Remove("Text");

                base.PostFilterProperties(properties);
            }
        }
        #endregion
    }
}
