﻿using LayoutEditor.Forms.ToolWindows;
using LayoutEditor.Utility;
using LECore.Manipulator;
using LECore.Structures;
using LECore.Structures.Core;
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using static LayoutEditor.src.Controls.BindAnimationLabel;

namespace LayoutEditor.src.Controls
{
    /// <summary>
    /// プロパティウインドウ上のアニメーションマークを表現するUIです。
    /// </summary>
    public partial class AnimationMark : Panel
    {
        #region フィールド

        private Pen AnimationMarkOutlinePen = new Pen(Color.LightSlateGray);
        private Brush AnimationMarkNoticeMarkBrush = new SolidBrush(Color.FromArgb(255, 90, 90, 255));

        private Dictionary<string, ToolStripItem> _toolStripItemList = new Dictionary<string, ToolStripItem>();
        private List<IAnmAttribute> _target = new List<IAnmAttribute>();
        private AnmCurveManipulator _anmCurveMnp = new AnmCurveManipulator();
        static private Cursor _cursor = null;

        private System.Windows.Forms.ContextMenuStrip contextMenuStrip1;

        public Action<ViewManagerMessage> SendMessageToViewManager;

        /// <summary>
        /// 必要なデザイナー変数です。
        /// </summary>
        private System.ComponentModel.IContainer components = null;

        #endregion フィールド

        #region プロパティ

        /// <summary>
        /// 操作対象アトリビュート。
        /// </summary>
        public IAnmAttribute[] Target
        {
            get { return _target.ToArray(); }
        }

        /// <summary>
        /// コンテキストメニュー。
        /// </summary>
        public ContextMenuStrip ContextMenuStripInst
        {
            get { return contextMenuStrip1; }
        }

        /// <summary>
        /// マウスオーバー時のカーソル。
        /// </summary>
        static public Cursor MouseOverCursor
        {
            get
            {
                // マウスホバー時のカーソルを設定
                if (_cursor == null)
                {
                    _cursor = CursorMgr.CreateCursor(
                    Properties.Resources.Icon_ContextCursor_01,
                    Cursors.Default.HotSpot.X, Cursors.Default.HotSpot.Y);
                }

                return _cursor;
            }
        }

        /// <summary>
        /// 通知の表示・非表示。
        /// </summary>
        public bool Notice
        {
            get
            {
                return (this.BackColor == Color.Pink);
            }
        }

        #endregion

        /// <summary>
        /// 使用中のリソースをすべてクリーンアップします。
        /// </summary>
        /// <param name="disposing">マネージ リソースを破棄する場合は true を指定し、その他の場合は false を指定します。</param>
        protected override void Dispose(bool disposing)
        {
            if (disposing && (components != null))
            {
                components.Dispose();
            }
            base.Dispose(disposing);
        }

        #region コンポーネント デザイナーで生成されたコード

        /// <summary>
        /// デザイナー サポートに必要なメソッドです。このメソッドの内容を
        /// コード エディターで変更しないでください。
        /// </summary>
        private void InitializeComponent()
        {
            this.components = new System.ComponentModel.Container();
            this.contextMenuStrip1 = new System.Windows.Forms.ContextMenuStrip(this.components);
            this.SuspendLayout();
            //
            // contextMenuStrip1
            //
            this.contextMenuStrip1.Name = "contextMenuStrip1";
            this.contextMenuStrip1.Size = new System.Drawing.Size(61, 4);
            this.contextMenuStrip1.Opening += new System.ComponentModel.CancelEventHandler(this.Event_ContextMenuOpening);
            //
            // AnimationMark
            //
            this.ContextMenuStrip = this.contextMenuStrip1;
            this.ResumeLayout(false);

        }

        #endregion

        #region オーバーライド

        /// <summary>
        /// OnPaint
        /// </summary>
        protected override void OnPaint(PaintEventArgs e)
        {
            base.OnPaint(e);

            using (Graphics g = this.CreateGraphics())
            {
                // 枠線を描く
                int r = this.ClientRectangle.Right - 1;
                int b = this.ClientRectangle.Bottom - 1;
                g.DrawRectangle(AnimationMarkOutlinePen, new Rectangle(0,0,r,b));

                // 通知マークを描く
                if (this.Notice)
                {
                    PointF[] points = {
                    new Point(this.ClientRectangle.Right - 5, 0),
                    new Point(this.ClientRectangle.Right, 0),
                    new Point(this.ClientRectangle.Right, 6)
                };
                    g.FillPolygon(AnimationMarkNoticeMarkBrush, points, System.Drawing.Drawing2D.FillMode.Winding);
                }
            }
        }

        protected override void OnMouseEnter(EventArgs e)
        {
            base.OnMouseEnter(e);

            this.Cursor = _cursor;
        }

        protected override void OnMouseHover(EventArgs e)
        {
            base.OnMouseHover(e);

            if (this.Notice)
            {
                this.contextMenuStrip1.Show(System.Windows.Forms.Cursor.Position);
            }
        }

        protected override void OnMouseLeave(EventArgs e)
        {
            base.OnMouseLeave(e);

            this.Cursor = System.Windows.Forms.Cursors.Default;
        }

        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public AnimationMark()
        {
            InitializeComponent();

            Initialize_();
        }

        private void Initialize_()
        {
            // デフォルトで使用するコンテキストメニューを登録します
            AnimationMarkContextMenuBuilder.BuildBasicContexMenu(this);
        }

        /// <summary>
        /// コンテキストメニューを登録します
        /// </summary>
        public void RegistContextMenu(string name, Image image, EventHandler handler)
        {
            ToolStripItem item = contextMenuStrip1.Items.Add(name, image, handler);

            if (item != null && name != "-")
            {
                _toolStripItemList.Add(name, item);
            }
        }

        /// <summary>
        /// 操作対象のアトリビュートを設定します
        /// </summary>
        public void SetTargetAttribute(IAnmAttribute attr)
        {
            _target.Clear();
            _target.AddRange(new IAnmAttribute[1] { attr });
        }
        public void SetTargetAttribute(IAnmAttribute[] attr)
        {
            _target.Clear();
            _target.AddRange(attr);
        }
        public void AddTargetAttribute(IAnmAttribute attr)
        {
            _target.Add(attr);
        }

        /// <summary>
        /// 操作対象のアトリビュートをクリアします
        /// </summary>
        public void ClearTargetAttribute()
        {
            _target.Clear();
        }

        /// <summary>
        /// アニメーションマークの色を更新します
        /// </summary>
        public void UpdateMarkColor()
        {
            // マーク色を更新します
            ISubScene subScene = LECore.LayoutEditorCore.Scene.CurrentISubScene;
            Color c;
            if (subScene != null)
            {
                List<Color> colors = new List<Color>();
                foreach (IAnmAttribute attr in _target)
                {
                    colors.Add(AnimationMarkColorConverter.GetColor(attr, subScene.IsAnimEditSeparateMode()));
                }
                c = AnimationMarkColorConverter.SelectColor(colors.ToArray());
            }
            else
            {
                // サブシーンがnullの場合はデフォルト色を表示しておく
                c = AnimationMarkColorConverter.DefaultColor;
            }

            if (this.BackColor != c)
            {
                this.BackColor = c;
            }
        }

        /// <summary>
        /// コンテキストメニューを更新します
        /// </summary>
        public void UpdateContextMenu()
        {

        }

        /// <summary>
        /// コンテキストメニューアイテムを取得します
        /// </summary>
        public ToolStripItem GetToolStripItem(string key)
        {
            if (_toolStripItemList.ContainsKey(key))
            {
                return _toolStripItemList[key];
            }

            return null;
        }

        #region イベント

        /// <summary>
        /// コンテキストメニューが開いたときに呼び出されます
        /// </summary>
        private void Event_ContextMenuOpening(object sender, CancelEventArgs e)
        {
            // エディットボックスの編集を確定するためにフォーカスを取る
            this.Focus();
        }

        /// <summary>
        /// キーを追加します
        /// </summary>
        public void Event_AddKey(object sender, EventArgs e)
        {
            LECore.LayoutEditorCore.Scene.CurrentISubScene.BeginMassiveModify();

            foreach (IAnmAttribute target in _target)
            {
                if (target.NumSubAttribute == 0)
                {
                    MakeKeyFrame_(
                        target.ICurrentAnimationCurve,
                        GlobalTime.Inst.Time,
                        target.Value);

                    continue;
                }

                for (int i = 0; i < target.NumSubAttribute; i++)
                {
                    IAnmAttribute subAttr = target.FindSubAttributeByIdx(i);

                    MakeKeyFrame_(
                        subAttr.ICurrentAnimationCurve,
                        GlobalTime.Inst.Time,
                        subAttr.Value);
                }
            }

            LECore.LayoutEditorCore.Scene.CurrentISubScene.EndMassiveModify();
        }

        private void MakeKeyFrame_(IAnmCurve curve, int time, object val)
        {
            if (val is float)
            {
                // FloatColor対応
                val = (float)val * curve.ViewScale;
            }

            _anmCurveMnp.BindTarget(curve);
            _anmCurveMnp.MakeKeyFrame(time, val);
        }

        /// <summary>
        /// カーブエディタを開きます
        /// </summary>
        public void Event_OpenCurveEditor(object sender, EventArgs e)
        {
            if (SendMessageToViewManager != null)
            {
                var arg = new ViewManagerMessage(ViewManagerMessageKind.ShowCurveEditor, _target.ToArray());
                SendMessageToViewManager(arg);
            }
            else
            {
                Debug.Assert(false);
            }
        }

        #endregion
    }

    /// <summary>
    /// アニメーションマークのコンテキストメニューを生成します。
    /// </summary>
    public class AnimationMarkContextMenuBuilder
    {
        /// <summary>
        /// アニメーションマークに基本コンテキストメニューを追加します。
        /// </summary>
        public static void BuildBasicContexMenu(AnimationMark mark)
        {
            mark.RegistContextMenu(StringResMgr.Get("ANIMCONTEXTMENU_ADD_KEY"), null, mark.Event_AddKey);
            mark.RegistContextMenu("-", null, null);
            mark.RegistContextMenu(StringResMgr.Get("ANIMCONTEXTMENU_SHOW_CURVEEDITOR"), null, mark.Event_OpenCurveEditor);
        }

        /// <summary>
        /// ラベルに複数同時操作用のコンテキストメニューを追加します。
        /// </summary>
        public static void BuildMultiControlMenu(BindAnimationLabel label, AnimationMark[] marks)
        {
            label.RegistContextMenu(StringResMgr.Get("ANIMCONTEXTMENU_ADD_ALL_KEY"), null,
                (s, e) => BindAnimationLabel.Event_AddKeyAll(s, new BindAnimationLabelArgs(marks)));
            label.RegistContextMenu("-", null, null);
            label.RegistContextMenu(StringResMgr.Get("ANIMCONTEXTMENU_SHOW_CURVEEDITOR"), null,
                (s, e) => BindAnimationLabel.Event_ShowCurveEditorAll(s, new BindAnimationLabelArgs(marks)));
        }
    }

    /// <summary>
    /// アニメーションマークが利用可能なコントロールに対し、
    /// アニメーションマークを使うかどうかの設定を行ないます
    /// </summary>
    public interface IAnimationMarkUsable
    {
        /// <summary>
        /// アニメーションマークを利用するか
        /// </summary>
        bool UseAnimationMark { get; set; }

        /// <summary>
        /// アニメーションマークを更新します
        /// </summary>
        void UpdateAnimationMark();
    }
}
